/*
 * FreeBSD sysinstall internationalization support
 *
 * Copyright (c) 1996-2000
 *   Tatsumi Hosokawa <hosokawa@FreeBSD.org>   All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer,
 *    verbatim and that no modifications are made prior to this
 *    point in the file.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY TATSUMI HOSOKAWA ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL JORDAN HUBBARD OR HIS PETS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, LIFE OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/*
 *                         -- GUIDELINE --
 *
 * Public interface of this file is "catalog()".  It translates messages
 * into another language specified by variable "language" defined in
 * sysinstall.h.  The translation is based on message files.  Each
 * message have its own ID number expressed in five decimal
 * characters.  Message file have to be placed on /stand/catalog
 * and its filename is "messages.(LOCALE)".  Currently messages.ja_JP
 * (Japanese, EUC code) and messages.zh_TW (Traditional Chinese, Big5
 * code) are supported.  If you want to support another languages,
 * please read these message files.
 *
 * Messages translated by "catalog()" have to satisfy the following
 * format. If messages don't satisfy following format, "catalog()" simply
 * be echoed back to its callers.
 *
 * Format:
 * "$$$$xxxxx English message" (xxxxx: ID number)
 * (don't forget to put a space character between ID and English
 *  message) 
 *
 * Currently, first two digits are allocated to each source file, and
 * last three digits are allocated for each message.
 *
 * If variable "lanuguage" is set to LANG_ENGLISH, or corresponding
 * message does not exist on message file, it simply returns the
 * English message follows ID string.
 * 
 * If you want to extend the functionality of sysinstall, you don't
 * have to worry about L10N support.  If you put new messages in code, 
 * it simply ignored by "catalog()".  You don't have to put "$$$$xxxxx" new 
 * ID numbers.  Messages without "$$$$xxxxx" simply echoed back by
 * "catalog()", so English messages are displayed after all.
 *
 * Again, DON'T WORRY ABOUT L10N.  It affects you nothing, even if you
 * can't translated messages into any other languages.  We'll do that.
 *
 * Notes:
 * Currently, ad-hoc implementation of support of multibyte characters 
 * for libncurses and libdialog is not smart, and it will affect the
 * iso8859 European charset support.  It should be replaced with
 * another implementation.
 *
 * Tatsumi Hosokawa <hosokawa@FreeBSD.org>, Yokohama, Japan.
 */

#include "sysinstall.h"
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <ctype.h>

#undef	HASH_DEBUG

/* sysinstall messages are stored to these files */
#define	CATALOG_MSG_FILE	"/stand/catalog/%s/messages.cat"

/* definition of hash table to which the messages are stored */

#define	N_CATALOG_MSG_HASH	256

typedef struct catalog_msg_hash {
    struct catalog_msg_hash *next;
    char	*ptr;
} catalog_msg_hash;

typedef struct catalog_msg_data {
    char	*msg;
    size_t	size;
    catalog_msg_hash *hash[N_CATALOG_MSG_HASH];
} catalog_msg_data;

/* this structure contains messages themselves and hash tables for messages */
struct catalog_msg_data catalog_msgs[LANG_NLANG];

/* number of ring buffers catalog() uses */
#define	N_CATALOG_RBUF	256


/* calculate hash table */
static int
hash(char *msg)
{
    int	d;
    char buf[8];
    
    if (strncmp(msg, "$$$$", 4)) {
	return -1;
    }
    strncpy(buf, msg + 4, 5);
    buf[5] = 0;
    sscanf(buf, "%d", &d);

    return d % N_CATALOG_MSG_HASH;
}

/* register messages into hash table */
static void
register_msg(char *msg, catalog_msg_data *data)
{
    int hash_val;
    static catalog_msg_hash *p, *q;

    p = safe_malloc(sizeof(*p));
    hash_val = hash(msg);
#ifdef	HASH_DEBUG
    {
	char buf[10];
	
	strncpy(buf, msg, 9);
	buf[9] = 0;
	msgDebug("Registering: %s (%d)\n", buf, hash_val);
    }
#endif	/* HASH_DEBUG */
    q = data->hash[hash_val];
    data->hash[hash_val] = p;
    p->next = q;
    p->ptr = msg;
}

/* load messages from file and store them into hash table */
static int 
load_msg(char *catalog, catalog_msg_data *data) 
{
    int		i, fd;
    char	msgfile[64];
    struct stat	sb;
 
    bzero(data, sizeof(*data));
    
    sprintf(msgfile, CATALOG_MSG_FILE, catalog);
    
    if ((fd = open(msgfile, O_RDONLY)) < 0) {
	msgDebug("Can't open catalog message file %s\n", msgfile);
	return -1;
    }

    if (fstat(fd, &sb) < 0) {
	msgDebug("%s: fstat failed", msgfile);
	close(fd);
	return -1;
    }

    data->size = sb.st_size;

    data->msg = safe_malloc(data->size + 32);
    if (read(fd, data->msg, data->size) < 0) {
	msgDebug("Can't read catalog message file %s\n", msgfile);
	safe_free(data->msg);
	data->msg = 0;
	close(fd);
	return -1;
    }

    for (i = 0; i < data->size; i++) {
	if (strncmp(data->msg + i, "$$$$", 4) == 0 &&
	    isdigit(data->msg[i + 4]) && isdigit(data->msg[i + 5]) && 
	    isdigit(data->msg[i + 6]) && isdigit(data->msg[i + 7]) && 
	    isdigit(data->msg[i + 8])) {
	    register_msg(data->msg + i, data);
	}
    }
   
    close(fd);
    return 0;
}

static char	*rbuf[N_CATALOG_RBUF] = {0};

static char *
catalog_alloc(int len)
{
    static int	idx = 0;
    char *r;

    if (rbuf[idx]) {
	safe_free(rbuf[idx]);
	rbuf[idx] = 0;
    }
    r = rbuf[idx] = safe_malloc(len);
    idx++;
    if (idx >= N_CATALOG_RBUF) {
	idx = 0;
    }

    return r;
}

/* private routine called from catalog() function:
   get coressponding messages from hash table for current language */

static char *
private_catalog(char *str, catalog_msg_data *data)
{
    int		hash_val;
    static catalog_msg_hash *p;
    static char	buffer[1024];

    if (!str) {
	return str;
    }

    if (strncmp(str, "$$$$", 4) || str[9] != ' ') {
	return str;
    }

    if ((hash_val = hash(str)) < 0) {
	return str + 10;
    }
    
    for (p = data->hash[hash_val]; p; p = p->next) {
	if (strncmp(str + 4, p->ptr + 4, 5) == 0) {
	    char *q, *r;
	    int skip;
	    
	    skip = 1;
#ifdef	HASH_DEBUG
	    msgDebug("Found %s (%d).\n", str, hash_val);
#endif	/* HASH_DEBUG */
	    r = buffer;

	    for (q = p->ptr + 10; !(q[0] == '\\' && q[1] == '\\'); q++) {
		if (skip && !isspace(*q)) {
		    skip = 0;
		}
		if (!skip) {
		    *r = *q;
		    r++;
		}
	    }
	    *r = 0;

	    return buffer;
	}
    }
#ifdef	HASH_DEBUG
    msgDebug("Not Found %s (%d).\n", str, hash_val);
#endif	/* HASH_DEBUG */
    return str + 10;
}

/* 'public' interfaces for catalog message translation */

char *
catalog(char *msg)
{
    char *translated;

    if (!msg) {
	return msg;
    }
#ifdef MULTILINGUAL
    if (language == LANG_ENGLISH) {
	if (strncmp(msg, "$$$$", 4) == 0 && msg[9] == ' ') {
	    return msg + 10;
	}
    }
    translated = private_catalog(msg, &catalog_msgs[language]);
    if (!translated || !strlen(translated)) {
	translated = "--";	/* translation failed */
    }
    else {
	char *tmp;
	
	tmp = catalog_alloc(strlen(translated) + 1);
	strcpy(tmp, translated);
	translated = tmp;
    }
    return translated;
#else
    if (strncmp(msg, "$$$$", 4) == 0 && msg[9] == ' ') {
      return msg + 10;
    }
    return msg;
#endif
}


int
cataloglen(char *str)
{
    return strlen(catalog(str));
}

/* create a string that repeat 'c' strlen(catalog(str)) times */

char *
catalogrep(char *str, int c)
{
    int	 i, l;
    static char	buf[128];
    
    buf[0] = 0;
    l = cataloglen(str);

    if (l < sizeof(buf)) {
	for (i = 0; i < l; i++) {
	    buf[i] = (char)c;
	}
	buf[i] = 0;
    }
    
    return buf;
}

/* initialize localization support code */
void 
init_catalog(void)
{
    /* Initialize ring buffer */
    bzero(rbuf, sizeof(rbuf));

#ifdef MULTILINGUAL
    /* Load message files */
    switch (language) {
    case LANG_JAPANESE:
	load_msg("ja", &catalog_msgs[LANG_JAPANESE]);
	break;
    case LANG_KOREAN:
	load_msg("ko", &catalog_msgs[LANG_KOREAN]);
        break;
    case LANG_PORTUGUESE_BR:
	load_msg("pt_BR", &catalog_msgs[LANG_PORTUGUESE_BR]);
        break;
    case LANG_RUSSIAN:
	load_msg("ru", &catalog_msgs[LANG_RUSSIAN]);
        break;
    case LANG_TRAD_CHINESE:
	load_msg("zh_TW", &catalog_msgs[LANG_TRAD_CHINESE]);
        break;
    default:
	/* empty */
        break;
    }
#endif

    /* replace static variables */
    menus_catalog();
    options_catalog();
    anonFTP_catalog();
    index_catalog();
    tcpip_catalog();
    user_catalog();
    tzsetup_catalog();
    pccard_catalog();
}
