/*
 * The new sysinstall program.
 *
 * This is probably the last attempt in the `sysinstall' line, the next
 * generation being slated for what's essentially a complete rewrite.
 *
 * $FreeBSD: src/release/sysinstall/main.c,v 1.57.2.2 2000/07/04 12:33:56 nyan Exp $
 *
 * Copyright (c) 1995
 *	Jordan Hubbard.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer,
 *    verbatim and that no modifications are made prior to this
 *    point in the file.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY JORDAN HUBBARD ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL JORDAN HUBBARD OR HIS PETS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, LIFE OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include "sysinstall.h"
#include <sys/signal.h>
#include <sys/fcntl.h>

static void
screech(int sig)
{
    msgDebug("\007Signal %d caught!  That's bad!\n", sig);
    longjmp(BailOut, sig);
}

#ifdef MULTILINGUAL
int use_kon = 0;
int kon_term = 0;
#endif

int
main(int argc, char *argv[])
{
    int choice, scroll, curr, max, status;
    int lang_input;
    char *lang_var = 0;
    char *lang_env = 0;

    /* Catch fatal signals and complain about them if running as init */
    if (getpid() == 1) {
	signal(SIGBUS, screech);
	signal(SIGSEGV, screech);
    }
    signal(SIGPIPE, SIG_IGN);

    /* We don't work too well when running as non-root anymore */
    if (geteuid() != 0) {
	fprintf(stderr, "Error: This utility should only be run as root.\n");
	return 1;
    }

#ifdef MULTILINGUAL
    lang_env = getenv(SYSINSTALL_LANG_ENV);
    if (lang_env) {
        if (strcmp(lang_env, "ja") == 0) {
	    language = LANG_JAPANESE;
	}
	else if (strcmp(lang_env, "ko") == 0) {
	    language = LANG_KOREAN;
	}
	else if (strcmp(lang_env, "ru") == 0) {
	    language = LANG_RUSSIAN;
	}
	else if (strcmp(lang_env, "zh_TW") == 0) {
	    language = LANG_TRAD_CHINESE;
	}
    }
    else {
        language = LANG_ENGLISH;
    }
    if (USE_KON_LANG(language)) {
        use_kon = 1;
    }
#endif /* MULTILINGUAL */

#ifdef PC98
    {
	/* XXX */
	char *p = getenv("TERM");
	if (p && strcmp(p, "cons25") == 0)
	    putenv("TERM=cons25w");
    }
#endif

    /* Set up whatever things need setting up */
    systemInitialize(argc, argv);

#ifdef MULTILINGUAL
    if (!lang_env) {
        dmenuOpenSimple(&MenuLocalization, FALSE);
	lang_var = variable_get("_language");
	if (lang_var) {
#ifdef JAPANESE
	    if (strcmp(lang_var, "japanese") == 0) {
	        setenv(SYSINSTALL_LANG_ENV, "ja", 1);
		execv(argv[0], argv);
		/*NOTREACHED*/
	    }
#endif
#ifdef KOREAN
	    if (strcmp(lang_var, "korean") == 0) {
	        language = LANG_KOREAN;
	        setenv(SYSINSTALL_LANG_ENV, "ko", 1);
		execv(argv[0], argv);
		/*NOTREACHED*/
	    }
#endif
#ifdef RUSSIAN
	    if (strcmp(lang_var, "russian") == 0) {
	        language = LANG_RUSSIAN;
	        setenv(SYSINSTALL_LANG_ENV, "ru", 1);
		execv(argv[0], argv);
		/*NOTREACHED*/
	    }
#endif
#ifdef TRAD_CHINESE
	    if (strcmp(lang_var, "trad_chinese") == 0) {
	        language = LANG_TRAD_CHINESE;
	        setenv(SYSINSTALL_LANG_ENV, "zh_TW", 1);
		execv(argv[0], argv);
		/*NOTREACHED*/
	    }
#endif
	}
    }

#ifdef USE_KON
    /* this if () condition is not complete, but I think it's */
    /* acceptable for almost all cases */
    if (use_kon && kon_term) {
	kon2_main();
    }
#endif /* USE_KON */
    if (language == LANG_RUSSIAN) {
	vsystem("vidcontrol -f 8x16 /stand/fonts/koi8-r-8x16.fnt");
    }
#endif /* MULTILINGUAL */

    /* Initialize the localized messages */
    init_catalog();

    /* Set default flag and variable values */
    installVarDefaults(NULL);
    /* only when multi-user is it reasonable to do this here */
    if (!RunningAsInit)
	installEnvironment();

    if (argc > 1 && !strcmp(argv[1], "-fake")) {
	variable_set2(VAR_DEBUG, "YES", 0);
	Fake = TRUE;
	msgConfirm("$$$$10000 I'll be just faking it from here on out, OK?");
    }

    /* Try to preserve our scroll-back buffer */
    if (OnVTY) {
	for (curr = 0; curr < 25; curr++)
	    putchar('\n');
    }
    /* Move stderr aside */
    if (DebugFD)
	dup2(DebugFD, 2);

    /* Initialize PC-card */
    pccardInitialize();

    /* Initialize USB */
    usbInitialize();

    /* Probe for all relevant devices on the system */
    deviceGetAll();

    /* First, see if we have any arguments to process (and argv[0] counts if it's not "sysinstall") */
    if (!RunningAsInit) {
	int i, start_arg;

	if (!strstr(argv[0], "sysinstall"))
	    start_arg = 0;
	else if (Fake)
	    start_arg = 2;
	else
	    start_arg = 1;
	for (i = start_arg; i < argc; i++) {
	    if (DITEM_STATUS(dispatchCommand(argv[i])) != DITEM_SUCCESS)
		systemShutdown(1);
	}
	if (argc > start_arg)
	    systemShutdown(0);
    }
    else
	dispatch_load_file_int(TRUE);

    status = setjmp(BailOut);
    if (status) {
	msgConfirm("$$$$10003 A signal %d was caught - I'm saving what I can and shutting\n"
		   "If you can reproduce the problem, please turn Debug on in\n"
		   "the Options menu for the extra information it provides in\n"
		   "debugging problems like this.", status);
	systemShutdown(status);
    }

    /* Begin user dialog at outer menu */
    dialog_clear();
    while (1) {
	choice = scroll = curr = max = 0;
	dmenuOpen(&MenuInitial, &choice, &scroll, &curr, &max, TRUE);
	if (getpid() != 1
#ifdef __alpha__
	    || !msgYesNo("$$$$10002 Are you sure you wish to exit?  The system will halt.")
#else
	    || !msgYesNo("$$$$10001 Are you sure you wish to exit?  The system will reboot\n"
		         "(be sure to remove any floppies/CDROMs from the drives).")
#endif
	    )
	    break;
    }

    /* Say goodnight, Gracie */
    systemShutdown(0);

    return 0; /* We should never get here */
}
