/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core.policies;

import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.Host;
import com.datastax.driver.core.HostDistance;
import com.datastax.driver.core.LatencyTracker;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.policies.ChainableLoadBalancingPolicy;
import com.datastax.driver.core.policies.CloseableLoadBalancingPolicy;
import com.datastax.driver.core.policies.LoadBalancingPolicy;
import com.google.common.collect.AbstractIterator;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.util.ArrayDeque;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LatencyAwarePolicy
implements ChainableLoadBalancingPolicy,
CloseableLoadBalancingPolicy {
    private static final Logger logger = LoggerFactory.getLogger(LatencyAwarePolicy.class);
    private final LoadBalancingPolicy childPolicy;
    private final Tracker latencyTracker;
    private final ScheduledExecutorService updaterService = Executors.newSingleThreadScheduledExecutor(LatencyAwarePolicy.threadFactory("LatencyAwarePolicy updater"));
    private final double exclusionThreshold;
    private final long scale;
    private final long retryPeriod;
    private final long minMeasure;

    private LatencyAwarePolicy(LoadBalancingPolicy childPolicy, double exclusionThreshold, long scale, long retryPeriod, long updateRate, int minMeasure) {
        this.childPolicy = childPolicy;
        this.retryPeriod = retryPeriod;
        this.scale = scale;
        this.latencyTracker = new Tracker();
        this.exclusionThreshold = exclusionThreshold;
        this.minMeasure = minMeasure;
        this.updaterService.scheduleAtFixedRate(new Updater(), updateRate, updateRate, TimeUnit.NANOSECONDS);
    }

    @Override
    public LoadBalancingPolicy getChildPolicy() {
        return this.childPolicy;
    }

    public static Builder builder(LoadBalancingPolicy childPolicy) {
        return new Builder(childPolicy);
    }

    private static double inMS(long nanos) {
        return (double)nanos / 1000000.0;
    }

    private static double inMS(double nanos) {
        return nanos / 1000000.0;
    }

    private static ThreadFactory threadFactory(String nameFormat) {
        return new ThreadFactoryBuilder().setNameFormat(nameFormat).build();
    }

    @Override
    public void init(Cluster cluster, Collection<Host> hosts) {
        this.childPolicy.init(cluster, hosts);
        cluster.register(this.latencyTracker);
    }

    @Override
    public HostDistance distance(Host host) {
        return this.childPolicy.distance(host);
    }

    @Override
    public Iterator<Host> newQueryPlan(String loggedKeyspace, Statement statement) {
        final Iterator<Host> childIter = this.childPolicy.newQueryPlan(loggedKeyspace, statement);
        return new AbstractIterator<Host>(){
            private Queue<Host> skipped;

            protected Host computeNext() {
                long min = LatencyAwarePolicy.this.latencyTracker.getMinAverage();
                long now = System.nanoTime();
                while (childIter.hasNext()) {
                    Host host = (Host)childIter.next();
                    TimestampedAverage latency = LatencyAwarePolicy.this.latencyTracker.latencyOf(host);
                    if (min < 0L || latency == null || latency.nbMeasure < LatencyAwarePolicy.this.minMeasure || now - latency.timestamp > LatencyAwarePolicy.this.retryPeriod) {
                        return host;
                    }
                    if (latency.average <= (long)(LatencyAwarePolicy.this.exclusionThreshold * (double)min)) {
                        return host;
                    }
                    if (this.skipped == null) {
                        this.skipped = new ArrayDeque<Host>();
                    }
                    this.skipped.offer(host);
                }
                if (this.skipped != null && !this.skipped.isEmpty()) {
                    return this.skipped.poll();
                }
                return (Host)this.endOfData();
            }
        };
    }

    public Snapshot getScoresSnapshot() {
        Map<Host, TimestampedAverage> currentLatencies = this.latencyTracker.currentLatencies();
        ImmutableMap.Builder builder = ImmutableMap.builder();
        long now = System.nanoTime();
        for (Map.Entry<Host, TimestampedAverage> entry : currentLatencies.entrySet()) {
            Host host = entry.getKey();
            TimestampedAverage latency = entry.getValue();
            Snapshot.Stats stats = new Snapshot.Stats(now - latency.timestamp, latency.average, latency.nbMeasure);
            builder.put((Object)host, (Object)stats);
        }
        return new Snapshot((Map)builder.build());
    }

    @Override
    public void onUp(Host host) {
        this.childPolicy.onUp(host);
    }

    @Override
    public void onSuspected(Host host) {
        this.childPolicy.onSuspected(host);
    }

    @Override
    public void onDown(Host host) {
        this.childPolicy.onDown(host);
        this.latencyTracker.resetHost(host);
    }

    @Override
    public void onAdd(Host host) {
        this.childPolicy.onAdd(host);
    }

    @Override
    public void onRemove(Host host) {
        this.childPolicy.onRemove(host);
        this.latencyTracker.resetHost(host);
    }

    @Override
    public void close() {
        if (this.childPolicy instanceof CloseableLoadBalancingPolicy) {
            ((CloseableLoadBalancingPolicy)this.childPolicy).close();
        }
        this.updaterService.shutdown();
    }

    public static class Builder {
        private static final double DEFAULT_EXCLUSION_THRESHOLD = 2.0;
        private static final long DEFAULT_SCALE = TimeUnit.MILLISECONDS.toNanos(100L);
        private static final long DEFAULT_RETRY_PERIOD = TimeUnit.SECONDS.toNanos(10L);
        private static final long DEFAULT_UPDATE_RATE = TimeUnit.MILLISECONDS.toNanos(100L);
        private static final int DEFAULT_MIN_MEASURE = 50;
        private final LoadBalancingPolicy childPolicy;
        private double exclusionThreshold = 2.0;
        private long scale = DEFAULT_SCALE;
        private long retryPeriod = DEFAULT_RETRY_PERIOD;
        private long updateRate = DEFAULT_UPDATE_RATE;
        private int minMeasure = 50;

        public Builder(LoadBalancingPolicy childPolicy) {
            this.childPolicy = childPolicy;
        }

        public Builder withExclusionThreshold(double exclusionThreshold) {
            if (exclusionThreshold < 1.0) {
                throw new IllegalArgumentException("Invalid exclusion threshold, must be greater than 1.");
            }
            this.exclusionThreshold = exclusionThreshold;
            return this;
        }

        public Builder withScale(long scale, TimeUnit unit) {
            if (scale <= 0L) {
                throw new IllegalArgumentException("Invalid scale, must be strictly positive");
            }
            this.scale = unit.toNanos(scale);
            return this;
        }

        public Builder withRetryPeriod(long retryPeriod, TimeUnit unit) {
            if (retryPeriod < 0L) {
                throw new IllegalArgumentException("Invalid retry period, must be positive");
            }
            this.retryPeriod = unit.toNanos(retryPeriod);
            return this;
        }

        public Builder withUpdateRate(long updateRate, TimeUnit unit) {
            if (updateRate <= 0L) {
                throw new IllegalArgumentException("Invalid update rate value, must be strictly positive");
            }
            this.updateRate = unit.toNanos(updateRate);
            return this;
        }

        public Builder withMininumMeasurements(int minMeasure) {
            if (minMeasure < 0) {
                throw new IllegalArgumentException("Invalid minimum measurements value, must be positive");
            }
            this.minMeasure = minMeasure;
            return this;
        }

        public LatencyAwarePolicy build() {
            return new LatencyAwarePolicy(this.childPolicy, this.exclusionThreshold, this.scale, this.retryPeriod, this.updateRate, this.minMeasure);
        }
    }

    private static class HostLatencyTracker {
        private final long thresholdToAccount;
        private final double scale;
        private final AtomicReference<TimestampedAverage> current = new AtomicReference();

        HostLatencyTracker(long scale, long thresholdToAccount) {
            this.scale = scale;
            this.thresholdToAccount = thresholdToAccount;
        }

        public void add(long newLatencyNanos) {
            TimestampedAverage previous;
            TimestampedAverage next;
            while ((next = this.computeNextAverage(previous = this.current.get(), newLatencyNanos)) != null && !this.current.compareAndSet(previous, next)) {
            }
        }

        private TimestampedAverage computeNextAverage(TimestampedAverage previous, long newLatencyNanos) {
            long nbMeasure;
            long currentTimestamp = System.nanoTime();
            long l = nbMeasure = previous == null ? 1L : previous.nbMeasure + 1L;
            if (nbMeasure < this.thresholdToAccount) {
                return new TimestampedAverage(currentTimestamp, -1L, nbMeasure);
            }
            if (previous == null || previous.average < 0L) {
                return new TimestampedAverage(currentTimestamp, newLatencyNanos, nbMeasure);
            }
            long delay = currentTimestamp - previous.timestamp;
            if (delay <= 0L) {
                return null;
            }
            double scaledDelay = (double)delay / this.scale;
            double prevWeight = Math.log(scaledDelay + 1.0) / scaledDelay;
            long newAverage = (long)((1.0 - prevWeight) * (double)newLatencyNanos + prevWeight * (double)previous.average);
            return new TimestampedAverage(currentTimestamp, newAverage, nbMeasure);
        }

        public TimestampedAverage getCurrentAverage() {
            return this.current.get();
        }
    }

    private static class TimestampedAverage {
        private final long timestamp;
        private final long average;
        private final long nbMeasure;

        TimestampedAverage(long timestamp, long average, long nbMeasure) {
            this.timestamp = timestamp;
            this.average = average;
            this.nbMeasure = nbMeasure;
        }
    }

    private class Tracker
    implements LatencyTracker {
        private final ConcurrentMap<Host, HostLatencyTracker> latencies = new ConcurrentHashMap<Host, HostLatencyTracker>();
        private volatile long cachedMin = -1L;

        private Tracker() {
        }

        @Override
        public void update(Host host, long newLatencyNanos) {
            HostLatencyTracker old;
            HostLatencyTracker hostTracker = (HostLatencyTracker)this.latencies.get(host);
            if (hostTracker == null && (old = this.latencies.putIfAbsent(host, hostTracker = new HostLatencyTracker(LatencyAwarePolicy.this.scale, 30L * LatencyAwarePolicy.this.minMeasure / 100L))) != null) {
                hostTracker = old;
            }
            hostTracker.add(newLatencyNanos);
        }

        public void updateMin() {
            long newMin = Long.MAX_VALUE;
            long now = System.nanoTime();
            for (HostLatencyTracker tracker : this.latencies.values()) {
                TimestampedAverage latency = tracker.getCurrentAverage();
                if (latency == null || latency.average < 0L || latency.nbMeasure < LatencyAwarePolicy.this.minMeasure || now - latency.timestamp > LatencyAwarePolicy.this.retryPeriod) continue;
                newMin = Math.min(newMin, latency.average);
            }
            if (newMin != Long.MAX_VALUE) {
                this.cachedMin = newMin;
            }
        }

        public long getMinAverage() {
            return this.cachedMin;
        }

        public TimestampedAverage latencyOf(Host host) {
            HostLatencyTracker tracker = (HostLatencyTracker)this.latencies.get(host);
            return tracker == null ? null : tracker.getCurrentAverage();
        }

        public Map<Host, TimestampedAverage> currentLatencies() {
            HashMap<Host, TimestampedAverage> map = new HashMap<Host, TimestampedAverage>(this.latencies.size());
            for (Map.Entry entry : this.latencies.entrySet()) {
                map.put((Host)entry.getKey(), ((HostLatencyTracker)entry.getValue()).getCurrentAverage());
            }
            return map;
        }

        public void resetHost(Host host) {
            this.latencies.remove(host);
        }
    }

    public static class Snapshot {
        private final Map<Host, Stats> stats;

        private Snapshot(Map<Host, Stats> stats) {
            this.stats = stats;
        }

        public Map<Host, Stats> getAllStats() {
            return this.stats;
        }

        public Stats getStats(Host host) {
            return this.stats.get(host);
        }

        public static class Stats {
            private final long lastUpdatedSince;
            private final long average;
            private final long nbMeasurements;

            private Stats(long lastUpdatedSince, long average, long nbMeasurements) {
                this.lastUpdatedSince = lastUpdatedSince;
                this.average = average;
                this.nbMeasurements = nbMeasurements;
            }

            public long lastUpdatedSince() {
                return this.lastUpdatedSince;
            }

            public long getLatencyScore() {
                return this.average;
            }

            public long getMeasurementsCount() {
                return this.nbMeasurements;
            }
        }
    }

    private class Updater
    implements Runnable {
        private Set<Host> excludedAtLastTick = Collections.emptySet();

        private Updater() {
        }

        @Override
        public void run() {
            try {
                logger.trace("Updating LatencyAwarePolicy minimum");
                LatencyAwarePolicy.this.latencyTracker.updateMin();
                if (logger.isDebugEnabled()) {
                    HashSet<Host> excludedThisTick = new HashSet<Host>();
                    double currentMin = LatencyAwarePolicy.this.latencyTracker.getMinAverage();
                    for (Map.Entry<Host, Snapshot.Stats> entry : LatencyAwarePolicy.this.getScoresSnapshot().getAllStats().entrySet()) {
                        Host host = entry.getKey();
                        Snapshot.Stats stats = entry.getValue();
                        if (stats.getMeasurementsCount() < LatencyAwarePolicy.this.minMeasure) continue;
                        if (stats.lastUpdatedSince() > LatencyAwarePolicy.this.retryPeriod) {
                            if (!this.excludedAtLastTick.contains(host)) continue;
                            logger.debug(String.format("Previously avoided host %s has not be queried since %.3fms: will be reconsidered.", host, LatencyAwarePolicy.inMS(stats.lastUpdatedSince())));
                            continue;
                        }
                        if (stats.getLatencyScore() > (long)(LatencyAwarePolicy.this.exclusionThreshold * currentMin)) {
                            excludedThisTick.add(host);
                            if (this.excludedAtLastTick.contains(host)) continue;
                            logger.debug(String.format("Host %s has an average latency score of %.3fms, more than %f times more than the minimum %.3fms: will be avoided temporarily.", host, LatencyAwarePolicy.inMS(stats.getLatencyScore()), LatencyAwarePolicy.this.exclusionThreshold, LatencyAwarePolicy.inMS(currentMin)));
                            continue;
                        }
                        if (!this.excludedAtLastTick.contains(host)) continue;
                        logger.debug("Previously avoided host {} average latency has come back within accepted bounds: will be reconsidered.", (Object)host);
                    }
                    this.excludedAtLastTick = excludedThisTick;
                }
            }
            catch (RuntimeException e) {
                logger.error("Error while updating LatencyAwarePolicy minimum", (Throwable)e);
            }
        }
    }
}

