/*
    avicore
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#ifndef __WAVE_H__
#define __WAVE_H__


#include "common.h"


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


#define WAVE_FMT_UNKNOWN 0
#define WAVE_FMT_PCM     1


#define WFX_SIZE 18


#ifdef USE_GCC


typedef struct _WaveFormatEx
{
  guint16 format_tag __attribute__((packed));
  guint16 channels __attribute__((packed));
  guint32 samples_per_sec __attribute__((packed));
  guint32 average_bytes_per_sec __attribute__((packed));
  guint16 block_align __attribute__((packed));
  guint16 bits_per_sample __attribute__((packed));
  guint16 size __attribute__((packed));
} WaveFormatEx;


#else /* not USE_GCC */


typedef struct _WaveFormatEx { guint8 dummy[WFX_SIZE]; } WaveFormatEx;


#endif /* not USE_GCC */


typedef struct _WaveConv WaveConv;


/******************************************************************************
*                                                                             *
* ja:Wave構造体マクロ                                                         *
*                                                                             *
******************************************************************************/
#define wfx_get_format_tag(wfx) GUINT16_FROM_LE(*(guint16 *)(wfx))
#define wfx_get_channels(wfx) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(wfx)+2))
#define wfx_get_samples_per_sec(wfx)                                        \
                            GUINT32_FROM_LE(*(guint32 *)((guint8 *)(wfx)+4))
#define wfx_get_average_bytes_per_sec(wfx)                                  \
                            GUINT32_FROM_LE(*(guint32 *)((guint8 *)(wfx)+8))
#define wfx_get_block_align(wfx)                                            \
                            GUINT16_FROM_LE(*(guint16 *)((guint8 *)(wfx)+12))
#define wfx_get_bits_per_sample(wfx)                                        \
                            GUINT16_FROM_LE(*(guint16 *)((guint8 *)(wfx)+14))
#define wfx_get_size(wfx) GUINT16_FROM_LE(*(guint16 *)((guint8 *)(wfx)+16))
#define wfx_set_format_tag(wfx,format_tag)                                  \
                                (*(guint16 *)(wfx)=GUINT16_TO_LE(format_tag))
#define wfx_set_channels(wfx,channels)                                      \
                    (*(guint16 *)((guint8 *)(wfx)+2)=GUINT16_TO_LE(channels))
#define wfx_set_samples_per_sec(wfx,samples_per_sec)                        \
            (*(guint32 *)((guint8 *)(wfx)+4)=GUINT32_TO_LE(samples_per_sec))
#define wfx_set_average_bytes_per_sec(wfx,average_bytes_per_sec)            \
        (*(guint32 *)((guint8 *)(wfx)+8)=GUINT32_TO_LE(average_bytes_per_sec))
#define wfx_set_block_align(wfx,block_align)                                \
                (*(guint16 *)((guint8 *)(wfx)+12)=GUINT16_TO_LE(block_align))
#define wfx_set_bits_per_sample(wfx,bits_per_sample)                        \
            (*(guint16 *)((guint8 *)(wfx)+14)=GUINT16_TO_LE(bits_per_sample))
#define wfx_set_size(wfx,size)                                              \
                        (*(guint16 *)((guint8 *)(wfx)+16)=GUINT16_TO_LE(size))


/******************************************************************************
*                                                                             *
* ja:Waveマクロ                                                               *
*                                                                             *
******************************************************************************/
/*  ja:WaveFormatEx構造体とそれに続くデータのバイト数を求める
    wfx,WaveFormatEx構造体へのポインタ
    RET,バイト数                                                            */
#define wf_header_bytes(wfx) (wfx_get_size(wfx)+WFX_SIZE)


/*  ja:WaveFormatEx構造体を比較する
    wfx0,WaveFormatEx構造体へのポインタ
    wfx1,WaveFormatEx構造体へのポインタ
     RET,TRUE:同じ,FALSE:異なる                                             */
#define wf_header_cmp(wfx0,wfx1)                                            \
                            (wf_header_bytes(wfx0)==wf_header_bytes(wfx1)   \
                                &&g_memcmp(wfx0,wfx1,wf_header_bytes(wfx0))==0)


/******************************************************************************
*                                                                             *
* ja:PCM生成関数                                                              *
*                                                                             *
******************************************************************************/
/*  ja:無音PCMを作る
        wfx,WaveFormatEx構造体へのポインタ
    samples,出力データのサンプル数
        RET,無音のPCM                                                       */
gpointer
wave_fill_silent_pcm (WaveFormatEx *wfx,
                      const gsize   samples);


/******************************************************************************
*                                                                             *
* ja:PCM加工関数                                                              *
*                                                                             *
******************************************************************************/
/*  ja:PCMの変換を開始する
    wfx0,元のWaveFormatEx構造体へのポインタ
    wfx1,新しいWaveFormatEx構造体へのポインタ
     RET,変換ハンドル,NULL:変換不可能                                       */
WaveConv *
wave_convert_begin (WaveFormatEx *wfx0,
                    WaveFormatEx *wfx1);


/*  ja:PCMの変換をする
            wcv,変換ハンドル
         in_buf,入力データ
     in_samples,入力データのサンプル数
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
wave_convert (WaveConv      *wcv,
              gconstpointer  in_buf,
              const gsize    in_samples,
              gpointer      *out_buf,
              gsize         *out_samples);


/*  ja:PCMの変換を終了する
      wave_conv,変換ハンドル
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
wave_convert_end (WaveConv *wcv,
                  gpointer *out_buf,
                  gsize    *out_samples);


/*  ja:PCMの変換をする
           wfx0,元のWaveFormatEx構造体へのポインタ
           wfx1,新しいWaveFormatEx構造体へのポインタ
         in_buf,入力データ
     in_samples,入力データのサンプル数
        out_buf,出力データ
    out_samples,出力データのサンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
wave_convert_direct (WaveFormatEx  *wfx0,
                     WaveFormatEx  *wfx1,
                     gconstpointer  in_buf,
                     const gsize    in_samples,
                     gpointer      *out_buf,
                     gsize         *out_samples);


#ifdef __cplusplus
}
#endif /* __cplusplus */


#endif /* __WAVE_H__ */
