#ifndef DIGITS_HPP
#define DIGITS_HPP

#include <string>
#include <algorithm>

/* digits<V, C>
 *
 * std::string s = "1234";
 * if (digits<size_t, char>::check(s)) {
 *   size_t value = digits<size_t, char>::eval(s);
 * }
 */

/* capacity<V, C>
 *
 * The SIZE argument is an integer
 *  and optional unit (example: 10K is 10*1024).
 * Units are K,M,G,T,P,E,Z,Y (powers of 1024)
 *  or KB,MB,...(powers of 1000).
 *
 * std::string s = "128M";
 * if (capacity<size_t, char>::check(s)) {
 *   size_t size = capacity<size_t, char>::eval(s);
 * }
 */

template <typename V,  typename C> class digits {
public:
  static C toupper(C v) {
     if ((C('a') <= v) && (v <= C('z'))) return (v - C(0x20));
     return v;
  }
  static C tolower(C v) {
     if ((C('A') <= v) && (v <= C('Z'))) return (v + C(0x20));
     return v;
  }
  static size_t count(const std::basic_string<C> &s) {
    for (size_t i = 0; i < s.size(); i++) {
      if (!((C('0') <= s[i]) && (s[i] <= C('9')))) return i;
    }
    return s.size();
  }
  static bool check(const std::basic_string<C> &s) {
    return count(s) == s.size();
  }
  static V eval(const std::basic_string<C> &s) {
    V t = V(0);
    for (size_t i = 0; i < s.size(); i++) {
      t *= 10;
      t += s[i] - C('0');
    }
    return t;
  }
};

template <typename V, typename C> class capacity: public digits<V, C> {
public:
  static V unit(const std::basic_string<C> &s) {
    static C units[] = {
      C('K'), C('M'), C('G'), C('T'), C('P'), C('E'), C('Z'), C('Y'),
    };
    const size_t units_size = (sizeof units) / (sizeof (C));
    V base;
    if (s.size() == 0) {
      return 1;
    } else if ((s.size() == 2)
     && (digits<V, C>::toupper(s[1]) == C('B'))) {
      base = V(1000);
    } else if (s.size() == 1) {
      base = V(1024);
    } else {
      return 0;
    }
    V uv = V(base);
    for (size_t i = 0; i < units_size; i++) {
      if (digits<V, C>::toupper(s[0]) == units[i]) {
        return uv;
      }
      uv = uv * base;
    }
    return 0;
  }
  static bool check(const std::basic_string<C> &s) {
    size_t l = digits<V, C>::count(s);
    if (l == 0) {
      return false;
    } else if (l == s.size()) {
      return true;
    } else if (unit(s.substr(l))) {
      return true;
    }
    return false;
  }
  static V eval(const std::basic_string<C> &s) {
    size_t l = digits<V, C>::count(s);
    V v = digits<V, C>::eval(s.substr(0, l));
    V u = unit(s.substr(l));
    return u * v;
  }
};

#endif
