#include <cstdio>
#include <cerrno>
#include <stdexcept>
#include <string>
#include <memory>

#include <fcntl.h>
#include <io.h>
#include <sys/stat.h>
#include <sys/types.h>

#define CRYPTOPP_ENABLE_NAMESPACE_WEAK 1
#include <md5.h>
#include <sha.h>
#include "cryptopp-lib.hpp"

#include "digits.hpp"
#include "popt.hpp"

class sum_type {
public:
  virtual void update(const byte *p, size_t l) = 0;
  virtual void restart() = 0;
  virtual void final() = 0;
  virtual void calc(FILE *fp, std::vector<char> &buff) = 0;
  virtual std::string dump() = 0;
  virtual ~sum_type() {}
};

template <class C> class sum: public sum_type {
  C hash;
  byte digest[C::DIGESTSIZE];
public:
  virtual void update(const byte *p, size_t l) { hash.Update(p, l); }
  virtual void restart() { hash.Restart(); }
  virtual void final() { hash.Final(digest); }
  virtual void calc(FILE *fp, std::vector<char> &buff) {
    restart();
    while (!feof(fp) && !ferror(fp)) {
      size_t n = fread(&(buff[0]), 1, buff.size(), fp);
      if (n > 0) {
        update((byte *)&(buff[0]), n);
      }
    }
    if (!ferror(fp)) {
      final();
    } else {
      throw std::runtime_error(strerror(errno));
    }
  }
  virtual std::string dump() { 
    char buff[C::DIGESTSIZE * 2 + 1];
    for (size_t i = 0; i < C::DIGESTSIZE; i++) {
      sprintf(&(buff[i * 2]), "%02x", digest[i] & 0xff);
    }
    return (std::string)buff;
  }
  sum() {};
  virtual ~sum() {};
};

bool iscdev(FILE *fp) {
# ifdef _WIN32
  struct _stat stt;
  if (_fstat(fileno(fp), &stt)) throw std::runtime_error("fstat");
  return (stt.st_mode & _S_IFCHR) != 0;
# else
  struct stat stt;
  if (fstat(fileno(fp), &stt)) throw std::runtime_error("fstat");
  return (stt.st_mode & S_IFCHR) != 0;
# endif
}

int main(int argc, char *argv[]) {
  try {
    int n;
    popts<char> opts;
    opts.option("type:t", true);
    opts.option("text:T", false);
    opts.option("buffer-size:B", true);
    opts.option("string:s", true);
    opts.option("help:h", false);
    if (n = opts.parse(argc, argv)) {
      printf("option error:%s\n", argv[n - 1]);
      return 1;
    }
    if (opts.check("help")) {
      printf(
       "usage: checksum [options] [file]...\n"
       " options:\n"
       "  --type/-t typename ... hash type(default:md5)\n"
       "   typename ::= (md5|sha1|sha224|sha256|sha384|sha512);\n"
       "  --text/-T ... read files in text mode(*1).\n"
       "  --buffer-size/-B /[0-9]+([KMGTPEZYkmgtpezy][Bb]?)?/\n"
       "   specifying a buffer size(*2).\n"
       "  --string/-s \"string\" ... read string from arg.\n"
       "  --help/-h ... print this message.\n"
       "  *1 read/write in text mode from console.\n"
       "     read/write in binary mode from other(default).\n"
       "  *2 The SIZE argument is an integer\n"
       "      and optional unit (example: 10K is 10*1024).\n"
       "     Units are K,M,G,T,P,E,Z,Y (powers of 1024)\n"
       "      or KB,MB,... (powers of 1000).\n");
      return 1;
    }
    std::unique_ptr<sum_type> hash;
    if (opts.check("type")) {
      if (opts.value("type").size() != 1) {
        throw std::runtime_error("hash-type: you can specify only one");
      }
      std::string tname = opts.value("type")[0];
      if (strcmpi(tname.c_str(), "md5") == 0) {
        hash.reset((sum_type *)new sum<CryptoPP::Weak::MD5>());
      } else if (strcmpi(tname.c_str(), "sha1") == 0) {
        hash.reset((sum_type *)new sum<CryptoPP::SHA1>());
      } else if (strcmpi(tname.c_str(), "sha224") == 0) {
        hash.reset((sum_type *)new sum<CryptoPP::SHA224>());
      } else if (strcmpi(tname.c_str(), "sha256") == 0) {
        hash.reset((sum_type *)new sum<CryptoPP::SHA256>());
      } else if (strcmpi(tname.c_str(), "sha384") == 0) {
        hash.reset((sum_type *)new sum<CryptoPP::SHA384>());
      } else if (strcmpi(tname.c_str(), "sha512") == 0) {
        hash.reset((sum_type *)new sum<CryptoPP::SHA512>());
      } else {
        throw std::runtime_error(
         ((std::string)"bad type:" + tname).c_str());
      }
    } else {
      hash.reset(new sum<CryptoPP::Weak::MD5>());
    }
    if (opts.check("string")) {
      std::vector<std::string> &strs = opts.value("string");
      for (size_t i = 0; i < strs.size(); i++) {
        hash->restart();
        hash->update((byte *)strs[i].c_str(), strs[i].size());
        hash->final();
        printf("%s \"%s\"\n", hash->dump().c_str(),
         (byte *)opts.value("string")[i].c_str());
      }
    }
    size_t bsize = 8192;
    if (opts.check("B")) {
      if ((opts.count_values("B") == 1)
       && (capacity<size_t, char>::check(opts.value("B")[0]))) {
        bsize = capacity<size_t, char>::eval(opts.value("B")[0]);
        /* printf("DEBUG:bsize=%ld\n", bsize); */
      } else {
        throw std::runtime_error("bad arg(option -B)");
      }
    }
    std::vector<char> buff(bsize);
    for (size_t i = 0; i < opts.args.size(); i++) {
      try {
        FILE *fp;
        const char *mode = (opts.check("text") ? "rt" : "rb");
        if ((fp = fopen(opts.args[i].c_str(), mode)) != NULL) {
          std::unique_ptr<FILE, decltype(&fclose)> keep(fp, fclose);
          if (setvbuf(fp, NULL, _IONBF, 0)) {
            fprintf(stderr, "warning:setvbuf:%s\n", strerror(errno));
          }
          hash->calc(fp, buff);
          printf("%s *%s\n", hash->dump().c_str(), opts.args[i].c_str());
        }
      } catch (std::exception &e) {
        printf("error:%s:%s\n", opts.args[i].c_str(), e.what());
      }
    }
    if ((opts.args.size() + 
     (opts.check("string") ? opts.value("string").size() : 0)) < 1) {
      try {
        if (iscdev(stdin) && (!opts.check("text"))) {
          _setmode(fileno(stdin), _O_BINARY);
          if (errno) {
            fprintf(stderr, "warning:_setmode:%s\n", strerror(errno));
          }
        }
        hash->calc(stdin, buff);
        printf("%s\n", hash->dump().c_str());
      } catch (std::exception &e) {
        printf("error:stdin:%s\n", e.what());
      }
    }
  } catch (std::exception &e) {
    printf("%s\n", e.what());
    return 2;
  }
  return 0;
}
