/* Copyright(C) 2004 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifndef _SEN_STORE_H
#define _SEN_STORE_H

#ifndef _SENNA_H
#include "senna_in.h"
#endif /* _SENNA_H */

#ifndef _SEN_SET_H
#include "set.h"
#endif /* _SEN_SET_H */

#ifndef _SEN_IO_H
#include "io.h"
#endif /* _SEN_IO_H */

#ifdef __cplusplus
extern "C" {
#endif

#define SEN_ST_APPEND 1

/**** fixed sized elements ****/

typedef struct _sen_ra sen_ra;

struct _sen_ra {
  sen_io *io;
  int element_width;
  int element_mask;
  struct sen_ra_header *header;
};

sen_ra *sen_ra_create(const char *path, unsigned int element_size);
sen_ra *sen_ra_open(const char *path);
sen_rc sen_ra_info(sen_ra *ra, unsigned int *element_size, sen_id *curr_max);
sen_rc sen_ra_close(sen_ra *ra);
sen_rc sen_ra_remove(const char *path);
void *sen_ra_get(sen_ra *ra, sen_id id);
void *sen_ra_at(sen_ra *ra, sen_id id);

/**** variable sized elements ****/

typedef struct _sen_ja sen_ja;
typedef struct _sen_ja_einfo sen_ja_einfo;

struct _sen_ja {
  sen_io *io;
  struct sen_ja_header *header;
};

sen_ja *sen_ja_create(const char *path, unsigned int max_element_size);
sen_ja *sen_ja_open(const char *path);
sen_rc sen_ja_info(sen_ja *ja, unsigned int *max_element_size);
sen_rc sen_ja_close(sen_ja *ja);
sen_rc sen_ja_remove(const char *path);
sen_rc sen_ja_put(sen_ja *ja, sen_id id, const void *value, int value_len, int flags);
int sen_ja_at(sen_ja *ja, sen_id id, void *valbuf, int buf_size);

const void *sen_ja_ref(sen_ja *ja, sen_id id, uint32_t *value_len);
sen_rc sen_ja_unref(sen_ja *ja, sen_id id);

int sen_ja_size(sen_ja *ja, sen_id id);
sen_rc sen_ja_alloc(sen_ja *ja, int element_size, sen_ja_einfo *einfo, void **value);
sen_rc sen_ja_replace(sen_ja *ja, sen_id id, sen_ja_einfo *ei);

/**** store ****/

#define SEN_STORE_CTX_HEAD 1
#define SEN_STORE_CTX_TAIL 2

#define car(c) (((c) && (c)->type == sen_store_list) ? (c)->u.l.car : NULL)
#define cdr(c) (((c) && (c)->type == sen_store_list) ? (c)->u.l.cdr : NULL)
#define caar(c) (car(car(c)))
#define cadr(c) (car(cdr(c)))
#define cdar(c) (cdr(car(c)))
#define cddr(c) (cdr(cdr(c)))

typedef struct _sen_store sen_store;
typedef struct _sen_store_obj sen_store_obj;
typedef struct _sen_store_ctx sen_store_ctx;
typedef struct _sen_store_spec sen_store_spec;

typedef enum {
  sen_store_nil = 0,
  sen_store_builtin_class,
  sen_store_structured_class,
  sen_store_obj_slot,
  sen_store_ra_slot,
  sen_store_ja_slot,
  sen_store_idx_slot,
  sen_store_object,
  sen_store_records,
  sen_store_bulk,
  sen_store_list,
  sen_store_int,
  sen_store_str,
  sen_store_native_method,
  sen_store_method,
  sen_store_cont
} sen_store_type;

struct _sen_store_spec {
  sen_store_type type;
  union {
    struct {
      unsigned int key_size;
      unsigned int flags;
      sen_encoding encoding;
    } sc;
    struct {
      unsigned int element_size;
    } bc;
    struct {
      sen_id class;
      unsigned int collection_type;
    } os;
    struct {
      sen_id class;
      unsigned int element_size;
    } fs;
    struct {
      sen_id class;
      unsigned int max_element_size;
    } vs;
    struct {
      sen_id target;
      int initial_n_segments;
    } is;
  } u;
};

typedef sen_store_obj *sen_store_method_func(sen_store_ctx *, sen_store_obj *);
typedef sen_store_obj *sen_store_parser(sen_store_ctx *, const char *, uint32_t);

struct _sen_store_obj {
  sen_store_type type;
  sen_id class;
  union {
    struct {
      sen_id self;
    } o;
    struct {
      sen_records *records;
    } r;
    struct {
      const void *value;
      uint32_t size;
    } b;
    struct {
      sen_store_obj *car;
      sen_store_obj *cdr;
    } l;
    struct {
      char *value;
      uint32_t size;
    } s;
    struct {
      sen_store_obj *func;
    } f;
    struct {
      sen_store_method_func *func;
    } n;
    struct {
      int32_t i;
    } i;
  } u;
};

sen_store *sen_store_create(const char *path, int flags, sen_encoding encoding);
sen_store *sen_store_open(const char *path);
sen_rc sen_store_close(sen_store *s);

sen_store_ctx *sen_store_ctx_open(sen_store *s, sen_store_parser *parser);

sen_rc sen_store_ctx_close(sen_store_ctx *c);

sen_rc sen_store_at(sen_store_ctx *c, sen_store_obj *obj,
                    const char *key, const sen_store_obj *args, sen_store_obj **res);
sen_rc sen_store_send(sen_store_ctx *c, sen_store_obj *obj,
                    const char *message, const sen_store_obj *args, sen_store_obj **res);

sen_store_obj *sen_store_ctx_eval(sen_store_ctx *c, sen_store_obj *sexp);
void sen_store_ctx_add_native_method(sen_store_ctx *c, const char *name,
                                     sen_store_method_func *func);

const char *_sen_store_obj_key(sen_store *store, sen_store_obj *obj);

sen_store_obj *sen_store_ctx_feed(sen_store_ctx *c, const char *str, uint32_t str_size, int mode);

sen_store_obj *sen_store_sexp_parser(sen_store_ctx *c, const char *str, uint32_t str_size);

#define SEN_OBJ2VALUE(o,v,s) ((v) = (o)->u.b.value, (s) = (o)->u.b.size)
#define SEN_VALUE2OBJ(o,v,s) ((o)->u.b.value = (v), (o)->u.b.size = (s))

/**** vgram ****/

typedef struct _sen_vgram_vnode
{
  struct _sen_vgram_vnode *car;
  struct _sen_vgram_vnode *cdr;
  sen_id tid;
  sen_id vid;
  int freq;
  int len;
} sen_vgram_vnode;

struct _sen_vgram {
  sen_sym *vgram;
};

struct _sen_vgram_buf {
  size_t len;
  sen_id *tvs;
  sen_id *tvp;
  sen_id *tve;
  sen_vgram_vnode *vps;
  sen_vgram_vnode *vpp;
  sen_vgram_vnode *vpe;
};

sen_vgram *sen_vgram_create(const char *path);
sen_vgram *sen_vgram_open(const char *path);
sen_rc sen_vgram_close(sen_vgram *vgram);
sen_rc sen_vgram_update(sen_vgram *vgram, sen_id rid, sen_vgram_buf *b, sen_set *terms);

sen_vgram_buf *sen_vgram_buf_open(size_t len);
sen_rc sen_vgram_buf_add(sen_vgram_buf *b, sen_id tid);
sen_rc sen_vgram_buf_close(sen_vgram_buf *b);

#ifdef __cplusplus
}
#endif

#endif /* _SEN_STORE_H */
