/* Copyright(C) 2004,2005,2006 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "senna_in.h"
#include "str.h"
#include "inv.h"
#include "sym.h"
#include "str.h"
#include "store.h"
#include <string.h>
#include <stdio.h>

/* rectangular arrays */

#define SEN_RA_IDSTR "SENNA:RA:01.000"
#define SEN_RA_SEGMENT_SIZE (1 << 22)

#define SEN_RA_MAX_CACHE (4294967295U)

struct sen_ra_header {
  char idstr[16];
  unsigned element_size;
  sen_id curr_max;
  uint32_t reserved[10];
};

sen_ra *
sen_ra_create(const char *path, unsigned int element_size)
{
  sen_io *io;
  int max_segments, n_elm, w_elm;
  sen_ra *ra = NULL;
  struct sen_ra_header *header;
  unsigned actual_size;
  if (element_size > SEN_RA_SEGMENT_SIZE) {
    SEN_LOG(sen_log_error, "element_size too large (%d)", element_size);
    return NULL;
  }
  for (actual_size = 1; actual_size < element_size; actual_size *= 2) ;
  max_segments = ((SEN_ID_MAX + 1) / SEN_RA_SEGMENT_SIZE) * actual_size;
  io = sen_io_create(path, sizeof(struct sen_ra_header),
                     SEN_RA_SEGMENT_SIZE, max_segments, sen_io_auto, SEN_RA_MAX_CACHE);
  if (!io) { return NULL; }
  header = sen_io_header(io);
  memcpy(header->idstr, SEN_RA_IDSTR, 16);
  header->element_size = actual_size;
  header->curr_max = 0;
  if (!(ra = SEN_MALLOC(sizeof(sen_ra)))) {
    sen_io_close(io);
    return NULL;
  }
  n_elm = SEN_RA_SEGMENT_SIZE / header->element_size;
  for (w_elm = 22; (1 << w_elm) > n_elm; w_elm--);
  ra->io = io;
  ra->header = header;
  ra->element_mask =  n_elm - 1;
  ra->element_width = w_elm;
  return ra;
}

sen_ra *
sen_ra_open(const char *path)
{
  sen_io *io;
  int n_elm, w_elm;
  sen_ra *ra = NULL;
  struct sen_ra_header *header;
  io = sen_io_open(path, sen_io_auto, SEN_RA_MAX_CACHE);
  if (!io) { return NULL; }
  header = sen_io_header(io);
  if (memcmp(header->idstr, SEN_RA_IDSTR, 16)) {
    SEN_LOG(sen_log_error, "ra_idstr (%s)", header->idstr);
    sen_io_close(io);
    return NULL;
  }
  if (!(ra = SEN_MALLOC(sizeof(sen_ra)))) {
    sen_io_close(io);
    return NULL;
  }
  n_elm = SEN_RA_SEGMENT_SIZE / header->element_size;
  for (w_elm = 22; (1 << w_elm) > n_elm; w_elm--);
  ra->io = io;
  ra->header = header;
  ra->element_mask =  n_elm - 1;
  ra->element_width = w_elm;
  return ra;
}

sen_rc
sen_ra_info(sen_ra *ra, unsigned int *element_size, sen_id *curr_max)
{
  if (!ra) { return sen_invalid_argument; }
  if (element_size) { *element_size = ra->header->element_size; }
  if (curr_max) { *curr_max = ra->header->curr_max; }
  return sen_success;
}

sen_rc
sen_ra_close(sen_ra *ra)
{
  sen_rc rc;
  if (!ra) { return sen_invalid_argument; }
  rc = sen_io_close(ra->io);
  SEN_FREE(ra);
  return rc;
}

sen_rc
sen_ra_remove(const char *path)
{
  if (!path) { return sen_invalid_argument; }
  sen_io_remove(path);
  return sen_success;
}

void *
sen_ra_get(sen_ra *ra, sen_id id)
{
  void *p;
  uint16_t seg;
  if (id > SEN_ID_MAX) { return NULL; }
  seg = id >> ra->element_width;
  SEN_IO_SEG_MAP(ra->io, seg, p);
  if (!p) { return NULL; }
  if (id > ra->header->curr_max) { ra->header->curr_max = id; }
  return (void *)(((byte *)p) + ((id & ra->element_mask) * ra->header->element_size));
}

void *
sen_ra_at(sen_ra *ra, sen_id id)
{
  void *p;
  uint16_t seg;
  if (id > ra->header->curr_max) { return NULL; }
  seg = id >> ra->element_width;
  SEN_IO_SEG_MAP(ra->io, seg, p);
  if (!p) { return NULL; }
  return (void *)(((byte *)p) + ((id & ra->element_mask) * ra->header->element_size));
}

/**** jagged arrays ****/

#define SEN_JA_IDSTR "SENNA:JA:01.000"

#define W_OF_JA_MAX 38
#define W_OF_JA_SEGMENT 22
#define W_OF_JA_MAX_SEGMENTS (W_OF_JA_MAX - W_OF_JA_SEGMENT)

#define W_OF_JA_EINFO 3
#define W_OF_JA_EINFO_IN_A_SEGMENT (W_OF_JA_SEGMENT - W_OF_JA_EINFO)
#define N_OF_JA_EINFO_IN_A_SEGMENT (1U << W_OF_JA_EINFO_IN_A_SEGMENT)
#define JA_EINFO_MASK (N_OF_JA_EINFO_IN_A_SEGMENT - 1)

#define JA_SEGMENT_SIZE (1U << W_OF_JA_SEGMENT)
#define JA_MAX_SEGMENTS (1U << W_OF_JA_MAX_SEGMENTS)

#define JA_BSA_SIZE (1U << (W_OF_JA_SEGMENT - 7))
#define JA_N_BSEGMENTS (1U << (W_OF_JA_MAX_SEGMENTS - 7))

#define JA_N_ESEGMENTS (1U << 9)

#define SEN_JA_MAX_CACHE (4294967295U)

struct _sen_ja_einfo {
  union {
    uint64_t ll;
    struct {
      uint16_t seg;
      uint16_t pos;
      uint16_t size;
      uint8_t tail[2];
    } s;
  } u;
};

#define EINFO_SET(e,_seg,_pos,_size) {\
  (e)->u.s.seg = _seg;\
  (e)->u.s.pos = (_pos) >> 4;\
  (e)->u.s.size = _size;\
  (e)->u.s.tail[0] = (((_pos) >> 14) & 0xc0) + ((_size) >> 16);\
  (e)->u.s.tail[1] = 0;\
}

#define EINFO_GET(e,_seg,_pos,_size) {\
  _seg = (e)->u.s.seg;\
  _pos = ((e)->u.s.pos + (((e)->u.s.tail[0] & 0xc0) << 10)) << 4;\
  _size = (e)->u.s.size + (((e)->u.s.tail[0] & 0x3f) << 16);\
}

typedef struct {
  uint32_t seg;
  uint32_t pos;
} ja_pos;

struct sen_ja_header {
  char idstr[16];
  unsigned max_element_size;
  unsigned max_segments;
  ja_pos free_elements[24];
  uint8_t segments[JA_MAX_SEGMENTS];
  uint32_t esegs[JA_N_ESEGMENTS];
  uint32_t bsegs[JA_N_BSEGMENTS];
};


#define JA_SEG_ESEG 1;
#define JA_SEG_BSEG 2;
#define SEG_NOT_ASSIGNED 0xffffffff

sen_ja *
sen_ja_create(const char *path, unsigned int max_element_size)
{
  int i;
  sen_io *io;
  int max_segments;
  sen_ja *ja = NULL;
  struct sen_ja_header *header;
  if (max_element_size > JA_SEGMENT_SIZE) {
    SEN_LOG(sen_log_error, "max_element_size too large (%d)", max_element_size);
    return NULL;
  }
  max_segments = max_element_size * 128;
  if (max_segments > JA_MAX_SEGMENTS) { max_segments = JA_MAX_SEGMENTS; }
  io = sen_io_create(path, sizeof(struct sen_ja_header),
                     JA_SEGMENT_SIZE, max_segments, sen_io_auto, SEN_JA_MAX_CACHE);
  if (!io) { return NULL; }
  header = sen_io_header(io);
  memcpy(header->idstr, SEN_JA_IDSTR, 16);
  for (i = 0; i < JA_N_ESEGMENTS; i++) { header->esegs[i] = SEG_NOT_ASSIGNED; }
  for (i = 0; i < JA_N_BSEGMENTS; i++) { header->bsegs[i] = SEG_NOT_ASSIGNED; }
  header->max_element_size = max_element_size;
  header->max_segments = max_segments;
  header->segments[0] = JA_SEG_ESEG;
  header->esegs[0] = 0;
  if (!(ja = SEN_MALLOC(sizeof(sen_ja)))) {
    sen_io_close(io);
    return NULL;
  }
  ja->io = io;
  ja->header = header;
  return ja;
}

sen_ja *
sen_ja_open(const char *path)
{
  sen_io *io;
  sen_ja *ja = NULL;
  struct sen_ja_header *header;
  io = sen_io_open(path, sen_io_auto, SEN_JA_MAX_CACHE);
  if (!io) { return NULL; }
  header = sen_io_header(io);
  if (memcmp(header->idstr, SEN_JA_IDSTR, 16)) {
    SEN_LOG(sen_log_error, "ja_idstr (%s)", header->idstr);
    sen_io_close(io);
    return NULL;
  }
  if (!(ja = SEN_MALLOC(sizeof(sen_ja)))) {
    sen_io_close(io);
    return NULL;
  }
  ja->io = io;
  ja->header = header;
  return ja;
}

sen_rc
sen_ja_info(sen_ja *ja, unsigned int *max_element_size)
{
  if (!ja) { return sen_invalid_argument; }
  return sen_success;
}

sen_rc
sen_ja_close(sen_ja *ja)
{
  sen_rc rc;
  if (!ja) { return sen_invalid_argument; }
  rc = sen_io_close(ja->io);
  SEN_FREE(ja);
  return rc;
}

sen_rc
sen_ja_remove(const char *path)
{
  if (!path) { return sen_invalid_argument; }
  sen_io_remove(path);
  return sen_success;
}

sen_rc
sen_ja_put(sen_ja *ja, sen_id id, const void *value, int value_len, int flags)
{
  int rc;
  void *buf;
  sen_ja_einfo einfo;
  sen_log("value='%s'", (char *) value);
  if ((flags & SEN_ST_APPEND)) {
    uint32_t old_len;
    const void *oldvalue = sen_ja_ref(ja, id, &old_len);
    if (oldvalue) {
      if ((rc = sen_ja_alloc(ja, value_len + old_len, &einfo, &buf))) { return rc; }
      memcpy(buf, oldvalue, old_len);
      memcpy((byte *)buf + old_len, value, value_len);
      sen_ja_unref(ja, id);
    } else {
      if ((rc = sen_ja_alloc(ja, value_len, &einfo, &buf))) { return rc; }
      memcpy(buf, value, value_len);
    }
  } else {
    if ((rc = sen_ja_alloc(ja, value_len, &einfo, &buf))) { return rc; }
    // printf("put id=%d, value_len=%d value=%p ei=%p(%d:%d)\n", id, value_len, buf, &einfo, einfo.u.s.pos, einfo.u.s.tail[0]);
    memcpy(buf, value, value_len);
  }
  return sen_ja_replace(ja, id, &einfo);
}

int
sen_ja_at(sen_ja *ja, sen_id id, void *valbuf, int buf_size)
{
  uint32_t len;
  const void *value = sen_ja_ref(ja, id, &len);
  if (!value) { return -1; }
  if (buf_size >= len) { memcpy(valbuf, value, len); }
  sen_ja_unref(ja, id);
  return (int) len;
}

const void *
sen_ja_ref(sen_ja *ja, sen_id id, uint32_t *value_len)
{
  sen_ja_einfo *einfo;
  uint32_t lseg, *pseg, pos;
  lseg = id >> W_OF_JA_EINFO_IN_A_SEGMENT;
  pos = id & JA_EINFO_MASK;
  pseg = &ja->header->esegs[lseg];
  if (*pseg == SEG_NOT_ASSIGNED) { *value_len = 0; return NULL; }
  SEN_IO_SEG_MAP(ja->io, *pseg, einfo);
  if (!einfo) { *value_len = 0; return NULL; }
  if (einfo[pos].u.s.tail[1] & 1) {
    *value_len = einfo[pos].u.s.tail[1] >> 1;
    return (void *) &einfo[pos];
  }
  {
    void *value;
    uint32_t jag, vpos, vsize;
    EINFO_GET(&einfo[pos], jag, vpos, vsize);
    SEN_IO_SEG_MAP(ja->io, jag, value);
    // printf("at id=%d value=%p jag=%d vpos=%d ei=%p(%d:%d)\n", id, value, jag, vpos, &einfo[pos], einfo[pos].u.s.pos, einfo[pos].u.s.tail[0]);
    if (!value) { *value_len = 0; return NULL; }
    *value_len = vsize;
    return (byte *)value + vpos;
  }
}

sen_rc
sen_ja_unref(sen_ja *ja, sen_id id)
{
  // todo
  return sen_success;
}

int
sen_ja_size(sen_ja *ja, sen_id id)
{
  sen_ja_einfo *einfo;
  uint32_t lseg, *pseg, pos;
  lseg = id >> W_OF_JA_EINFO_IN_A_SEGMENT;
  pos = id & JA_EINFO_MASK;
  pseg = &ja->header->esegs[lseg];
  if (*pseg == SEG_NOT_ASSIGNED) { return -1; }
  SEN_IO_SEG_MAP(ja->io, *pseg, einfo);
  if (!einfo) { return -1; }
  if (einfo[pos].u.s.tail[1] & 1) {
    return einfo[pos].u.s.tail[1] >> 1;
  } else {
    return einfo[pos].u.s.size + ((einfo[pos].u.s.tail[0] & 0x3f) << 16);
  }
}

sen_rc
sen_ja_alloc(sen_ja *ja, int element_size, sen_ja_einfo *einfo, void **value)
{
  int m, size;
  void *addr;
  ja_pos *vp;
  if (element_size < 8) {
    einfo->u.s.tail[1] = element_size * 2 + 1;
    *value = (void *)einfo;
    return sen_success;
  }
  if (element_size >= ja->header->max_element_size) {
    return sen_invalid_argument;
  }
  for (m = 4, size = 16; size < element_size; m++, size *= 2);
  vp = &ja->header->free_elements[m];
  if (!vp->seg) {
    int i = 0;
    while (ja->header->segments[i]) {
      if (++i >= ja->header->max_segments) { return sen_memory_exhausted; }
    }
    ja->header->segments[i] = m;
    vp->seg = i;
    vp->pos = 0;
  }
  EINFO_SET(einfo, vp->seg, vp->pos, element_size);
  SEN_IO_SEG_MAP(ja->io, vp->seg, addr);
  // printf("addr=%p seg=%d pos=%d\n", addr, vp->seg, vp->pos);
  if (!addr) { return sen_memory_exhausted; }
  *value = (byte *)addr + vp->pos;
  if ((vp->pos += size) == JA_SEGMENT_SIZE) {
    vp->seg = 0;
    vp->pos = 0;
  }
  return sen_success;
}

sen_rc
sen_ja_free(sen_ja *ja, sen_ja_einfo *einfo)
{
  uint32_t seg, pos, size;
  if (einfo->u.s.tail[1] & 1) { return sen_success; }
  EINFO_GET(einfo, seg, pos, size);
  // free
  return sen_success;
}

sen_rc
sen_ja_replace(sen_ja *ja, sen_id id, sen_ja_einfo *ei)
{
  uint32_t lseg, *pseg, pos;
  sen_ja_einfo *einfo, eback;
  lseg = id >> W_OF_JA_EINFO_IN_A_SEGMENT;
  pos = id & JA_EINFO_MASK;
  pseg = &ja->header->esegs[lseg];
  if (*pseg == SEG_NOT_ASSIGNED) {
    int i = 0;
    while (ja->header->segments[i]) {
      if (++i >= ja->header->max_segments) { return sen_memory_exhausted; }
    }
    ja->header->segments[i] = 1;
    *pseg = i;
  }
  SEN_IO_SEG_MAP(ja->io, *pseg, einfo);
  if (!einfo) { return sen_memory_exhausted; }
  eback = einfo[pos];
  einfo[pos] = *ei;
  // todo: SEN_ATOMIC_SET64
  sen_ja_free(ja, &eback);
  return sen_success;
}

/**** store ****/

typedef struct _sen_store_trigger sen_store_trigger;
typedef struct _sen_store_entity sen_store_entity;

struct _sen_store {
  sen_sym *keys;
  sen_ja *values;
  sen_set *entities;
};

struct _sen_store_ctx {
  uint32_t mode;
  uint32_t seqno;
  sen_store *store;
  sen_set *objects;
  sen_set *bindings;
  void *opaque;
  sen_store_obj *(*parser)(sen_store_ctx *, const char *, uint32_t);
  sen_store_method_func *doing;
};

struct _sen_store_trigger {
  sen_store_trigger *next;
  int32_t type;
  void *data;
};

struct _sen_store_entity {
  sen_store_type type;
  sen_store *store;
  sen_id id;
  sen_store_trigger *triggers;
  union {
    struct {
      sen_sym *keys;
    } c;
    struct {
      sen_id class;
      sen_ra *ra;
    } o;
    struct {
      sen_ra *ra;
    } f;
    struct {
      sen_ja *ja;
    } v;
    struct {
      sen_id target;
      sen_index *index;
    } i;
  } u;
};

inline static void
gen_pathname(const char *path, char *buffer, int fno)
{
  size_t len = strlen(path);
  memcpy(buffer, path, len);
  if (fno >= 0) {
    buffer[len] = '.';
    sen_str_itoh(fno, buffer + len + 1, 7);
  } else {
    buffer[len] = '\0';
  }
}

sen_rc
sen_class_at(sen_store_entity *c, const void *key, int flags, sen_store_obj *res)
{
  sen_id id = flags ? sen_sym_get(c->u.c.keys, key) : sen_sym_at(c->u.c.keys, key);
  if (id) {
    res->u.o.self = id;
    res->type = sen_store_object;
    res->class = c->id;
    return sen_success;
  } else {
    res->type = sen_store_nil;
    return sen_other_error;
  }
}

sen_store_entity *sen_slot_class(sen_store *s, sen_id slot);

sen_store_entity *
sen_store_entity_open(sen_store *s, const char *name)
{
  sen_id id;
  sen_set_eh *ep;
  uint32_t spec_len;
  sen_store_entity *slot;
  const char *spec, *spe;
  if (!(id = sen_sym_at(s->keys, name))) { return NULL; }
  if (sen_set_at(s->entities, &id, (void **) &slot)) { return slot; }
  if (!(spec = sen_ja_ref(s->values, id, &spec_len))) { return NULL; }
  spe = spec + spec_len;
  ep = sen_set_get(s->entities, &id, (void **) &slot);
  sen_log("name='%s' spec=%d,'%s'", name, spec_len, spec);
  slot->store = s;
  slot->id = id;
  slot->triggers = NULL;
  {
    char buffer[PATH_MAX];
    gen_pathname(s->keys->io->path, buffer, id);
    if (!strchr(name, '.')) {
      sen_store_spec *ss = (sen_store_spec *)spec;
      switch (ss->type) {
      case sen_store_builtin_class :
        break;
      case sen_store_structured_class :
        if (!(slot->u.c.keys = sen_sym_open(buffer))) { goto exit; }
        break;
      default :
        goto exit;
      }
      slot->type = ss->type;
    } else {
      switch (*spec++) {
      case 'o' :
        slot->type = sen_store_obj_slot;
        if (!(slot->u.o.class = sen_sym_at(s->keys, spec))) { goto exit; }
        if (!(slot->u.o.ra = sen_ra_open(buffer))) { goto exit; }
        break;
      case 'f' :
        slot->type = sen_store_ra_slot;
        if (!(slot->u.f.ra = sen_ra_open(buffer))) { goto exit; }
        break;
      case 'v' :
        slot->type = sen_store_ja_slot;
        if (!(slot->u.v.ja = sen_ja_open(buffer))) { goto exit; }
        break;
      case 'i' :
        slot->type = sen_store_idx_slot;
        {
          sen_store_entity *target, *l, *k;
          if (!(target = sen_store_entity_open(s, spec))) { goto exit; }
          if (!(l = sen_slot_class(s, id))) { goto exit; }
          if (!(k = sen_slot_class(s, target->id))) { goto exit; }
          slot->u.i.target = target->id;
          if (!(slot->u.i.index =
                sen_index_open_with_keys_lexicon(buffer, k->u.c.keys, l->u.c.keys))) {
            goto exit;
          }
        }
        break;
      }
    }
  }
  while ((spec += strlen(spec) + 1) < spe) {
    sen_store_entity *index = sen_store_entity_open(s, spec);
    if (index) {
      sen_store_trigger *t = SEN_MALLOC(sizeof(sen_store_trigger));
      t->next = slot->triggers;
      t->type = 1;
      t->data = (void *) index;
      slot->triggers = t;
    }
  }
  sen_ja_unref(s->values, id);
  return slot;
exit :
  sen_set_del(s->entities, ep);
  sen_ja_unref(s->values, id);
  return NULL;
}

sen_store_entity *
sen_store_entity_create(sen_store *s, const char *name, const sen_store_obj *args)
{
  sen_id id;
  sen_set_eh *ep;
  sen_store_entity *slot;
  const char *spec = args->u.b.value;
  char buffer[PATH_MAX];
  // todo : replace spec if the slot already exists.
  if (strlen(name) >= SEN_SYM_MAX_KEY_SIZE) { return NULL; }
  if (!(id = sen_sym_get(s->keys, name))) { return NULL; }
  ep = sen_set_get(s->entities, &id, (void **) &slot);
  slot->store = s;
  slot->id = id;
  slot->triggers = NULL;
  if (sen_ja_put(s->values, id, spec, args->u.b.size + 1, 0)) { goto exit; }
  // todo : don't expect nul terminated string.
  gen_pathname(s->keys->io->path, buffer, id);
  if (!strchr(name, '.')) {
    sen_store_spec *ss = (sen_store_spec *)spec;
    switch (ss->type) {
    case sen_store_builtin_class :
      break;
    case sen_store_structured_class :
      if (!(slot->u.c.keys = sen_sym_create(buffer,
                                            ss->u.sc.key_size,
                                            ss->u.sc.flags,
                                            ss->u.sc.encoding))) { goto exit; }
      break;
    default :
      goto exit;
    }
    slot->type = ss->type;
  } else {
    switch (*spec++) {
    case 'o' :
      slot->type = sen_store_obj_slot;
      if (!(slot->u.o.class = sen_sym_get(s->keys, spec))) { goto exit; }
      if (!(slot->u.o.ra = sen_ra_create(buffer, sizeof(sen_id)))) { goto exit; }
      break;
    case 'f' :
      slot->type = sen_store_ra_slot;
      if (!(slot->u.f.ra = sen_ra_create(buffer, atoi(spec)))) { goto exit; }
      break;
    case 'v' :
      slot->type = sen_store_ja_slot;
      if (!(slot->u.v.ja = sen_ja_create(buffer, atoi(spec)))) { goto exit; }
      break;
    case 'i' :
      slot->type = sen_store_idx_slot;
      {
        sen_store_entity *target, *l, *k;
        if (!(target = sen_store_entity_open(s, spec))) { goto exit; }
        if (!(l = sen_slot_class(s, id))) { goto exit; }
        if (!(k = sen_slot_class(s, target->id))) { goto exit; }
        if (sen_ja_put(s->values, target->id, name, strlen(name) + 1, SEN_ST_APPEND)) {
          goto exit;
        }
        slot->u.i.target = target->id;
        if (!(slot->u.i.index =
              sen_index_create_with_keys_lexicon(buffer, k->u.c.keys, l->u.c.keys, 0))) {
          goto exit;
        }
        {
          sen_store_trigger *t;
          if (!(t = SEN_MALLOC(sizeof(sen_store_trigger)))) { goto exit; }
          t->next = target->triggers;
          t->type = 1;
          t->data = (void *) slot;
          target->triggers = t;
        }
      }
      break;
    }
  }
  return slot;
exit :
  sen_set_del(s->entities, ep);
  sen_sym_del(s->keys, name);
  return NULL;
}

sen_store_entity *
sen_store_entity_by_id(sen_store *s, sen_id id)
{
  sen_store_entity *slot;
  const char *name;
  if (sen_set_at(s->entities, &id, (void **) &slot)) { return slot; }
  if (!(name = _sen_sym_key(s->keys, id))) { return NULL; }
  return sen_store_entity_open(s, name);
}

sen_store_entity *
sen_slot_class(sen_store *s, sen_id slot)
{
  int i = SEN_SYM_MAX_KEY_SIZE;
  char buf[SEN_SYM_MAX_KEY_SIZE], *dst = buf;
  const char *src = _sen_sym_key(s->keys, slot);
  while (*src != '.') {
    if (!*src || !--i) { return NULL; }
    *dst++ = *src++;
  }
  *dst = '\0';
  return sen_store_entity_open(s, buf);
}

sen_store_entity *
sen_class_slot(sen_store *s, sen_id class, const char *name)
{
  char buf[SEN_SYM_MAX_KEY_SIZE], *dst;
  const char *src = _sen_sym_key(s->keys, class);
  if (!src) { return NULL; }
  strcpy(buf, src);
  dst = buf + strlen(src);
  *dst++ = '.';
  strcpy(dst, name);
  return sen_store_entity_open(s, buf);
}

sen_rc
sen_store_entity_close(sen_store_entity *slot, int all)
{
  sen_store *s = slot->store;
  sen_store_trigger *t, *t_;
  switch (slot->type) {
  case sen_store_obj_slot :
    // sen_class_close(slot->u.o.class);
    sen_ra_close(slot->u.o.ra);
    break;
  case sen_store_ra_slot :
    sen_ra_close(slot->u.f.ra);
    break;
  case sen_store_ja_slot :
    sen_ja_close(slot->u.v.ja);
    break;
  case sen_store_idx_slot :
    sen_index_close(slot->u.i.index);
    break;
  case sen_store_structured_class :
    sen_sym_close(slot->u.c.keys);
    break;
  default :
    return sen_invalid_argument;
  }
  for (t = slot->triggers; t; t = t_) {
    t_ = t->next;
    SEN_FREE(t);
  }
  if (!all) {
    sen_set_eh *ep;
    if ((ep = sen_set_at(s->entities, &slot->id, NULL))) {
      sen_set_del(s->entities, ep);
    }
  }
  return sen_success;
}

sen_store *
sen_store_create(const char *path, int flags, sen_encoding encoding)
{
  sen_store *s;
  char buffer[PATH_MAX];
  if (strlen(path) > PATH_MAX - 14) { return NULL; }
  if (!(s = SEN_MALLOC(sizeof(sen_store)))) { return NULL; }
  if ((s->entities = sen_set_open(sizeof(sen_id), sizeof(sen_store_entity), 0))) {
    if ((s->keys = sen_sym_create(path, 0, flags, encoding))) {
      gen_pathname(path, buffer, 0);
      if ((s->values = sen_ja_create(buffer, JA_SEGMENT_SIZE))) {
        SEN_LOG(sen_log_notice, "store created (%s) flags=%x", path, s->keys->flags);
        return s;
      }
      sen_sym_close(s->keys);
    }
    sen_set_close(s->entities);
  }
  SEN_FREE(s);
  return NULL;
}

sen_store *
sen_store_open(const char *path)
{
  sen_store *s;
  char buffer[PATH_MAX];
  if (strlen(path) > PATH_MAX - 14) { return NULL; }
  if (!(s = SEN_MALLOC(sizeof(sen_store)))) { return NULL; }
  if ((s->entities = sen_set_open(sizeof(sen_id), sizeof(sen_store_entity), 0))) {
    if ((s->keys = sen_sym_open(path))) {
      gen_pathname(path, buffer, 0);
      if ((s->values = sen_ja_open(buffer))) {
        SEN_LOG(sen_log_notice, "store opened (%s) flags=%x", path, s->keys->flags);
        return s;
      }
      sen_sym_close(s->keys);
    }
    sen_set_close(s->entities);
  }
  SEN_FREE(s);
  return NULL;
}

sen_rc
sen_store_close(sen_store *s)
{
  sen_store_entity *e;
  sen_set_cursor *c;
  sen_sym_close(s->keys);
  sen_ja_close(s->values);
  if ((c = sen_set_cursor_open(s->entities))) {
    while (sen_set_cursor_next(c, NULL, (void **) &e)) {
      sen_store_entity_close(e, 1);
    }
    sen_set_cursor_close(c);
  }
  sen_set_close(s->entities);
  SEN_FREE(s);
  return sen_success;
}

inline static sen_id
slot_class_at(sen_store *s, sen_id slot, const char *key, const sen_store_obj *args)
{
  sen_store_entity *c;
  if (!(c = sen_slot_class(s, slot))) { return 0; }
  return args ? sen_sym_get(c->u.c.keys, key) : sen_sym_at(c->u.c.keys, key);
}

inline static sen_rc
slot_value_obj(sen_store_entity *slot, sen_id id, const sen_store_obj *args, sen_store_obj *res)
{
  sen_rc rc = sen_success;
  sen_id *ip = args ? sen_ra_get(slot->u.o.ra, id) : sen_ra_at(slot->u.o.ra, id);
  if (!ip) {
    res->type = sen_store_nil;
    goto exit;
  }
  if (args) {
    switch (args->type) {
    case sen_store_object :
      if (args->class != slot->u.o.class) {
        rc = sen_invalid_argument;
        res->type = sen_store_nil;
        goto exit;
      }
      *ip = args->u.o.self;
      break;
    case sen_store_bulk :
      {
        sen_store_entity *c;
        if (!(c = sen_store_entity_by_id(slot->store, slot->u.o.class)) ||
            !(*ip = sen_sym_get(c->u.c.keys, args->u.b.value))) {
          rc = sen_invalid_argument;
          res->type = sen_store_nil;
          goto exit;
        }
      }
      break;
    default :
      rc = sen_invalid_argument;
      res->type = sen_store_nil;
      goto exit;
      break;
    }

  }
  res->type = (*ip) ? sen_store_object : sen_store_nil;
  res->class = slot->u.o.class;
  res->u.o.self = *ip;
exit :
  return rc;
}

inline static sen_rc
slot_value_ra(sen_store_entity *slot, sen_id id, const sen_store_obj *args, sen_store_obj *res)
{
  sen_rc rc = sen_success;
  void *vp = args ? sen_ra_get(slot->u.f.ra, id) : sen_ra_at(slot->u.f.ra, id);
  if (!vp) {
    res->type = sen_store_nil;
    rc = sen_invalid_argument;
    goto exit;
  }
  if (args) {
    switch (args->type) {
    case sen_store_bulk :
      if (sizeof(int32_t) == slot->u.f.ra->header->element_size) {
        int32_t i = sen_atoi(args->u.b.value,
                             (char *)args->u.b.value + args->u.b.size, NULL);
        memcpy(vp, &i, sizeof(int32_t));
      } else {
        if (args->u.b.size != slot->u.f.ra->header->element_size) {
          res->type = sen_store_nil;
          rc = sen_invalid_argument;
          goto exit;
        }
        memcpy(vp, args->u.b.value, args->u.b.size);
      }
      break;
    case sen_store_int :
      if (sizeof(int32_t) != slot->u.f.ra->header->element_size) {
        res->type = sen_store_nil;
        rc = sen_invalid_argument;
        goto exit;
      }
      memcpy(vp, &args->u.i.i, sizeof(int32_t));
      break;
    default :
      res->type = sen_store_nil;
      rc = sen_invalid_argument;
      goto exit;
    }
  }
  if (slot->u.f.ra->header->element_size == sizeof(int32_t)) {
    res->type = sen_store_int;
    memcpy(&res->u.i.i, vp, sizeof(int32_t));
  } else {
    res->type = sen_store_bulk;
    res->u.b.size = slot->u.f.ra->header->element_size;
    res->u.b.value = vp;
  }
exit :
  return rc;
}

inline static sen_rc
slot_value_ja(sen_store_entity *slot, sen_id id, const sen_store_obj *args, sen_store_obj *res)
{
  sen_rc rc = sen_success;
  uint32_t value_len;
  const void *vp = sen_ja_ref(slot->u.v.ja, id, &value_len);
  if (args) {
    sen_store_trigger *t;
    // todo : support append and so on..
    if (args->type != sen_store_bulk) {
      rc = sen_invalid_argument;
      goto exit;
    }
    for (t = slot->triggers; t; t = t->next) {
      if (t->type == 1) {
        sen_store_entity *index = (sen_store_entity *)t->data;
        sen_log("updating %d => %d", value_len, args->u.b.size);
        sen_index_upd(index->u.i.index, _sen_sym_key(index->u.i.index->keys, id),
                      vp, value_len, args->u.b.value, args->u.b.size);
      }
    }
    res->type = sen_store_nil;
    rc = sen_ja_put(slot->u.v.ja, id, args->u.b.value, args->u.b.size, 0);
  } else {
    if (vp) {
      res->type = sen_store_bulk;
      res->u.b.size = value_len;
      res->u.b.value = vp;
    } else {
      res->type = sen_store_nil;
    }
  }
exit :
  return rc;
}

inline static sen_store_obj *
obj_new(sen_store_ctx *c)
{
  sen_store_obj *o;
  sen_set_get(c->objects, &c->seqno, (void **) &o);
  c->seqno++;
  return o;
}

inline static sen_store_obj *
obj_clear(sen_store_obj *o)
{
  switch (o->type) {
  case sen_store_records :
    if (o->u.r.records) { sen_records_close(o->u.r.records); }
    break;
  case sen_store_str :
    if (o->u.s.value) { SEN_FREE(o->u.s.value); }
    break;
  default :
    break;
  }
  return o;
}

#define obj_res(c,r,s) ((r) ? (*(r) ? obj_clear(*(r)) : (*(r) = obj_new(c))) : &(s))

sen_rc
sen_store_at(sen_store_ctx *c, sen_store_obj *obj,
             const char *key, const sen_store_obj *args, sen_store_obj **res)
{
  sen_rc rc = sen_success;
  sen_store *s = c->store;
  sen_store_obj rs;
  if (obj) {
    sen_id id;
    sen_store_entity *slot;
    switch (obj->type) {
    case sen_store_structured_class :
      if (!(slot = sen_store_entity_by_id(s, obj->u.o.self))) {
        rc = sen_invalid_argument;
      } else {
        rc = sen_class_at(slot, key, args ? 1 : 0, obj_res(c, res, rs));
      }
      break;
    case sen_store_obj_slot :
      if (!(id = slot_class_at(s, obj->u.o.self, key, args))) {
        rc = sen_other_error;
        break;
      }
      if (!(slot = sen_store_entity_by_id(s, obj->u.o.self))) {
        rc = sen_invalid_argument;
        break;
      }
      rc = slot_value_obj(slot, id, args, obj_res(c, res, rs));
      break;
    case sen_store_ra_slot :
      if (!(id = slot_class_at(s, obj->u.o.self, key, args))) {
        rc = sen_other_error;
        break;
      }
      if (!(slot = sen_store_entity_by_id(s, obj->u.o.self))) {
        rc = sen_invalid_argument;
        break;
      }
      rc = slot_value_ra(slot, id, args, obj_res(c, res, rs));
      break;
    case sen_store_ja_slot :
      if (!(id = slot_class_at(s, obj->u.o.self, key, args))) {
        rc = sen_other_error;
        break;
      }
      if (!(slot = sen_store_entity_by_id(s, obj->u.o.self))) {
        rc = sen_invalid_argument;
        break;
      }
      rc = slot_value_ja(slot, id, args, obj_res(c, res, rs));
      break;
    case sen_store_idx_slot :
      // todo : simply pickup a term
      if (!(slot = sen_store_entity_by_id(s, obj->u.o.self))) {
        rc = sen_invalid_argument;
        break;
      }
      if (res) {
        sen_store_obj *rp = obj_res(c, res, rs);
        sen_store_entity *class = sen_slot_class(c->store, slot->u.i.target);
        rp->u.r.records = sen_index_sel(slot->u.i.index, key, strlen(key));
        rp->type = rp->u.r.records ? sen_store_records : sen_store_nil;
        rp->class = class->id;
      }
      break;
    case sen_store_records :
      // todo : break;
    default :
      rc = sen_invalid_argument;
      break;
    }
  } else {
    sen_store_entity *slot;
    if (args && args->type == sen_store_bulk) {
      slot = sen_store_entity_create(s, key, args);
      // todo : remove_entity if args->type == sen_store_nil
    } else {
      slot = sen_store_entity_open(s, key);
    }
    if (slot) {
      if (res) {
        sen_store_obj *rp = obj_res(c, res, rs);
        rp->type = slot->type;
        rp->class = 0;
        rp->u.o.self = slot->id;
      }
      rc = sen_success;
    } else {
      if (res) {
        sen_store_obj *rp = obj_res(c, res, rs);
        rp->type = sen_store_nil;
      }
      rc = sen_other_error;
    }
  }
  return rc;
}

sen_rc
sen_store_send(sen_store_ctx *c, sen_store_obj *obj,
               const char *message, const sen_store_obj *args, sen_store_obj **res)
{
  sen_rc rc = sen_success;
  sen_store *s = c->store;
  sen_store_obj rs, *rp = obj_res(c, res, rs);
  if (obj) {
    sen_store_entity *slot;
    switch (obj->type) {
    case sen_store_structured_class :
      slot = sen_class_slot(s, obj->u.o.self, message);
      if (slot) {
        rp->type = slot->type;
        rp->u.o.self = slot->id;
      } else {
        rp->type = sen_store_nil;
        rc = sen_invalid_argument;
      }
      break;
    case sen_store_object :
      slot = sen_class_slot(s, obj->class, message);
      switch (slot->type) {
      case sen_store_obj_slot :
        rc = slot_value_obj(slot, obj->u.o.self, args, rp);
        break;
      case sen_store_ra_slot :
        rc = slot_value_ra(slot, obj->u.o.self, args, rp);
        break;
      case sen_store_ja_slot :
        rc = slot_value_ja(slot, obj->u.o.self, args, rp);
        break;
      case sen_store_idx_slot :
        {
          sen_store_entity *class = sen_slot_class(c->store, slot->u.i.target);
          const char *key = _sen_sym_key(slot->u.i.index->lexicon, obj->u.o.self);
          if (!key) {
            rp->type = sen_store_nil;
            rc = sen_invalid_argument;
            break;
          }
          rp->u.r.records = sen_index_sel(slot->u.i.index, key, strlen(key));
          rp->type = rp->u.r.records ? sen_store_records : sen_store_nil;
          rp->class = class->id;
        }
        break;
      default :
        rp->type = sen_store_nil;
        rc = sen_invalid_argument;
        break;
      }
      break;
    default :
      rp->type = sen_store_nil;
      rc = sen_invalid_argument;
      break;
    }
  } else {
    rp->type = sen_store_nil;
    rc = sen_invalid_argument;
  }
  return rc;
}

sen_store_obj *sen_store_ctx_at(sen_store_ctx *c, const char *key, sen_store_obj *value);

static sen_store_obj *
_native_method_set(sen_store_ctx *c, sen_store_obj *args)
{
  sen_store_obj *res = NULL;
  sen_store_obj *key = car(args);
  sen_store_obj *val = cadr(args);
  if (key->type == sen_store_bulk || key->type == sen_store_str) {
    if (val) { val = sen_store_ctx_eval(c, val); }
    res = sen_store_ctx_at(c, key->u.b.value, val);
  }
  return res;
}

static sen_store_obj *
_native_method_defclass(sen_store_ctx *c, sen_store_obj *args)
{
  sen_store_spec spec;
  sen_store_obj arg;
  sen_store_obj *res = NULL;
  sen_store_obj *car = car(args);
  sen_store_obj *cdr = cdr(args);
  if (car->type == sen_store_bulk) {
    if (cdr && cdr->type == sen_store_list) {
      spec.type = sen_store_structured_class;
      // todo : check more strictly
      spec.u.sc.key_size = car(cdr)->u.i.i;
      cdr = cdr(cdr);
      spec.u.sc.flags = car(cdr)->u.i.i;
      cdr = cdr(cdr);
      spec.u.sc.encoding = car(cdr)->u.i.i;
    } else {
      spec.type = sen_store_builtin_class;
      // spec.u.bc.element_size = atoi(tokens[2]);
    }
    arg.type = sen_store_bulk;
    arg.u.b.value = &spec;
    arg.u.b.size = sizeof spec;
    sen_store_at(c, NULL, car->u.b.value, &arg, &res);
  }
  return res;
}

static sen_store_obj *
_native_method_defslot(sen_store_ctx *c, sen_store_obj *args)
{
  sen_store_obj *res = NULL;
  sen_store_obj *car = car(args);
  if (car->type == sen_store_bulk) {
    sen_store_at(c, NULL, car->u.b.value, cadr(args), &res);
  }
  return res;
}

static sen_store_obj *
_native_method_at(sen_store_ctx *c, sen_store_obj *args)
{
  sen_store_obj *res = NULL;
  sen_store_obj *arg = cdr(args);
  if (arg->type == sen_store_list) {
    sen_store_obj *msg = car(arg);
    if (msg && msg->type == sen_store_bulk) {
      sen_store_at(c, sen_store_ctx_eval(c, car(args)),
                   msg->u.b.value, cadr(arg), &res);
    }
  }
  return res;
}

static sen_store_obj *
_native_method_send(sen_store_ctx *c, sen_store_obj *args)
{
  sen_store_obj *res = NULL;
  sen_store_obj *arg = cdr(args);
  if (arg->type == sen_store_list) {
    sen_store_obj *msg = car(arg);
    if (msg && msg->type == sen_store_bulk) {
      sen_store_send(c, sen_store_ctx_eval(c, car(args)),
                     msg->u.b.value, cdr(arg), &res);
    }
  }
  return res;
}

const char *
_sen_store_obj_key(sen_store *store, sen_store_obj *obj)
{
  sen_store_entity *cls;
  switch (obj->type) {
  case sen_store_object :
    if (!(cls = sen_store_entity_by_id(store, obj->class))) { return NULL; }
    return _sen_sym_key(cls->u.c.keys, obj->u.o.self);
  case sen_store_builtin_class :
  case sen_store_structured_class :
  case sen_store_obj_slot :
  case sen_store_ra_slot :
  case sen_store_ja_slot :
  case sen_store_idx_slot :
    return _sen_sym_key(store->keys, obj->u.o.self);
  default :
    return NULL;
  }
}

static sen_store_obj *
_native_method__put(sen_store_ctx *c, sen_store_obj *args)
{
  if (args->type == sen_store_bulk) {
    fwrite(args->u.b.value, 1, args->u.b.size, stdout);
    putchar('\n');
  }
  return NULL;
}

// from index.c
typedef struct {
  int score;
  int n_subrecs;
  byte subrecs[1];
} recinfo;

inline static sen_store_method_func *
name2method(sen_store_ctx *c, const char *name)
{
  sen_store_obj *obj;
  if (!(obj = sen_store_ctx_at(c, name, NULL)) ||
      !obj->type == sen_store_native_method) { return NULL; }
  return obj->u.n.func;
}

inline static sen_store_obj *
rbuf2obj(sen_rbuf *buf, sen_store_obj *obj)
{
  obj->type = sen_store_bulk;
  obj->u.b.value = buf->head;
  obj->u.b.size = buf->curr - buf->head;
  return obj;
}

static sen_store_obj *
_native_method_put(sen_store_ctx *c, sen_store_obj *args)
{
  sen_id *rp;
  recinfo *ri;
  sen_rbuf buf;
  sen_records *r;
  const sen_recordh *rh;
  int i, ofrat = 0, limit = 10;
  sen_store_obj obj, bobj, *rec, *s, *slots, *res = NULL;
  sen_store_method_func *_put;
  if (!(_put = name2method(c, "_put"))) { return NULL; }
  if (sen_rbuf_init(&buf, -1)) { return NULL; }
  rec = sen_store_ctx_eval(c, car(args));
  if (!rec || rec->type != sen_store_records) { goto exit; }
  r = rec->u.r.records;
  sen_rbuf_itoa(&buf, sen_records_nhits(r));
  _put(c, rbuf2obj(&buf, &bobj));
  args = cdr(args);
  if (!args || args->type != sen_store_list) { goto exit; }
  slots = car(args);
  if (!slots || slots->type != sen_store_list) { goto exit; }
  // check && compile slots
  args = cdr(args);
  if (car(args) && car(args)->type == sen_store_int) {
    ofrat = car(args)->u.i.i;
  }
  args = cdr(args);
  if (car(args) && car(args)->type == sen_store_int) {
    limit = car(args)->u.i.i;
  }
  sen_records_rewind(r);
  for (i = 0; i < ofrat; i++) {
    if (!sen_records_next(r, NULL, 0, NULL)) { goto exit; }
  }
  obj.type = sen_store_object;
  obj.class = rec->class;
  for (i = 0; i < limit; i++) {
    if (!sen_records_next(r, NULL, 0, NULL) ||
        !(rh = sen_records_curr_rec(r)) ||
        sen_set_element_info(r->records, rh, (void **)&rp, (void **)&ri)) { goto exit; }
    obj.u.o.self = *rp;
    SEN_RBUF_REWIND(&buf);
    for (s = slots;;) {
      if (car(s) && car(s)->type == sen_store_bulk) {
        char *value = (char *)car(s)->u.b.value;
        if (value[0] == ':') {
          switch (value[1]) {
          case 'k' :
            {
              const char *key = _sen_store_obj_key(c->store, &obj);
              if (key) { SEN_RBUF_PUTS(&buf, key); }
            }
            break;
          case 's' :
            sen_rbuf_itoa(&buf, ri->score);
            break;
          default :
            break;
          }
        } else {
          sen_store_send(c, &obj, value, NULL, &res);
          switch(res->type) {
          case sen_store_object :
            {
              const char *key = _sen_store_obj_key(c->store, res);
              if (key) { SEN_RBUF_PUTS(&buf, key); }
            }
            break;
          case sen_store_bulk :
          case sen_store_str :
            sen_rbuf_write(&buf, res->u.b.value, res->u.b.size);
            break;
          case sen_store_int :
            sen_rbuf_itoa(&buf, res->u.i.i);
            break;
          default :
            break;
          }
        }
      }
      s = cdr(s);
      if (!s || s->type != sen_store_list) { break; }
      SEN_RBUF_PUTC(&buf, '\t');
    }
    _put(c, rbuf2obj(&buf, &bobj));
  }
exit :
  return NULL;
}

static sen_store_obj *
_native_method_quote(sen_store_ctx *c, sen_store_obj *args)
{
  return car(args);
}

#define MAXSLOTS 0x100
#define BUFSIZE 0x100000

static sen_store_obj *
_native_method_ins(sen_store_ctx *c, sen_store_obj *args)
{
  int i, n;
  sen_store_obj *res = NULL;
  sen_store_obj *s, *car, val;
  // sen_store_entity *class;
  char buf[BUFSIZE], *tokbuf[MAXSLOTS];
  if (!args || args->type != sen_store_list) { goto exit; }
  car = sen_store_ctx_eval(c, car(args));
  if (!car || car->type != sen_store_structured_class) { goto exit; }
  // if (!(class = sen_store_entity_by_id(c->store, car->u.o.self))) { goto exit; }
  args = cdr(args);
  for (s = args, n = 0; s && s->type == sen_store_list; s = cdr(s), n++) {
    if (car(s)->type != sen_store_bulk) { goto exit; }
    sen_store_send(c, car, car(s)->u.b.value, NULL, &s->u.l.car);
    if (!car(s) || car(s)->type == sen_store_nil) { goto exit; }
  }
  res = obj_new(c);
  res->type = sen_store_int;
  res->u.i.i = 0;
  val.type = sen_store_bulk;
  while (fgets(buf, BUFSIZE, stdin)) {
    size_t len = strlen(buf);
    if (buf[len - 1] == '\n') { buf[len - 1] = '\0'; }
    if (sen_str_tok(buf, "\t", tokbuf, MAXSLOTS, NULL) != n + 1) { continue; }
    // if (sen_class_at(class, tokbuf[0], 1, &res)) { continue; }
    for (s = args, i = 1; i < n + 1; s = cdr(s), i++) {
      val.u.b.value = tokbuf[i];
      val.u.b.size = strlen(tokbuf[i]);
      sen_store_at(c, car(s), tokbuf[0], &val, NULL);
    }
    res->u.i.i++;
  }
exit :
  return res;
}

static sen_store_obj *
_native_method_sort(sen_store_ctx *c, sen_store_obj *args)
{
  int limit = 10;
  sen_sort_optarg arg;
  sen_store_obj *res = NULL, *rec = sen_store_ctx_eval(c, car(args));
  if (rec->type != sen_store_records) { goto exit; }
  arg.compar = NULL;
  arg.compar_arg = (void *)(intptr_t)rec->u.r.records->records->key_size;
  arg.mode = sen_sort_descending;
  args = cdr(args);
  if (car(args) && car(args)->type == sen_store_int) {
    limit = car(args)->u.i.i;
  }
  args = cdr(args);
  if (car(args) && car(args)->type == sen_store_bulk) {
    const char *str = car(args)->u.b.value;
    if (*str == 'a') {
      arg.mode = sen_sort_ascending;
    }
  }
  if (!sen_records_sort(rec->u.r.records, limit, &arg)) { res = rec; }
exit :
  return res;
}

static sen_store_obj *
_native_method_sel(sen_store_ctx *c, sen_store_obj *args)
{
  sen_records *r;
  sen_store_entity *cls, *slot;
  sen_id id = SEN_SYM_NIL;
  sen_store_obj o, *res = NULL, *cl = sen_store_ctx_eval(c, car(args));
  if (cl->type != sen_store_structured_class ||
      !(cls = sen_store_entity_by_id(c->store, cl->u.o.self))) { goto exit; }
  args = cadr(args);
  if (!args || args->type != sen_store_list) { goto exit; }
  if (car(args)->type != sen_store_bulk) { goto exit; } // must be "eq"
  args = cdr(args);
  if (!args || car(args)->type != sen_store_bulk) { goto exit; } // must be slot name
  if (!(slot = sen_class_slot(c->store, cls->id, car(args)->u.b.value)) ||
      slot->type != sen_store_ra_slot) { // todo : support other types
    goto exit;
  }
  args = cadr(args);
  if (!args || args->type != sen_store_int) { goto exit; } // todo : support other types
  if (!(r = sen_records_open(sen_rec_document, sen_rec_none, 0))) { goto exit; }
  r->keys = cls->u.c.keys;
  res = obj_new(c);
  res->type = sen_store_records;
  res->class = cls->id;
  res->u.r.records = r;
  // while ((id = sen_sym_next(cls->u.c.keys, id)))
  // todo : delete check
  for (id = 1; id <= slot->u.f.ra->header->curr_max; id++) {
    if (slot_value_ra(slot, id, NULL, &o)) { break; }
    if (o.u.i.i == args->u.i.i) {
      sen_set_get(r->records, &id, NULL);
    }
  }
exit :
  return res;
}

void
sen_store_ctx_add_native_method(sen_store_ctx *c, const char *name,
                                sen_store_method_func *func)
{
  sen_store_obj *o = obj_new(c);
  o->type = sen_store_native_method;
  o->u.n.func = func;
  sen_store_ctx_at(c, name, o);
}

sen_store_ctx *
sen_store_ctx_open(sen_store *s, sen_store_parser *parser)
{
  sen_store_ctx *c = SEN_MALLOC(sizeof(sen_store_ctx));
  if (c) {
    c->parser = parser;
    c->doing = NULL;
    c->mode = 0;
    c->opaque = NULL;
    c->seqno = 0;
    c->store = s;
    if (!(c->objects = sen_set_open(sizeof(int), sizeof(sen_store_obj), 0))) {
      SEN_FREE(c);
      c = NULL;
      goto exit;
    }
    if (!(c->bindings = sen_set_open(0, sizeof(sen_store_obj *), 0))) {
      sen_set_close(c->objects);
      SEN_FREE(c);
      c = NULL;
      goto exit;
    }
    sen_store_ctx_add_native_method(c, "set", _native_method_set);
    sen_store_ctx_add_native_method(c, "defclass", _native_method_defclass);
    sen_store_ctx_add_native_method(c, "defslot", _native_method_defslot);
    sen_store_ctx_add_native_method(c, "at", _native_method_at);
    sen_store_ctx_add_native_method(c, "send", _native_method_send);
    sen_store_ctx_add_native_method(c, "put", _native_method_put);
    sen_store_ctx_add_native_method(c, "quote", _native_method_quote);
    sen_store_ctx_add_native_method(c, "ins", _native_method_ins);
    sen_store_ctx_add_native_method(c, "sort", _native_method_sort);
    sen_store_ctx_add_native_method(c, "sel", _native_method_sel);
    sen_store_ctx_add_native_method(c, "_put", _native_method__put);
  }
exit :
  return c;
}

sen_rc
sen_store_ctx_close(sen_store_ctx *c)
{
  sen_store_obj *o;
  sen_set_cursor *sc;
  if ((sc = sen_set_cursor_open(c->objects))) {
    while (sen_set_cursor_next(sc, NULL, (void **) &o)) { obj_clear(o); }
    sen_set_cursor_close(sc);
  }
  sen_set_close(c->objects);
  sen_set_close(c->bindings);
  SEN_FREE(c);
  return sen_success;
}

sen_store_obj *
sen_store_ctx_at(sen_store_ctx *c, const char *key, sen_store_obj *value)
{
  sen_store_obj **res;
  // todo : delete
  if (value) {
    sen_set_get(c->bindings, key, (void **) &res);
    *res = value;
  } else {
    if (!sen_set_at(c->bindings, key, (void **) &res)) {
      return NULL;
    }
  }
  return *res;
}

sen_store_obj *
sen_store_ctx_eval(sen_store_ctx *c, sen_store_obj *expr)
{
  sen_store_obj *res = expr;
  if (!expr) { return NULL; }
  switch (expr->type) {
  case sen_store_nil :
    break;
  case sen_store_builtin_class :
    break;
  case sen_store_structured_class :
    break;
  case sen_store_obj_slot :
    break;
  case sen_store_ra_slot :
    break;
  case sen_store_ja_slot :
    break;
  case sen_store_idx_slot :
    break;
  case sen_store_object :
    break;
  case sen_store_records :
    break;
  case sen_store_bulk :
    res = sen_store_ctx_at(c, expr->u.b.value, NULL);
    if (!res) {
      sen_store_at(c, NULL, expr->u.b.value, NULL, &res);
    }
    break;
  case sen_store_list :
    {
      sen_store_obj *car = sen_store_ctx_eval(c, car(expr));
      sen_store_obj *cdr = cdr(expr);
      if (!car) { res = NULL; break; }
      if (car->type == sen_store_native_method) {
        c->doing = car->u.n.func;
        res = c->doing(c, cdr);
      }
    }
    break;
  case sen_store_str :
    res = sen_store_ctx_at(c, expr->u.s.value, NULL);
    if (!res) {
      sen_store_at(c, NULL, expr->u.s.value, NULL, &res);
    }
    break;
  case sen_store_native_method :
    break;
  case sen_store_method :
    break;
  case sen_store_int :
    break;
  }
  return res;
}

sen_store_obj *
sen_store_ctx_feed(sen_store_ctx *c, const char *str, uint32_t str_size, int mode)
{
  sen_store_obj *r;
  if ((mode & SEN_STORE_CTX_HEAD) && c->doing) {
    c->mode = SEN_STORE_CTX_TAIL;
    r = c->doing(c, NULL);
    c->doing = NULL;
  }
  c->mode = mode;
  if (c->doing) {
    sen_store_obj arg;
    arg.type = sen_store_bulk;
    arg.u.b.value = str;
    arg.u.b.size = str_size;
    r = c->doing(c, &arg);
  } else {
    sen_store_obj *expr = c->parser(c, str, str_size);
    r = sen_store_ctx_eval(c, expr);
  }
  if (c->mode & SEN_STORE_CTX_TAIL) { c->doing = NULL; }
  return r;
}

/**** sexp parser ****/

typedef struct {
  char *cur;
  char *str_end;
  char *last;
  sen_encoding encoding;
} parseinfo;

typedef sen_store_obj cons;

inline static cons *
cons_new(sen_store_ctx *c)
{
  cons *o = obj_new(c);
  o->type = sen_store_list;
  o->u.l.car = NULL;
  o->u.l.cdr = NULL;
  return o;
}

inline static cons *
token_new(sen_store_ctx *c)
{
  cons *o = obj_new(c);
  o->type = sen_store_bulk;
  return o;
}

static cons *get_expr(sen_store_ctx *q);

inline static void
skip_space(sen_store_ctx *c)
{
  unsigned int len;
  parseinfo *i = (parseinfo *) c->opaque;
  while (i->cur < i->str_end && sen_isspace(i->cur, i->encoding)) {
    /* null check and length check */
    if (!(len = sen_str_charlen_nonnull(i->cur, i->str_end, i->encoding))) {
      i->cur = i->str_end;
      break;
    }
    i->cur += len;
  }
}

inline static cons *
get_phrase(sen_store_ctx *c)
{
  cons *token;
  parseinfo *i = (parseinfo *) c->opaque;
  char *start = i->cur, *end;
  for (end = i->cur;; end++) {
    if (end >= i->str_end) {
      i->cur = end;
      break;
    }
    if (*end == SEN_QUERY_QUOTER) {
      i->cur = end + 1;
      break;
    }
  }
  if (start < end && (token = token_new(c))) {
    token->u.b.value = start;
    token->u.b.size = end - start;
    i->last = end;
    return token;
  }
  return NULL;
}

inline static cons *
get_word(sen_store_ctx *c)
{
  cons *token;
  parseinfo *i = (parseinfo *) c->opaque;
  char *start = i->cur, *end;
  unsigned int len;
  for (end = i->cur;; ) {
    /* null check and length check */
    if (!(len = sen_str_charlen_nonnull(end, i->str_end, i->encoding))) {
      i->cur = i->str_end;
      break;
    }
    if (sen_isspace(end, i->encoding) ||
        *end == SEN_QUERY_PARENR) {
      i->cur = end;
      break;
    }
    end += len;
  }
  if (start < end && (token = token_new(c))) {
    token->u.b.value = start;
    token->u.b.size = end - start;
    i->last = end;
    return token;
  }
  return NULL;
}

inline static cons *
get_int(sen_store_ctx *c)
{
  cons *token = obj_new(c);
  parseinfo *i = (parseinfo *) c->opaque;
  token->type = sen_store_int;
  token->u.i.i = sen_atoi(i->cur, i->str_end, (const char **) &i->cur);
  return token;
}

inline static cons *
get_token(sen_store_ctx *c)
{
  cons *token = NULL;
  parseinfo *i = (parseinfo *) c->opaque;
  char cur;
  skip_space(c);
  if (i->cur >= i->str_end) { return NULL; }
  cur = *i->cur;
  if (i->last) { *i->last = '\0'; }
  switch (cur) {
  case '\0' :
    return NULL;
  case SEN_QUERY_PARENR :
    i->cur++;
    return NULL;
  case SEN_QUERY_QUOTEL :
    i->cur++;
    token = cons_new(c);
    token->u.l.car = get_phrase(c);
    break;
  case SEN_QUERY_PARENL :
    i->cur++;
    token = cons_new(c);
    token->u.l.car = get_expr(c);
    break;
  case '0' :
  case '1' :
  case '2' :
  case '3' :
  case '4' :
  case '5' :
  case '6' :
  case '7' :
  case '8' :
  case '9' :
    token = cons_new(c);
    token->u.l.car = get_int(c);
    break;
  case '-' :
    token = cons_new(c);
    if ('0' <= *(i->cur + 1) && *(i->cur + 1) <= '9') {
      token->u.l.car = get_int(c);
    } else {
      token->u.l.car = get_word(c);
    }
    break;
  default :
    token = cons_new(c);
    token->u.l.car = get_word(c);
    break;
  }
  return token;
}

static cons *
get_expr(sen_store_ctx *c)
{
  cons *head, *expr, *last;
  if (!(head = get_token(c))) {
    return NULL;
  }
  for (last = head; (expr = get_token(c)); last = expr) {
    last->u.l.cdr = expr;
  }
  return head;
}

sen_store_obj *
sen_store_sexp_parser(sen_store_ctx *c, const char *str, uint32_t str_size)
{
  parseinfo info;
  sen_store_obj *o = obj_new(c);
  o->type = sen_store_str;
  o->u.s.size = str_size + 1;
  o->u.s.value = SEN_MALLOC(o->u.s.size);
  memcpy(o->u.s.value, str, str_size);
  o->u.s.value[str_size] = '\0';
  info.cur = o->u.s.value;
  info.str_end = info.cur + str_size;
  info.last = NULL;
  info.encoding = c->store->keys->encoding;
  c->opaque = &info;
  o = get_expr(c);
  if (info.last) { *info.last = '\0'; }
  return o;
}

/**** vgram ****/

static int len_sum = 0;
static int img_sum = 0;
static int simple_sum = 0;
static int skip_sum = 0;

sen_vgram *
sen_vgram_create(const char *path)
{
  sen_vgram *s;
  if (!(s = SEN_MALLOC(sizeof(sen_vgram)))) { return NULL; }
  s->vgram = sen_sym_create(path, sizeof(sen_id) * 2, 0, sen_enc_none);
  if (!s->vgram) {
    SEN_FREE(s);
    return NULL;
  }
  return s;
}

sen_vgram *
sen_vgram_open(const char *path)
{
  sen_vgram *s;
  if (!(s = SEN_MALLOC(sizeof(sen_vgram)))) { return NULL; }
  s->vgram = sen_sym_open(path);
  if (!s->vgram) {
    SEN_FREE(s);
    return NULL;
  }
  return s;
}

sen_vgram_buf *
sen_vgram_buf_open(size_t len)
{
  sen_vgram_buf *b;
  if (!(b = SEN_MALLOC(sizeof(sen_vgram_buf)))) { return NULL; }
  b->len = len;
  b->tvs = b->tvp = SEN_MALLOC(sizeof(sen_id) * len);
  if (!b->tvp) { SEN_FREE(b); return NULL; }
  b->tve = b->tvs + len;
  b->vps = b->vpp = SEN_MALLOC(sizeof(sen_vgram_vnode) * len * 2);
  if (!b->vpp) { SEN_FREE(b->tvp); SEN_FREE(b); return NULL; }
  b->vpe = b->vps + len;
  return b;
}

sen_rc
sen_vgram_buf_add(sen_vgram_buf *b, sen_id tid)
{
  uint8_t dummybuf[8], *dummyp;
  if (b->tvp < b->tve) { *b->tvp++ = tid; }
  dummyp = dummybuf;
  SEN_B_ENC(tid, dummyp);
  simple_sum += dummyp - dummybuf;
  return sen_success;
}

typedef struct {
  sen_id vid;
  sen_id tid;
} vgram_key;

sen_rc
sen_vgram_update(sen_vgram *vgram, sen_id rid, sen_vgram_buf *b, sen_set *terms)
{
  sen_inv_updspec **u;
  if (b && b->tvs < b->tvp) {
    sen_id *t0, *tn;
    for (t0 = b->tvs; t0 < b->tvp - 1; t0++) {
      sen_vgram_vnode *v, **vp;
      sen_set_at(terms, t0, (void **) &u);
      vp = &(*u)->vnodes;
      for (tn = t0 + 1; tn < b->tvp; tn++) {
        for (v = *vp; v && v->tid != *tn; v = v->cdr) ;
        if (!v) {
          if (b->vpp < b->vpe) {
            v = b->vpp++;
          } else {
            // todo;
            break;
          }
          v->car = NULL;
          v->cdr = *vp;
          *vp = v;
          v->tid = *tn;
          v->vid = 0;
          v->freq = 0;
          v->len = tn - t0;
        }
        v->freq++;
        if (v->vid) {
          vp = &v->car;
        } else {
          break;
        }
      }
    }
    {
      sen_set *th = sen_set_open(sizeof(sen_id), sizeof(int), 0);
      if (!th) { return sen_memory_exhausted; }
      if (t0 == b->tvp) { SEN_LOG(sen_log_debug, "t0 == tvp"); }
      for (t0 = b->tvs; t0 < b->tvp; t0++) {
        sen_id vid, vid0 = *t0, vid1 = 0;
        sen_vgram_vnode *v, *v2 = NULL, **vp;
        sen_set_at(terms, t0, (void **) &u);
        vp = &(*u)->vnodes;
        for (tn = t0 + 1; tn < b->tvp; tn++) {
          for (v = *vp; v; v = v->cdr) {
            if (!v->vid && (v->freq < 2 || v->freq * v->len < 4)) {
              *vp = v->cdr;
              v->freq = 0;
            }
            if (v->tid == *tn) { break; }
            vp = &v->cdr;
          }
          if (v) {
            if (v->freq) {
              v2 = v;
              vid1 = vid0;
              vid0 = v->vid;
            }
            if (v->vid) {
              vp = &v->car;
              continue;
            }
          }
          break;
        }
        if (v2) {
          if (!v2->vid) {
            vgram_key key;
            key.vid = vid1;
            key.tid = v2->tid;
            v2->vid = sen_sym_get(vgram->vgram, (char *)&key);
          }
          vid = *t0 = v2->vid * 2 + 1;
          memset(t0 + 1, 0, sizeof(sen_id) * v2->len);
          t0 += v2->len;
        } else {
          vid = *t0 *= 2;
        }
        {
          int *tf;
          sen_set_get(th, &vid, (void **) &tf);
          (*tf)++;
        }
      }
      if (!th->n_entries) { SEN_LOG(sen_log_debug, "th->n_entries == 0"); }
      {
        int j = 0;
        int skip = 0;
        sen_set_eh *ehs, *ehp, *ehe;
        sen_set_sort_optarg arg;
        uint8_t *ps = SEN_MALLOC(b->len * 2), *pp, *pe;
        if (!ps) {
          sen_set_close(th);
          return sen_memory_exhausted;
        }
        pp = ps;
        pe = ps + b->len * 2;
        arg.mode = sen_sort_descending;
        arg.compar = NULL;
        arg.compar_arg = (void *)(intptr_t)sizeof(sen_id);
        arg.compar_arg0 = NULL;
        ehs = sen_set_sort(th, 0, &arg);
        if (!ehs) {
          SEN_FREE(ps);
          sen_set_close(th);
          return sen_memory_exhausted;
        }
        SEN_B_ENC(th->n_entries, pp);
        for (ehp = ehs, ehe = ehs + th->n_entries; ehp < ehe; ehp++, j++) {
          int *id = (int *)SEN_SET_INTVAL(*ehp);
          SEN_B_ENC(*SEN_SET_INTKEY(*ehp), pp);
          *id = j;
        }
        for (t0 = b->tvs; t0 < b->tvp; t0++) {
          if (*t0) {
            int *id;
            if (!sen_set_at(th, t0, (void **) &id)) {
              SEN_LOG(sen_log_error, "lookup error (%d)", *t0);
            }
            SEN_B_ENC(*id, pp);
          } else {
            skip++;
          }
        }
        len_sum += b->len;
        img_sum += pp - ps;
        skip_sum += skip;
        SEN_FREE(ehs);
        SEN_FREE(ps);
      }
      sen_set_close(th);
    }
  }
  return sen_success;
}

sen_rc
sen_vgram_buf_close(sen_vgram_buf *b)
{
  if (!b) { return sen_invalid_argument; }
  if (b->tvs) { SEN_FREE(b->tvs); }
  if (b->vps) { SEN_FREE(b->vps); }
  SEN_FREE(b);
  return sen_success;
}

sen_rc
sen_vgram_close(sen_vgram *vgram)
{
  if (!vgram) { return sen_invalid_argument; }
  SEN_LOG(sen_log_debug, "len=%d img=%d skip=%d simple=%d", len_sum, img_sum, skip_sum, simple_sum);
  sen_sym_close(vgram->vgram);
  SEN_FREE(vgram);
  return sen_success;
}
