#include "global.h"
#include "preferences.h"
#include "preferencedialog.h"
#include "ui_preferencedialog.h"

#include <QColorDialog>
#include <QDebug>
#include <QDir>
#include <QFileDialog>
#include <QFontDialog>
#include <QPainter>
#include <QStandardPaths>

///////////////////////////////////////////////////////////////////////////////
/// \brief PreferenceDialog::PreferenceDialog
/// \param parent   親ウィジェット
///
/// コンストラクタ
///
PreferenceDialog::PreferenceDialog(QWidget *parent) :
    QDialog(parent),
    ui(new Ui::PreferenceDialog)
{
    ui->setupUi(this);
    ui->tabWidget->setCurrentIndex(0);
    ui->lBoxSample->setText(QDir::homePath());
    ui->fvHiddenSample->installEventFilter(this);
    ui->fvMarkedSample->installEventFilter(this);
    ui->fvNormalSample->installEventFilter(this);
    ui->fvReadOnlySample->installEventFilter(this);
    ui->fvSystemSample->installEventFilter(this);
    ui->lBoxSample->installEventFilter(this);
    ui->sBoxNormalSample->installEventFilter(this);
    ui->sBoxUnmatchSample->installEventFilter(this);
    ui->tvSample->installEventFilter(this);
    ui->hvSample->installEventFilter(this);
    ui->gvSample->installEventFilter(this);

    connect(ui->fvNormal, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->fvNormalBg, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->fvMarked, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->fvMarkedBg, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->fvSystem, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->fvHidden, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->fvReadOnly, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->lBoxBg, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->lBoxFg, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->sBoxNormal, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->sBoxNormalBg, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->sBoxUnmatch, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->sBoxUnmatchBg, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->tvBg, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->tvCtrl, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->tvFg, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->hvBg, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->hvFg, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->gvBg, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->allBg, SIGNAL(clicked()), this, SLOT(chooseColor()));
    connect(ui->allFg, SIGNAL(clicked()), this, SLOT(chooseColor()));

    connect(ui->fvFont, SIGNAL(clicked()), this, SLOT(chooseFont()));
    connect(ui->lBoxFont, SIGNAL(clicked()), this, SLOT(chooseFont()));
    connect(ui->sBoxFont, SIGNAL(clicked()), this, SLOT(chooseFont()));
    connect(ui->tvFont, SIGNAL(clicked()), this, SLOT(chooseFont()));
    connect(ui->hvFont, SIGNAL(clicked()), this, SLOT(chooseFont()));
    connect(ui->allFont, SIGNAL(clicked()), this, SLOT(chooseFont()));

    connect(ui->browseArchiver, SIGNAL(clicked()), this, SLOT(choosePath()));
    connect(ui->browseEditor, SIGNAL(clicked()), this, SLOT(choosePath()));
    connect(ui->browseTerminal, SIGNAL(clicked()), this, SLOT(choosePath()));

    connect(ui->exportAppearance, SIGNAL(clicked()), this, SLOT(exportAppearance()));
    connect(ui->importAppearance, SIGNAL(clicked()), this, SLOT(importAppearance()));

    Preferences prefs(this);

    // [全般]
    ui->checkUpdate->setChecked(prefs.isCheckUpdate());
    ui->confirmQuit->setChecked(prefs.isConfirmQuit());
    ui->moveAfterCreation->setChecked(prefs.isMoveAfterCreation());
    ui->openAfterCreation->setChecked(prefs.isOpenAfterCreation());
    ui->confirmCopy->setChecked(prefs.isConfirmCopy());
    ui->confirmMove->setChecked(prefs.isConfirmMove());
    ui->confirmDelete->setChecked(prefs.isConfirmDelete());
    ui->confirmRename->setChecked(prefs.isConfirmRename());
    ui->autoCloseCopy->setChecked(prefs.isAutoCloseCopy());
    ui->autoCloseMove->setChecked(prefs.isAutoCloseMove());
    ui->autoCloseDelete->setChecked(prefs.isAutoCloseDelete());
    ui->autoCloseRename->setChecked(prefs.isAutoCloseRename());
    QRadioButton *radio = findChild<QRadioButton*>("rb" + prefs.getCopyBehavior());
    if (radio)
        radio->setChecked(true);
    else
        ui->rbOverWriteIfNew->setChecked(true);
    ui->preferExtensions->setPlainText(prefs.getPreferExtensions());
    ui->resetOnBoot->setChecked(prefs.isReset());

    // [外観(色)]
    loadAppearance(prefs);

    // [外観(フォント)]
    setFont(ui->fvFontSpec, prefs.getFolderViewFont());
    setFont(ui->lBoxFontSpec, prefs.getLocationBoxFont());
    setFont(ui->sBoxFontSpec, prefs.getSearchBoxFont());
    setFont(ui->tvFontSpec, prefs.getTextViewFont());
    setFont(ui->hvFontSpec, prefs.getHexViewFont());

    ui->lineHeight->setValue(prefs.getLineHeight());
    ui->tvLineHeight->setValue(prefs.getTextViewLineHeight());

    // [プログラムパス]
    ui->editorPath->setText(prefs.getEditorPath());
    ui->terminalPath->setText(prefs.getTerminalPath());
    ui->archiverPath->setText(prefs.getArchiverPath());
}

///////////////////////////////////////////////////////////////////////////////
/// \brief PreferenceDialog::~PreferenceDialog
///
/// デストラクタ
///
PreferenceDialog::~PreferenceDialog()
{
    delete ui;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief PreferenceDialog::loadAppearance
/// \param prefs    プリファレンスオブジェクト
///
/// 外観(色)設定を読み込みます。
///
void PreferenceDialog::loadAppearance(Preferences &prefs)
{
    setPalette(ui->fvNormalSample, QPalette::Base, prefs.getFolderViewBgColor());
    setPalette(ui->fvNormalSample, QPalette::Text, prefs.getFolderViewFgColor());

    setPalette(ui->fvMarkedSample, QPalette::Base, prefs.getFolderViewMarkedBgColor());
    setPalette(ui->fvMarkedSample, QPalette::Text, prefs.getFolderViewMarkedFgColor());

    setPalette(ui->fvSystemSample, QPalette::Base, prefs.getFolderViewBgColor());
    setPalette(ui->fvSystemSample, QPalette::Text, prefs.getFolderViewSystemColor());

    setPalette(ui->fvHiddenSample, QPalette::Base, prefs.getFolderViewBgColor());
    setPalette(ui->fvHiddenSample, QPalette::Text, prefs.getFolderViewHiddenColor());

    setPalette(ui->fvReadOnlySample, QPalette::Base, prefs.getFolderViewBgColor());
    setPalette(ui->fvReadOnlySample, QPalette::Text, prefs.getFolderViewReadOnlyColor());

    setPalette(ui->lBoxSample, QPalette::Base, prefs.getLocationBoxBgColor());
    setPalette(ui->lBoxSample, QPalette::Text, prefs.getLocationBoxFgColor());

    setPalette(ui->sBoxNormalSample, QPalette::Base, prefs.getSearchBoxBgColor());
    setPalette(ui->sBoxNormalSample, QPalette::Text, prefs.getSearchBoxFgColor());

    setPalette(ui->sBoxUnmatchSample, QPalette::Base, prefs.getSearchBoxUnmatchBgColor());
    setPalette(ui->sBoxUnmatchSample, QPalette::Text, prefs.getSearchBoxUnmatchFgColor());

    setPalette(ui->tvSample, QPalette::Base, prefs.getTextViewBgColor());
    setPalette(ui->tvSample, QPalette::Text, prefs.getTextViewFgColor());
    setPalette(ui->tvSample, QPalette::BrightText, prefs.getTextViewCtrlColor());

    setPalette(ui->hvSample, QPalette::Base, prefs.getHexViewBgColor());
    setPalette(ui->hvSample, QPalette::Text, prefs.getHexViewFgColor());

    setPalette(ui->gvSample, QPalette::Base, prefs.getImageViewBgColor());

    ui->darkFactor->setValue(prefs.getDarkFacotr());
}

///////////////////////////////////////////////////////////////////////////////
/// \brief PreferenceDialog::saveAppearance
/// \param prefs    プリファレンスオブジェクト
///
/// 外観(色)設定を保存します。
///
void PreferenceDialog::saveAppearance(Preferences &prefs)
{
    prefs.setFolderViewBgColor(ui->fvNormalSample->palette().base().color());
    prefs.setFolderViewFgColor(ui->fvNormalSample->palette().text().color());

    prefs.setFolderViewMarkedBgColor(ui->fvMarkedSample->palette().base().color());
    prefs.setFolderViewMarkedFgColor(ui->fvMarkedSample->palette().text().color());

    prefs.setFolderViewHiddenColor(ui->fvHiddenSample->palette().text().color());
    prefs.setFolderViewReadOnlyColor(ui->fvReadOnlySample->palette().text().color());
    prefs.setFolderViewSystemColor(ui->fvSystemSample->palette().text().color());

    prefs.setLocationBoxBgColor(ui->lBoxSample->palette().base().color());
    prefs.setLocationBoxFgColor(ui->lBoxSample->palette().text().color());

    prefs.setSearchBoxBgColor(ui->sBoxNormalSample->palette().base().color());
    prefs.setSearchBoxFgColor(ui->sBoxNormalSample->palette().text().color());

    prefs.setSearchBoxUnmatchBgColor(ui->sBoxUnmatchSample->palette().base().color());
    prefs.setSearchBoxUnmatchFgColor(ui->sBoxUnmatchSample->palette().text().color());

    prefs.setTextViewBgColor(ui->tvSample->palette().base().color());
    prefs.setTextViewFgColor(ui->tvSample->palette().text().color());
    prefs.setTextViewCtrlColor(ui->tvSample->palette().brightText().color());

    prefs.setHexViewBgColor(ui->hvSample->palette().base().color());
    prefs.setHexViewFgColor(ui->hvSample->palette().text().color());

    prefs.setImageViewBgColor(ui->gvSample->palette().base().color());

    prefs.setDarkFacotr(ui->darkFactor->value());
}

///////////////////////////////////////////////////////////////////////////////
/// \brief PreferenceDialog::setFont
/// \param label    ラベルオブジェクト
/// \param font     フォントオブジェクト
///
/// ラベルにフォントを設定します。
///
void PreferenceDialog::setFont(QLabel *label, const QFont &font)
{
    label->setFont(font);
    label->setText(QString("%1, %2pt").arg(font.family()).arg(font.pointSize()));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief PreferenceDialog::setPalette
/// \param w        ウィジェット
/// \param role     ロール
/// \param color    色
///
/// パレットに色を設定します。
///
void PreferenceDialog::setPalette(QWidget *w, QPalette::ColorRole role, const QColor &color)
{
    QPalette pal = w->palette();
    pal.setColor(role, color);
    w->setPalette(pal);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief PreferenceDialog::chooseColor
///
/// 色選択ダイアログを表示します。
///
void PreferenceDialog::chooseColor()
{
    QColor color;

    if (sender() == ui->fvNormal) {
        color = ui->fvNormalSample->palette().text().color();
    }
    else if (sender() == ui->fvNormalBg) {
        color = ui->fvNormalSample->palette().base().color();
    }
    else if (sender() == ui->fvMarked) {
        color = ui->fvMarkedSample->palette().text().color();
    }
    else if (sender() == ui->fvMarkedBg) {
        color = ui->fvMarkedSample->palette().base().color();
    }
    else if (sender() == ui->fvSystem) {
        color = ui->fvSystemSample->palette().text().color();
    }
    else if (sender() == ui->fvHidden) {
        color = ui->fvHiddenSample->palette().text().color();
    }
    else if (sender() == ui->fvReadOnly) {
        color = ui->fvReadOnlySample->palette().text().color();
    }
    else if (sender() == ui->lBoxBg) {
        color = ui->lBoxSample->palette().base().color();
    }
    else if (sender() == ui->lBoxFg) {
        color = ui->lBoxSample->palette().text().color();
    }
    else if (sender() == ui->sBoxNormal) {
        color = ui->sBoxNormalSample->palette().text().color();
    }
    else if (sender() == ui->sBoxNormalBg) {
        color = ui->sBoxNormalSample->palette().base().color();
    }
    else if (sender() == ui->sBoxUnmatch) {
        color = ui->sBoxUnmatchSample->palette().text().color();
    }
    else if (sender() == ui->sBoxUnmatchBg) {
        color = ui->sBoxUnmatchSample->palette().base().color();
    }
    else if (sender() == ui->tvBg) {
        color = ui->tvSample->palette().base().color();
    }
    else if (sender() == ui->tvFg) {
        color = ui->tvSample->palette().text().color();
    }
    else if (sender() == ui->tvCtrl) {
        color = ui->tvSample->palette().brightText().color();
    }
    else if (sender() == ui->hvBg) {
        color = ui->hvSample->palette().base().color();
    }
    else if (sender() == ui->hvFg) {
        color = ui->hvSample->palette().text().color();
    }
    else if (sender() == ui->gvBg) {
        color = ui->gvSample->palette().base().color();
    }
    else if (sender() == ui->allBg) {
        color = ui->fvNormalSample->palette().base().color();
    }
    else if (sender() == ui->allFg) {
        color = ui->fvNormalSample->palette().text().color();
    }

    color = QColorDialog::getColor(color, this, tr("色を選択"));
    if (!color.isValid()) {
        return;
    }

    QPalette pal;
    if (sender() == ui->fvNormal) {
        setPalette(ui->fvNormalSample, QPalette::Text, color);
    }
    else if (sender() == ui->fvNormalBg) {
        setPalette(ui->fvNormalSample, QPalette::Base, color);
        setPalette(ui->fvHiddenSample, QPalette::Base, color);
        setPalette(ui->fvReadOnlySample, QPalette::Base, color);
        setPalette(ui->fvSystemSample, QPalette::Base, color);
    }
    else if (sender() == ui->fvMarked) {
        setPalette(ui->fvMarkedSample, QPalette::Text, color);
    }
    else if (sender() == ui->fvMarkedBg) {
        setPalette(ui->fvMarkedSample, QPalette::Base, color);
    }
    else if (sender() == ui->fvSystem) {
        setPalette(ui->fvSystemSample, QPalette::Text, color);
    }
    else if (sender() == ui->fvHidden) {
        setPalette(ui->fvHiddenSample, QPalette::Text, color);
    }
    else if (sender() == ui->fvReadOnly) {
        setPalette(ui->fvReadOnlySample, QPalette::Text, color);
    }
    else if (sender() == ui->lBoxBg) {
        setPalette(ui->lBoxSample, QPalette::Base, color);
    }
    else if (sender() == ui->lBoxFg) {
        setPalette(ui->lBoxSample, QPalette::Text, color);
    }
    else if (sender() == ui->sBoxNormal) {
        setPalette(ui->sBoxNormalSample, QPalette::Text, color);
    }
    else if (sender() == ui->sBoxNormalBg) {
        setPalette(ui->sBoxNormalSample, QPalette::Base, color);
    }
    else if (sender() == ui->sBoxUnmatch) {
        setPalette(ui->sBoxUnmatchSample, QPalette::Text, color);
    }
    else if (sender() == ui->sBoxUnmatchBg) {
        setPalette(ui->sBoxUnmatchSample, QPalette::Base, color);
    }
    else if (sender() == ui->tvBg) {
        setPalette(ui->tvSample, QPalette::Base, color);
    }
    else if (sender() == ui->tvFg) {
        setPalette(ui->tvSample, QPalette::Text, color);
    }
    else if (sender() == ui->tvCtrl) {
        setPalette(ui->tvSample, QPalette::BrightText, color);
    }
    else if (sender() == ui->hvBg) {
        setPalette(ui->hvSample, QPalette::Base, color);
    }
    else if (sender() == ui->hvFg) {
        setPalette(ui->hvSample, QPalette::Text, color);
    }
    else if (sender() == ui->gvBg) {
        setPalette(ui->gvSample, QPalette::Base, color);
    }
    else if (sender() == ui->allBg) {
        setPalette(ui->fvNormalSample, QPalette::Base, color);
        setPalette(ui->fvHiddenSample, QPalette::Base, color);
        setPalette(ui->fvReadOnlySample, QPalette::Base, color);
        setPalette(ui->fvSystemSample, QPalette::Base, color);
        setPalette(ui->lBoxSample, QPalette::Base, color);
        setPalette(ui->sBoxNormalSample, QPalette::Base, color);
        setPalette(ui->tvSample, QPalette::Base, color);
        setPalette(ui->hvSample, QPalette::Base, color);
        setPalette(ui->gvSample, QPalette::Base, color);
    }
    else if (sender() == ui->allFg) {
        setPalette(ui->fvNormalSample, QPalette::Text, color);
        setPalette(ui->lBoxSample, QPalette::Text, color);
        setPalette(ui->sBoxNormalSample, QPalette::Text, color);
        setPalette(ui->tvSample, QPalette::Text, color);
        setPalette(ui->hvSample, QPalette::Text, color);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief PreferenceDialog::chooseFont
///
/// フォント選択ダイアログを表示します。
///
void PreferenceDialog::chooseFont()
{
    QFont font;
    if (sender() == ui->fvFont) {
        font = ui->fvFontSpec->font();
    }
    else if (sender() == ui->lBoxFont) {
        font = ui->lBoxFontSpec->font();
    }
    else if (sender() == ui->sBoxFont) {
        font = ui->sBoxFontSpec->font();
    }
    else if (sender() == ui->tvFont) {
        font = ui->tvFontSpec->font();
    }
    else if (sender() == ui->hvFont) {
        font = ui->hvFontSpec->font();
    }
    else if (sender() == ui->allFont) {
        font = ui->fvFontSpec->font();
    }

    bool ok;
    font = QFontDialog::getFont(&ok, font, this, tr("フォントを選択"));

    if (!ok) {
        return;
    }

    if (sender() == ui->fvFont) {
        setFont(ui->fvFontSpec, font);
    }
    else if (sender() == ui->lBoxFont) {
        setFont(ui->lBoxFontSpec, font);
    }
    else if (sender() == ui->sBoxFont) {
        setFont(ui->sBoxFontSpec, font);
    }
    else if (sender() == ui->tvFont) {
        setFont(ui->tvFontSpec, font);
    }
    else if (sender() == ui->hvFont) {
        setFont(ui->hvFontSpec, font);
    }
    else if (sender() == ui->allFont) {
        setFont(ui->fvFontSpec, font);
        setFont(ui->lBoxFontSpec, font);
        setFont(ui->sBoxFontSpec, font);
        setFont(ui->tvFontSpec, font);
        setFont(ui->hvFontSpec, font);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief PreferenceDialog::choosePath
///
/// プログラム選択ダイアログを表示します。
///
void PreferenceDialog::choosePath()
{
    QStringList list = QStandardPaths::standardLocations(
                QStandardPaths::ApplicationsLocation);
#ifdef Q_OS_WIN
    QString path = QFileDialog::getOpenFileName(
                this, tr("アプリケーションを選択"), getenv("PROGRAMFILES"),
                tr("実行ファイル (*.exe *.com *.bat *.pif);;すべてのファイル (*)"));
#elif defined(Q_OS_MAC)
    QString path = QFileDialog::getOpenFileName(
                this, tr("アプリケーションを選択"), list.at(0),
                tr("実行ファイル (*.app);;すべてのファイル (*)"));
#else
    QString path = QFileDialog::getOpenFileName(
                this, tr("アプリケーションを選択"), list.at(0),
                tr("すべてのファイル (*)"));
#endif
    if (path.isEmpty()) {
        return;
    }

    if (path.indexOf(" ") != -1) {
        path = QQ(path);
    }

    if (sender() == ui->browseEditor) {
        ui->editorPath->setText(path);
    }
    else if (sender() == ui->browseTerminal) {
        ui->terminalPath->setText(path);
    }
    else if (sender() == ui->browseArchiver) {
        ui->archiverPath->setText(path);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief PreferenceDialog::exportAppearance
///
/// 「エクスポート」ボタンクリック時の処理を行います。
///
void PreferenceDialog::exportAppearance()
{
    QStringList list = QStandardPaths::standardLocations(
                QStandardPaths::DocumentsLocation);
    QString path = QFileDialog::getSaveFileName(
                this, tr("ファイルを選択"), list.at(0) + "/gefu_appearance.ini",
                tr("設定ファイル (*.ini);;すべてのファイル (*)"));
    if (path.isEmpty()) {
        return;
    }

    Preferences prefs(path, this);
    saveAppearance(prefs);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief PreferenceDialog::importAppearance
///
/// 「インポート」ボタンクリック時の処理を行います。
///
void PreferenceDialog::importAppearance()
{
    QStringList list = QStandardPaths::standardLocations(
                QStandardPaths::DocumentsLocation);
    QString path = QFileDialog::getOpenFileName(
                this, tr("ファイルを選択"), list.at(0),
                tr("設定ファイル (*.ini);;すべてのファイル (*)"));
    if (path.isEmpty()) {
        return;
    }

    Preferences prefs(path, this);
    loadAppearance(prefs);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief PreferenceDialog::accept
///
/// OKボタンクリック時の処理を行います。
///
void PreferenceDialog::accept()
{
    Preferences prefs(this);

    // [全般]
    prefs.setCheckUpdate(ui->checkUpdate->isChecked());
    prefs.setConfirmQuit(ui->confirmQuit->isChecked());
    prefs.setMoveAfterCreation(ui->moveAfterCreation->isChecked());
    prefs.setOpenAfterCreation(ui->openAfterCreation->isChecked());
    prefs.setConfirmCopy(ui->confirmCopy->isChecked());
    prefs.setConfirmMove(ui->confirmMove->isChecked());
    prefs.setConfirmDelete(ui->confirmDelete->isChecked());
    prefs.setConfirmRename(ui->confirmRename->isChecked());
    prefs.setAutoCloseCopy(ui->autoCloseCopy->isChecked());
    prefs.setAutoCloseMove(ui->autoCloseMove->isChecked());
    prefs.setAutoCloseDelete(ui->autoCloseDelete->isChecked());
    prefs.setAutoCloseRename(ui->autoCloseRename->isChecked());

    QAbstractButton *checked = ui->copyBehavior->checkedButton();
    prefs.setCopyBehavior(checked->objectName().mid(2));

    prefs.setPreferExtensions(ui->preferExtensions->toPlainText());
    prefs.setReset(ui->resetOnBoot->isChecked());

    // [外観(色)]
    saveAppearance(prefs);

    // [外観(フォント)]
    prefs.setFolderViewFont(ui->fvFontSpec->font());
    prefs.setLocationBoxFont(ui->lBoxFontSpec->font());
    prefs.setSearchBoxFont(ui->sBoxFontSpec->font());
    prefs.setTextViewFont(ui->tvFontSpec->font());
    prefs.setHexViewFont(ui->hvFontSpec->font());

    prefs.setLineHeight(ui->lineHeight->value());
    prefs.setTextViewLineHeight(ui->tvLineHeight->value());

    // [プログラムパス]
    prefs.setEditorPath(ui->editorPath->text());
    prefs.setTerminalPath(ui->terminalPath->text());
    prefs.setArchiverPath(ui->archiverPath->text());

    QDialog::accept();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief PreferenceDialog::eventFilter
/// \param obj      イベントが発生したオブジェクト
/// \param event    イベントオブジェクト
/// \return イベントを処理した場合はtrueを返します。
///
bool PreferenceDialog::eventFilter(QObject *obj, QEvent *event)
{
    if (event->type() == QEvent::Paint) {
        QLineEdit *edit = static_cast<QLineEdit*>(obj);

        QPainter painter(edit);

        painter.fillRect(edit->rect(), edit->palette().base());
        painter.setPen(edit->palette().text().color());
        painter.drawText(edit->rect(), Qt::AlignVCenter, edit->text());

        if (edit->objectName() == "tvSample") {
            QRect rcCtrl(edit->rect());
            int textWidth = edit->fontMetrics().width(edit->text());
            rcCtrl.setLeft(edit->rect().left() + textWidth);
            painter.setPen(edit->palette().brightText().color());
            painter.drawText(rcCtrl, Qt::AlignVCenter, QChar(0x21B5));
        }
        return true;
    }

    return false;
}
