#include "bookmarkdialog.h"
#include "copymoveworker.h"
#include "deleteworker.h"
#include "foldermodel.h"
#include "folderpanel.h"
#include "global.h"
#include "historydialog.h"
#include "operationdialog.h"
#include "overwritedialog.h"
#include "preferences.h"
#include "preferencedialog.h"
#include "renameworker.h"
#include "renamesingledialog.h"
#include "renamemultidialog.h"
#include "sortdialog.h"
#include "version.h"
#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QCheckBox>
#include <QClipboard>
#include <QCloseEvent>
#include <QDebug>
#include <QDesktopServices>
#include <QFileDialog>
#include <QInputDialog>
#include <QMessageBox>
#include <QMimeData>
#include <QNetworkAccessManager>
#include <QProcess>
#include <QScrollArea>

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::MainWindow
/// \param parent   親ウィジェット
///
/// コンストラクタ
///
MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow),
    m_overwriteDialog(this),
    m_viewMode(ModeBasic),
    m_prevMode(ModeBasic),
    m_actions()
{
    ui->setupUi(this);
    connect(qApp, SIGNAL(focusChanged(QWidget*,QWidget*)), this, SLOT(app_focusChange(QWidget*,QWidget*)));

    // ステータスバーにウィジェットを設定する
    QLabel *label = new QLabel(this);
    label->setAlignment(Qt::AlignRight);
    label->setObjectName("Right");
    statusBar()->addPermanentWidget(label, 0);

    // アプリケーション情報を初期化する
    qApp->setApplicationName("Gefu");
    qApp->setApplicationVersion(QString("%1").arg(VERSION_VALUE));
    qApp->setOrganizationDomain("sourceforge.jp");
    qApp->setOrganizationName("gefu");
    qApp->setWindowIcon(QIcon("://images/file-manager.png"));

    // 設定の初期化
    Preferences prefs(this);
    if (prefs.isReset()) {
        prefs.clear();
    }

    // パネルにモデルを割り当てる
    ui->LPanel->setModel(new FolderModel(this));
    ui->RPanel->setModel(new FolderModel(this));
    setActiveModel(ui->LPanel->model());

    // パネルを初期化する
    ui->LPanel->initialize(this);
    ui->RPanel->initialize(this);
    ui->FPanel->initialize(this);
    ui->FPanel->setVisible(false);

    // モデルを初期化する
    prefs.restoreModel("Left", ui->LPanel->model());
    prefs.restoreModel("Right", ui->RPanel->model());

    // アクションを初期化する
    initActions();

    // ブックマークメニューを初期化する
    initBookmarkMenu();

    // メインウィンドウを初期化する
    setWindowTitle(tr("げふぅ Ver%1").arg(VERSION_VALUE));
    setWindowIcon(QIcon("://images/file-manager.png"));
    prefs.restoreWindow(this);

    // 最新バージョンをチェックする
    if (prefs.isCheckUpdate()) {
        onCheckUpdate(true);
    }

    prefs_updated();

    connect(ui->FPanel, SIGNAL(showed(QWidget*)), this, SLOT(view_showed(QWidget*)));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::~MainWindow
///
/// デストラクタ
///
MainWindow::~MainWindow()
{
    delete ui;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::app_focusChange
/// \param old  フォーカスを失うウィジェット
/// \param now  フォーカスを得るウィジェット
///
/// フォーカス変更時の処理
///
void MainWindow::app_focusChange(QWidget *old, QWidget *now)
{
    qDebug() << "MainWindow::app_focusChange()";
    if (old) qDebug() << "  old is" << old->objectName();
    if (now) qDebug() << "  now is" << now->objectName();

    // フォルダビューがフォーカスを得た場合
    if (now && (now->objectName() == "folderView" ||
                now->objectName() == "thumbnailView"))
    {
        connect(focusItemView()->selectionModel(), SIGNAL(currentChanged(QModelIndex,QModelIndex)),
                this, SLOT(view_currentChanged(QModelIndex,QModelIndex)));
        FolderModel *m = static_cast<FolderModel*>(focusItemView()->model());
        setActiveModel(m);
        statusBar()->showMessage(m->filePath(focusItemView()->currentIndex()));
    }

    // 検索ボックスがフォーカスを失った場合
    if (old && old->objectName() == "searchBox") {
        // 検索モード終了
        ui->toggle_Search->setChecked(false);
    }

    // アクションの有効／無効を設定する
    if (now) {
        updateActions();
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::model_PreReset
///
/// FolderModelリセット前の処理
///
void MainWindow::model_PreReset()
{
    qDebug() << "MainWindow::model_PreReset()";

    statusBar()->showMessage(tr("ファイルリストを取得しています..."));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::model_PostReset
///
/// FolderModelリセット後の処理
///
void MainWindow::model_PostReset()
{
    qDebug() << "MainWindow::model_PostReset()";

    FolderModel *m = static_cast<FolderModel*>(sender());
    if (!m->error().isEmpty()) {
        statusBar()->showMessage(m->error());
    }
    else {
        statusBar()->showMessage(tr("レディ"));
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onRunCommand
///
/// 入力されたコマンドを実行します。
///
void MainWindow::onRunCommand()
{
    qDebug() << "MainWindow::onRunCommand";

    QFileInfoList list = selectedItems();
    QString command = QString::null;
    foreach (const QFileInfo &info, list) {
#ifdef Q_OS_MAC
        QString path = info.absoluteFilePath();
#else
        QString path = info.fileName();
#endif
        if (info.isExecutable()) {
            command = QQ(path) + " " + command;
        }
        else {
            command += " " + QQ(path);
        }
    }

    QInputDialog dlg(this);
    dlg.setInputMode(QInputDialog::TextInput);
    dlg.setWindowTitle(tr("コマンドを実行"));
    dlg.setLabelText(tr("コマンド："));
    dlg.setTextValue(command);
    dlg.resize(width() * 0.8, dlg.height());

    int ret = dlg.exec();
    command = dlg.textValue();
    if (ret == QDialog::Accepted && !command.isEmpty()) {
        startProcess(command, tr("コマンドの実行に失敗しました。"));
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onExpandLeft
///
/// 左パネルを拡大します。
///
void MainWindow::onExpandLeft()
{
    qDebug() << "MainWindow::onExpandLeft()";

    QList<int> sizes = ui->splitter->sizes();
    QList<int> newSizes;

    newSizes << sizes[0] + 30;
    newSizes << sizes[1] - 30;

    ui->splitter->setSizes(newSizes);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onExpandRight
///
/// 右パネルを拡大します。
///
void MainWindow::onExpandRight()
{
    qDebug() << "MainWindow::onExpandLeft()";

    QList<int> sizes = ui->splitter->sizes();
    QList<int> newSizes;

    newSizes << sizes[0] - 30;
    newSizes << sizes[1] + 30;

    ui->splitter->setSizes(newSizes);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onToggleMark
///
/// マーク状態を反転します。
///
void MainWindow::onToggleMark()
{
    qDebug() << "MainWindow::onToggleMark()";

    QModelIndex index = focusItemView()->currentIndex();
    if (activeModel()->isMarked(index)) {
        activeModel()->setData(index, Qt::Unchecked, Qt::CheckStateRole);
    }
    else {
        activeModel()->setData(index, Qt::Checked, Qt::CheckStateRole);
    }

    if (index.row() < activeModel()->rowCount() - 1) {
        index = activeModel()->index(index.row() + 1, 0);
        focusItemView()->setCurrentIndex(index);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::view_statusChanged
/// \param text ステータス文字列
///
/// ビューから受け取った文字列をステータスバーに表示します。
///
void MainWindow::view_statusChanged(const QString &text)
{
    statusBar()->findChild<QLabel*>("Right")->setText(text);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::view_showed
/// \param w    表示されたビュー
///
/// パネルのビューが表示された場合の処理を行います。
///
void MainWindow::view_showed(QWidget *w)
{
    w->setFocus();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::view_currentChanged
/// \param current  新しいカレントインデックス
/// \param previous (使用しません)
///
/// カレントインデックスが変更された場合の処理を行います。
///
void MainWindow::view_currentChanged(const QModelIndex &current, const QModelIndex &previous)
{
    qDebug() << "MainWindow::view_currentChanged()" << current;
    Q_UNUSED(previous);

    if (focusItemView() != sender()->parent()) {
        return;
    }

    statusBar()->showMessage(activeModel()->filePath(current));

    if (m_viewMode == ModePreview) {
        view_statusChanged();
        inactivePanel()->setViewItem(current);
    }

    updateActions();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::view_finished
///
/// ビューアの終了
///
void MainWindow::view_finished()
{
    qDebug() << "MainWindow::view_finished()";

    statusBar()->findChild<QLabel*>("Right")->setText("");

    setViewMode(m_prevMode);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onMove
///
/// 選択アイテムを隣のパネルに移動します。
///
void MainWindow::onMove()
{
    qDebug() << "MainWindow::onMove";

    QStringList list;
    foreach (const QFileInfo &fi, selectedItems()) {
        list << fi.absoluteFilePath();
    }
    if (list.isEmpty()) {
        return;
    }

    int row = focusItemView()->currentIndex().row();

    QString tgtPath = inactiveModel()->rootPath();
    moveItems(list, tgtPath);

    if (row >= activeModel()->rowCount()) {
        row = activeModel()->rowCount() - 1;
    }
    focusItemView()->setCurrentIndex(activeModel()->index(row, 1));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onKeyLeft
///
/// ←キー押下時の処理を行います。
///
void MainWindow::onKeyLeft()
{
    qDebug() << "MainWindow::onKeyLeft()";

    QWidget *w = qApp->focusWidget();
    if (w->objectName() == "folderView")
    {
        QWidget *panel = w->parentWidget()->parentWidget();
        if (panel == ui->LPanel) {
            activeModel()->onCdUp();
        }
        else if (ui->LPanel->folderPanel()->isVisible()){
            ui->LPanel->folderPanel()->itemView()->setFocus();
        }
    }
    else {
        QKeyEvent event = QKeyEvent(QEvent::KeyPress, Qt::Key_PageUp, Qt::NoModifier);
        qApp->sendEvent(w, &event);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onKeyRight
///
/// →キー押下時の処理を行います。
///
void MainWindow::onKeyRight()
{
    qDebug() << "MainWindow::onKeyRight()";

    QWidget *w = qApp->focusWidget();
    if (w->objectName() == "folderView")
    {
        QWidget *panel = w->parentWidget()->parentWidget();
        if (panel == ui->RPanel) {
            activeModel()->onCdUp();
        }
        else if (ui->RPanel->folderPanel()->isVisible()){
            ui->RPanel->folderPanel()->itemView()->setFocus();
        }
    }
    else {
        QKeyEvent event = QKeyEvent(QEvent::KeyPress, Qt::Key_PageDown, Qt::NoModifier);
        qDebug() << w->objectName();
        qApp->sendEvent(w, &event);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onChooseFolder
///
/// 選択したフォルダに移動します。
///
void MainWindow::onChooseFolder()
{
    qDebug() << "MainWindow::onChooseFolder()";

    QString path = QFileDialog::getExistingDirectory(
                this, tr("フォルダを選択"), activeModel()->rootPath());
    if (!path.isEmpty()) {
        activeModel()->setRootPath(path);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onCopyFileName
///
/// ファイル名をクリップボードにコピーします。
///
void MainWindow::onCopyFileName()
{
    qDebug() << "MainWindow::onCopyFileName()";

    QClipboard *clipboard = qApp->clipboard();
    clipboard->setText(activeModel()->fileName(focusItemView()->currentIndex()));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onCopyFilePath
///
/// フルパスをクリップボードにコピーします。
///
void MainWindow::onCopyFilePath()
{
    qDebug() << "MainWindow::onCopyFilePath()";

    QClipboard *clipboard = qApp->clipboard();
    clipboard->setText(activeModel()->filePath(focusItemView()->currentIndex()));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::askOverWrite
/// \param copyMethod   上書き方法
/// \param alias        別名
/// \param srcPath      コピー元パス
/// \param tgtPath      コピー先パス
///
/// 上書きの処理方法を選択するダイアログを表示します。
///
void MainWindow::askOverWrite(QString *copyMethod,
                              QString *alias,
                              const QString &srcPath,
                              const QString &tgtPath)
{
    qDebug() << "MainWindow::askOverWrite()";

    CopyMoveWorker *worker = static_cast<CopyMoveWorker*>(sender());
    if (!m_overwriteDialog.isKeepSetting() ||
        m_overwriteDialog.copyMethod() == "rbRename")
    {
        m_overwriteDialog.setFileInfo(srcPath, tgtPath);
        if (m_overwriteDialog.exec() == QDialog::Rejected) {
            worker->abort();
        }
    }
    *copyMethod = m_overwriteDialog.copyMethod();
    *alias = m_overwriteDialog.alias();

    worker->endAsking();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onAddBookmark
///
/// ブックマークを追加します。
///
void MainWindow::onAddBookmark()
{
    qDebug() << "MainWindow::onAddBookmark()";
    qDebug() << sender()->objectName();

    FolderModel *m;
    if (sender()->objectName() == "bookmarkBtn") {
        FolderPanel *p = static_cast<FolderPanel*>(sender()->parent());
        m = p->model();
    }
    else {
        m = activeModel();
    }

    QFileInfo fi(m->rootPath());
    Preferences(this).addBookmark(fi.fileName(), fi.absoluteFilePath());
    initBookmarkMenu();

    activePanel()->folderPanel()->itemView()->setFocus();
    statusBar()->showMessage(tr("%1をブックマークに追加しました").arg(fi.absoluteFilePath()));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::view_copyAvailable
/// \param yes  コピー可能ならtrue
///
/// 選択範囲の有無をメニューに反映します。
///
void MainWindow::view_copyAvailable(bool yes)
{
    ui->text_Copy->setEnabled(yes);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::copyItems
/// \param list     コピー元リスト
/// \param tgtDir   コピー先フォルダ
///
/// アイテムをコピーします。
///
void MainWindow::copyItems(const QStringList &list, const QString &tgtDir)
{
    qDebug() << "MainWindow::copyItems()" << tgtDir;

    Preferences prefs(this);
    if (prefs.isConfirmCopy() &&
        QMessageBox::question(this, tr("確認"), tr("コピーを実行しますか？"))
            != QMessageBox::Yes)
    {
        return;
    }

    // 上書き確認ダイアログを初期化する
    m_overwriteDialog.reset();

    // ワーカースレッドを作成する
    CopyMoveWorker *worker = new CopyMoveWorker();
    connect(worker, SIGNAL(askOverWrite(QString*,QString*,QString,QString)),
            this, SLOT(askOverWrite(QString*,QString*,QString,QString)));
    worker->setCopyList(list);
    worker->setTargetDir(tgtDir);
    worker->setMoveMode(false);

    // 進捗ダイアログを表示して、処理を開始する
    OperationDialog opDlg(this);
    opDlg.setWindowTitle(tr("コピー"));
    opDlg.setWorker(worker);
    opDlg.setAutoClose(prefs.isAutoCloseCopy());

    opDlg.exec();

    prefs.setAutoCloseCopy(opDlg.autoClose());
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::moveItems
/// \param list     移動元リスト
/// \param tgtDir   移動先フォルダ
///
/// アイテムを移動します。
///
void MainWindow::moveItems(const QStringList &list, const QString &tgtDir)
{
    qDebug() << "MainWindow::moveItems()" << tgtDir;

    Preferences prefs(this);
    if (prefs.isConfirmMove() &&
        QMessageBox::question(this, tr("確認"), tr("移動を実行しますか？"))
            != QMessageBox::Yes)
    {
        return;
    }

    // 上書き確認ダイアログを初期化する
    m_overwriteDialog.reset();

    // ワーカースレッドを作成する
    CopyMoveWorker *worker = new CopyMoveWorker();
    connect(worker, SIGNAL(askOverWrite(QString*,QString*,QString,QString)),
            this, SLOT(askOverWrite(QString*,QString*,QString,QString)));
    worker->setCopyList(list);
    worker->setTargetDir(tgtDir);
    worker->setMoveMode(true);

    // 進捗ダイアログを表示して、処理を開始する
    OperationDialog opDlg(this);
    opDlg.setWindowTitle(tr("移動"));
    opDlg.setWorker(worker);
    opDlg.setAutoClose(prefs.isAutoCloseMove());

    opDlg.exec();

    prefs.setAutoCloseMove(opDlg.autoClose());
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onContextMenuEvent
/// \param obj  イベントが発生したオブジェクト
/// \param e    コンテキストメニューイベント
/// \return 処理した場合はtrue, 処理しなかった場合はfalseを返します。
///
bool MainWindow::onContextMenuEvent(QObject *obj, QContextMenuEvent *e)
{
    QMenu menu(this);
    if (obj->objectName() == "textView") {
        menu.addAction(ui->text_ConvertFromEUC);
        menu.addAction(ui->text_ConvertFromJIS);
        menu.addAction(ui->text_ConvertFromSJIS);
        menu.addAction(ui->text_ConvertFromUTF8);
        menu.addAction(ui->text_ConvertFromUTF16);
        menu.addAction(ui->text_ConvertFromUTF16BE);
        menu.addAction(ui->text_ConvertFromUTF16LE);
        menu.addSeparator();
        menu.addAction(ui->view_SelectAll);
        menu.addAction(ui->text_Copy);
        if (m_viewMode == ModeView) {
            menu.addSeparator();
            menu.addAction(ui->view_Back);
        }
    }
    else if (obj->objectName() == "hexView") {
        menu.addAction(ui->view_SelectAll);
        menu.addAction(ui->text_Copy);
        if (m_viewMode == ModeView) {
            menu.addSeparator();
            menu.addAction(ui->view_Back);
        }
    }
    else if (obj->objectName() == "imageView") {
        menu.addAction(ui->image_FitToWindow);
        menu.addAction(ui->action_ScaleDown);
        menu.addAction(ui->action_ScaleUp);
        menu.addAction(ui->image_ScaleNormal);
        menu.addSeparator();
        menu.addAction(ui->image_Rotate90);
        menu.addAction(ui->image_Rotate180);

        if (m_viewMode == ModeView) {
            menu.addSeparator();
            menu.addAction(ui->view_Back);
        }
    }
    else if (obj->objectName() == "folderView" || obj->objectName() == "thumbnailView") {
        menu.addAction(ui->action_OpenWith);
        menu.addAction(ui->action_OpenEditor);
        menu.addAction(ui->action_OpenTerminal);
        menu.addAction(ui->action_OpenArchiver);
        menu.addSeparator();
        menu.addAction(ui->action_Copy);
        menu.addAction(ui->action_Move);
        menu.addSeparator();
        menu.addAction(ui->action_Delete);
        menu.addSeparator();
        menu.addAction(ui->action_CopyFileName);
        menu.addAction(ui->action_CopyFilePath);
        menu.addSeparator();
        menu.addAction(ui->action_Rename);
    }
    else {
        qDebug() << "No context menu" << obj->objectName();
        return false;
    }

    QString statusText = statusBar()->currentMessage();
    menu.exec(e->globalPos());
    statusBar()->showMessage(statusText);

    return true;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onKeyPressEvent
/// \param obj  イベントが発生したオブジェクト
/// \param e    キーイベントオブジェクト
/// \return 処理した場合はtrue, 処理しなかった場合はfalseを返します。
///
bool MainWindow::onKeyPressEvent(QObject *obj, QKeyEvent *e)
{
    qDebug() << "MainWindow::onKeyPressEvent()";

    QString modifier = QString::null;
    if (e->modifiers() & Qt::ShiftModifier)     { modifier += "Shift+"; }
    if (e->modifiers() & Qt::ControlModifier)   { modifier += "Ctrl+"; }
    if (e->modifiers() & Qt::AltModifier)       { modifier += "Alt+"; }
    if (e->modifiers() & Qt::MetaModifier)      { modifier += "Meta+"; }

    QString key = modifier + QKeySequence(e->key()).toString();

    foreach (QAction *action, findChildren<QAction*>()) {
        if (action->isEnabled()) {
            foreach (const QKeySequence &ks, action->shortcuts()) {
                if (ks.toString() == key) {
                    qDebug() << "emit" << action->objectName();
                    if (action->isCheckable()) {
                        action->toggle();
                    }
                    else {
                        action->trigger();
                    }
                    return true;
                }
             }
        }
    }

    if (obj->objectName() == "folderView" || obj->objectName() == "thumbnailView")
    {
        // カーソル移動系のみ有効にする
        switch (e->key()) {
        case Qt::Key_Down:
        case Qt::Key_Up:
        case Qt::Key_Left:
        case Qt::Key_Right:
        case Qt::Key_Home:
        case Qt::Key_End:
        case Qt::Key_PageDown:
        case Qt::Key_PageUp:
        case Qt::Key_Tab:
            return false;
        }
        return true;
    }

    return false;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onCopy
///
/// 選択アイテムを隣のパネルにコピーします。
///
void MainWindow::onCopy()
{
    qDebug() << "MainWindow::onCopy";

    QStringList list;
    foreach (const QFileInfo &fi, selectedItems()) {
        list << fi.absoluteFilePath();
    }
    if (list.isEmpty()) {
        return;
    }

    QString tgtPath = inactiveModel()->rootPath();
    copyItems(list, tgtPath);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onCreateFile
///
/// ファイルを作成します。
///
void MainWindow::onCreateFile()
{
    qDebug() << "MainWindow::onCreateFile";

    bool bOk;
    QString name = QInputDialog::getText(
                this, tr("ファイルを作成"), tr("ファイル名："),
                QLineEdit::Normal, "", &bOk);
    if (!bOk || name.isEmpty()) {
        return;
    }

    QModelIndex index = activeModel()->touch(name);
    if (!index.isValid()) {
        QMessageBox::critical(
                    this, tr("エラー"),
                    tr("ファイルの作成に失敗しました。"));
        return;
    }

    focusItemView()->setCurrentIndex(index);
    if (Preferences(this).isOpenAfterCreation()) {
        onOpenEditor(index);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onCreateFolder
///
/// フォルダを作成します。
///
void MainWindow::onCreateFolder()
{
    qDebug() << "MainWindow::onCreateFolder";

    bool bOk;
    QString name = QInputDialog::getText(
                this, tr("フォルダを作成"), tr("フォルダ名："),
                QLineEdit::Normal, "", &bOk);
    if (!bOk || name.isEmpty()) {
        return;
    }

    QModelIndex index = activeModel()->mkdir(name);
    if (!index.isValid()) {
        QMessageBox::critical(
                    this, tr("エラー"),
                    tr("フォルダの作成に失敗しました。"));
        return;
    }

    focusItemView()->setCurrentIndex(index);
    if (Preferences(this).isMoveAfterCreation()) {
        onOpen(index);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onDelete
///
/// 選択アイテムを削除します。
///
void MainWindow::onDelete()
{
    qDebug() << "MainWindow::onDelete";

    QFileInfoList list = selectedItems();
    if (list.isEmpty()) {
        return;
    }

    Preferences prefs(this);
    if (prefs.isConfirmDelete()) {
        QString msg;
        if (list.size() == 1) {
            msg = list[0].fileName();
        }
        else {
            msg = tr("%1個のアイテム").arg(list.size());
        }
        int ret = QMessageBox::question(
                    this, tr("確認"),
                    tr("%1を削除しますか？").arg(msg));
        if (ret != QMessageBox::Yes) {
            return;
        }
    }

    DeleteWorker *worker = new DeleteWorker();
    worker->setDeleteList(list);

    OperationDialog opDlg(this);
    opDlg.setWindowTitle(tr("削除"));
    opDlg.setWorker(worker);
    opDlg.setAutoClose(prefs.isAutoCloseDelete());

    opDlg.exec();

    prefs.setAutoCloseDelete(opDlg.autoClose());
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onOpen
/// \param index    (使用しません)
///
/// アイテムを開きます。
///
void MainWindow::onOpen(const QModelIndex &index)
{
    Q_UNUSED(index);
    qDebug() << "MainWindow::onOpen()" << index;

    QModelIndex current = focusItemView()->currentIndex();
    if (activeModel()->isDir(current)) {
        activeModel()->setRootPath(activeModel()->filePath(current));
        return;
    }

    // 外部アプリを優先する場合
    QString ext = activeModel()->fileInfo(current).suffix().toLower();
    QStringList list = Preferences(this).getPreferExtensions().split(",", QString::SkipEmptyParts);
    foreach (const QString& s, list) {
        if (ext == s.trimmed().toLower()) {
            QString path = QDir::toNativeSeparators(activeModel()->filePath(current));
            QDesktopServices::openUrl(QUrl("file:///" + path));
            return;
        }
    }

    // 内蔵ビューアで表示する
    setViewMode(ModeView);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::prefs_updated
///
/// 設定が変更された場合の処理を行います。
///
void MainWindow::prefs_updated()
{
    Preferences prefs(this);
    ui->LPanel->updateAppearance(prefs);
    ui->RPanel->updateAppearance(prefs);
    ui->FPanel->updateAppearance(prefs);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onOpenEditor
/// \param path
///
/// 選択アイテムを外部エディタで開きます。
///
void MainWindow::onOpenEditor(const QModelIndex &index)
{
    qDebug() << "MainWindow::onOpenEditor";

    QString exe = Preferences(this).getEditorPath();
    if (exe.isEmpty()) {
        QMessageBox::critical(
                    this, tr("エラー"),
                    tr("外部エディタのパスが未定義です。"));
        return;
    }

    QFileInfoList list;
    if (index.isValid()) {
        list << activeModel()->fileInfo(index);
    }
    else {
        list = selectedItems();
    }

    QString files;
    foreach (const QFileInfo &info, list) {
        files += " " + QQ(info.absoluteFilePath());
    }
#ifdef Q_OS_MAC
    QString command = "open -a " + exe + files;
#else
    QString command = exe + files;
#endif
    if (!startProcess(command, tr("外部エディタの起動に失敗しました。"))) {
        qDebug() << command;
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onOpenTerminal
///
/// 選択アイテムをターミナルで開きます。
///
void MainWindow::onOpenTerminal()
{
    qDebug() << "MainWindow::onOpenTerminal";

    QString exe = Preferences(this).getTerminalPath();
    if (exe.isEmpty()) {
        QMessageBox::critical(
                    this, tr("エラー"),
                    tr("ターミナルのパスが未定義です。"));
        return;
    }

    QSet<QString> dirs;
    foreach (const QFileInfo &info, selectedItems()) {
        if (info.isDir()) {
            dirs.insert(info.absoluteFilePath());
        }
        else {
            dirs.insert(info.absolutePath());
        }
    }

    foreach (const QString &dir, dirs) {
#ifdef Q_OS_MAC
        QString command = "open -n -a " + exe + " " + QQ(dir);
#else
        QString command = exe + " " + QQ(dir);
#endif
        if (!startProcess(command, tr("ターミナルの起動に失敗しました。"))) {
            qDebug() << command;
            break;
        }
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onOpenArchiver
///
/// 選択アイテムをアーカイバで開きます。
///
void MainWindow::onOpenArchiver()
{
    qDebug() << "MainWindow::onOpenArchiver";

    QString exe = Preferences(this).getArchiverPath();
    if (exe.isEmpty()) {
        QMessageBox::critical(
                    this, tr("エラー"),
                    tr("アーカイバのパスが未定義です。"));
        return;
    }

    QString files;
    foreach (const QFileInfo &info, selectedItems()) {
        files += " " + QQ(info.absoluteFilePath());
    }
#ifdef Q_OS_MAC
    QString command = "open -a " + exe + files;
#else
    QString command = exe + files;
#endif
    if (!startProcess(command, tr("アーカイバの起動に失敗しました。"))) {
        qDebug() << command;
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onOpenBookmark
///
/// ブックマークのメニュー項目をクリックしたときの処理を行います。
///
void MainWindow::onOpenBookmark()
{
    qDebug() << "MainWindow::onOpenBookmark()";

    QAction *action = qobject_cast<QAction*>(sender());
    Q_CHECK_PTR(action);

    int i = action->data().toInt();
    activeModel()->setRootPath(Preferences(this).getBookmarkPath(i));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onRename
///
/// 選択アイテムの名前を変更します。
///
void MainWindow::onRename()
{
    qDebug() << "MainWindow::onRename";

    QFileInfoList list = selectedItems();
    if (list.isEmpty()) {
        return;
    }

    AbstractRenameDialog *dlg;
    if (list.size() == 1) {
        dlg = new RenameSingleDialog(this);
    }
    else {
        dlg = new RenameMultiDialog(this);
    }
    dlg->setWorkingDirectory(activeModel()->rootPath());
    dlg->setNames(list);
    int dlgResult = dlg->exec();
    if (dlgResult != QDialog::Accepted || dlg->renameMap().isEmpty()) {
        return;
    }

    Preferences prefs(this);
    if (prefs.isConfirmRename() &&
        QMessageBox::question(this, tr("確認"), tr("名前の変更を実行しますか？"))
            != QMessageBox::Yes)
    {
        return;
    }

    RenameWorker *worker = new RenameWorker();
    worker->setRenameMap(&dlg->renameMap());

    OperationDialog opDlg(this);
    opDlg.setWindowTitle(tr("名前を変更"));
    opDlg.setWorker(worker);
    opDlg.setAutoClose(prefs.isAutoCloseRename());

    opDlg.exec();

    QFileInfo fi(dlg->renameMap().first());
    focusItemView()->setCurrentIndex(activeModel()->search(fi.fileName()));

    prefs.setAutoCloseRename(opDlg.autoClose());
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onOpenWith
///
/// 関連付けられたアプリで開きます。
///
void MainWindow::onOpenWith()
{
    qDebug() << "MainWindow::onOpenWith";

    foreach (const QFileInfo &info, selectedItems()) {
        QString path = QDir::toNativeSeparators(info.absoluteFilePath());
        QDesktopServices::openUrl(QUrl("file:///" + path));
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::showBookmarkDialog
///
/// ブックマーク一覧ダイアログを表示します。
///
void MainWindow::showBookmarkDialog()
{
    qDebug() << "MainWindow::showBookmarkDialog()";

    BookmarkDialog dlg(this);
    dlg.setEditMode(false);
    if (dlg.exec() == QDialog::Accepted) {
        int n = dlg.selectedIndex();
        activeModel()->setRootPath(Preferences(this).getBookmarkPath(n + 1));
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onEditBookmark
///
/// ブックマークの編集ダイアログを表示します。
///
void MainWindow::onEditBookmark()
{
    qDebug() << "MainWindow::onEditBookmark()";

    BookmarkDialog dlg(this);
    dlg.setEditMode(true);

    if (dlg.exec() == QDialog::Accepted) {
        initBookmarkMenu();
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::showFilterDialog
///
/// フィルタ設定ダイアログを表示します。
///
void MainWindow::showFilterDialog()
{
    qDebug() << "MainWindow::showFilterDialog()";

    QString filters = activeModel()->nameFilters().join(" ");

    QInputDialog dlg(this);
    dlg.setInputMode(QInputDialog::TextInput);
    dlg.setWindowTitle(tr("フィルタを設定"));
    dlg.setLabelText(tr("フィルタ："));
    dlg.setTextValue(filters);
    dlg.resize(width() * 0.8, dlg.height());

    if (dlg.exec() == QDialog::Accepted) {
        filters = dlg.textValue();
        if (filters.isEmpty()) {
            filters = "*";
        }

        activeModel()->setNameFilters(filters.split(" ", QString::SkipEmptyParts));
        activeModel()->refresh();
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::showHistoryDialog
///
/// 履歴選択ダイアログを表示します。
///
void MainWindow::showHistoryDialog()
{
    qDebug() << "MainWindow::showHistoryDialog()";

    HistoryDialog dlg(this);
    dlg.setModel(ui->LPanel->model(), ui->RPanel->model(), activeModel());
    dlg.exec();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::showSortDialog
///
/// ソート方法選択ダイアログを表示します。
///
void MainWindow::showSortDialog()
{
    qDebug() << "MainWindow::showSortDialog()";

    SortDialog dlg(this);
    dlg.setModel(activeModel());
    dlg.exec();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onSplitCenter
///
/// パネルを中央で分割します。
///
void MainWindow::onSplitCenter()
{
    qDebug() << "MainWindow::onSplitCenter()";

    QList<int> sizes = ui->splitter->sizes();
    int sizeTotal = sizes[0] + sizes[1];
    sizes[0] = sizeTotal / 2;
    sizes[1] = sizeTotal - sizes[0];
    ui->splitter->setSizes(sizes);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onSwap
///
/// 左右のパネルでモデルを入れ替えます。
///
void MainWindow::onSwap()
{
    qDebug() << "MainWindow::onSwap()";

    FolderModel *tmp = ui->LPanel->model();
    ui->LPanel->setModel(ui->RPanel->model());
    ui->RPanel->setModel(tmp);

    // アクティブ状態も変更する
    if (ui->LPanel->model()->isActive()) {
        setActiveModel(ui->RPanel->model());
    }
    else {
        setActiveModel(ui->LPanel->model());
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onTogglePreviewMode
/// \param checked  メニューのチェック状態
///
/// 通常モード／プレビューモードを切り替えます。
///
void MainWindow::onTogglePreviewMode(bool checked)
{
    qDebug() << "MainWindow::onTogglePreviewMode()" << checked;

    if (checked) {
        setViewMode(ModePreview);
    }
    else {
        setViewMode(ModeBasic);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onSearchNext
///
/// 次のアイテムを検索します。
///
void MainWindow::onSearchNext()
{
    qDebug() << "MainWindow::onSearchNext";

    static_cast<FolderPanel*>(qApp->focusWidget()->parentWidget())->searchNext();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onSearchPrev
///
/// 前のアイテムを検索します。
///
void MainWindow::onSearchPrev()
{
    qDebug() << "MainWindow::onSearchPrev";

    static_cast<FolderPanel*>(qApp->focusWidget()->parentWidget())->searchNext(-1);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onKeyDown
///
/// フォーカスウィジェットに↓キーイベントを送信します。
///
void MainWindow::onKeyDown()
{
    qDebug() << "MainWindow::onKeyDown()";

    QKeyEvent event = QKeyEvent(QEvent::KeyPress, Qt::Key_Down, Qt::NoModifier);
    qApp->sendEvent(qApp->focusWidget(), &event);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onKeyEnd
///
/// フォーカスウィジェットにEND, Ctrl+ENDキーイベントを送信します。
///
void MainWindow::onKeyEnd()
{
    qDebug() << "MainWindow::onKeyEnd()";

    QKeyEvent event1 = QKeyEvent(QEvent::KeyPress, Qt::Key_End, Qt::NoModifier);
    qApp->sendEvent(qApp->focusWidget(), &event1);

    QKeyEvent event2 = QKeyEvent(QEvent::KeyPress, Qt::Key_End, Qt::ControlModifier);
    qApp->sendEvent(qApp->focusWidget(), &event2);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onKeyHome
///
/// フォーカスウィジェットにHOME, Ctrl+HOMEキーイベントを送信します。
///
void MainWindow::onKeyHome()
{
    qDebug() << "MainWindow::onKeyHome()";

    QKeyEvent event1 = QKeyEvent(QEvent::KeyPress, Qt::Key_Home, Qt::NoModifier);
    qApp->sendEvent(qApp->focusWidget(), &event1);

    QKeyEvent event2 = QKeyEvent(QEvent::KeyPress, Qt::Key_Home, Qt::ControlModifier);
    qApp->sendEvent(qApp->focusWidget(), &event2);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onKeyUp
///
/// フォーカスウィジェットに↑キーイベントを送信します。
///
void MainWindow::onKeyUp()
{
    qDebug() << "MainWindow::onKeyUp()";

    QKeyEvent event = QKeyEvent(QEvent::KeyPress, Qt::Key_Up, Qt::NoModifier);
    qApp->sendEvent(qApp->focusWidget(), &event);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onKeyHomeOther
///
/// 非フォーカスビューにHOME, Ctrl+HOMEキーイベントを送信します。
///
void MainWindow::onKeyHomeOther()
{
    qDebug() << "MainWindow::onKeyHomeOther()";

    QKeyEvent event1 = QKeyEvent(QEvent::KeyPress, Qt::Key_Home, Qt::NoModifier);
    sendEventOther(&event1);

    QKeyEvent event2 = QKeyEvent(QEvent::KeyPress, Qt::Key_Home, Qt::ControlModifier);
    sendEventOther(&event2);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onKeyDownOther
///
/// 非フォーカスビューに↓キーイベントを送信します。
///
void MainWindow::onKeyDownOther()
{
    qDebug() << "MainWindow::onKeyDownOther()";

    QKeyEvent event = QKeyEvent(QEvent::KeyPress, Qt::Key_Down, Qt::NoModifier);
    sendEventOther(&event);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onKeyUpOther
///
/// 非フォーカスビューに↑キーイベントを送信します。
///
void MainWindow::onKeyUpOther()
{
    qDebug() << "MainWindow::onKeyUpOther()";

    QKeyEvent event = QKeyEvent(QEvent::KeyPress, Qt::Key_Up, Qt::NoModifier);
    sendEventOther(&event);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onKeyEndOther
///
/// 非フォーカスビューにEND, Ctrl+ENDキーイベントを送信します。
///
void MainWindow::onKeyEndOther()
{
    qDebug() << "MainWindow::onKeyEndOther()";

    QKeyEvent event1 = QKeyEvent(QEvent::KeyPress, Qt::Key_End, Qt::NoModifier);
    sendEventOther(&event1);

    QKeyEvent event2 = QKeyEvent(QEvent::KeyPress, Qt::Key_End, Qt::ControlModifier);
    sendEventOther(&event2);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::initBookmarkMenu
///
/// ブックマークメニューを初期化します。
///
void MainWindow::initBookmarkMenu()
{
    qDebug() << "MainWindow::initBookmarkMenu()";

    Preferences prefs(this);
    QFileIconProvider iconProvider;

    ui->menu_Bookmark->clear();
    ui->menu_Bookmark->addAction(ui->action_AddBookmark);
    ui->menu_Bookmark->addAction(ui->action_EditBookmark);
    ui->menu_Bookmark->addSeparator();
    for (int n = 1; ; n++) {
        QString name = prefs.getBookmarkEntry(n);
        if (name.isEmpty()) {
            break;
        }
        QString path = prefs.getBookmarkPath(n);

        QAction *action = ui->menu_Bookmark->addAction(
                    iconProvider.icon(QFileInfo(path)), name, this,
                    SLOT(onOpenBookmark()));
        action->setData(n);
    }

    m_actions = findChildren<QAction*>();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::sendEventOther
/// \param event    送信するイベント
///
/// 非フォーカスのビューにイベントを送信します。
///
void MainWindow::sendEventOther(QEvent *event)
{
    qDebug() << "MainWindow::sendEventOther()";

    qApp->sendEvent(inactivePanel()->visibleView(), event);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onSyncPanel
///
/// 隣のパネルと同じフォルダを表示します。
///
void MainWindow::onSyncPanel()
{
    qDebug() << "MainWindow::onSyncPanel()";

    activeModel()->setRootPath(inactiveModel()->rootPath());
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onSyncPanelTo
///
/// 隣のパネルに同じフォルダを表示します。
///
void MainWindow::onSyncPanelTo()
{
    qDebug() << "MainWindow::onSyncPanelTo()";

    inactiveModel()->setRootPath(activeModel()->rootPath());
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onToggleFullMode
/// \param checked  メニューのチェック状態
///
/// 単画面／二画面を切り替えます。
///
void MainWindow::onToggleFullMode(bool checked)
{
    qDebug() << "MainWindow::onToggleFullMode()" << checked;

    if (checked) {
        setViewMode(ModeFull);
    }
    else {
        setViewMode(ModeBasic);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onToggleSearch
/// \param checked  メニューのチェック状態
///
/// 検索ボックスの表示／非表示を切り替えます。
///
void MainWindow::onToggleSearch(bool checked)
{
    qDebug() << "MainWindow::onToggleSearch" << checked;

    // 丸投げ
    ui->LPanel->folderPanel()->toggleSearch(checked);
    ui->RPanel->folderPanel()->toggleSearch(checked);
    ui->FPanel->folderPanel()->toggleSearch(checked);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::showPreferenceDialog
///
/// 環境設定ダイアログを表示します。
///
void MainWindow::showPreferenceDialog()
{
    qDebug() << "MainWindow::showPreferenceDialog";

    PreferenceDialog dlg(this);
    if (dlg.exec() == QDialog::Accepted) {
        prefs_updated();
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onToggleHidden
/// \param checked  メニューのチェック状態
///
/// 隠しファイルの表示／非表示を切り替えます。
///
void MainWindow::onToggleHidden(bool checked)
{
    qDebug() << "MainWindow::onToggleHidden" << checked;

    if (checked) {
        activeModel()->setFilter(activeModel()->filter() | QDir::Hidden);
    }
    else {
        activeModel()->setFilter(activeModel()->filter() ^ QDir::Hidden);
    }

    activeModel()->refresh();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onToggleSystem
/// \param checked  メニューのチェック状態
///
/// システムファイルの表示／非表示を切り替えます。
///
void MainWindow::onToggleSystem(bool checked)
{
    qDebug() << "MainWindow::onToggleSystem" << checked;

    if (checked) {
        activeModel()->setFilter(activeModel()->filter() | QDir::System);
    }
    else {
        activeModel()->setFilter(activeModel()->filter() ^ QDir::System);
    }

    activeModel()->refresh();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onToggleThumbnailMode
/// \param checked  メニューのチェック状態
///
/// サムネイルモードの切り替えを行います。
///
void MainWindow::onToggleThumbnailMode(bool checked)
{
    qDebug() << "MainWindow::onToggleThumbnailMode()" << checked;

    activePanel()->folderPanel()->toggleView(checked);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onToggleToolbar
/// \param checked メニューのチェック状態
///
/// ツールバーの表示／非表示を切り替えます。
///
void MainWindow::onToggleToolbar(bool checked)
{
    ui->mainToolBar->setVisible(checked);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onCheckUpdate
/// \param silent   最新版を使用している場合に何も表示しないならtrue
///
/// アップデートの有無を確認します。
///
void MainWindow::onCheckUpdate(bool silent)
{
    qDebug() << "MainWindow::onCheckUpdate()" << silent;

    QNetworkAccessManager *manager = new QNetworkAccessManager(this);

    if (silent) {
        connect(manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(checkUpdateFinishedSilent(QNetworkReply*)));
    }
    else {
        connect(manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(checkUpdateFinished(QNetworkReply*)));
    }
#ifdef Q_OS_WIN
    manager->get(QNetworkRequest(QUrl("http://gefu.sourceforge.jp/gefu_latest_win.html")));
#elif defined(Q_OS_MAC)
    manager->get(QNetworkRequest(QUrl("http://gefu.sourceforge.jp/gefu_latest_mac.html")));
#endif
}

void MainWindow::checkUpdateFinishedSilent(QNetworkReply *reply)
{
    qDebug() << "MainWindow::checkUpdateFinishedSilent";

    checkUpdateFinished(reply, true);
}

void MainWindow::checkUpdateFinished(QNetworkReply *reply, bool silent)
{
    qDebug() << "MainWindow::checkUpdateFinished" << silent;

    if (reply->error() == QNetworkReply::NoError) {
        QString str = reply->readAll();
        str = str.split(QRegExp("[\r\n]"), QString::SkipEmptyParts).at(0);
        QRegExp rx("Gefu([0-9]+)");
        rx.indexIn(str);
        QString version = rx.cap(1);
        if (version.toInt() > VERSION_VALUE * 100) {
            QMessageBox::information(
                        this, tr("情報"),
                        tr("最新のバージョンが存在します。<br/>") +
                        tr("<a href='%1'>こちらからダウンロードしてください。</a>").arg(str));
        }
        else if (!silent) {
            QMessageBox::information(
                        this, tr("情報"),
                        tr("お使いのバージョンは最新です。"));
        }
    }
    else if (!silent){
        QMessageBox::critical(
                    this, tr("エラー"),
                    tr("最新バージョンのチェックに失敗しました。<br/>") +
                    reply->errorString());
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::initActions
///
/// アクションを初期化します。
///
void MainWindow::initActions()
{
    qDebug() << "MainWindow::initActions()";

    QList<QKeySequence> shortcuts;

    //>>>>> 追加のショートカットキーを設定する
    appendActionShortcut(ui->action_Open, "M");
    appendActionShortcut(ui->action_OpenWith, "Shift+M");
    appendActionShortcut(ui->view_Back, "Backspace");
    // MacだとShift+の形で認識されてしまうもの
    appendActionShortcut(ui->action_ScaleUp, "Shift++");
    appendActionShortcut(ui->action_Filter, "Shift+*");
    appendActionShortcut(ui->action_OpenTerminal, "Shift+>");
    appendActionShortcut(ui->action_About, "Shift+?");
    appendActionShortcut(ui->image_ScaleNormal, "Shift+=");

    // SIGNAL -> SLOT
    connect(ui->action_About, SIGNAL(triggered()), this, SLOT(onAbout()));
    connect(ui->action_AddBookmark, SIGNAL(triggered()), this, SLOT(onAddBookmark()));
    connect(ui->action_Cd, SIGNAL(triggered()), this, SLOT(onChooseFolder()));
    connect(ui->action_CheckUpdate, SIGNAL(triggered()), this, SLOT(onCheckUpdate()));
    connect(ui->action_Copy, SIGNAL(triggered()), this, SLOT(onCopy()));
    connect(ui->action_CopyFileName, SIGNAL(triggered()), this, SLOT(onCopyFileName()));
    connect(ui->action_CopyFilePath, SIGNAL(triggered()), this, SLOT(onCopyFilePath()));
    connect(ui->action_CreateFile, SIGNAL(triggered()), this, SLOT(onCreateFile()));
    connect(ui->action_CreateFolder, SIGNAL(triggered()), this, SLOT(onCreateFolder()));
    connect(ui->action_Delete, SIGNAL(triggered()), this, SLOT(onDelete()));
    connect(ui->action_EditBookmark, SIGNAL(triggered()), this, SLOT(onEditBookmark()));
    connect(ui->action_ExpandLeft, SIGNAL(triggered()), this, SLOT(onExpandLeft()));
    connect(ui->action_ExpandRight, SIGNAL(triggered()), this, SLOT(onExpandRight()));
    connect(ui->action_Filter, SIGNAL(triggered()), this, SLOT(showFilterDialog()));
    connect(ui->action_History, SIGNAL(triggered()), this, SLOT(showHistoryDialog()));
    connect(ui->action_KeyDown, SIGNAL(triggered()), this, SLOT(onKeyDown()));
    connect(ui->action_KeyDownOther, SIGNAL(triggered()), this, SLOT(onKeyDownOther()));
    connect(ui->action_KeyEnd, SIGNAL(triggered()), this, SLOT(onKeyEnd()));
    connect(ui->action_KeyEndOther, SIGNAL(triggered()), this, SLOT(onKeyEndOther()));
    connect(ui->action_KeyHome, SIGNAL(triggered()), this, SLOT(onKeyHome()));
    connect(ui->action_KeyHomeOther, SIGNAL(triggered()), this, SLOT(onKeyHomeOther()));
    connect(ui->action_KeyLeft, SIGNAL(triggered()), this, SLOT(onKeyLeft()));
    connect(ui->action_KeyUp, SIGNAL(triggered()), this, SLOT(onKeyUp()));
    connect(ui->action_KeyUpOther, SIGNAL(triggered()), this, SLOT(onKeyUpOther()));
    connect(ui->action_Move, SIGNAL(triggered()), this, SLOT(onMove()));
    connect(ui->action_Open, SIGNAL(triggered()), this, SLOT(onOpen()));
    connect(ui->action_OpenArchiver, SIGNAL(triggered()), this, SLOT(onOpenArchiver()));
    connect(ui->action_OpenEditor, SIGNAL(triggered()), this, SLOT(onOpenEditor()));
    connect(ui->action_OpenTerminal, SIGNAL(triggered()), this, SLOT(onOpenTerminal()));
    connect(ui->action_OpenWith, SIGNAL(triggered()), this, SLOT(onOpenWith()));
    connect(ui->action_Quit, SIGNAL(triggered()), this, SLOT(close()));
    connect(ui->action_Rename, SIGNAL(triggered()), this, SLOT(onRename()));
    connect(ui->action_KeyRight, SIGNAL(triggered()), this, SLOT(onKeyRight()));
    connect(ui->action_RunCommand, SIGNAL(triggered()), this, SLOT(onRunCommand()));
    connect(ui->action_SearchNext, SIGNAL(triggered()), this, SLOT(onSearchNext()));
    connect(ui->action_SearchPrev, SIGNAL(triggered()), this, SLOT(onSearchPrev()));
    connect(ui->action_Setting, SIGNAL(triggered()), this, SLOT(showPreferenceDialog()));
    connect(ui->action_ShowBookmark, SIGNAL(triggered()), this, SLOT(showBookmarkDialog()));
    connect(ui->action_Sort, SIGNAL(triggered()), this, SLOT(showSortDialog()));
    connect(ui->action_SplitCenter, SIGNAL(triggered()), this, SLOT(onSplitCenter()));
    connect(ui->action_Swap, SIGNAL(triggered()), this, SLOT(onSwap()));
    connect(ui->action_SyncPanel, SIGNAL(triggered()), this, SLOT(onSyncPanel()));
    connect(ui->action_SyncPanelTo, SIGNAL(triggered()), this, SLOT(onSyncPanelTo()));

    connect(ui->toggle_FullMode, SIGNAL(toggled(bool)), this, SLOT(onToggleFullMode(bool)));
    connect(ui->toggle_Hidden, SIGNAL(toggled(bool)), this, SLOT(onToggleHidden(bool)));
    connect(ui->toggle_Mark, SIGNAL(triggered()), this, SLOT(onToggleMark()));
    connect(ui->toggle_PreviewMode, SIGNAL(toggled(bool)), this, SLOT(onTogglePreviewMode(bool)));
    connect(ui->toggle_Search, SIGNAL(toggled(bool)), this, SLOT(onToggleSearch(bool)));
    connect(ui->toggle_System, SIGNAL(toggled(bool)), this, SLOT(onToggleSystem(bool)));
    connect(ui->toggle_ThumbnailMode, SIGNAL(toggled(bool)), this, SLOT(onToggleThumbnailMode(bool)));
    connect(ui->toggle_Toolbar, SIGNAL(toggled(bool)), this, SLOT(onToggleToolbar(bool)));

    connect(ui->view_Back, SIGNAL(triggered()), this, SLOT(view_finished()));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::setActiveModel
/// \param m    新しいアクティブモデル
///
/// アクティブモデルを変更します。
///
void MainWindow::setActiveModel(FolderModel *m)
{
    qDebug() << "MainWindow::setActiveModel()" << m;

    if (activeModel()) {
        activeModel()->disconnect(this);
        ui->action_CdHome->disconnect();
        ui->action_CdRoot->disconnect();
        ui->action_CdUp->disconnect();
        ui->action_historyBack->disconnect();
        ui->action_HistoryForward->disconnect();
        ui->action_MarkAll->disconnect();
        ui->action_MarkAllFiles->disconnect();
        ui->action_MarkAllOff->disconnect();
        ui->action_MarkInvert->disconnect();
        ui->action_Refresh->disconnect();
    }

    m->setActive();

    connect(activeModel(), SIGNAL(modelAboutToBeReset()), this, SLOT(model_PreReset()));
    connect(activeModel(), SIGNAL(modelReset()), this, SLOT(model_PostReset()));
    connect(ui->action_CdHome, SIGNAL(triggered()), activeModel(), SLOT(onCdHome()));
    connect(ui->action_CdRoot, SIGNAL(triggered()), activeModel(), SLOT(onCdRoot()));
    connect(ui->action_CdUp, SIGNAL(triggered()), activeModel(), SLOT(onCdUp()));
    connect(ui->action_historyBack, SIGNAL(triggered()), activeModel(), SLOT(onHistoryBack()));
    connect(ui->action_HistoryForward, SIGNAL(triggered()), activeModel(), SLOT(onHistoryForward()));
    connect(ui->action_MarkAll, SIGNAL(triggered()), activeModel(), SLOT(onMarkAll()));
    connect(ui->action_MarkAllFiles, SIGNAL(triggered()), activeModel(), SLOT(onMarkAllFiles()));
    connect(ui->action_MarkAllOff, SIGNAL(triggered()), activeModel(), SLOT(onMarkAllOff()));
    connect(ui->action_MarkInvert, SIGNAL(triggered()), activeModel(), SLOT(onMarkInvert()));
    connect(ui->action_Refresh, SIGNAL(triggered()), activeModel(), SLOT(refresh()));

    prefs_updated();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::activeModel
/// \return アクティブなフォルダモデルを返します。
///
FolderModel *MainWindow::activeModel() const
{
    return FolderModel::activeModel();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::activePanel
/// \return アクティブモデルを所有している可視状態のパネルを返します。
///
Panel *MainWindow::activePanel() const
{
    if (ui->FPanel->model()->isActive() && ui->FPanel->isVisible()) {
        return ui->FPanel;
    }
    if (ui->RPanel->model()->isActive() && ui->RPanel->isVisible()) {
        return ui->RPanel;
    }
    if (ui->LPanel->model()->isActive() && ui->LPanel->isVisible()) {
        return ui->LPanel;
    }

    qDebug() << ">>>>>>>>>> activePanel() Logic error <<<<<<<<<<";
    return NULL;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::inactiveModel
/// \return 非アクティブなフォルダモデルを返します。
///
FolderModel *MainWindow::inactiveModel() const
{
    if (ui->LPanel->model()->isActive()) {
        return ui->RPanel->model();
    }
    Q_ASSERT(ui->RPanel->model()->isActive());
    return ui->LPanel->model();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::inactivePanel
/// \return 非アクティブモデルを所有している可視状態のパネルを返します。
///
Panel *MainWindow::inactivePanel() const
{
    if (ui->LPanel->model() == inactiveModel() && ui->LPanel->isVisible()) {
        return ui->LPanel;
    }
    if (ui->RPanel->model() == inactiveModel() && ui->RPanel->isVisible()) {
        return ui->RPanel;
    }
    if (ui->FPanel->model() == inactiveModel() && ui->FPanel->isVisible()) {
        return ui->FPanel;
    }

    qDebug() << ">>>>>>>>>> inactivePanel() Logic error <<<<<<<<<<";
    return NULL;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::selectedItems
/// \return マークされているアイテムまたはカレントアイテムのリスト
///
QFileInfoList MainWindow::selectedItems() const
{
    QFileInfoList list = activeModel()->markedItems();
    if (list.isEmpty()) {
        Q_ASSERT(focusItemView()->currentIndex().isValid());
        list << activeModel()->fileInfo(focusItemView()->currentIndex());
    }

    return list;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::focusItemView
/// \return フォーカスを持つアイテムビュー
///
QAbstractItemView *MainWindow::focusItemView() const
{
    return static_cast<QAbstractItemView*>(qApp->focusWidget());
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::setViewMode
/// \param mode 新しいビューモード
///
/// ビューモードを変更します。
///
void MainWindow::setViewMode(Mode mode)
{
    qDebug() << "MainWindow::setViewMode()" << mode;

    QWidget *newFocusWidget = NULL;

    m_prevMode = m_viewMode;
    m_viewMode = mode;

    switch (mode) {
    case ModeBasic:
        ui->FPanel->setVisible(false);
        ui->splitter->setVisible(true);
        newFocusWidget = activePanel()->folderPanel()->itemView();

        ui->LPanel->setViewItem();
        ui->RPanel->setViewItem();
        ui->FPanel->setViewItem();
        if (m_prevMode == ModeFull) {
            activePanel()->folderPanel()->setItemView(focusItemView()->objectName());
            activePanel()->folderPanel()->itemView()->setCurrentIndex(
                        focusItemView()->currentIndex());
            ui->FPanel->setModel(NULL);
        }
        break;

    case ModeFull:
        if (m_prevMode == ModeBasic) {
            ui->FPanel->setModel(activeModel());
            ui->FPanel->folderPanel()->setItemView(focusItemView()->objectName());
            ui->FPanel->folderPanel()->itemView()->setCurrentIndex(
                            focusItemView()->currentIndex());
        }
        ui->splitter->setVisible(false);
        ui->FPanel->setVisible(true);
        ui->FPanel->setViewItem();
        newFocusWidget = activePanel()->folderPanel()->itemView();
        break;

    case ModeView:
        ui->FPanel->setViewItem(focusItemView()->currentIndex());
        ui->splitter->setVisible(false);
        ui->FPanel->setVisible(true);
        newFocusWidget = ui->FPanel->visibleView();
        break;

    case ModePreview:
        inactivePanel()->setViewItem(focusItemView()->currentIndex());
        break;
    }

    if (newFocusWidget)
        newFocusWidget->setFocus();
    else
        updateActions();

    qDebug() << "MainWindow::setViewMode() end";
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::startProcess
/// \param cmd      実行するコマンド
/// \param errMsg   エラー時の表示メッセージ
/// \return true：成功, false：失敗
///
bool MainWindow::startProcess(const QString &cmd, const QString &errMsg)
{
    qDebug() << "MainWindow::startProcess" << cmd << errMsg;

    QProcess process(this);
    process.setWorkingDirectory(activeModel()->rootPath());
    if (!process.startDetached(cmd)) {
        QMessageBox::critical(this, tr("エラー"), errMsg + "<br/>" + cmd);
        return false;
    }
    return true;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::updateActions
///
/// アクションの有効／無効を設定します。
///
void MainWindow::updateActions()
{
    qDebug() << "MainWindow::updateActions";

    bool isView = true;
    bool isSearch = false;
    bool isImageView = false;
    bool toggleSearch = true;

    QWidget *w = qApp->focusWidget();
    if (w->objectName() == "scrollArea") {
        w = static_cast<QScrollArea*>(w)->widget();
    }

    if (w->objectName() == "folderView" || w->objectName() == "thumbnailView") {
        setEnabledAllActions(true);
        // 履歴
        ui->action_historyBack->setEnabled(!activeModel()->isHistoryBegin());
        ui->action_HistoryForward->setEnabled(!activeModel()->isHistoryEnd());
        // チェック状態
        ui->toggle_Hidden->blockSignals(true);
        ui->toggle_Hidden->setChecked(activeModel()->filter() & QDir::Hidden);
        ui->toggle_Hidden->blockSignals(false);

        ui->toggle_System->blockSignals(true);
        ui->toggle_System->setChecked(activeModel()->filter() & QDir::System);
        ui->toggle_System->blockSignals(false);

        ui->toggle_ThumbnailMode->blockSignals(true);
        ui->toggle_ThumbnailMode->setChecked(w->objectName() == "thumbnailView");
        ui->toggle_ThumbnailMode->blockSignals(false);

        // 現在の選択アイテムが".."の場合、ファイル操作系アクションは無効にする
        QFileInfoList list = selectedItems();
        if (list.size() == 1 && list[0].fileName() == "..") {
            ui->action_Copy->setEnabled(false);
            ui->action_Delete->setEnabled(false);
            ui->action_Move->setEnabled(false);
            ui->action_Rename->setEnabled(false);
        }

        // 単画面またはプレビューモードの場合
        if (!ui->LPanel->folderPanel()->isVisible() ||
            !ui->RPanel->folderPanel()->isVisible())
        {
            ui->action_Copy->setEnabled(false);
            ui->action_Move->setEnabled(false);
            ui->action_Swap->setEnabled(false);
            ui->action_SyncPanel->setEnabled(false);
            ui->action_SyncPanelTo->setEnabled(false);
        }

        // 単画面の場合
        if (m_viewMode == ModeFull) {
            ui->toggle_PreviewMode->setEnabled(false);
            ui->action_SplitCenter->setEnabled(false);
            ui->action_ExpandLeft->setEnabled(false);
            ui->action_ExpandRight->setEnabled(false);
            ui->action_KeyDownOther->setEnabled(false);
            ui->action_KeyEndOther->setEnabled(false);
            ui->action_KeyHomeOther->setEnabled(false);
            ui->action_KeyUpOther->setEnabled(false);
        }

        if (w->objectName() == "thumbnailView") {
            isImageView = true;
        }
    }
    else if (w->objectName() == "searchBox"){
        setEnabledAllActions(false);
        isView = false;
        isSearch = true;
    }
    else if (w->objectName() == "locationBox") {
        setEnabledAllActions(false);
        isView = false;
    }
    else if (w->objectName() == "textView") {
        setEnabledAllActions(false);
        toggleSearch = false;

        reconnectAction(ui->text_ConvertFromEUC, SIGNAL(triggered()), w, SLOT(onConvertFromEUC()));
        reconnectAction(ui->text_ConvertFromJIS, SIGNAL(triggered()), w, SLOT(onConvertFromJIS()));
        reconnectAction(ui->text_ConvertFromSJIS, SIGNAL(triggered()), w, SLOT(onConvertFromSJIS()));
        reconnectAction(ui->text_ConvertFromUTF8, SIGNAL(triggered()), w, SLOT(onConvertFromUTF8()));
        reconnectAction(ui->text_ConvertFromUTF16, SIGNAL(triggered()), w, SLOT(onConvertFromUTF16()));
        reconnectAction(ui->text_ConvertFromUTF16BE, SIGNAL(triggered()), w, SLOT(onConvertFromUTF16BE()));
        reconnectAction(ui->text_ConvertFromUTF16LE, SIGNAL(triggered()), w, SLOT(onConvertFromUTF16LE()));
        reconnectAction(ui->view_SelectAll, SIGNAL(triggered()), w, SLOT(onSelectAll()));
        reconnectAction(ui->text_Copy, SIGNAL(triggered()), w, SLOT(onCopy()));
    }
    else if (w->objectName() == "imageView") {
        setEnabledAllActions(false);
        toggleSearch = false;
        isImageView = true;

        reconnectAction(ui->image_FitToWindow, SIGNAL(triggered()), w, SLOT(onFitToWindow()));
        reconnectAction(ui->image_ScaleNormal, SIGNAL(triggered()), w, SLOT(onScaleNormal()));
        reconnectAction(ui->image_Rotate90, SIGNAL(triggered()), w, SLOT(onRotate90()));
        reconnectAction(ui->image_Rotate180, SIGNAL(triggered()), w, SLOT(onRotate180()));
    }
    else if (w->objectName() == "hexView") {
        setEnabledAllActions(false);
        toggleSearch = false;

        reconnectAction(ui->view_SelectAll, SIGNAL(triggered()), w, SLOT(onSelectAll()));
        reconnectAction(ui->text_Copy, SIGNAL(triggered()), w, SLOT(onCopy()));
    }
    else if (w->objectName() == "progressBar") {
        setEnabledAllActions(false);
        toggleSearch = false;
        isView = false;
    }
    else {
        toggleSearch = false;
        isView = false;
    }

    ui->toggle_Toolbar->blockSignals(true);
    ui->toggle_Toolbar->setChecked(ui->mainToolBar->isVisible());
    ui->toggle_Toolbar->blockSignals(false);

    ui->action_About->setEnabled(true);
    ui->action_CheckUpdate->setEnabled(true);
    ui->action_KeyDown->setEnabled(isView);
    ui->action_KeyEnd->setEnabled(isView);
    ui->action_KeyHome->setEnabled(isView);
    ui->action_KeyUp->setEnabled(isView);
    ui->action_Quit->setEnabled(true);
    ui->action_SearchNext->setEnabled(isSearch);
    ui->action_SearchPrev->setEnabled(isSearch);
    ui->action_Setting->setEnabled(true);
    ui->toggle_Search->setEnabled(toggleSearch);
    ui->view_Back->setEnabled(m_viewMode == ModeView);

    if (isImageView) {
        ui->action_ScaleDown->setText(tr("縮小"));
        ui->action_ScaleUp->setText(tr("拡大"));
        ui->action_KeyLeft->setEnabled(false);
        ui->action_KeyRight->setEnabled(false);
    }
    else {
        ui->action_ScaleDown->setText(tr("文字を小さく"));
        ui->action_ScaleUp->setText(tr("文字を大きく"));
        ui->action_KeyLeft->setEnabled(isView);
        ui->action_KeyRight->setEnabled(isView);
    }
    if (isView) {
        reconnectAction(ui->action_ScaleDown, SIGNAL(triggered()), w, SLOT(onScaleDown()));
        reconnectAction(ui->action_ScaleUp, SIGNAL(triggered()), w, SLOT(onScaleUp()));
    }
    else {
        ui->action_ScaleDown->setEnabled(false);
        ui->action_ScaleUp->setEnabled(false);
    }

    ui->toggle_FullMode->setEnabled(m_viewMode == ModeBasic || m_viewMode == ModeFull);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::setEnabledAllActions
/// \param enable   有効にする場合はtrue
///
/// 全てのアクションを有効または無効にします。
///
void MainWindow::setEnabledAllActions(bool enable)
{
    qDebug() << "MainWindow::setEnabledAllActions()" << enable;
    foreach (QAction *action, m_actions) {
        action->setEnabled(enable);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::onAbout
///
/// Aboutダイアログを表示します。
///
void MainWindow::onAbout()
{
    QMessageBox::about(
                this,
                tr("げふぅ について"),
                windowTitle() +
                tr("<center>Gefu is an Experimental File Utility.<br/>"
                   "<small>（げふぅは実験的なファイルユーティリティです）</small></center>"
                   "<p>最新版の情報は<a href='http://gefu.sourceforge.jp/'>公式サイト</a>で公開しています。</p>"
                   "<p><small>Copyright 2014 @miyabi_satoh All rights reserved.</small></p>"));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::closeEvent
/// \param event    クローズイベント
///
/// 終了処理を行います。
///
void MainWindow::closeEvent(QCloseEvent *event)
{
    qDebug() << "MainWindow::closeEvent()";
    Preferences prefs(this);

    if (prefs.isConfirmQuit()) {
        QMessageBox msgBox;
        QCheckBox *checkBox = new QCheckBox();
        checkBox->setText(tr("次回以降は確認しない"));
        msgBox.setCheckBox(checkBox);
        msgBox.setText(tr("終了しますか？"));
        msgBox.setStandardButtons(QMessageBox::Yes | QMessageBox::No);
        msgBox.setIcon(QMessageBox::Question);

        if (msgBox.exec() == QMessageBox::No) {
            qDebug() << ">>>>> ユーザによりキャンセルされました <<<<<";
            event->ignore();
            return;
        }
        prefs.setConfirmQuit(!checkBox->isChecked());
    }

    prefs.saveModel("Left", ui->LPanel->model());
    prefs.saveModel("Right", ui->RPanel->model());
    prefs.saveWindow(this);
    QMainWindow::closeEvent(event);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief MainWindow::eventFilter
/// \param watched  イベントの発生元オブジェクト
/// \param e        発生したイベント
/// \return イベントを処理した場合はtrueを返します。
///
bool MainWindow::eventFilter(QObject *watched, QEvent *e)
{
    switch (e->type()) {
    case QEvent::KeyPress:
        // キーイベントの処理
        return onKeyPressEvent(watched, static_cast<QKeyEvent*>(e));

    case QEvent::ContextMenu:
        // コンテキストメニューイベントの処理
        return onContextMenuEvent(watched, static_cast<QContextMenuEvent*>(e));

    default:
        break;
    }

    return false;
}
