#include "preferences.h"
#include "imageview.h"

#include <QApplication>
#include <QDebug>
#include <QMouseEvent>
#include <QPainter>
#include <QScrollArea>
#include <QScrollBar>

///////////////////////////////////////////////////////////////////////////////
/// \brief ImageView::ImageView
/// \param parent   親ウィジェット
///
/// コンストラクタ
///
ImageView::ImageView(QScrollArea *parent) :
    QWidget(parent),
    m_dragStartPos(-1, -1)
{
    m_scrollArea = parent;
    m_scrollArea->setWidget(this);
    setObjectName("imageView");
    setCursor(Qt::OpenHandCursor);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ImageView::setData
/// \param pixmap   ピクスマップ
///
/// ピクスマップを設定します。
///
void ImageView::setData(const QPixmap &pixmap)
{
    m_pixmap = pixmap;
    m_rotateDeg = 0;
    m_scaleFactor = -1;
    resizePixmap();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ImageView::scaleFactor
/// \return 倍率を返します。
///
double ImageView::scaleFactor()
{
    if (m_scaleFactor <= 0) {
        // 等倍または縮小
        double scaleX, scaleY;
        if (m_rotateDeg == 90 || m_rotateDeg == 270) {
            scaleX = 1.0 * m_scrollArea->size().height() / m_pixmap.width();
            scaleY = 1.0 * m_scrollArea->size().width() / m_pixmap.height();
        }
        else {
            scaleX = 1.0 * m_scrollArea->size().width() / m_pixmap.width();
            scaleY = 1.0 * m_scrollArea->size().height() / m_pixmap.height();
        }
        if (m_scaleFactor == -1 && scaleX >= 1 && scaleY >= 1) {
            return 1;
        }
        return qMin(scaleX, scaleY);
    }

    return m_scaleFactor;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ImageView::resizePixmap
///
/// ピクスマップのサイズを変更します。
///
void ImageView::resizePixmap()
{
    double scaleFactor = this->scaleFactor();
    m_scaledPixmap = m_pixmap.scaled(m_pixmap.size() * scaleFactor,
                                     Qt::IgnoreAspectRatio,
                                     Qt::SmoothTransformation);
    if (m_rotateDeg == 90 || m_rotateDeg == 270) {
        setMinimumSize(m_scaledPixmap.height(), m_scaledPixmap.width());
    }
    else {
        setMinimumSize(m_scaledPixmap.width(), m_scaledPixmap.height());
    }

    emit statusChanged(QString("%1 x %2 x %3(%4%)")
                       .arg(m_pixmap.width())
                       .arg(m_pixmap.height())
                       .arg(m_pixmap.depth())
                       .arg(static_cast<int>(scaleFactor * 100)));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ImageView::onFitToWindow
///
/// ピクスマップのサイズをウィンドウに合わせます。
///
void ImageView::onFitToWindow()
{
    qDebug() << "ImageView::onFitToWindow()";

    m_scaleFactor = 0;
    m_scaleFactor = this->scaleFactor();
    resizePixmap();
    update();
    m_scaleFactor = 0;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ImageView::onRotate90
///
/// 右に90度回転します。
///
void ImageView::onRotate90()
{
    m_rotateDeg += 90;
    m_rotateDeg %= 360;
    resizePixmap();
    update();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ImageView::onRotate180
///
/// 180度回転します。
///
void ImageView::onRotate180()
{
    m_rotateDeg += 180;
    m_rotateDeg %= 360;
    resizePixmap();
    update();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ImageView::onScaleDown
///
/// 縮小します。
///
void ImageView::onScaleDown()
{
    m_scaleFactor = this->scaleFactor() * 0.8;
    resizePixmap();
    update();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ImageView::onScaleNormal
///
/// 等倍表示します。
///
void ImageView::onScaleNormal()
{
    m_scaleFactor = 1;
    resizePixmap();
    update();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ImageView::onScaleUp
///
/// 拡大します。
///
void ImageView::onScaleUp()
{
    m_scaleFactor = this->scaleFactor() * 1.25;
    resizePixmap();
    update();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ImageView::setVisible
/// \param visible  表示(true)/非表示(false)
///
/// 表示時の処理を行います。
///
void ImageView::setVisible(bool visible)
{
    if (visible) {
        Preferences prefs(this);
        QPalette pal = this->palette();
        pal.setColor(this->backgroundRole(), prefs.getImageViewBgColor());
        this->setPalette(pal);
        this->setAutoFillBackground(true);
    }

    QWidget::setVisible(visible);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ImageView::mousePressEvent
/// \param e    マウスイベントオブジェクト
///
/// マウスクリック時の処理を行います。
///
void ImageView::mousePressEvent(QMouseEvent *e)
{
    if (e->button() == Qt::LeftButton) {
        setCursor(Qt::ClosedHandCursor);
        m_dragStartPos = e->pos();
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ImageView::mouseReleaseEvent
///
/// マウスボタンが離された場合の処理を行います。
///
void ImageView::mouseReleaseEvent(QMouseEvent *)
{
    setCursor(Qt::OpenHandCursor);
    m_dragStartPos = QPoint(-1, -1);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ImageView::mouseMoveEvent
/// \param e    マウスイベントオブジェクト
///
/// マウス移動時の処理を行います。
///
void ImageView::mouseMoveEvent(QMouseEvent *e)
{
    if (m_dragStartPos.x() == -1 && m_dragStartPos.y() == -1)
        return;

    if ((e->pos() - m_dragStartPos).manhattanLength() < qApp->startDragDistance())
        return;

    QScrollBar *hBar = m_scrollArea->horizontalScrollBar();
    QScrollBar *vBar = m_scrollArea->verticalScrollBar();
    QPoint delta = e->pos() - m_dragStartPos;
    hBar->setValue(hBar->value() - delta.x());
    vBar->setValue(vBar->value() - delta.y());
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ImageView::paintEvent
///
/// 描画イベントを処理します。
///
void ImageView::paintEvent(QPaintEvent *)
{
    QPainter painter(this);

    painter.translate(size().width() / 2, size().height() / 2);
    painter.rotate(m_rotateDeg);
    painter.drawPixmap(-m_scaledPixmap.width() / 2,
                       -m_scaledPixmap.height() / 2,
                       m_scaledPixmap);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief ImageView::resizeEvent
/// \param e    リサイズイベントオブジェクト
///
/// リサイズ時の処理を行います。
///
void ImageView::resizeEvent(QResizeEvent *e)
{
    QWidget::resizeEvent(e);

    resizePixmap();
}
