#include "global.h"
#include "preferences.h"
#include "mainwindow.h"
#include "folderpanel.h"
#include "ui_folderpanel.h"

#include <QDebug>
#include <QSettings>
#include <QStatusBar>

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderPanel::FolderPanel
/// \param parent   親ウィジェット
///
/// コンストラクタ
///
FolderPanel::FolderPanel(QWidget *parent) :
    QWidget(parent),
    ui(new Ui::FolderPanel),
    m_mainWnd(NULL)
{
    ui->setupUi(this);
    ui->searchBox->setText("");
    ui->searchBox->setVisible(false);
    ui->thumbnailView->setVisible(false);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderPanel::~FolderPanel
///
/// デストラクタ
///
FolderPanel::~FolderPanel()
{
    delete ui;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderPanel::initialize
/// \param w    メインウィンドウオブジェクト
///
/// 初期化処理を行います。
///
void FolderPanel::initialize(MainWindow *w)
{
    qDebug() << "FolderPanel::initialize();";

    // フォルダビューを初期化する
    ui->folderView->initialize(w);
    // サムネイルビューを初期化する
    ui->thumbnailView->initialize(w);

    connect(ui->bookmarkBtn, SIGNAL(clicked()), w, SLOT(onAddBookmark()));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderPanel::itemView
/// \return アイテムビューを返します。
///
QAbstractItemView *FolderPanel::itemView() const
{
    if (ui->thumbnailView->isVisible()) {
        return ui->thumbnailView;
    }
    return ui->folderView;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderPanel::model
/// \return 関連付けられたフォルダモデルを返します。
///
FolderModel *FolderPanel::model() const
{
    return static_cast<FolderModel*>(ui->folderView->model());
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderPanel::setModel
/// \param m    設定するモデル
///
/// モデルを設定します。
///
void FolderPanel::setModel(FolderModel *m)
{
    qDebug() << "FolderPanel::setModel()" << m;
    if (model()) {
        model()->disconnect(this);
    }

    ui->folderView->setModel(m);
    ui->thumbnailView->setModel(m);

    if (model()) {
        connect(model(), SIGNAL(dataChanged(QModelIndex,QModelIndex)),
                this, SLOT(model_dataChanged(QModelIndex,QModelIndex)));
        connect(model(), SIGNAL(modelReset()), this, SLOT(model_Reset()));

        model_Reset();
        model_dataChanged(QModelIndex(), QModelIndex());
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderPanel::toggleSearch
/// \param checked  メニューのチェック状態
///
/// 検索ボックスの表示／非表示を切り替えます。
///
void FolderPanel::toggleSearch(bool checked)
{
    qDebug() << "FolderPanel::toggleSearch()" << checked;
    if (!isVisible()) {
        return;
    }

    if (checked) {
        if (!model()->isActive()) {
            return;
        }

        ui->searchBox->setVisible(true);
        ui->searchBox->setFocus();
        ui->searchBox->selectAll();
    }
    else {
        if (!ui->searchBox->isVisible()) {
            return;
        }

        if (ui->searchBox->hasFocus()) {
            itemView()->setFocus();
        }
        ui->searchBox->setVisible(false);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderPanel::toggleView
/// \param checked  メニューのチェック状態
///
/// リスト／サムネイルを切り替えます。
///
void FolderPanel::toggleView(bool checked)
{
    QModelIndex index = itemView()->currentIndex();
    if (checked) {
        ui->folderView->setVisible(false);
        ui->thumbnailView->setVisible(true);
        ui->thumbnailView->setFocus();
    }
    else {
        ui->thumbnailView->setVisible(false);
        ui->folderView->setVisible(true);
        ui->folderView->setFocus();
    }
    itemView()->setCurrentIndex(index);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderPanel::updateAppearance
///
/// 外観を変更します。
///
void FolderPanel::updateAppearance(const Preferences &prefs)
{
    qDebug() << "FolderPanel::updateAppearance()";

    QPalette pal;

    pal = ui->filterLabel->palette();
    pal.setColor(ui->filterLabel->backgroundRole(), prefs.folderViewBgColor(model()->isActive()));
    pal.setColor(ui->filterLabel->foregroundRole(), prefs.folderViewFgColor(model()->isActive()));
    ui->filterLabel->setAutoFillBackground(true);
    ui->filterLabel->setPalette(pal);

    pal = ui->locationBox->palette();
    pal.setColor(QPalette::Base, prefs.locationBoxBgColor(model()->isActive()));
    pal.setColor(QPalette::Text, prefs.locationBoxFgColor(model()->isActive()));
    ui->locationBox->setPalette(pal);
    ui->locationBox->setFont(prefs.getLocationBoxFont());

    pal = ui->searchBox->palette();
    pal.setColor(QPalette::Base, prefs.getSearchBoxBgColor());
    pal.setColor(QPalette::Text, prefs.getSearchBoxFgColor());
    ui->searchBox->setPalette(pal);
    ui->searchBox->setFont(prefs.getSearchBoxFont());

    pal = ui->folderView->palette();
    pal.setColor(QPalette::Base, prefs.folderViewBgColor(model()->isActive()));
    ui->folderView->setPalette(pal);

    pal = ui->thumbnailView->palette();
    pal.setColor(QPalette::Base, prefs.folderViewBgColor(model()->isActive()));
    ui->thumbnailView->setPalette(pal);

    QHeaderView *header = ui->folderView->verticalHeader();
    QFont font = prefs.getFolderViewFont();
    header->setDefaultSectionSize(QFontMetrics(font).height() * prefs.getLineHeight());
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderPanel::searchNext
/// \param step 次なら1, 前なら-1
///
/// 次または前のアイテムを検索します。
///
void FolderPanel::searchNext(int step)
{
    qDebug() << "FolderPanel::searchNext()" << step;

    QModelIndex index = model()->search(ui->searchBox->text(),
                                        itemView()->currentIndex(),
                                        step);
    showSearchResult(index);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderPanel::setItemView
/// \param name "folderView"または"thumbnailView"
///
/// 指定されたビューを可視状態にします。
///
void FolderPanel::setItemView(const QString &name)
{
    ui->folderView->setVisible(false);
    ui->thumbnailView->setVisible(false);
    findChild<QWidget*>(name)->setVisible(true);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderPanel::showSearchResult
/// \param index    検索されたアイテム
///
/// 検索結果を表示します。
///
void FolderPanel::showSearchResult(const QModelIndex &index)
{
    Preferences prefs(this);

    QPalette pal = ui->searchBox->palette();
    if (index.isValid()) {
        itemView()->setCurrentIndex(index);
        pal.setColor(QPalette::Base, prefs.getSearchBoxBgColor());
        pal.setColor(QPalette::Text, prefs.getSearchBoxFgColor());
    }
    else {
        pal.setColor(QPalette::Base, prefs.getSearchBoxUnmatchBgColor());
        pal.setColor(QPalette::Text, prefs.getSearchBoxUnmatchFgColor());
    }
    ui->searchBox->setPalette(pal);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderPanel::model_dataChanged
/// \param topLeft      (使用しません)
/// \param bottomRight  (使用しません)
///
/// アイテムの選択状態変更に伴う処理を行います。
///
void FolderPanel::model_dataChanged(const QModelIndex &topLeft, const QModelIndex &bottomRight)
{
    Q_UNUSED(topLeft);
    Q_UNUSED(bottomRight);

    QFileInfoList list = model()->markedItems();
    if (list.isEmpty()) {
        ui->filterLabel->setText(tr("フィルタ：%1").arg(model()->nameFilters().join(" ")));
    }
    else {
        int numFolders = 0;
        int numFiles = 0;
        qint64 totalSize = 0;
        foreach (const QFileInfo &fi, list) {
            if (fi.isDir()) {
                ++numFolders;
            }
            else {
                ++numFiles;
                totalSize += fi.size();
            }
        }

        QString text;
        if (numFolders > 0) {
            text += tr("%1個のフォルダ ").arg(numFolders);
        }
        if (numFiles > 0) {
            text += tr("%1個のファイル ").arg(numFiles);
        }
        if (!text.isEmpty()) {
            text += tr("を選択 合計%1").arg(fileSizeToString(totalSize));
        }
        ui->filterLabel->setText(text);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderPanel::model_Reset
///
/// モデルリセット後の処理を行います。
///
void FolderPanel::model_Reset()
{
    ui->locationBox->setText(model()->rootPath());
    ui->filterLabel->setText(tr("フィルタ：%1").arg(model()->nameFilters().join(" ")));
}

///////////////////////////////////////////////////////////////////////////////
/// \brief FolderPanel::on_searchBox_textEdited
/// \param arg1 入力テキスト
///
/// 検索ボックスのテキストが変更された場合の処理を行います。
///
void FolderPanel::on_searchBox_textEdited(const QString &arg1)
{
    if (arg1.right(1) == "/") {
        ui->searchBox->setText(arg1.left(arg1.size() - 1));
        // 検索終了
        itemView()->setFocus();
    }
    else {
        showSearchResult(model()->search(arg1));
    }
}
