package cerot.blight.sequence;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;

import cerot.blight.btrace.MethodCall;
import cerot.blight.btrace.MethodExpression;
import cerot.blight.btrace.MethodReturn;
import cerot.blight.sequence.entity.Connector;
import cerot.blight.sequence.entity.ConnectorType;
import cerot.blight.sequence.entity.EndPoint;
import cerot.blight.sequence.entity.Node;

/**
 * メソッド表現をシーケンス図用モデルに変換するコンバータクラス。
 * @author cero-t
 * 
 */
public final class SequenceConverter {
	/**
	 * メソッド表現のリストを結線のリストに変換します。
	 * @param methodExpressionList メソッド表現のリスト
	 * @return 結線のリスト
	 */
	public static List<Connector> toConnectorList(
			List<MethodExpression> methodExpressionList) {
		if (methodExpressionList == null) {
			return null;
		}

		List<Connector> result = new ArrayList<Connector>(methodExpressionList
				.size());

		Map<String, Node> nodeMap = new HashMap<String, Node>();
		Node startNode = new Node();
		startNode.setName("START");
		Stack<Node> nodeStack = new Stack<Node>();
		nodeStack.push(startNode);

		for (MethodExpression method : methodExpressionList) {

			Connector connector = null;
			if (method instanceof MethodCall) {
				connector = processMethodCall((MethodCall) method, nodeMap,
						nodeStack);
			} else if (method instanceof MethodReturn) {
				connector = processMethodReturn((MethodReturn) method,
						nodeStack);
			}

			if (connector != null) {
				result.add(connector);
			}
		}

		return result;
	}

	/**
	 * メソッド呼び出しを処理します。
	 * @param method メソッド呼び出し。
	 * @param nodeMap ノードのマップ
	 * @param nodeStack ノードのスタック
	 * @return メソッド呼び出しを表現する結線
	 */
	private static Connector processMethodCall(MethodCall method,
			Map<String, Node> nodeMap, Stack<Node> nodeStack) {
		Node previousNode = nodeStack.lastElement();
		String className = method.getClassName();

		if (className.equals(previousNode.getName())) {
			nodeStack.push(previousNode);
			return null;
		}

		Node toNode = nodeMap.get(className);
		if (toNode == null) {
			toNode = new Node();
			toNode.setName(className);
			nodeMap.put(className, toNode);
		}

		Connector result = new Connector();
		EndPoint from = new EndPoint();
		result.setFrom(from);
		from.setParentNode(previousNode);

		EndPoint to = new EndPoint();
		result.setTo(to);
		to.setParentNode(toNode);

		String methodName = method.getMethodName();
		result.setName(methodName);
		result.setConnectorType(ConnectorType.METHOD_CALL);

		nodeStack.push(toNode);

		return result;
	}

	/**
	 * メソッドリターンを処理します。
	 * @param method メソッドリターン
	 * @param nodeStack ノードのスタック
	 * @return メソッドリターンを表現する結線
	 */
	private static Connector processMethodReturn(MethodReturn method,
			Stack<Node> nodeStack) {
		Node currentNode = nodeStack.pop();
		Node previousNode = nodeStack.lastElement();

		if (currentNode.getName().equals(previousNode.getName())) {
			return null;
		}

		Connector result = new Connector();
		EndPoint from = new EndPoint();
		result.setFrom(from);
		from.setParentNode(currentNode);

		EndPoint to = new EndPoint();
		result.setTo(to);
		to.setParentNode(previousNode);

		result.setName("Return");
		result.setConnectorType(ConnectorType.METHOD_RETURN);

		return result;
	}

	/**
	 * プライベートコンストラクタ。呼び出し禁止。
	 */
	private SequenceConverter() {
		// Do Nothing.
	}
}
