/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBConstValue;
import org.opengion.fukurou.util.StringUtil ;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.Map;
import java.util.LinkedHashMap;
import java.util.Locale ;										// 6.4.1.2 (2016/01/22)

/**
 * TableUpdateTag にパラメーターを渡す為のタグクラスです。
 *
 * 汎用的なデータベース登録処理を行えるタグ tableUpdate タグを新規作成します。
 * これは、具体的なSLQを作成する tableUpdateParam タグと組み合わせて使用できます。
 * tableUpdate タグは、queryType に JDBCTableUpdate を指定します。基本的にこれだけ
 * です。tableUpdateParam では、sqlType に、INSERT,COPY,UPDATE,MODIFY,DELETE の
 * どれかを指定する事で、SQL文のタイプを指定します。COPY,MODIFY は command と
 * 関連を持たす為に追加しているタイプで、INSERTやUPDATE と同じ処理を行います。
 * tableUpdateParam の table には、作成したい SQL のテーブルを指定します。
 * where 属性は、検索結果の DBTableModel の更新時に使用する条件を指定します。
 *
 * @og.formSample
 * ●形式：&lt;og:tableUpdate command="{&#064;command}" queryType="JDBCTableUpdate" &gt;
 *            &lt;og:tableUpdateParam
 *                sqlType       = "{&#064;sqlType}"       // INSERT,COPY,UPDATE,MODIFY,DELETE
 *                table         = "{&#064;TABLE_NAME}"    // 処理対象のテーブル名
 *                names         = "{&#064;names}"         // 処理対象のカラム名
 *                omitNames     = "{&#064;omitNames}"     // 処理対象外のカラム名
 *                where         = "{&#064;where}"         // 処理対象を特定するキー
 *                whereNames    = "{&#064;whereNames}"    // 処理対象を特定するキー条件(where句)をCSV形式
 *                constKeys     = "{&#064;constKeys}"     // 処理カラム名の中の固定情報カラム名
 *                constVals     = "{&#064;constVals}"     // 処理カラム名の中の固定情報設定値
 *                asNames       = "{&#064;asNames}"       // 別名を付けたカラム名(select A as B from TBL の B を指定)
 *                orgNames      = "{&#064;orgNames}"      // tableの実際のカラム名(select A as B from TBL の A を指定)
 *                funcKeys      = "{&#064;funcKeys}"      // 関数等を設定するカラム名
 *                funcVals      = "{&#064;funcVals}"      // 関数等の設定値
 *                logicalDelete = "{&#064;logicalDelete}" // sqlTypeがDELETEの場合にもUPDATE文を発行
 *            /&gt;
 *         &lt;/og:tableUpdate&gt;
 *
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:tableUpdateParam
 *       sqlType          ○【TAG】BODY部に書かれている SQLタイプを指定します(必須)
 *       table            ○【TAG】処理対象のテーブル名を指定します(必須)
 *       names              【TAG】処理対象のカラム名をCSV形式で複数指定します
 *       omitNames          【TAG】処理対象外のカラム名をCSV形式で複数指定します
 *       where              【TAG】処理対象を特定するキー条件(where句)を指定します
 *       whereNames         【TAG】処理対象を特定するキー条件(where句)をCSV形式で複数指定します
 *       constKeys          【TAG】設定値を固定値と置き換える対象となるカラム名をCSV形式で複数指定します
 *       constVals          【TAG】設定値を固定値と置き換える対象となる設定値をCSV形式で複数指定します
 *       funcKeys           【TAG】関数等を設定するカラム名をCSV形式で複数指定します
 *       funcVals           【TAG】関数等の設定値をCSV形式で複数指定します
 *       asNames            【TAG】別名を付けたカラム名(select A as B from TBL の B を指定)をCSV形式で複数指定します
 *       orgNames           【TAG】tableの実際のカラム名(select A as B from TBL の A を指定)をCSV形式で複数指定します
 *       quotCheck          【TAG】リクエスト情報の シングルクォート(') 存在チェックを実施するかどうか[true/false]を設定します (初期値:USE_SQL_INJECTION_CHECK[=true])
 *       constObjKey        【TAG】固定情報カラムの処理オブジェクトを特定するキーを設定します(初期値:SYSTEM_ID)
 *       logicalDelete      【TAG】sqlType="DELETE"の場合に論理削除(UPDATE)を行うかどうかを指定します(初期値:false)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *    ・【entry.jsp】
 *         &lt;og:tableUpdate command="{&#064;command}" queryType="JDBCTableUpdate" &gt;
 *            &lt;og:tableUpdateParam
 *               sqlType  = "{&#064;sqlType}"
 *               table    = "{&#064;MEM.TABLE_NAME}"
 *               where    = "ROWID = [ROWID]"
 *            /&gt;
 *         &lt;/og:tableUpdate&gt;
 *
 * @og.rev 3.8.8.0 (2007/12/22) 新規作成
 * @og.rev 4.1.2.0 (2008/03/12) 実装の大幅な修正
 * @og.group ＤＢ登録
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableUpdateParamTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.9.8.0 (2018/05/28)" ;
	private static final long serialVersionUID = 698020180528L ;

	/** sqlType属性に設定できる値			{@value} */
	public static final String SQL_TYPE  = "|INSERT|COPY|UPDATE|MODIFY|DELETE|" ;

	// 3.8.0.4 (2005/08/08) 印刷時に使用するシステムID
	private static final String SYSTEM_ID =HybsSystem.sys( "SYSTEM_ID" );

	// 4.3.6.0 (2009/05/01) デフォルトで利用するconstObjのシステムリソース名
	private static final String DEFAULT_CONST_OBJ = HybsSystem.sys( "DEFAULT_CONST_CLASS" );

	private String		sqlType		;				// INSERT,COPY,UPDATE,MODIFY,DELETE
	private String		table		;				// 処理対象のテーブル名
	private String[]	names		;				// 処理対象のカラム名
	private String		omitNames	= ",ROWID,ROWNUM,WRITABLE,";		// 処理対象外のカラム名
	private String		where		;				// 処理対象を特定するキー
	private String		whereNames	;				// 5.5.8.5 (2012/11/27) 処理対象を特定するCSV形式のカラム名
	private String[]	constKeys	;				// 処理カラム名の中の固定情報カラム名
	private String[]	constVals	;				// 処理カラム名の中の固定情報設定値
	private String[]	funcKeys	;				// 5.5.1.9 (2012/04/19) 関数等を設定するカラム名
	private String[]	funcVals	;				// 5.5.1.9 (2012/04/19) 関数等の設定値
	private String[]	asNames		;				// 5.5.1.9 (2012/04/19) 別名を付けたカラム名(select A as B from TBL の B を指定)
	private String[]	orgNames	;				// 5.5.1.9 (2012/04/19) tableの実際のカラム名(select A as B from TBL の A を指定)
	private String		constObjKey	= SYSTEM_ID;	// 固定情報カラムの処理オブジェクトを特定するキー
	private boolean		quotCheck	= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );
	private boolean		logicalDelete;				// 4.3.7.0 (2009/06/01) sqlTypeがDELETEの場合にもUPDATE文を発行

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public TableUpdateParamTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 5.5.1.9 (2012/04/19) エラーチェックを先に行います。
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 *
	 * @return	後続処理の指示( SKIP_BODY )
	 */
	@Override
	public int doStartTag() {
		if( !useTag() ) { return SKIP_BODY ; }	// 6.3.4.0 (2015/08/01)

		// constKeys,constVals の個数チェック
		if( constKeys != null ) {
			if( constVals == null || constKeys.length != constVals.length ) {
				final String errMsg = "<b>constKeys と、constVals の個数が異なります。</b><br />"
									+ " constKeys=[" + StringUtil.array2csv( constKeys ) + "]"
									+ " constVals=[" + StringUtil.array2csv( constVals ) + "]" ;
				throw new HybsSystemException( errMsg );
			}
		}

		// funcKeys,funcVals の個数チェック
		if( funcKeys != null ) {
			if( funcVals == null || funcKeys.length != funcVals.length ) {
				final String errMsg = "<b>funcKeys と、funcVals の個数が異なります。</b><br />"
									+ " funcKeys=[" + StringUtil.array2csv( funcKeys ) + "]"
									+ " funcVals=[" + StringUtil.array2csv( funcVals ) + "]" ;
				throw new HybsSystemException( errMsg );
			}
		}

		// asNames,orgNames の個数チェック
		if( orgNames != null ) {
			if( asNames == null || orgNames.length != asNames.length ) {
				final String errMsg = "<b>orgNames と、asNames の個数が異なります。</b><br />"
									+ " orgNames=[" + StringUtil.array2csv( orgNames ) + "]"
									+ " asNames=[" + StringUtil.array2csv( asNames ) + "]" ;
				throw new HybsSystemException( errMsg );
			}
		}

		return SKIP_BODY ;				// Body を評価しない
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 4.3.7.0 (2009/06/01) 論理削除対応
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();
		if( !useTag() ) { return EVAL_PAGE ; }	// 6.3.4.0 (2015/08/01)

		final TableUpdateTag updateTag = (TableUpdateTag)findAncestorWithClass( this,TableUpdateTag.class );
		if( updateTag == null ) {
			final String errMsg = "<b>" + getTagName() + "タグは、TableUpdateTagの内側(要素)に記述してください。</b>";
			throw new HybsSystemException( errMsg );
		}

		final String upSqlType = updateTag.getSqlType() ;
		if( upSqlType == null || upSqlType.equals( sqlType ) ) {
			// 通常の names カラム配列を設定します。
			if( names == null ) { names = updateTag.getNames(); }
			final NamesData namesData = makeNamesData( names );

			String query = null;
			if( "INSERT".equalsIgnoreCase( sqlType ) || "COPY".equalsIgnoreCase( sqlType ) ) {
				query = getInsertSQL( namesData );
			}
			else if( "UPDATE".equalsIgnoreCase( sqlType ) || "MODIFY".equalsIgnoreCase( sqlType )
					|| "DELETE".equalsIgnoreCase( sqlType ) && logicalDelete ) { // 4.3.7.0 (2009/06/01)
				query = getUpdateSQL( namesData );
			}
			else if( "DELETE".equalsIgnoreCase( sqlType ) ) {
				query = getDeleteSQL();
			}

			jspPrint( query );
		}

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 4.3.7.0 (2009/06/01) logicalDelete属性追加
	 * @og.rev 5.5.1.9 (2012/04/19) asNames、orgNames、funcKeys、funcVals属性追加
	 * @og.rev 5.5.8.5 (2012/11/27) 処理対象を特定するCSV形式のカラム名
	 */
	@Override
	protected void release2() {
		super.release2();			// 3.5.6.0 (2004/06/18) 追加(抜けていた)
		sqlType		= null;			// INSERT,COPY,UPDATE,MODIFY,DELETE
		table		= null;			// 処理対象のテーブル名
		names		= null;			// 処理対象のカラム名
		omitNames	= ",ROWID,ROWNUM,WRITABLE,";		// 処理対象外のカラム名
		where		= null;			// 処理対象を特定するキー
		whereNames	= null;			// 5.5.8.5 (2012/11/27) 処理対象を特定するCSV形式のカラム名
		constKeys	= null;			// 処理カラム名の中の固定情報カラム名
		constVals	= null;			// 処理カラム名の中の固定情報設定値
		quotCheck	= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );
		constObjKey	= SYSTEM_ID;	// 固定情報カラムの処理オブジェクトを特定するキー
		logicalDelete = false;		// 4.3.7.0 (2009/06/01)
		funcKeys	= null;			// 5.5.1.9 (2012/04/19) 関数等を設定するカラム名
		funcVals	= null;			// 5.5.1.9 (2012/04/19) 関数等の設定値
		asNames		= null;			// 5.5.1.9 (2012/04/19) 別名を付けたカラム名(select A as B from TBL の B を指定)
		orgNames	= null;			// 5.5.1.9 (2012/04/19) tableの実際のカラム名(select A as B from TBL の A を指定)
	}

	/**
	 * 【TAG】BODY部に書かれている SQLタイプを指定します。
	 *
	 * @og.tag
	 * SQLタイプは、INSERT,COPY,UPDATE,MODIFY,DELETE の中から指定する
	 * 必要があります。これらは、内部に書かれるSQLの形式を指定するのに使用します。
	 * 内部処理は、DBTableModelの改廃コード(A,C,D)に対して使用される
	 * SQL を選択する場合の情報に使用されます。
	 * なお、COPY と MODIFY は、command で指定できる簡易機能として用意しています。
	 * 上位の TableUpdateTag の sqlType 属性 と同じsqlType 属性の場合のみ、SQL文を
	 * 合成・出力します。(上位のsqlTypeがnullの場合は、無条件実行します。)
	 * 指定のタイプが、異なる場合は、なにも処理を行いません。
	 *
	 * @param	type SQLタイプ [INSERT/COPY/UPDATE/MODIFY/DELETE]
	 */
	public void setSqlType( final String type ) {
		sqlType = nval( getRequestParameter( type ),sqlType );
		if( sqlType != null && SQL_TYPE.indexOf( "|" + sqlType + "|" ) < 0 ) {
			sqlType = null;
	//	final String errMsg = "sqlType属性には、" + SQL_TYPE + "以外設定できません。"
	//					+ " typeIn=[" + type + "]"
	//					+ " sqlType=[" + sqlType + "]" ;
	//		throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】処理対象のテーブル名を指定します。
	 *
	 * @og.tag
	 * テーブル名を指定することで、sqlTypeに応じた QUERYを生成することが出来ます。
	 * 生成する場合のカラムを特定する場合は、names 属性で指定できます。
	 * また、WHERE条件は、where属性で指定します。
	 *
	 * @param	tbl テーブル名
	 * @see		#setNames( String )
	 * @see		#setWhere( String )
	 * @see		#setSqlType( String )
	 */
	public void setTable( final String tbl ) {
		table = nval( getRequestParameter( tbl ),table );
	}

	/**
	 * 【TAG】処理対象のカラム名をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * 生成するQUERYのカラム名をCSV形式(CSV)で複数指定します。
	 * 指定がない場合は、DBTableModel の全カラム(※)を使用して、QUERYを構築します。
	 * 一般に、テーブル結合してDBTableModelを構築した場合は、登録すべきカラムを
	 * 指定する必要があります。
	 * (※)正確には、DBTableModel の全カラムのうち、ROWID,ROWNUM,WRITABLE カラムは
	 * 無視します。
	 * 分解方法は、通常のパラメータ取得後に、CSV分解します。
	 *
	 * @og.rev 3.8.8.5 (2007/03/09) 通常のパラメータ取得後に、CSV分解に戻します。
	 *
	 * @param	nms カラム名 (CSV形式)
	 * @see		#setTable( String )
	 * @see		#setOmitNames( String )
	 */
	public void setNames( final String nms ) {
		names = StringUtil.csv2Array( getRequestParameter( nms ) );
		if( names.length == 0 ) { names = null; }
	}

	/**
	 * 【TAG】処理対象外のカラム名をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * 生成するQUERYのカラム名に指定しないカラム名をCSV形式(CSV)で複数指定します。
	 * 指定がない場合は、DBTableModel の全カラム(※)を使用して、QUERYを構築します。
	 * テーブル結合などで、処理したくないカラム数の方が少ない場合に、names ですべてを
	 * 指定するより少ない記述ですみます。
	 * (※)正確には、DBTableModel の全カラムのうち、ROWID,ROWNUM,WRITABLE カラムは
	 * 無視します。
	 *
	 * @param	nms カラム名 (CSV形式)
	 * @see		#setTable( String )
	 * @see		#setNames( String )
	 */
	public void setOmitNames( final String nms ) {
		omitNames = omitNames + nval( getRequestParameter( nms ),"" ) + ",";
	}

	/**
	 * 【TAG】処理対象を特定するキー条件(where句)を指定します。
	 *
	 * @og.tag
	 * 生成するQUERYのwhere 句を指定します。通常の WHERE 句の書き方と同じで、
	 * DBTableModelの値を割り当てたい箇所に[カラム名] を記述します。
	 * 文字列の場合、設定値をセットするときに、シングルコーテーションを
	 * 使用しますが、[カラム名]で指定する場合は、その前後に、(')シングル
	 * コーテーションは、不要です。
	 * {&#064;XXXX}変数を使用する場合は、パース時に固定文字に置き換えられる為、
	 * 文字列指定時の(')シングルコーテーションが必要になります。
	 * ※ 5.5.8.5 (2012/11/27) whereNames 属性と併用した場合は、where が、and を付けて、文字列結合されます。
	 * 例：FGJ='1' and CLM=[CLM] and SYSTEM_ID in ([SYSID],'**') and KBSAKU='{&#064;KBSAKU}'
	 *
	 * @param	wr 検索条件 (where句)
	 */
	public void setWhere( final String wr ) {
		where = nval( getRequestParameter( wr ),where );
	}

	/**
	 * 【TAG】処理対象を特定するキー条件(where句)をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * 生成するQUERYのwhere 句を指定する方法として、複数のカラム名をCSV指定し、内部で
	 * KEY=[KEY] 文字列を作成します。
	 * ここでは、カラム名は、データベースのカラム名と同じで、かつ、DBTableModel にも
	 * 同じカラムのデータが存在していること、という条件付きとします。
	 * また、where 条件との併用を行いますが、こちらの条件が先に使用され、where 条件は、
	 * and を付けて、文字列結合されます。
	 * 例： CLM,SYSTEM_ID,KBSAKU   ⇒   CLM=[CLM] and SYSTEM_ID=[SYSTEM_ID] and KBSAKU=[KBSAKU]
	 *
	 * @og.rev 5.5.8.5 (2012/11/27) 新規追加
	 *
	 * @param	wrnm 検索条件カラム (where句)作成のためのカラム名(CSV形式)
	 */
	public void setWhereNames( final String wrnm ) {
		whereNames = nval( getRequestParameter( wrnm ),whereNames );
	}

	/**
	 * 【TAG】設定値を固定値と置き換える対象となるカラム名をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * names 属性のカラムや table 属性より、QUERYを作成して、DBTableModelの値を
	 * 割り当てる場合、DBTableModelの値ではなく、外部から指定した固定値を
	 * 割り当てたい場合に、そのカラム名をCSV形式(CSV)で複数指定します。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param	keys 固定値カラム (CSV形式)
	 * @see		#setConstVals( String )
	 */
	public void setConstKeys( final String keys ) {
		constKeys = getCSVParameter( keys );
	}

	/**
	 * 【TAG】設定値を固定値と置き換える対象となる設定値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * names 属性のカラムや table 属性より、QUERYを作成して、DBTableModelの
	 * 値を割り当てる場合、DBTableModelの値ではなく、外部から指定した固定値を
	 * 割り当てたい場合に、そのカラム名に対応する設定値をCSV形式(CSV)で
	 * 複数指定します。ここで指定する設定値は、constKeys 属性と対応させます。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @param	vals 設定値(CSV形式)
	 * @see		#setConstKeys( String )
	 */
	public void setConstVals( final String vals ) {
		constVals = getCSVParameter( vals );
	}

	/**
	 * 【TAG】関数等を設定するカラム名をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * constVals 属性で設定する値は、必ずシングルクオートが付与されます。
	 * その場合、関数などを設定したい場合でも、文字列として設定しようとします。
	 * ここで指定するカラム名(funcKeys)自身は、constKeys と同じ書式です。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 5.5.1.9 (2012/04/19) 新規追加
	 *
	 * @param	keys 関数カラム (CSV形式)
	 * @see		#setFuncVals( String )
	 */
	public void setFuncKeys( final String keys ) {
		funcKeys = getCSVParameter( keys );
	}

	/**
	 * 【TAG】関数等の設定値をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * funcKeys 属性に対応する 関数などの設定値を割り当てます。
	 * constVals 属性との違いは、funcVals の設定値は、そのままの形で、SQL文の
	 * 構築に使われます。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 5.5.1.9 (2012/04/19) 新規追加
	 *
	 * @param	vals 関数設定値 (CSV形式)
	 * @see		#setFuncKeys( String )
	 */
	public void setFuncVals( final String vals ) {
		funcVals = getCSVParameter( vals );
	}

	/**
	 * 【TAG】別名を付けたカラム名(select A as B from TBL の B を指定)をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * SELECT 文を記述したとき、別名を付けていたり、SELECTしたテーブルと別のテーブルに
	 * DBTableModelの値を書き込む場合、DBTableModel の持っているカラム名と、実際に
	 * 書き込むカラム名が異なります。そのようなケースに、元の別名カラムを指定します。
	 * orgNames属性の並び順と、asNames属性の並び順を合わせておく必要があります。
	 * このカラム名は、DBTableModel には持っているが、テーブル側には持っていない値
	 * なので、内部的に omitNames 属性に値を設定します。利用者は、omitNames に
	 * 書き込む必要はありません。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 5.5.1.9 (2012/04/19) 新規追加
	 *
	 * @param	keys 別名カラム (CSV形式)
	 * @see		#setOrgNames( String )
	 */
	public void setAsNames( final String keys ) {
		asNames = getCSVParameter( keys );
	}

	/**
	 * 【TAG】tableの実際のカラム名(select A as B from TBL の A を指定)をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * SELECT 文を記述したとき、別名を付けていたり、SELECTしたテーブルと別のテーブルに
	 * DBTableModelの値を書き込む場合、DBTableModel の持っているカラム名と、実際に
	 * 書き込むカラム名が異なります。そのようなケースに、テーブルの実カラムを指定します。
	 * orgNames属性の並び順と、asNames属性の並び順を合わせておく必要があります。
	 * このカラム名は、DBTableModel には持っていませんが、テーブル側には持っている値
	 * なので、このカラム名で、SQL文を構築します。 UPDATE TBL SET A=[B] WHERE … となります。
	 * 分解方法は、CSV変数を先に分解してから、getRequestParameter で値を取得します。
	 * こうしないとデータ自身にカンマを持っている場合に分解をミスる為です。
	 *
	 * @og.rev 5.5.1.9 (2012/04/19) 新規追加
	 *
	 * @param	keys 実カラム (CSV形式)
	 * @see		#setAsNames( String )
	 */
	public void setOrgNames( final String keys ) {
		orgNames = getCSVParameter( keys );
	}

	/**
	 * 【TAG】リクエスト情報の シングルクォート(') 存在チェックを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_SQL_INJECTION_CHECK[={@og.value SystemData#USE_SQL_INJECTION_CHECK}])。
	 *
	 * @og.tag
	 * ＳＱＬインジェクション対策の一つとして、暫定的ではありますが、SQLのパラメータに
	 * 渡す文字列にシングルクォート(') を許さない設定にすれば、ある程度は防止できます。
	 * 数字タイプの引数には、 or 5=5 などのシングルクォートを使用しないコードを埋めても、
	 * 数字チェックで検出可能です。文字タイプの場合は、必ず (')をはずして、
	 * ' or 'A' like 'A のような形式になる為、(')チェックだけでも有効です。
	 * (') が含まれていたエラーにする(true)／かノーチェックか(false)を指定します。
	 * (初期値:システム定数のUSE_SQL_INJECTION_CHECK[={@og.value SystemData#USE_SQL_INJECTION_CHECK}])。
	 *
	 * @param   flag クォートチェック [true:する/それ以外:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_SQL_INJECTION_CHECK
	 */
	public void setQuotCheck( final String flag ) {
		quotCheck = nval( getRequestParameter( flag ),quotCheck );
	}

	/**
	 * 【TAG】固定情報カラムの処理オブジェクトを特定するキーを設定します(初期値:SYSTEM_ID)。
	 *
	 * @og.tag
	 * 固定情報カラム をシステム単位にJavaクラスで管理できます。
	 * そのクラスオブジェクトは、org.opengion.hayabusa.db.DBConstValue インターフェースを
	 * 継承した、plugin クラスになります。
	 * そのクラスを特定するキーワードを指定します。
	 * 初期値は、SYSTEM_ID でシステム単位にクラスを作成します。
	 * もし、他のシステムと共通の場合は、継承だけさせることも可能です。
	 * 対応したDBConstValueクラスがプラグインとして存在しない場合は、
	 * システムリソースのDEFAULT_CONST_CLASSで指定されたクラスが利用されます。
	 * 固定情報カラムを使用しない場合は、constObjKey="" をセットしてください。
	 *
	 * 初期値は、SYSTEM_ID です。
	 *
	 * @og.rev 6.9.8.0 (2018/05/28) 固定情報カラムを使用しない場合は、constObjKey="" をセット。
	 *
	 * @param   key 固定カラムキー
	 */
	public void setConstObjKey( final String key ) {
//		constObjKey = nval( getRequestParameter( key ),constObjKey );
		final String objKey = getRequestParameter( key );
		if( objKey != null ) { constObjKey = objKey; }
	}

	/**
	 * 【TAG】sqlType="DELETE"の場合に論理削除(UPDATE)を行うかどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * sqlType="DELETE"の場合に論理削除(UPDATE)を行うかどうかを指定します。
	 * trueが指定された場合は、DELETE文ではなく、UPDATE文が発行されます。
	 * falseが指定された場合は、DELETE文が発行されます。
	 * さらに論理削除を行う場合、org.opengion.hayabusa.db.DBConstValue インターフェースに
	 * 定義されている、getLogicalDeleteKeys()及びgetLogicalDeleteValsを実装することで、
	 * 論理削除する際のフラグの更新方法を統一的に管理することが可能になります。
	 * 初期値は、false(物理削除する)です
	 *
	 * @param   flag 論理削除可否 [true:UPDATE文/false:DELETE文]
	 */
	public void setLogicalDelete( final String flag ) {
		logicalDelete = nval( getRequestParameter( flag ),logicalDelete );
	}

	/**
	 * データをインサートする場合に使用するSQL文を作成します。
	 *
	 * @og.rev 4.1.2.1 (2008/03/17) DBConstValue による固定値セットを採用
	 * @og.rev 4.3.6.4 (2009/05/01) デフォルト設定をシステムリソースで設定可能にする
	 * @og.rev 5.3.4.0 (2011/04/01) DEFAULT_CONST_OBJの初期値変更(null→ゼロ文字列)
	 * @og.rev 6.2.3.0 (2015/05/01) CSV形式の作成を、String#join( CharSequence , CharSequence... )を使用。
	 *
	 * @param   namesData NamesDataオブジェクト
	 *
	 * @return  インサートSQL
	 * @og.rtnNotNull
	 */
	private String getInsertSQL( final NamesData namesData ) {
		String cls = HybsSystem.sys( "DBConstValue_" + constObjKey ) ;

		// 4.3.6.4 (2009/05/01) 標準の追加
		if( cls == null){
			cls = DEFAULT_CONST_OBJ;
		}

		if( cls != null && !cls.isEmpty() ) {
			final DBConstValue constVal = HybsSystem.newInstance( cls );
	 		// 4.2.1.0 (2008/04/16) 初期化追加
			constVal.init( table,getUser().getUserID(),getGUIInfoAttri( "KEY" ) );
			final String[] keys = constVal.getInsertKeys();
			final String[] vals = constVal.getInsertVals();
			namesData.add( keys,vals );
		}

		final String[] nms = namesData.getNames();
		final String[] vls = namesData.getVals();

		// 6.2.3.0 (2015/05/01) CSV形式の作成を、String#join( CharSequence , CharSequence... )を使用。
		final StringBuilder sql = new StringBuilder( BUFFER_MIDDLE )
			.append( "INSERT INTO " ).append( table )
			.append( " ( " )
			.append( String.join( "," , nms ) )		// 6.2.3.0 (2015/05/01)
			.append( " ) VALUES ( " )
			.append( String.join( "," , vls ) )		// 6.2.3.0 (2015/05/01)
			.append( " )" );

		return sql.toString();
	}

	/**
	 * データをアップデートする場合に使用するSQL文を作成します。
	 *
	 * where と whereNames が同時に指定された場合は、whereNames が先に処理され
	 * where 条件は、and 結合されます。
	 *
	 * @og.rev 4.1.2.1 (2008/03/17) DBConstValue による固定値セットを採用
	 * @og.rev 4.3.6.4 (2009/05/01) デフォルト設定をシステムリソースで設定可能にする
	 * @og.rev 4.3.7.0 (2009/06/01) 論理削除対応
	 * @og.rev 5.3.7.0 (2011/07/01) DEFAULT_CONST_OBJの初期値変更(null→ゼロ文字列) 対応忘れ
	 * @og.rev 5.5.8.5 (2012/11/27) whereNames 対応
	 *
	 * @param   namesData NamesDataオブジェクト
	 *
	 * @return  アップデートSQL
	 * @og.rtnNotNull
	 */
	private String getUpdateSQL( final NamesData namesData ) {
		String cls = HybsSystem.sys( "DBConstValue_" + constObjKey ) ;

		// 4.3.6.4 (2009/05/01) 標準の追加
		if( cls == null){
			cls = DEFAULT_CONST_OBJ;
		}

		if( cls != null && !cls.isEmpty() ) {		// 5.3.7.0 (2011/07/01)
			final DBConstValue constVal = HybsSystem.newInstance( cls );
	 		// 4.2.1.0 (2008/04/16) 初期化追加
			constVal.init( table,getUser().getUserID(),getGUIInfoAttri( "KEY" ) );
			// 4.3.7.0 (2009/06/01) 論理削除対応
			String[] keys = null;
			String[] vals = null;
			if( "DELETE".equalsIgnoreCase( sqlType ) ) {
				keys = constVal.getLogicalDeleteKeys();
				vals = constVal.getLogicalDeleteVals();
			}
			else {
				keys = constVal.getUpdateKeys();
				vals = constVal.getUpdateVals();
			}
			namesData.add( keys,vals );
		}

		final String[] nms = namesData.getNames();
		final String[] vls = namesData.getVals();

		final StringBuilder sql = new StringBuilder( BUFFER_MIDDLE )
			.append( "UPDATE " ).append( table ).append( " SET " )
			.append( nms[0] ).append( '=' ).append( vls[0] );			// 6.0.2.5 (2014/10/31) char を append する。

		for( int i=1; i<nms.length; i++ ) {
			sql.append( ',' ).append( nms[i] ).append( '=' ).append( vls[i] );		// 6.0.2.5 (2014/10/31) char を append する。
		}

		// 5.5.8.5 (2012/11/27) whereNames 対応
		String whereAnd = " WHERE " ;
		if( whereNames != null && whereNames.length() > 0 ) {
			final String[] wnms = whereNames.split(",");
			sql.append( whereAnd ).append( wnms[0] ).append( "=[" ).append( wnms[0] ).append( ']' );		// 6.0.2.5 (2014/10/31) char を append する。

			for( int i=1; i<wnms.length; i++ ) {
				sql.append( " AND " ).append( wnms[i] ).append( "=[" ).append( wnms[i] ).append( ']' );		// 6.0.2.5 (2014/10/31) char を append する。
			}
			whereAnd = " AND " ;		// whereNames 優先。ここを通らなければ、初期値のまま、" WHERE " が使われる
		}

		// 5.5.8.5 (2012/11/27) whereNames 対応。whereNames が登録されていれば、AND で繋げる。
		if( where != null && where.length() > 0 ) {
			sql.append( whereAnd ).append( where );
		}

		return sql.toString();
	}

	/**
	 * データをデリートする場合に使用するSQL文を作成します。
	 *
	 * where と whereNames が同時に指定された場合は、whereNames が先に処理され
	 * where 条件は、and 結合されます。
	 *
	 * @og.rev 5.5.8.5 (2012/11/27) whereNames 対応
	 *
	 * @return  デリートSQL
	 * @og.rtnNotNull
	 */
	private String getDeleteSQL() {
		final StringBuilder sql = new StringBuilder( BUFFER_MIDDLE );
		sql.append( "DELETE FROM " ).append( table );

		// 5.5.8.5 (2012/11/27) whereNames 対応
		String whereAnd = " WHERE " ;
		if( whereNames != null && whereNames.length() > 0 ) {
			final String[] wnms = whereNames.split(",");
			sql.append( whereAnd ).append( wnms[0] ).append( "=[" ).append( wnms[0] ).append( ']' );		// 6.0.2.5 (2014/10/31) char を append する。

			for( int i=1; i<wnms.length; i++ ) {
				sql.append( " AND " ).append( wnms[i] ).append( "=[" ).append( wnms[i] ).append( ']' );		// 6.0.2.5 (2014/10/31) char を append する。
			}
			whereAnd = " AND " ;		// whereNames 優先。ここを通らなければ、初期値のまま、" WHERE " が使われる
		}

		// 5.5.8.5 (2012/11/27) whereNames 対応。whereNames が登録されていれば、AND で繋げる。
		if( where != null && where.length() > 0 ) {
			sql.append( whereAnd ).append( where );
		}
		return sql.toString();
	}

	/**
	 * names,constKeys,omitNames から、必要なキー情報と、属性情報を持った NamesData を作成します。
	 *
	 * @og.rev 4.1.2.1 (2008/03/17) 固定値の constVals の前後に、"'" を入れる。
	 * @og.rev 5.5.1.9 (2012/04/19) asNames、orgNames、funcKeys、funcVals属性追加
	 *
	 * @param   nms カラム名配列(可変長引数)
	 *
	 * @return	属性情報を持ったNamesData
	 */
	private NamesData makeNamesData( final String... nms ) {

		final NamesData namesData = new NamesData();

		// 5.5.1.9 (2012/04/19) omitNames に、asNames配列の値を設定しておきます。
		if( asNames != null ) {
			for( int i=0; i<asNames.length; i++ ) {
				if( asNames[i] != null && asNames[i].length() > 0 ) {
					omitNames = omitNames + asNames[i] + ",";
				}
			}
		}

		// names で指定されたカラム名
		for( int i=0; i<nms.length; i++ ) {
			final String nm = nms[i];
			if( nm != null && nm.length() > 0 && omitNames.indexOf( "," + nm + "," ) < 0 ) {
				namesData.add( nm,"[" + nm + "]" ) ;
			}
		}

		// 固定値の constKeys カラム配列を設定します。
		if( constKeys != null && constKeys.length > 0 ) {
			for( int j=0; j<constKeys.length; j++ ) {
				final String nm = constKeys[j];
				if( nm != null && nm.length() > 0 ) {
					namesData.add( nm,"'" + constVals[j] + "'" ) ;	// constVals は、シングルクオートで囲います。
				}
			}
		}

		// 関数値の funcKeys カラム配列を設定します。
		if( funcKeys != null && funcKeys.length > 0 ) {
			for( int j=0; j<funcKeys.length; j++ ) {
				final String nm = funcKeys[j];
				if( nm != null && nm.length() > 0 ) {
					namesData.add( nm, funcVals[j] ) ;		// funcVals は、シングルクオートで囲いません。
				}
			}
		}

		// 別名の asNames,orgNames カラム配列を設定します。
		if( orgNames != null && orgNames.length > 0 ) {
			for( int j=0; j<orgNames.length; j++ ) {
				final String onm = orgNames[j];
				if( onm != null && onm.length() > 0 ) {
					namesData.add( onm,"[" + asNames[j] + "]" ) ;
				}
			}
		}

		return namesData ;
	}

	/**
	 * 内部データを受け渡す為の、簡易クラスです。
	 * 更新するカラム名と値のセット配列を管理しています。
	 *
	 * @og.rev 6.3.9.1 (2015/11/27) 修飾子を、なし → private に変更。
	 *    ※ classのfinal化と変数のprivate化、メソッドの修飾子なし(パッケージプライベート)化を行います。
	 * @og.rev 6.4.1.2 (2016/01/22) nameの値を、大文字小文字の区別をなくすために、常に大文字で登録します。
	 */
	private static final class NamesData {
		private final Map<String,String> nameMap = new LinkedHashMap<>() ;

		/**
		 * キーと値のセットを追加します。
		 *
		 * @og.rev 6.3.9.1 (2015/11/27) 修飾子を、なし → private に変更。
		 * @og.rev 6.4.1.2 (2016/01/22) nameの値を、大文字小文字の区別をなくすために、常に大文字で登録します。。
		 *
		 * @param   nm  キー（大文字のみ。内部で変換しておきます。）
		 * @param   val 値
		 */
		/* default */ void add( final String nm,final String val ) {
			nameMap.put( nm.toUpperCase(Locale.JAPAN),val );
		}

		/**
		 * キー配列と対応する、値配列のセットを追加します。
		 *
		 * @og.rev 6.3.9.1 (2015/11/27) 修飾子を、なし → private に変更。
		 *
		 * @param   nms  キー配列
		 * @param   vals 値配列
		 */
		/* default */ void add( final String[] nms,final String[] vals ) {
			if( nms != null ) {
				for( int i=0; i<nms.length; i++ ) {
					nameMap.put( nms[i].toUpperCase(Locale.JAPAN),vals[i] );
				}
			}
		}

		/**
		 * キー配列を返します。
		 *
		 * @og.rev 6.3.9.1 (2015/11/27) 修飾子を、なし → private に変更。
		 *
		 * @return	キー配列
		 * @og.rtnNotNull
		 */
		/* default */ String[] getNames() {
			return nameMap.keySet().toArray( new String[nameMap.size()] );
		}

		/**
		 * 値配列を返します。
		 *
		 * @og.rev 6.3.9.1 (2015/11/27) 修飾子を、なし → private に変更。
		 *
		 * @return	値配列
		 * @og.rtnNotNull
		 */
		/* default */ String[] getVals()  {
			return nameMap.values().toArray( new String[nameMap.size()] );
		}
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "sqlType"			,sqlType		)
				.println( "table"			,table			)
				.println( "names"			,names			)
				.println( "omitNames"		,omitNames		)
				.println( "where"			,where			)
				.println( "whereNames"		,whereNames		)
				.println( "constKeys"		,constKeys		)
				.println( "constVals"		,constVals		)
				.println( "logicalDelete"	,logicalDelete	)
				.fixForm().toString() ;
	}
}
