/*
 * PreferencesWindowController.m
 *
 * Copyright (C) 2008 MikuInstaller Project. All rights reserved.
 * http://mikuinstaller.sourceforge.jp/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in
 *     the documentation and/or other materials provided with the
 *     distribution.
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#import "PreferencesController.h"
#import "ApplicationInfo.h"

@interface NSObject (BeginSheetWithStartBackend)
- (void)beginSheetWithStartBackend:(NSString *)name
			 arguments:(NSArray *)arguments
		       environment:(NSDictionary *)environment
			   message:(NSString *)message
		    modalForWindow:(NSWindow *)docWindow
		     modalDelegate:(id)delegate
		    didEndSelector:(SEL)didEndSelector
		       contextInfo:(void *)contextInfo;
@end

@implementation PreferencesController

+ (void)initialize
{
	[self exposeBinding:@"application"];
	[self exposeBinding:@"winePrefixSelection"];
}

- (id)init
{
	if ((self = [self initWithWindowNibName:@"Preferences"])) {
		winePrefixList = [[NSMutableArray alloc] initWithCapacity:16];
	}
	return self;
}

- (void)dealloc
{
	[winePrefixList release];
	[[NSNotificationCenter defaultCenter] removeObserver:self];
	[application unbind:@"winePrefix"];
	[super dealloc];
}

- (void)awakeFromNib
{
	if (winePrefixTableView) {
		/* do only when "Preferences" window is awake. */
		[application bind:@"winePrefix"
			 toObject:self
		      withKeyPath:@"winePrefixSelection"
			  options:nil];

		[[NSNotificationCenter defaultCenter]
		 addObserver:self
		 selector:@selector(textDidEndEditing:)
		 name:NSControlTextDidEndEditingNotification
		 object:winePrefixTableView];
	}
}

- (void)windowDidLoad
{
	[[self window] center];
}

- (id)application
{
	return application;
}

- (NSString *)winePrefixSelection
{
	NSInteger row = [winePrefixTableView selectedRow];
	if (row >= 0 && row < [winePrefixList count])
		return [[winePrefixList objectAtIndex:row]
			objectForKey:@"path"];
	else
		return nil;
}

- (void)setWinePrefixSelection:(NSString *)path
{
	NSEnumerator *e = [winePrefixList objectEnumerator];
	NSUInteger i = 0;
	NSDictionary *item;

	while ((item = [e nextObject])) {
		if ([[item objectForKey:@"path"] isEqual:path]) {
			[winePrefixTableView
			 selectRowIndexes:[NSIndexSet indexSetWithIndex:i]
			 byExtendingSelection:NO];
			break;
		}
		i++;
	}
}

- (void)tableViewSelectionDidChange:(NSNotification *)notification
{
	if ([notification object] == winePrefixTableView) {
		[self willChangeValueForKey:@"winePrefixSelection"];
		[self didChangeValueForKey:@"winePrefixSelection"];
	}
}

- (void)updateWinePrefixList
{
	NSString *base = [application valueForKey:@"winePrefixBase"];
	NSArray *prefixList = [[NSFileManager defaultManager]
			       directoryContentsAtPath:base];

	NSString *selection = [[[self winePrefixSelection] retain] autorelease];
	[winePrefixList removeAllObjects];

	if (!prefixList)
		return;

	NSEnumerator *e = [prefixList objectEnumerator];
	NSString *name;

	while ((name = [e nextObject])) {
		[winePrefixList
		 addObject:[NSDictionary
			    dictionaryWithObjectsAndKeys:
			    [base stringByAppendingPathComponent:name], @"path",
			    name, @"name",
			    nil]];
	}

	[winePrefixList
	 sortUsingDescriptors:[winePrefixTableView sortDescriptors]];
	[winePrefixTableView reloadData];

	[self setWinePrefixSelection:selection];
}

- (NSInteger)numberOfRowsInTableView:(NSTableView *)tableView
{
	return [winePrefixList count];
}

- (id)tableView:(NSTableView *)tableView
objectValueForTableColumn:(NSTableColumn *)tableColumn
	    row:(int)row
{
	if (!(row >= 0 && row < [winePrefixList count]))
		return nil;
	else
		return [[winePrefixList objectAtIndex:row]
			objectForKey:[tableColumn identifier]];
}

- (void)tabView:(NSTabView *)tabView
willSelectTabViewItem:(NSTabViewItem *)tabViewItem
{
	if ([[tabViewItem identifier] isEqual:@"WINEPREFIX"])
		[self updateWinePrefixList];
}

- (void)windowDidBecomeKey:(NSNotification *)notification
{
	[self updateWinePrefixList];
}

- (void)tableView:(NSTableView *)tableView
sortDescriptorsDidChange:(NSArray *)oldDescriptors
{
	[winePrefixList sortUsingDescriptors:[tableView sortDescriptors]];
	[tableView reloadData];
}

- (void)addWinePrefix:(id)sender
{
	/* add a dummy item to winePrefixList as a placeholder for editting
	 * the name of new WINEPREFIX. */
	NSDictionary *newItem =
		[NSDictionary dictionaryWithObject:@"Untitled"
					    forKey:@"name"];

	NSUInteger row = [winePrefixList count];
	[winePrefixList addObject:newItem];
	[winePrefixTableView reloadData];
	[winePrefixTableView
	 selectRowIndexes:[NSIndexSet indexSetWithIndex:row]
	 byExtendingSelection:NO];
	[winePrefixTableView editColumn:0
				    row:row
			      withEvent:nil
				 select:YES];
}

- (void)createWinePrefix:(NSString *)newName
{
	NSString *base = [application valueForKey:@"winePrefixBase"];
	NSString *path = [base stringByAppendingPathComponent:newName];
	[application setValue:path forKey:@"winePrefix"];

	[application
	 beginSheetWithStartBackend:@"prefixcreate"
	 arguments:nil
	 environment:nil
	 message:[NSString
		  stringWithFormat:
		  NSLocalizedString(@"Initializing WINEPREFIX %@...", nil),
		  newName]
	 modalForWindow:[self window]
	 modalDelegate:nil
	 didEndSelector:nil
	 contextInfo:NULL];

	[self updateWinePrefixList];
	[self setWinePrefixSelection:path];
}

- (void)removeWinePrefix:(id)sender
{
	NSString *prefix = [[self winePrefixSelection] lastPathComponent];

	NSAlert *alert =
		[NSAlert
		 alertWithMessageText:
		 [NSString stringWithFormat:
			   NSLocalizedString(@"Are you sure you want to"
					     " remove WINEPREFIX %@?", nil),
			   prefix]
		 defaultButton:nil
		 alternateButton:NSLocalizedString(@"Cancel", nil)
		 otherButton:nil
		 informativeTextWithFormat:
		 NSLocalizedString(@"You can't undo this action.", nil)];

	[alert
	 beginSheetModalForWindow:[self window]
	 modalDelegate:self
	 didEndSelector:@selector(removeConfirm:returnCode:contextInfo:)
	 contextInfo:prefix];
}

- (void)removeConfirm:(NSAlert *)alert
	   returnCode:(int)returnCode
	  contextInfo:(void *)contextInfo
{
	[[alert window] orderOut:self];

	if (returnCode == NSAlertDefaultReturn) {
		[application
		 beginSheetWithStartBackend:@"prefixremove"
		 arguments:nil
		 environment:nil
		 message:[NSString
			  stringWithFormat:
			  NSLocalizedString(@"Removing WINEPREFIX %@...", nil),
			  [[self winePrefixSelection] lastPathComponent]]
		 modalForWindow:[self window]
		 modalDelegate:nil
		 didEndSelector:nil
		 contextInfo:NULL];
	}

	[self updateWinePrefixList];
}

#if 0
- (void)renameWinePrefix:(NSString *)oldPath
		 newName:(NSString *)newName
{
	NSArray *path = [oldPath pathComponents];
	path = [path subarrayWithRange:NSMakeRange(0, [path count] - 1)];
	path = [path arrayByAddingObject:newName];
	NSString *newPath = [NSString pathWithComponents:path];

	if (![oldPath isEqual:newPath]) {
		[[NSFileManager defaultManager] movePath:oldPath
						  toPath:newPath
						 handler:self];
		/* NOTE: updateWinePrefixList may release oldPath. */
		[self updateWinePrefixList];
		[self setWinePrefixSelection:newPath];
	}
}

- (BOOL)fileManager:(NSFileManager *)manager
shouldProceedAfterError:(NSDictionary *)errorInfo
{
	NSBeginAlertSheet(
		[NSString
		 stringWithFormat:
		 NSLocalizedString(@"Rename WINEPREFIX %@", nil),
		 [[errorInfo objectForKey:@"Path"] lastPathComponent]],
		nil, nil, nil, [self window], self, nil, NULL,
		@"%@", [errorInfo objectForKey:@"Error"]);
	return YES;
}
#endif

- (void)textDidEndEditing:(NSNotification *)notification
{
	NSText *text = [[notification userInfo] objectForKey:@"NSFieldEditor"];
	NSString *newName = [NSString stringWithString:[text string]];
	NSInteger row = [winePrefixTableView editedRow];
	NSParameterAssert(row >= 0 && row < [winePrefixList count]);
	NSDictionary *oldItem = [winePrefixList objectAtIndex:row];
	NSParameterAssert([oldItem objectForKey:@"path"] == nil);

	if ([newName length] == 0
	    || [newName rangeOfString:@"/"].location != NSNotFound) {
		NSBeginAlertSheet(
			NSLocalizedString(@"Edit WINEPREFIX name", nil),
			nil, nil, nil, [self window], self, nil, nil, NULL,
			NSLocalizedString(@"Invalid WINEPREFIX name.", nil));

		/* remove dummy entries from winePrefixList */
		[self updateWinePrefixList];
		return;
	}

	[self createWinePrefix:newName];
}

- (void)openWinecfg:(id)sender
{
	[application
	 beginSheetWithStartBackend:@"winecfg"
	 arguments:nil
	 environment:nil
	 message:[NSString
		  stringWithFormat:
		  NSLocalizedString(@"Starting winecfg of WINEPREFIX %@...",
				    nil),
		  [[self winePrefixSelection] lastPathComponent]]
	 modalForWindow:[self window]
	 modalDelegate:nil
	 didEndSelector:nil
	 contextInfo:NULL];
}

- (void)openDriveC:(id)sender
{
	NSString *path = [self winePrefixSelection];
	path = [path stringByAppendingPathComponent:@"dosdevices"];
	path = [path stringByAppendingPathComponent:@"c:"];
	[[NSWorkspace sharedWorkspace] openFile:path
				withApplication:@"Finder"];
}

- (void)createApplicationBundles:(id)sender
{
	[application
	 beginSheetWithStartBackend:@"addapp"
	 arguments:nil
	 environment:nil
	 message:[NSString
		  stringWithFormat:
		  NSLocalizedString(@"Making application bundles of"
				    " WINEPREFIX %@...",
				    nil),
		  [[self winePrefixSelection] lastPathComponent]]
	 modalForWindow:[self window]
	 modalDelegate:nil
	 didEndSelector:nil
	 contextInfo:NULL];
}

@end
