/* ---------------------------------------------------------------------------------------------- */
/* MHash-384 for Java 1.7+                                                                        */
/* Copyright(c) 2016-2018 LoRd_MuldeR <mulder2@gmx.de>                                            */
/*                                                                                                */
/* Permission is hereby granted, free of charge, to any person obtaining a copy of this software  */
/* and associated documentation files (the "Software"), to deal in the Software without           */
/* restriction, including without limitation the rights to use, copy, modify, merge, publish,     */
/* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the  */
/* Software is furnished to do so, subject to the following conditions:                           */
/*                                                                                                */
/* The above copyright notice and this permission notice shall be included in all copies or       */
/* substantial portions of the Software.                                                          */
/*                                                                                                */
/* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING  */
/* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND     */
/* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,   */
/* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, */
/* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.        */
/* ---------------------------------------------------------------------------------------------- */

package com.muldersoft.mhash384.utils;

import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

public class JavaBytes {

    // =========================================================================
    // INPUT DATA
    // =========================================================================

    private static final List<String> INPUT_INI = Collections.unmodifiableList(Arrays.asList(
            "EFDCB66D4ECC1AA4AF9B3F204698FAC6A306B411987C282AE89249C264A9E1C8A4AB93161F359677352FC8B56B6E37AE", /* 00 */
            "1EE14718ECF437F2814821C5710E28A3EF41E30D49087A379D50EFB00795758588F95DC7AEA0FA41BF819DEF28236378"  /* 01 */
    ));
    
    private static final List<String> INPUT_XOR = Collections.unmodifiableList(Arrays.asList(
            "01DCDF00414B3037B1B3AF661B8E96F8944D2873DB39312173DA9A36662AE7551F4F318C4ECB56B1F09743D99C2AA5BC", /* 00 */
            "A81FBBC6CBBFC95439DE43648959EDDB1A641A0BDA01822FB52E6072669326582C5B1731AC802084C2EF10671FC79DD4", /* 01 */
            "CF2A8D8E088100468B7E9B2089E268F6930461652C5D2DECF096E42223BFC8B98DD338964CFE0970269C342F7CEC60BD", /* 02 */
            "B970A920D2ED0552010F894A254DA7E18D5F205F9D1A40D850C33DCCC3FD5F58B49F31BDE7D82C577CDE04F62A959033", /* 03 */
            "49FAB2822B9C84AC34B8648CD68CBEF1E5121147BB4126DEC0B31F54B2FFE00F2F193DA38E8CC632058C984B429B8AFC", /* 04 */
            "E07F9DA44D8C9942BE2AF9B39ACA65F25D3D8FB1466DC295C0051E3EC3F962C7F8FDC3CCD4CE2BB69024C0EFC6199937", /* 05 */
            "A3111377EF01F5EF31F59B366C02A3FF61B82949760D16DCF6B958AF92BD3BDF7297AAEFEC69C0B2FE8A50AD9E8684CD", /* 06 */
            "3D3DD0C829EA9AA03E77597EEC96C7A38FD038231E7F168464617B131FB7EDE085C99CE4C5405874A58735D41F19C5E3", /* 07 */
            "8028628ACAF91B9A194A640538C9706404A15E018A4F1680F4BE1B04C2360955DFB24D4CEF581A203C59D0A0FD15879D", /* 08 */
            "88F8E2ECE78AF1BCB46B6E22297364C29339F17F926B99F331293489B3B8F07C3909CE1649C9BCF10C46103BFA31BCE9", /* 09 */
            "4D1A0B8CC7EFE2A9AFD7878CAD55E871C89CFBC858CF4153C4739486C74F75D60BF7192C130AC9F20084F2BC5E81BD9A", /* 0A */
            "4AFBE975014FCACF41DEAF8CFACC41E43C1EC23B53ED16E978B06EB30F1C3248DD11165D04285C726546D9B5609491E4", /* 0B */
            "FFD1E2E04DC8D26007B186948A74ECCEB5120E45121AC531BD0BC86330810C85DE93AFDDDB48773078DEB0DE6EB99196", /* 0C */
            "79BAA2AC3FDEBA55B1B7E3E1C92A567EA53F69AB4A5B0794F0DA7499954F6DDD58117C89E1132248D95DF5F794C51A6E", /* 0D */
            "89BC8A0C65091C33B04EEAEE063135C2F53267E04AB6E689B4C579B7207E8BF23FD31E51343CD2DF119E523E2F8197FE", /* 0E */
            "2E10CB1C6060F32DBC92E732A94E6B63F3220D831FD04267502C5F7414BCE87F89E0651DE91D2457759E56B04482D915", /* 0F */
            "43AEE32C6A84E8030C5007202C0BD7E3B4F464474205D32A7D17FC95DE386C06E8DFBF64567AA54555BD889D5853046F", /* 10 */
            "687ABE14EAB8DA27397B3AB50D72C344505EAA6D1FDE618D61BE79865DA13F6917BBAB29B5E90E2D010C921972FA8B2F", /* 11 */
            "8B7223A4F56FF453291B7B5CB98B6FE1FD421625786FBF7DE33D1020D0E8CDC8CA530C708B739E87878AF1F304B8A12F", /* 12 */
            "7ECE1F24E56DD7112E0869241B2FA6DF84B834DCC459B2FDE022EDA6319E7D3B59839D8CA03C9928644790F491BBC774", /* 13 */
            "CBDC6F49E6B0DD0D44BA2F8D0034673286BCC821586AE61CC7B7491285CEE55BED3912FFD97F3851F4AF0186BEBEBCBF", /* 14 */
            "CA8A48E54ECCE516BEDF1864B7F8F506D56A1F60A4B36AA449B25AB5FE0DD9D921377DBB5E49FCE1708F64F5D59D99E7", /* 15 */
            "E9F873A569AFE02CDA66BC05CA9973908C88174756D35385EAAAF16CF4FDA730B39F7A55653A3512F10AB727BC23E852", /* 16 */
            "93E96FF8C5BBE2AFA47785420253E97C704B25D1F77D074CC0B2093F1470559C8D5BFDD4E364AACF550518673F5B1BF7", /* 17 */
            "8DAC832E5BE81ACB588BFB202B7583D8A34D8A70DFD1D7E4D03B0627B687033EE3D0BE7EDBDC75CF818EE8C5B09F8BEA", /* 18 */
            "8E3B02E1489D7D315336752B64E3B532E4D4CC795C580A656DAB08F313ED767E8E567E88FDBA36BF259490F1D8E933D0", /* 19 */
            "EAB437D0D62CAA62C090AD28B982B003E255D768D25704ECA048511AF6256A43E79F078F4D498B5FC41735FB75B357FE", /* 1A */
            "70DA9FC3504FF29DB9AB1F388673FF2536922F4CD17357BAF09C3AB292E7E04A90CE0BC3D9BA13EC647C4CA63C918DE3", /* 1B */
            "D834A38DD1ECD688A97A0020DE46AB6A9CDEC7F6E62EA71F288A5A6FD74DC47ED44A2E2765CE50F1BB9B50B5DB9E4F3C", /* 1C */
            "C66DA90E41DDF2E65A3FE86F160C5C096F6AF0405108CCBEF938382EB627FC7F163DD634617F006C5184B1FEDC908497", /* 1D */
            "C95719ED07FCB21C5112DF043F6EE7EB50F9FD60012334CE589FA85104D96579B7129E44D71905A73314766E0733528D", /* 1E */
            "DC6C8014C5457CB8D635FDCD286A69B6D66F232CE27D01BF56AF4AC0F682EC0F57DF1D55B64328F5651ED4C52A87CACA", /* 1F */
            "26D9D1CC79EEC50269BF340A34B1EBFEFAA5AAAC8E397174D0A8F9BD426BCF6F5B131F464D6D2452122DD15660D0D6DA", /* 20 */
            "6E389AEC5F51A22F7EF68F1C54C127FB986D4D46E0485C30F0A47B39E7CF8A311D398DFDB7F2A78F2FC651D1FBB10D2E", /* 21 */
            "A44E4E8D1B49DCB007A4822049C2F343C40AC04A8D6505BAD9B91D3F0729B16CAAF39951B50F9015966EF5D3AD3F9076", /* 22 */
            "EA78CBAC0EB6D009A0FEA6725A23DEABCE729C7444CB94D940A994626627AA0D37F738CDE3D018D54C29491C01CDB3C5", /* 23 */
            "7C9792AEA745C87AD1FF5620C5BD8FD49ECA84E3004B56B95AFD3923C228B1D6E5DBF79EB3FD283B441712E354084B9F", /* 24 */
            "E35D288BD8E249BC91776C1453A366E5F5D1E1684E95EFEC9108E117D7DDF60681B30F9DA2CE7C8C6497DBD786818C0D", /* 25 */
            "C2F891FF45044BE075A1A76D2B87E2EB85CE65798AF3C2BF213F532B4EFD09DC0DAA1DF5A53A6C883028606A50D826B2", /* 26 */
            "609A62616379F33AA339A3BC53E4516DD7AD92616A5ADBECD043726D86E924AA8555B564F4C2986556AA12AB31C1D6B0", /* 27 */
            "CED9ED85C1C17BFFEB522325ACBAFFC204D3D8F4B2D15394D271504C04756EEA2DDBA4A91AF827F11F67D5C28F8002E4", /* 28 */
            "8B9C08AD432DC08F5A7543E29796BBC734A6DB5B3C1967DE016E3BC2A2804EE45B9BCACCE5172F757549598B80ADBDBA", /* 29 */
            "6F3FB117C5CDD15516C3B0A59CD6EEC5D9A1A411DE538769938C54979F4AC37C3737BCC1D55284DB6CAD9F8AF9156BB7", /* 2A */
            "EBBF284F9C75EBDFB383EBB406753DE8AA86127AEE7C403C10BFDD10523DE027138BF6C4EB4A8A13B1EFF67DDB78B067", /* 2B */
            "F6D1138D7AA3DA5EBAA8098D8FB66371DAE76D1B8B6CAAF2400F1034368D1EDC7C937F5172E8D2777D05BBF83CADE6EF", /* 2C */
            "0E9C2EA6CF34B0819036B30D58F60BA0DB3A2C5848F08BCAC87AD1B94250D5647C892E09EEF9616626DB85CF571085F3", /* 2D */
            "251EE3F58718C12AF9438D81178A2AE4F0929A889039A8A8F06B65225EBDCCFD2E4D14EDF7BF73C6A9369895BC1DFACF", /* 2E */
            "CAE302B41D6979CBBBFA5A58B51EE6235113B99DC81AB52F6093795BEC17A0568F71FB4D2E5E355E762F92EDBA34A2F2", /* 2F */
            "D130015265A4D9FF09BEA253D71F26C881B6EAEDC46521E6FAE268165682B8A9A89C3EC4774AB6230D2E45E055219DB2", /* 30 */
            "2B560284C3A692AB37008AD0B379A7B8AF11CD2C30F90BFC7FE87A250F2762EDC3FBD711647242C374ED8264F6B322BD", /* 31 */
            "28195CC8A7AD394353CBE808464C4FC4D58E3D7A765F8726D83052F60185AA4FEFCB0D85223BB4E75A31305E787FAC28", /* 32 */
            "725D0EE230F195439091D2C6BDDF34E0E3BE49C6C275460161BE300BA4AD566B02D79D7551FA7CC16543910F5F1CDA58", /* 33 */
            "4099ADC44481B43FFE1361922FD9EB81A989C09E441FCEAC449B3A13D3CB901945A9BE396F201134DC1AD05A046633FE", /* 34 */
            "1A563A6D522F3E69BE589E079F475A9E75A2A9638E4C0038DA3B6202577A0362211D3F1E0D727AF65E1FFC529AD99233", /* 35 */
            "47B61E86C6D6D01B437D6F83ADADC318D5A361028DED738CA00D4C630425164B1A69AFA5AF4C9DD2F99E1C67F951B582", /* 36 */
            "A66A7740B6BDEA79FEF7FF1496AF80A305AFD43EEACD898CB00C78ED31AD71340ED31A1AD784667374B96844161499BE", /* 37 */
            "46FA8D6CCBF6D12E31C2FC147F303956707F4401DE5F067F3AE5FEC7E33594E928E39F8A63531714B7B329EA1E9FCAB2", /* 38 */
            "EFD8F755825C78041F5A93870BD30CD1EFBF894671FF871628ED617FF22BDA58411289CCAE5CB62E95DD42F41801F2F9", /* 39 */
            "A8525B8645FC59E175E62DC00A5F7F0C09C56785210416AC50EF76E9B30D76262B3B2CDC19F5D665A41297CD11D8F4FF", /* 3A */
            "EAC99A649EEE5039A593C92F143C0065B314735203071206EA2761A0C764A4EC02AA7FD46CAC25B3C68CC182A96D03BF", /* 3B */
            "B2873F024EC83CA897470AB8FD8853EB18FE15C159B305BDB0AB08F687EAEAFD510A3FDE73602E4303E1B84DCCF0FCF0", /* 3C */
            "D85BBBDC8033C0D89223D9C39CA9F34F7D3BCB6D5B63C3FD1C30F974DA0C0FB58B24BC9EBEFB5143C58954925B7B84FC", /* 3D */
            "6ABD7C2E0844D7A7CCF2EA456CDF530DE8938CF52B3921B8BA023CA2F281657CEC635DA675D1EDAEB4AA52F22EE1BE6C", /* 3E */
            "981C3AC677CB59046A92B54C84877B49745BA6BB40C55815B7AF550D22A371EDD5E8BD87C65F537467874A37F0F538F5", /* 3F */
            "C23BBA2A9DECC0214E610E930B0E34501A681AA91477577E38A3209714EDC3760FD15563EEEB4AB67D57668A01D42178", /* 40 */
            "6AF88CE145A098B51AEB858CD88B8B46E8B733AFB8E2D6E8313FAA8C10A7EBFA127D375E77557CEA96BDA2F70B2F2155", /* 41 */
            "EC8903978FAFB636C7213C425C079763760384036AB6D17CE0C63A26385F1F49299877D6811A6DF5876F90FC5304B88D", /* 42 */
            "A6FABBC2D6E0BA169B70C9640080E6BC29B2D5265598B27B4A9657C726E4397EA801CCC6766678D5800EF7CC72619998", /* 43 */
            "235931A8CF5490BFE798F98E0E8F879FC6EEE29C38F30CA7929A79F2D53E002488F2E12749587A450B85B28F38891965", /* 44 */
            "165E0303E4A4D82767994F42D1E8436AE6CC8BCF6E130D1B50101711709DDEFC373BDEC40CD0532840B274A4AA5109F6", /* 45 */
            "A9F88BA008FDF8C8ECC897E3476EE05ABCE290AB69D57A74FA44DB1811E3115D6267AEFD64480C882697D04A2D3AECEB", /* 46 */
            "C0782AF2ABCD331302BA1290F2F9627363C82F1A56ADC2B910F8E8C03EFE51C4E3EB348625CCAFFD93D607969CB8E7AE", /* 47 */
            "CC6E179443E58FBCD21C93C655A7B8EE2B9834A31F2B8BA4C83B69516025ECEE9176EB7B427AAE948CB65B9E30B7A76E", /* 48 */
            "C1A33A0AD6EDD98918B3C5D95813B5F7B024BD263B359A8BC8C17C2216A99B5071F9A11D582377293AA67C7618284290", /* 49 */
            "99B7465E09201C7B9AF89FA01CA4FA81FC2EC63E761AD123E2A9A39585B17D1408394DE529F94E81479448E69794FAA4", /* 4A */
            "23CA3D1C4CBDCABBE3265436CE1A37E41BBF10F69E8A4CC905A66708048F5C4DE259DCDD9C5BFEFE439E65FAFD936EFD", /* 4B */
            "A24D73B6978F719C3F53F343CCB0BB8EBE3C72769EE07C6AFACB9E539CF558DD67B91D4E30DE986A1DB913D11698913A", /* 4C */
            "98BD4E140DC3C3C6142B1592BF3263E8CDBEAC59ED095B0E900763F0F625896AE213550F30324E398A13A4417A803195", /* 4D */
            "2ACD98ED8C6260731CAAA6B4C4CF323804DCB41EB677EB5DF88B5844A8105B68981D9E951A061A4DBC9471894C878EDB", /* 4E */
            "4959FEAD5D6C2DBD6ABD59E28C50304906D2C5494CAF8B3470E4541304A4293C520F3416CAF2F503B23D09D92613DB85", /* 4F */
            "26B5A815C32D17912C99E7555BB033C609CE9D6A0002514FD485282B2B8D79979C5B792F4A4A14C6851D9D02DC0BB4E7", /* 50 */
            "62FEB6CACFB060EC9D977D69D5C661EABF08EFD806D8155625F1EEA460EA5718A25346B51F5A9665D92F9ADC358CA274", /* 51 */
            "27E63DFC63E8FFA6CDB9CCE2CE99FDA3979D5B754974830D3298C8407D6693BE629D5FADA39B42B72654D31271CD84E1", /* 52 */
            "AB1FA4DAF66E583CEEB6B7A236D24766A90738CDFDF5C6B328CBA9E5648E2D4CFDE5BF6C0CFE0DA39D00B863D7D78485", /* 53 */
            "75FBBF094EEA16AA48931F027CD729F25D360679009B2E7FDDFCD148BD3DE21A4DBFF544B094D0E19C0E5C6294352C22", /* 54 */
            "283A27FF968853D2B0960C6CEA0D03F2172BBA07A473DB38688C87D296E6F4BB5CB7E9BC5D68CF0F57A5D71B0E47BFB4", /* 55 */
            "DE0108AAC1E4FF2FD346CFABEAC62B99B72E203F98B5F60881853D8CA54B29BEA6AED7C89FAA1680D2093B155C39D7ED", /* 56 */
            "0BAEAC99D4974B84C7F258A699C9B4DA6F622C5E4ACCF5C158AB397D9781BEAABF811F67E101FFE3AFBCC2881C3C0EF3", /* 57 */
            "26B211FB518D6C3E64BADAD51A10784AE6BE4E06A587186CD471F5C61343CD5C8389BB0DD6AAED5DC88112678914A17D", /* 58 */
            "2B2D0BC3BB88D27DC5A7D1FAFF517AD296F39056A09F82ADFB38A61A6CED4D4E9D308E4EA6F9B2649097CE294AECC6B3", /* 59 */
            "8FCA2B950690B1A2293EFCBF03D422DF8C9125B3E76353AB3D402092A1A701739BAB974CAB9BF6765EA8FCC55D8C586E", /* 5A */
            "408C92E8C2E1EC8C4AF4C914B71B43505186AEE0CDFB10692385EAFAB9657C67F708E4D3C898CA801EC8B9F89884907E", /* 5B */
            "46E8958B6A2C18782172FD410F78A6479D8E9DD83A299004390913C3265AD025D231F1E23077CBF1E7EE3E574E80D7F3", /* 5C */
            "5A8567A3D85E40B216ECF161133FCF7352DA5C6FBA3C0DD756E57983DEB34BFB83254FDCB768D1539A14F95F35C6B82D", /* 5D */
            "498A29C6E19D4AE62EF4AAF46027BA11BDBA7DAA84F39505940B2A04F6DC944D4E7ED35610FC0D53BADD94C2907E59E1", /* 5E */
            "14DF0FC43F475F8017E2AA8D264BF82F92625BDFE58B934D8384F415A4ACEA818E9C5EAEC5D8642B4D8EF55F1C826687", /* 5F */
            "4A2335C4F77128D9544E1476D29ABA94654EC86321785044B04AD9B02F80445AB0E01B6480C8D020596E325E88A3CBBF", /* 60 */
            "896955157448D0620DB08C4C0F236D683BA8FC5B3CD1C4A204F57C53E144535BB7D04DCC7BE468404BBE993192334646", /* 61 */
            "1D7837E6AB02CE273EA35BAED4493EA4D1CAFDB5DF94FABE98B580BB62170C4FC3C57A6CA9421C4368D65FC2C1201634", /* 62 */
            "FAEABABC48717536454251E8F62F7315B318E8A7FDCDC5237C2E832013C913444D9E5DAFD169905212262E8C870537A7", /* 63 */
            "8A3E5D0BEF8402A2A33BC5FAFA01990963CBE8ACD00762F5EA26A3F1819841786EEB78D1BB4AF6BB7ECF9671300E845F", /* 64 */
            "0811B67CCCF5D0FC9F8CAB3F3496BD6B57CB7D24F1355C2D58218594165BDE80FAF3368A653A78F8C04CD80176267762", /* 65 */
            "E6417CE75AAA23B034A7BFE3CBA617618C13E396F8C9B6ED5C9066464B09ED6376CB6A642C5CE283498E082A3EB449C6", /* 66 */
            "6F2ADEA6357B5AA054DA382B15557B69302BD81946237AAE8F0CBB82111EFEDC45DD2DADCE20F2D38A77A5E9E8A2D1D8", /* 67 */
            "E1EC332735862A2892B68B1A7E9C7C44F45618DC99E963E37CAC984502DD1A73C8650598CD70840D9A5DA584A26D4EFD", /* 68 */
            "16B19B010740C15CB4544AC01016439A221F749C9E2F99A5A63E8A279A65570FC7231669ADD072ADC5BC35BA740BC801", /* 69 */
            "6C44E75A4F378694D27ACE108A57764717C487FAFA7E15D66A3654D5C8E29EDF0CE35EEDCC611FFAD88A8C03C0095093", /* 6A */
            "CF106948BC4B1F2C91C0DC9990B99712193B21E3E109AB323340DE0608DD16668A5BB677BF602828402C410B1197B771", /* 6B */
            "EB080FF49CA5543EB4B9429542D6CA275999D45DC1533205F7EA9E398A1BEF3EBE8817775476DEC617064D7790C84100", /* 6C */
            "F3328E9150A7F8D652E3E61B04ACFDF851D82010F3CEB01559D673336676D5D84CB3BCEF1D91C3420C589AB58033BE49", /* 6D */
            "54B8E70EDCE038557BB590E99687FD576CFF08688D2B1FDDFD0F6D068BFE994FEB9BCE302489AE4466B21F200661E3E4", /* 6E */
            "2F5E0060189669AD473AF1D03C00CAE40278299268D1F3B4888714BC3A7EC9D29FF9C7F071EBD2D9875A5DC25DFFDB10", /* 6F */
            "E2A97A3E468399D83BF7EACA32C80DA113DCAC8EB6C2231D227EC90E1102EE97B2344832F03814348613888303B190EB", /* 70 */
            "3A3D3B6CE026BFFE18D4953B9A68ED5924BB7B574AB777A0E0CB7DD64983DCB1CF768C439869AC978062BC7A900E6033", /* 71 */
            "39D4C3B78A7A33C743D72EF22AB0B4EB54AE8184DDA503940C2A7DA083C385369DBC6F921D4AD8222CBB61FE182EAA42", /* 72 */
            "D8CE9A806C0BD24DF69D65A65845727CC3FF81CC76F2B04876B1FDC3CA67CE58CED0970AFBCBE78A57502941B726F5F3", /* 73 */
            "E006AEC17FCEFCF905CAA1629E003591B7050CC99F585312669260401E1594908442D25AA757CC5A228655CD4038770C", /* 74 */
            "93EE8D67D3F1F3A1BEA46D48DBF8D7F43C91F02B8646453C6C3D7C1F04188A58EFA97287F89CEF84CB40364E108BFF4B", /* 75 */
            "C6DCE3730D4FF82502AF54F87D9727907D69D20F6F4F788F90C255C64C166E8FA3529FBF4BF9C9A23ECEC41136694F6B", /* 76 */
            "3DE10A5EC6CA7B3F7E196081D085ACAADF5F0DE3705D60F7393E7C83DCC57075A5F33BC2DCB98F970AEB7F050D1204C0", /* 77 */
            "6F3B3B3D11A8BC05B52269AB2B95B8DC12EDE24EB1385F13202BBA6B5836B5E1EE3636C5925ACC4942224CF6EEB509BF", /* 78 */
            "5F0CC3BBC4BE9A92584313FCCC54DD2EC11FE90F003940363371667C72FC97239611990B62AC8D9F4CFCB9EB3C317FAD", /* 79 */
            "CA8E520A894A3FBABD9ED1B80098CC40BDF24507DFF3757C47AEC572E68D35ECF3D4523D27B373E41AB11E16973A05AB", /* 7A */
            "FFC293A6C26B817D2C9E9D134959D8287FA5216408199BBFA6F002DE0DCCD861BE8F9DC57F2CF35D1352E2DF86A47647", /* 7B */
            "84B55BE101708E743ADEC53721209F3EB18F9A1E68DFADBD09A050819774CF2DE4AB295D380A8762A3605B0C689C239F", /* 7C */
            "DDC7031FBFDFFE8F0B175DE65B832F0A31162ABC6571968551215E534BBC36B19F2F7D3B5D01AE44CF43A2426E83B61B", /* 7D */
            "7E32DB672B16F04ACE6F45DE0E6AB78825718548B8E70B41D7368BCF39A0FAC4956863EC49880C47720E335796341674", /* 7E */
            "06707A8E33D9D6C6B684BFE26CD576C644F47E5ECD5FC46CAF1B23A856D844B798A627916AC5657E040C3964A1127E19", /* 7F */
            "A5DAEC3134C0A39B0CA04160BD5ADB1FB50EC5A9F29E1ACBBE2FA1126AF7BFAFBEFC0AC4C9C5A4B3994739C71FB1EB29", /* 80 */
            "6FEC2D343E83A7635BDBA5715757F50CD6F6282EE46A11B3A8B501F5922A5524A782A21006B605CAA10BD2E896975C81", /* 81 */
            "B8AAE0532226D0ED891831C0470E84B774C824D648E8FF28B5E4E02EAD3906EB5ABB086ADA60A713A80C57666A9E29F1", /* 82 */
            "529E3E52B1E7230A0C148861C9F08E260CFC8A131BAD803D8C09F324902FAA9F0231EE49879998483B0688492E2B5457", /* 83 */
            "EFA6EAC5036814CD02773C1F8DAA5DF50E4EEDBD0702DE31BA7FD757D0D740EFA8805F0C74005F8B1448467BFF3E1EF8", /* 84 */
            "2A07B766016AC70D64215C35364219E9CD6F7EFE35FCF6F1F05CC06084C29267AB3BF2F32579A444AC75F42D9A25B9C9", /* 85 */
            "EF3A14B5EDDB84642314E0802D2DD0E914DEAEA9F928762A5763EBB480E15A0225F7CA14E8CDF5E68E594510DC61E6BC", /* 86 */
            "E62C38DCFD21000B7BB32AE917EE3DA7E49F15E24CC9B65656E28259DCA361D8B43B8008A9285F480DC6B4AF7E4AE61B", /* 87 */
            "703C64241142DCAE732D33342C45063A37877EA1624567CB2871D534614DD114E748092A1D94F5D14524056F0C6D1CB7", /* 88 */
            "E325B1823A595DF9742D0DD5C96F397C44361C9540A9F45102382F9BF6331FB98ECBAFBBE91A0467528EBF3811F904A8", /* 89 */
            "FD2BC6534631FB0D27A5F036FEEB9A6CD0F876D7911D077512EFB3A29C6E0B72DC4BCA3D5E871DA1028FB6E6E608F46F", /* 8A */
            "EF17ECC8930A7B4A9D97B34672FB273DC6AE835F35A25D8F6C27469530C21F5B2FBC16A26150E79502AD93AAE0B5C71A", /* 8B */
            "6D24BE43CF07DD5663681D62A38D2A2F9872C9B411724AA0B882B4857C19690A87B1BA8D2804C6F4D7B199CC36F40B49", /* 8C */
            "EEFB8D8573FD9E0F933403199B91560AFF0DB41665D5248C322EE105EA984196DB8CE0F83890D89B3A32F8983C901F80", /* 8D */
            "082CDAF93F215BAC67C118A1B9274FACAF74501CFB93198A53525CABA0E812D3C9AF3A005EFE8A6EF242DCB60DA7B2FE", /* 8E */
            "D3887FBFBB7314DFDDDCCCF0F720C342B2C4331C33C8C4151666010767F4785B8455B7C1FD5DE487A821C5EA181875F2", /* 8F */
            "7E289831418562F02AD12E3042B185C37C20D0D735A6AE96A68BEF98E22CBD41A1411D22F8D93243D813FB404F3D2F38", /* 90 */
            "E13FC0A76F6642947E21C9D9F7FDDDCB161E68B366D6B1F855BF957EB5743874B23213EF8364D766529BB98AF96643D4", /* 91 */
            "036D7ADDAADB5C330525835F802D032E7DF7D0D8D7A2BEF284927644B27696B7215E21E4D1F9B5B977743669C40EB7FD", /* 92 */
            "A9B3534BE88977845BFD4283541A509097AFFCCD121C9778C146C4C9637989C70820E72FCBDA59C75526E2F4A0AE4F4F", /* 93 */
            "A4739E20FD72BDC26D6EE5A5C1A54CA670A97A6FCB884E5C2B6108339E979C4893A63730D6BB23A75B1DCEAB00045EE5", /* 94 */
            "427C14E4F88C8BDB1D8630868E039BC233DB40A251502D1BE043C9CCB45D2B3D292B67B6EE077B2D1C3A2FBDE24C742A", /* 95 */
            "3DED69F37016D86A9A947B13AC66D7C3822D8645DF4CB39C2BA20F98F19E10DA6703138D422AC4C48D34D6138FA04A1D", /* 96 */
            "28E59C8B257D112C8747068CC5499FCFD6C16EB780F9191AB416151633F7AF08A230E00D6BA1A1C3FD066FB9965B83D2", /* 97 */
            "70F4BC1B7F8FFC3738DC0331E56B0FDCA9AB7290AD2B0BBDB307973C3D0783C6BDC455F6CDCA111F23F0E08317B8F0DC", /* 98 */
            "0AEEC24E9285C50F3BCDA47833B61ACE839986F959EE0723C959034A8D7F5EB9D4AD7E05B05C4FB56C37A3D39F7A0EC4", /* 99 */
            "0227B7230FBF2D0728D7D2AD632BED4707BD8F8B5012EFD048A0D43AE04034429B8939207F1449A1351EAD01B9FDF219", /* 9A */
            "A7119D2E311CEF251E532CD0C4ED04792272F878D8D30A0B769C412CED9C4C42262FFBFA65CBDDF5DB73D86721EA368E", /* 9B */
            "4BDBE90B3FBADCB21324EC3A8D6FEA576D9EFBE530850D00401A88AFF8A4C8F4655CB76B8A2E271C35505B6DBDE16F43", /* 9C */
            "6E15E57E23F570374962737362C1FA26C962372D1829B80BA1FE6832EA4D62116726E307F96E776304C761081677505B", /* 9D */
            "42E2FF3A8A6FC164FB85B2BC9D28B268C559CFF024533A282EC83F3911DAB3CEAE0FC74A9D736A27DB9CDD048BAB4CCF", /* 9E */
            "D79C52221D20E765499EDD73903CE7049B016D987DF48349FCFAB44AD12FC5C1811293F3B800FDF9511DC619CA53CEBE", /* 9F */
            "A059EE78B826EDDF4673AF294D17C85A5E527D4E4DF282B5DB5B9A2693F95CE36551D304FB54F296AB3EB70D65912FCC", /* A0 */
            "7D0C4F67B6C78135390CAEA7DE304D3749E19FABC8D494FE1A9E1B6437A04516886CC4BDAB6AF35A0529217344F502FE", /* A1 */
            "3CEDF34141B52CEE8133BA924753573FCB32BE22BC66025A0C480183DE403CB3BF5B84B427DFCF317251428DB0232156", /* A2 */
            "86FCE831C58E25CB5CC43FFE45CBFC7533877CC042F199BE1212FA7F0CC22E1C448EAB4B7D1F9823A7B1363A9FA7599E", /* A3 */
            "2D8C2FEDA0E5106D192E366838BBEB3F36226AA60ACEA0AFE7E1285DC1F3926A900371FA1883D9ECBAC33B1AF360EB66", /* A4 */
            "D4A2A11612BDE0E382AB0DA614CB4CB8189A4D50AC01F4C6E36A5DA1D9F6A647E43120D6B16B11B67D395F4236E75378", /* A5 */
            "C0C155CD47F3877F4B03BFE5C334CA7177710F1F4B844FF73443BBAB720E8DC5F03F8868C58634060FD60511C872EB50", /* A6 */
            "8C253DAAB52863069AA438F54A6196AC181D08C723A22C5E633C49C88E3910A1C9F54A67992675B01FDD98ACBD38D976", /* A7 */
            "A10893DA7575A9F78F5F4A025AB2A018D80538F0336BFFC00F9751D33889626F30383EB925BF911AE6149F68CE19CC60", /* A8 */
            "B9081DBAC6BE0598785DD9BC69C714928B035A0CA56E172B8946783500724888AF1E57C958650569E1DE4E944FF22261", /* A9 */
            "EA5EDC4D2718C0D2CB1C5D4DA15A8AE4C6272382F816301594A934E5057B54CE658E481A3D68D10DE8F24929E50A46A0", /* AA */
            "7DF146281AF482CD014B68E726407B066CE564938C70DDBC36DAD2DE72A5DAA26D573BF69C0B2980684DAB14B4AA0329", /* AB */
            "9C69DC064E738B5F83CC16BD5A1C36F5A99B365E6E141B122748FA5AD0FACCE826D073A047D99C49B005B182505B0C0C", /* AC */
            "15B6A2A20ED0FD1C9333AF729BD65A2522CC333293BD2C1BD724D949B15E8BE169D0DB0512B9711785ACA8980DD7653C", /* AD */
            "230EC629D77BB3F243115B991D297CB2A2F955792C53C76F48A76728EBE25BA77CE662A405384400DDC06B7E6BF49D66", /* AE */
            "20DDB9BD7644410B056391B1FA2E8C06CA4EDE51CF167C0046602B550536F8705040672597C21FF40AF8EC6E8AFB844B", /* AF */
            "0023C5749251B883335A4F86D66B7E00AE353DED3EFACE8F3FC80526D67B35DE0D9078FBDA80BC53467900DFF3FE4C14", /* B0 */
            "0F9CB2BE6A448113E38D541B6A9A5829673953DAF354FC0E3C818A277F8569E98D16EA77DB122A3BE40A860318B6EA84", /* B1 */
            "78CE11F42D7D5E5084F76DFF199C998D999B578E3AE935CBD9FD092C1BE6321231F33C63ACD316D85AA08030B8D65C0C", /* B2 */
            "0098DBE19CA84FE9E2426617D114213763C3C4166A78E21B74B145353E03B0E4F43C0824EAE508C458C1E6622528602A", /* B3 */
            "9E27EBE6D1426A6F2A6A600A6B5FA3428FF7E2306BA90370DF83D91A683EDDDD29572442F0225388E9CC0F1B6437320A", /* B4 */
            "054DF380E896064EFAB81A4AA3AD88A4F87426486CCA156FBB1B3C82374729607EC0B87CF73F960A5C57D7E6470F7808", /* B5 */
            "5758E103AC614A1A766AEE86F81358DF203FBA51DC74396A78C93DF969C5721FE69E32E230196597E287C6CECD8AB95B", /* B6 */
            "2A06A7C10C0DCC9799D5298268A6745FF2D818BB774858B3D52A820D4F64D8862F808EF87A263981BB91206E6347C676", /* B7 */
            "0847C6D71CE0C74686FD451B447C1E11C20623B0E2856FCC3ADDFA2D0398181E6736A0A06B336B46D1C70AEEB2B1257D", /* B8 */
            "5633260D141A9776D530805F596CA3DB8CE33EF69437CE46F62D54E97E747088DF5C9318489B45EAA4AAD29F0BA850CA", /* B9 */
            "BDBD7B16767F6D9FF7968427F1B7B6DD58C76599B35276EE286F4C7F6CADD7918188C0401742117BCEC4F1964266D163", /* BA */
            "97E4E8A6B5135B248A8BD785E52979774545C1A0975BC5BB13FAE3BD9F59E37DAFD5627C0E91DE2BA223AC778474E1A9", /* BB */
            "DE1BF1EAF86C6B3BA246A3ACD50035FE6F80179DD96A21CD3F8DB7CB17300D03497A798B5D94506CAD52DCC6F61AE841", /* BC */
            "F4A4E1D08E1F440B5E27633CD56422E01465C14F1DB414209A939043988D37C2CBE65CFA245DB3686340AEDE28DDA855", /* BD */
            "1F7AB65A3F892454D70AB4167EBEB5A19B2631E824C2028DD5D97BDEE31519BCEA2DC77449E4058CEB204F2D6D2FBAFF", /* BE */
            "6537E69171A2665D3FD2F835435A3F23ADD5DD3E622D6C8AC522CDD5E5E243F85AEC27F3DBFDA8A2477A65ED570E1445", /* BF */
            "3BA7CB01D32E9D639E335734E7B5416B0ED96A84F94539F645CEE2E46DF5A70DDE142EE1E9AFEC1C78D6121C4FDC72DD", /* C0 */
            "7BB30AF653390B772D394F2B7F8F7BB60277A3C213AF34897DF6E674DD56D0845643CD3073C42451FAB15F8BD1A1DC18", /* C1 */
            "42B453ABF5150D8B913F109C1188E18CC27BB7631FB43BF9EBDDE685EF10841976D67C87C56D33EA95EC73C0AF40F084", /* C2 */
            "BCE43D59A1F50BFBBA7027CA04D84600FB6FDB98A2BE644BD5DE777E993DED4AFCA39F1EDF710F3AA5E5893C858D8841", /* C3 */
            "C68AC776E6AEACFC538067C7866106EBD27B4A352F4EFDE3847DA2B3BF01E3783C79E3C136926D58F957BC8726AA1610", /* C4 */
            "95492C4203C7C6120DD60DB1EE8321FCE1D9EBA902F62B42EA2DBF7D0E37A4F2E11FB9098BF5DA48DBFE213F818EA338", /* C5 */
            "17CB21316D4756DDB88952498140146A648112F5808442884947ADC3F7D58F35651CCE28E26A53770B3803DAF337F89B", /* C6 */
            "BEAB16E2DCE6B6E38F39ECC8E39172DF607CC9553FF29C0E4BFD15154F4F0BA7EE6230B6BD408CE435B654110D164E99", /* C7 */
            "ADDDFF1BD2C11CD42A1A262CBA6E1AA00BF2291D09475A464C93A0ABADF4DE3273EE8E1327333E63F3AE2031F5D13B28", /* C8 */
            "246C7CABB2D9A55C50E9C7282C1EE0F62FBDA09565A0D3D7196552679C04A4EB137C66DA29A6DD8208A76B6B4BDA56BF", /* C9 */
            "7CA3C59BE3E286106ADD75CF1F7AE24801747450737A6435A1F2259CB2B4923BE0C8F55E8ECE7210D7964398F350B69B", /* CA */
            "E045864ED1825101AC54969193E1A1C523D85A934D0794C7B4FA88CB734A42137C5CBFD6BDA3D5F966607FE938748825", /* CB */
            "BAF36FD2A180D481EAC440AC1B9598F79AA24D80FFB7B06C79601F517358F163D1071831418BB63B819609A6AE7D3A03", /* CC */
            "3E9152D8CDBAE55186AD793F203DD016BE3AEB778AD4A8912810254DD76B66189B5DCDE36636C3270A8AAD65868BC58C", /* CD */
            "6D0672780D93152AEEE705247B8280919EBDB976F137463FA7DE3E73A2D0C1BFF871A00BA0046AC7484C96A803F23486", /* CE */
            "0FC7BCDABB06BFFBF75C3FFB3D6309B3ECA305D103109162373F503B204FFF61CE332C9F54963FA29A4420A52242CDB4", /* CF */
            "C71D481179D198C1505A2845CEE92569F339BFF6DD6755B58BEAD52B8DE892454B686E65920DCA2B99593FA43EE68A37", /* D0 */
            "D90A68D717E615019BB920AEA19161A62F3D6F96D90EB1E4DF15ECBA10513D7DE6E5D539B4F01831C7D17A7528FECE36", /* D1 */
            "A04FF0BEB4EBFBAFE5E90A5B3DDAA3CA8453542209F4A14580A6FFD72BB5A70714E0C4705A1ABF6AD699EC1FC18A677D", /* D2 */
            "7021A124E3181575DC7AAE2817AD945F8BB5521E7F0D565A6671D3792F0805EED3888EA394413A1ACE4D7E47B55BF9CC", /* D3 */
            "22F440263CAADE68E77BB287772EAC7B29493775962A40E91E06A27FA68CB91BDDEF02932ABDB9C779F03B88DC175233", /* D4 */
            "65F6D517B53E239197DB65A2F00B1C391D77AE9B85AA485519133B9B3E9B07716376D9F11A7DB3D4949AD02F5AE16184", /* D5 */
            "FE4434CDE09D923B03B0FCFD713B70522D713290D4A672382B56946FF629EE9660A15D01B2B3C4280B1D5EAF793933A0", /* D6 */
            "BC40FCFB0E0D494BA31C4648C7B3D1DEF1113C219A07EC8D2378BEB1A5C2BD1C190CC3478070A19463DAB6E1CCF56329", /* D7 */
            "901B6B9E82BABF91872A234C45D610016CA46A95C1CC6D6C22779315E0F0229560A59396346BE6ACFB67A503CB488846", /* D8 */
            "50D440F74C97660BE71ECABF64EDFE0C80201B895718CE22A05D89804D35D3068F700402A2B0D086326FCB334CA4DFC0", /* D9 */
            "BCFBD02EA005CDD5F0225A467555311508E18B3692A7AF6205D34A820C8CED0A51A8D7CEC33E80EA0AC007503FAE879C", /* DA */
            "F43EEFB5C83C521AE5E9B05FC48841AC79C52C38BF85B5F926CD0818AE3BF7A94F385C32CA8F5F74F17B22107B954752", /* DB */
            "1A48FC969198A4B0D9A78940BB0C4E1C42781D9BE60E769187D1CAF3680F8A30D09FF193606AAF294518DABC60048793", /* DC */
            "F05D48134A56A03489A65EEB91DC69B98FC7F43960E63C62FA1C6B9FF9415E927E219D4E563479352B6A48D6DE0AEF85", /* DD */
            "1A7FF9C54B045FFD44A0A9562E9468B2F11425A22D1EBF92208D33120BD28E0EF2D74197AF80E162CEEDCA73DFE66C93", /* DE */
            "D57190439D29C9A444C007DC2B5EAF9DEF6DDF48A780CEDC61B205E4A96024B11885B6CE84C3FE5DB8B56986B6E2CE21", /* DF */
            "F36DACFA34237E99BE45EB5253BCFED0402C6946B8B21AC02460A6FCE7E9CD67F89A6D5B162629FCF66CCEA374DB821E", /* E0 */
            "16E06074DCC31A1DF172017AC3FA38C3BBC1CE4BB784ED60DA89A8BCE82AE671A6DACFFB8D26C0BB185181AE9609F6D6", /* E1 */
            "F110DBDD94D17661F59FBB4CBA69F393463B60FB3F3C5E001C60B896FE8E78AC5EB3E26795DE5AB6997328D4654D6219", /* E2 */
            "21069118ABE24B61811CB8C48FCEFC6A483B032CFB56F902FB32E848198CC057A620815462A04F70900038D1894959E2", /* E3 */
            "5AD509789BFFECD0DDCD5E8325F69CA0154D8F1ACD9B8C82AC7DF75E94CE3CAF6D6554D1B38754BEB5DB64AF738486E7", /* E4 */
            "35A308A1AC9A43BF2647805AB3E6E4924BB74A616F61588FFA4602EE5BDBF54E3FDD62470A7174DB5795433CA808FAAC", /* E5 */
            "51A094B8774CA6055F07974C74EEF225022AFEF7AD81A9530967C44BBA336FD68AA327918AECBA3DF70B8436573C3F0A", /* E6 */
            "CF374F83420766C371F31901A13EF07C63AD56C7DEF9DC0F9E5BB5E859F5A231D0BF453BB9893E4CA1E14B66C2719760", /* E7 */
            "B41861CC73FD3E48461D79A138B04BE14010D37D37FBA8177D9622AA693225A42204454B8126799A33A5D487DCCD6EB6", /* E8 */
            "D291D0317A053320E27678F1E50D1F769A3D663A63159FC7AD7B4D3F67BAB452269CC05E2B33CE1C0FB8261CD734BCC3", /* E9 */
            "F3D0546D3D4A25EEB42874AD28C9B7F273EC788B29962D284AE73A48132B8553756C99D7A0910B66ECA7E2C2712D555C", /* EA */
            "559FA5BF24911FDDA1DDF5DE3770554BC7C3FD139366B9466E7ECC0C881D2BA414E76D6A27E54B877352D5FBC4FAB878", /* EB */
            "F19A622BED8DAC0A35548E5D7EFC5A2ECAC84974B4F057B2AB317ED03D0335AE710FC138F2C517389C90CC495A403416", /* EC */
            "9FA7DEB936F10461A1529B0B58462F9D9F109111C8B9EC6523A3EB28444E33EA554084CA75118937599D58A7C946EAC2", /* ED */
            "6EC3AABB7856AC4E980E6907C1CBCCAF1F8557ADC700CBF57DCB1CE0AF48D9F47FB3DADF8199AB8AE6B36DB8FADBF312", /* EE */
            "C00F0D3F7A101660605B94B12DB6C69779944F7BA2B65F3840858ADEDD47E2BC1E044BDB0E9FB02B86C79D01A3109539", /* EF */
            "9731893D5B98482AFB8DE267F97903268780F407143A505DA41CAEFCCCD3A8E3A042F0B3D7B7A7FE3E3151FEBB19A1AC", /* F0 */
            "E7EDF679003A6950BAFC97D4A8C6AB1213C096B49C79559AC3052501434B50191280FB23E7ADFB091959905D31BD2FC0", /* F1 */
            "575C0C46FCFCC65BFE625E873F34B4191696FDCC7F51B8A3C79C56F30E5AE7C014E3461CD27FAD151B7BCCB9CB472859", /* F2 */
            "3806FE58E5CC8F16F8244ED76734C1BF4E04940E0F5DDB565BD0AFDDC4158B7BA4C6BA949911C5C9FF6E2AC155AE9726", /* F3 */
            "49C7C844B8114144B450E41BCA35CB00302450EC67BEF97CA8662049DB1E0D8BDA69C022528EB8FA6ABBF16585C1A2F7", /* F4 */
            "37BB420DF67F044EDCC0E9F3E2EF07B34D100886187778410492E5379305DAAE3DA4791C37E4128F80688445CBA4EA17", /* F5 */
            "51398A7CE4CF8D9D49A5FCD891A69CA53D72A60EC2392DA50E8296B879AB55396BCB00AF2EDC0BDEBEB93848E54B3E90", /* F6 */
            "7AD7C52A18922E1929292C57C4F5B8F5F0CF1F98A577C10B072B9F293BB660CD09B8604F5575B6FBDECB396A81B9FCDB", /* F7 */
            "254AD7ADB4C220DE6C62E20F95A0070DADEB89F339309BD8A2F685CC178B289F9343905B5DEE95A5E0C30F34A2977C86", /* F8 */
            "669CD51AF7CFBFAAE3E0806F6880271D6934C259E098BF905DFEEAF0FBCA724989F74B948B4118B653640AEAFB6807C3", /* F9 */
            "DD3BACDCC04BE1206D4949BD64198E5131FDB39666598A74BBBC6DE9C0C15A81F27F201C61C062792738AFE3E84E5CDD", /* FA */
            "CDD71FD35A6411DE3CC012793E87523FB0CFF8720FCA36F393E85FE07300F012E894A085263F090B2DF60A01DAFA90EC", /* FB */
            "9DA50DB1EEB4FADDE524E49C9974799ADE09FFF26A24CBB9AF9D71E9F3ACE7CDEB62B1A62566EC9D06D02AB1217D3553", /* FC */
            "DD31E6391AE0352293ACD1065B35E915F4EB56CC03E792180717815C850C97F1BF4F6A8AC0540A6FFCF8AE5DE9507FF0", /* FD */
            "AB45B413DC50B20740B417369551D8D5CA32286A108E721003225E54D8D093AF4B6CA5591EA576E94E12AB774DC4E062", /* FE */
            "D9F4F850DF6CB96C8ABAD81B1667335DCB4079CFE79C72E5E5542F763E316996303E4B79B9D397C4E46933038B945111", /* FF */
            "75B15CC53B0D2502DA1BCA6BA05243589EDA977556C06B7E6C57727ECF0A1325DC613D5A78E5C3F8CE062D94A3B4945A" /* ZZ */
    ));

    private static final List<String> INPUT_MIX = Collections.unmodifiableList(Arrays.asList(
            "0204230C170010240E27051B0B1D0820291F2D1A2F182B25282E0F110A2A061C09191E12162C07142601210322130D15", /* 00 */
            "161C1215210A22020B0C1D082C11002A192B102809041307242D2E2F29260F17270D1B1A0305251E0123061F140E2018", /* 01 */
            "15180A1F2E2611282213251B090006082C14212D272B0229072F230B2A0C1619031D1C050D171A0F1E0424100120120E", /* 02 */
            "2710181E090B1A13210D122A2B29112E2F2502081B05031F2616060A192C141C170C0E07202215010F2404001D282D23", /* 03 */
            "0A0D082A1128291A1C05212513072E18171B0C2C231F1E2F06240F001D1003040109262B1920120E162D0B2214152702", /* 04 */
            "2B25111C1A12151F0F29130407061B0D0E0019091D2F2101262716032E2D24182A0B102C2202052008140C1E0A172328", /* 05 */
            "1A1D1827092621060A2C2A250E0801220B1E052F0F2E00160D0C2820231F102B071C24192D1303041115121B14022917", /* 06 */
            "2D02141E1D2B22180715110E10001A0D0916230B132520122E1B2C0103081C052117280C062F291F0A2A0F2426190427", /* 07 */
            "100E19231A13140C221D2E152A2829171F0806090B27042402122105162C00260F1E2D2F0A181B2B03250D071120011C", /* 08 */
            "29100F13220600110E0320181F2A192F0B022B2304072C250821241C010C171D1A2E121526092D1E050D27281B0A1614", /* 09 */
            "2125292B0F0E0914201E240C1510030B02010628081F11042F18192E0A22271723071A2C1C000D2A1D26122D131B0516", /* 0A */
            "2A22151D0E24160309192C01261F2718252014171E2B0D06050C29110B001B0F1002282F2D081A2E122104071C0A2313", /* 0B */
            "1E1B0412172A240503260B2F292E0A0827022D21190C0F1D1C0D0E06001416131F222C28110125070918231520102B1A", /* 0C */
            "290C22191409181D1C282E0D0513262B0F001E10042715081B1A122302250B2411071F0E2A2F170A202C2D0121160306", /* 0D */
            "1312012C232A21271D0E201C282B0C1B07220F08250504240A2909101F19022D00171E031516141126060B2E1A0D2F18", /* 0E */
            "112A1D210B181B2C002209032D010F0C1F141C1E2E25152F190D0A29131723120716101A20050604272B08240E260228", /* 0F */
            "08281E2B2C240C210E16050220272304181A25001014090F2F0B1C1F11011913222D0726151B1229060A030D2E2A1D17", /* 10 */
            "170E1B2115040B0A110C2F062209051D2E002A2D122C2610160318272914081A191F1E240F0D2B0220012823071C2513", /* 11 */
            "23190F22260311051A1F28162F2D0425201C1B212B06010C1D27242909070D1000132E0E17082A0A021418121E0B2C15", /* 12 */
            "110F260D130C1923142E1C06050807182A1B220A152C17291F280B1E271A0E092D251002011D2012031624212B002F04", /* 13 */
            "150C17191B072B051F10200E252C2F121806012403212E14221E0F1A0B1323292A160D2D091C28020427261D0011080A", /* 14 */
            "241F2E262D0C251521171E040E0A0D1109070829181214020B2B2027162C2A0F031D10190601131C2F28002223051B1A", /* 15 */
            "16120D14221527011C001A2002242D1F1E25262B03180C2928231D130807172F0E0B2A2E211B0A1106052C0F10190409", /* 16 */
            "142D2105100E2F2722150928071323081F16292518262A031E2E001C0C122C11240A0B061920170D0F011D021B1A2B04", /* 17 */
            "28130E020A191D2A1C2508091F211716272D05180F22202B2400142C011A0D2E1511292F040C1E07121B0B0626231003", /* 18 */
            "112E211A200728180929172A1B16141D260A2B250506082D0B1E1224100E03150F0201271C002C191F0C22232F13040D", /* 19 */
            "14261D241E110A160018202E0F060C1C042D092A0E1003011713221202052C27292B231B1A190B2F211F072508280D15", /* 1A */
            "0308292C222D02122B20111A261F1B25230C192700130E1C211E0D151416042A09281D07010F17182E24100A052F060B", /* 1B */
            "042C1F08271B16241C260E132321001418072006051A032F2A121D0D0F2B0C152E2D091102220110190B28251E170A29", /* 1C */
            "1F0C1E202A2E15221901240308060A251229211711162F13071A262D0D27041B2C18002810141D090E022B0F0B1C0523", /* 1D */
            "2429270F10060D13280420230A251E2A2B0E1A1501000514212D092C12032F11070817260C182E0B1D191F2202161C1B", /* 1E */
            "282409292C230111170A1B07020408060D122B0F131E1D220E0520160B210C0325141C2A1A1F192F102718262D15002E", /* 1F */
            "1B09151A25081204062B1E1728192F2D18140B012A0D102007290E03220F05230A112C1613211F2E1C1D0002260C2427", /* 20 */
            "11291B171D062703040D190C2D002228162308122502130B1F201E1C090521242F1A260F0714152A2B2E0E2C0A011018", /* 21 */
            "2B08172E1E1619060B07271D23120926141B00152D012F2A0E1F020C0D21112C1A1318051C292210250A042403280F20", /* 22 */
            "1A2F160018081C2B0A241D1729210C1926252A1E071F11230B2C032D201304011410270D2E0E022812091505220F061B", /* 23 */
            "1814132A0E2C152D240B272E250F2B290103052F0407020C1623111721281C2022081B0006191D0D1E1A0A0910261F12", /* 24 */
            "0C2214062B29180213111A202C1E0810001D07152E121C0D0A1F0405230F270B192503091721282A2401261B2D162F0E", /* 25 */
            "2E261F2315282D041E1C1007290902161413110D2F20030B00182722052112081B19252C061A172B010C0E2A0F241D0A", /* 26 */
            "0E03042412171B182A27280F0D020800212C161F291A092B0614111E260A052F15222D2513100C1C0B20071D2E190123", /* 27 */
            "2805031F270A2E0F0220221D191B0B150C14042C16212F092312071025242D2A18060D1C0E112B29261A081713001E01", /* 28 */
            "031B2026142C1912270E1C1E240701021A181F1D252F282110050F17132A220A0D2E230C0400162B0B15092D06291108", /* 29 */
            "21020001060C2922230D142F1C1E26121B2003090A18191F110E1D2E07161A28272B0F15080B04102C252D242A131705", /* 2A */
            "0F2506140B20000A2C21182301220D022F2D0E2B2413110719262915042E1C1B281A05101F0803172A1D1E120927160C", /* 2B */
            "2C2B112F00131803060715241C160A0521271714020901041B101E0B0D1A292E2D0F0C261D120E222319082A1F202528", /* 2C */
            "2F202A050126130E1E0A06021A172B041518100F2127251B090D2E281F160B141C1224000711082D030C2C1D19292223", /* 2D */
            "072F190E2316100C24200F2C2B020306120B1E251A0817130A18041F2D1C1D210127281B0522290D112E141500092A26", /* 2E */
            "1D032A1A0E1525042907210113200B2228061B2717000F1C2B020A101F2C09162D052E1223142F261E180C1911240D08", /* 2F */
            "151F2E1D052F00162D2C2B2321181012072A031C19140E261A20220A27280D0C1B011704110929130B1E0806240F2502", /* 30 */
            "1C1D272B212907131F0C1B002F261419092C160D24031A100F172A232D0E0A12021E2E18150622200504250B11280801", /* 31 */
            "0B100A1C24272A2D280F03232E0516140D01042F1A08220E001F29211D1E11092B2506192C17071518020C20121B2613", /* 32 */
            "2305152E02120E2117110D2D042B22131C202A25160A2803060B2614011810290C071A0F091D002C1B1F1E0819272F24", /* 33 */
            "0E14240717041C08021F00122C1D060C1E152B19220D1826110A2D2813272E1623032F2A290901200F1B25211A100B05", /* 34 */
            "0B222E0424131A261D2F092A08150F2320121B0602110A1E2D030C1C2B191001272C0D282521160700171F05180E2914", /* 35 */
            "19132821291A2E0F0902122F1400032B17270110110D180C1F2A15260422050E0B23200A1D1B081E252D0724162C061C", /* 36 */
            "13171C1E080B2F290A00252D031F022201122C2B15091B11240F05270C0D2816191A26042E1014211D0E23062A071820", /* 37 */
            "21162C082D1F28002A0B222E102F1D0D1B190C271C2017150129241A120225260A23052B0E061E1114180407030F1309", /* 38 */
            "1D2E2A062F030F1C2621191B15021217110C20141F132305162D1029081A220127242C04180A0D250709000B2B280E1E", /* 39 */
            "100E252203170123160F2B0913200A001B271F072D11242A0B18140D12081D1E2E06260C15041C292C211928052F021A", /* 3A */
            "200B160F1529041B0A2A012F0E281C2C0D031D2E062B0508100012252D0C02222617191A2114232413181E091F270711", /* 3B */
            "130D0B0E2B1122020F10230C061A010A12152D27071D2A1B1E0417292F16210018091F20280324140508252E262C1C19", /* 3C */
            "272C20212F251F002E160526080A15282403100B121E0C140F1D012B06071C2D021A0D2A180E1719091123042913221B", /* 3D */
            "20091A29161B051E0D1100191225130C0B040F261D1C2A1501222102270E2D100608282C2F2B0A1F072E032314241718", /* 3E */
            "0A1401151F222E2D072C2A29131E212F0618241A0D0819111204170B0020020C0510092627280F251C1B2B0E031D1623", /* 3F */
            "2708050D150F09252B011D111C0422030A0E13171028002606202C1E072923211924122D2F0C2E142A1A02181B1F0B16", /* 40 */
            "111A291C1B000B091E0E1F180A16261303120F2E0D220207231408202C271028172D2B24042A21060C192F251D010515", /* 41 */
            "02132D28001A012F2B031C142226242C292E0B191D17082A0D1B061815042125050E0F27072009231612111F0C1E0A10", /* 42 */
            "1C2F012D0A0F29152E04192B1D28000527131B090E232A221A08251F241718071E2603120C060B2C101421200D021611", /* 43 */
            "200F091B281F021C0B230C25042227112C0E0A24291D2F032D17060718100515262A1401001A2E121E2B1916210D0813", /* 44 */
            "0727182A030F291715112014260519232B012422091C210A25001B040D28062C1F1E0C082D10131A0B0E2E2F1D120216", /* 45 */
            "2E2D1A1407130400122B260A200B2C0D0C23031D27010F2A22160519241F180E11282915101C0908020625211B172F1E", /* 46 */
            "2D0C0513141A120E1D11292726010B192107102E2F081820171F06152B0D230024031C1E16221B282C250A0F0204092A", /* 47 */
            "262219251F011C1A110C10180903070D242008272A0E12290017020F2E2F2B1D2C132D0A0415231406282116050B1E1B", /* 48 */
            "05150A1E2F0010231B251622290F060E13261C091F2811120C272B24172E012A1D2D071820042C21030B14190D1A0802", /* 49 */
            "100B2B26152A02191329062124181622142E250C081B27050F122F04002D0709170A010E0D1D1F1A23281C11031E202C", /* 4A */
            "0B0A140C110F0903061C2B011325261B1904211E121D2D0E2729232A102F1F2C082024001828150522160D1707021A2E", /* 4B */
            "0C190B181A16170901262008071329280A1D0E2F061B11042B251C2124100205142C23120F0D2A0300152D272E1E1F22", /* 4C */
            "02091D2C0C1C1926072B0B0E01252724212F29232E22131B281F110A1403081A120604172A000F0D15161E0520182D10", /* 4D */
            "032314250E0A070B042E0022161A2D05292A18121F27260109241B06112F1021021D0C1E2C081C130D2B28150F172019", /* 4E */
            "09281F290C261B23120D2F11202A072C251D1A04190210162B15060B05222D180E0F21031E2713240A17002E081C0114", /* 4F */
            "1D1F0D0B172B2D0C0F2E07280121101C230A2419292A0209260E1B251606112200121A15052F14031E2720041308182C", /* 50 */
            "0A27141E05041B1D120703100816240C2209151C2820252101260B2F1A132B232E2D0E290D170018112C021F19062A0F", /* 51 */
            "2A1F2B1403272E01111E1B1A25001523211918290E090B1320051C12282D240D0422080C17072C06161D2F0A0F261002", /* 52 */
            "1F211A052517201B0C28132A022D1D01290F07091C1904141511230A032E0E00222B2F0D1226101E24062C1816080B27", /* 53 */
            "0F2A0500282F241A131C2C04251D26092D15082E1714111F0E121910180A1B072B0C1E290123270D160306200B210222", /* 54 */
            "0D1C2A0929281A111419161F18202D1D272F2301252B2C0A021B150E21100024070F052E03131E060B1217082204260C", /* 55 */
            "2D0C211C1106072C1220012E1F18040B132B2A2F240527291426190A0D230902161B100800280F1D1A15222517031E0E", /* 56 */
            "06171D0B191F2A2F29150C2C1A040D20181C28252209072B21140E02232608131E160A0527111B242E002D121001030F", /* 57 */
            "0E1D04220F1B2C2A2B06090D24282F130817031E05022E20290C27121A071C211618150B1026232D01190A1F25141100", /* 58 */
            "0D0325160A122A1910040F18051523110928260C0B13221C0E07142B2927062D1B2E1E2C202F2417021A0821011D001F", /* 59 */
            "03090C2410201F250E28080F2C19161822062E0B0104142105230A002D1C12022B131D27170D11291A1B1E2F2A261507", /* 5A */
            "050D23161D28120925141A1B0F10031E152B182D220E0C110421070A2C0B192E062A0817011C20241F2F020029272613", /* 5B */
            "07261C2B16202E102327040A021417191D1E1B010F2429130E062205031209111815001A082D252C0B280D2A212F0C1F", /* 5C */
            "130A2F0C1C2D1F1415051B10031118282E210D2A022500201D08190F04272226060E1701232B1E1A242C0929160B0712", /* 5D */
            "26172E180619090F1B28052716231A211D032C242014252F1E01022D1F132A070004080C12290E22152B100B0D0A111C", /* 5E */
            "06282B07181D2912011B0E030A20022C16170D1F002A1E05040B2D1A26232F2213101419251C1108092E0C240F211527", /* 5F */
            "1915062F011220022E072214171B2B1A08211E16292523180E091F262D282C0A051D270F042A10000D0B24031C0C1311", /* 60 */
            "2A21192E220B0D1F270C17022C0A11290713162F10242B14201A1E181B060001230415080E1D09122D0F0526031C2825", /* 61 */
            "1E18071C150C2F1024130D12002E140A200405021708112A29250B161923272C0E281F211B2D2B1A0F0926220601031D", /* 62 */
            "0B052C1B20090A2611101F032F141D02150D2D0704181A2223210E122A282B251E29241900271C16170F01132E060C08", /* 63 */
            "1906112D0C1C2A28051F1B211D0B1A2F07010A17160929242B0313221E0820100D180E0F02262725002C14042312152E", /* 64 */
            "2B091B161C0F21120B25222D2708280E23191E05260A15142A2420071F0D1811100C2E002F011D1317031A0206042C29", /* 65 */
            "2C120F0E07220811012E0C1F0D2621142F031B2A1823131C052528172B151D040B091E272D2919160600020A2420101A", /* 66 */
            "21170326111E222E0A2B2A19181C251A20230729022C0D050915162D06040F0B1F0E141027121D131B08012F00280C24", /* 67 */
            "0400280F261A230D2A0E222B0614290325011F1C1305240A0C091D152F111B1727101216212E2D19071E182C2002080B", /* 68 */
            "1804082123290C2B071E1409031017000D0F1522200A061A27121B261925280B1C01052E11162C2D0E13022A2F1F1D24", /* 69 */
            "25271713012E16190D29151E1100062309260B070C1A14220A201F1802100E080F1221282C2D2B241B051D1C04032F2A", /* 6A */
            "160723200F2B2D0802002E2C22251319041815060D011E272A21290C2F091405241B0E101226281A0B03170A1F111C1D", /* 6B */
            "24060B1F10021A2125161E2700220C09010D190F07281218262E13080320171B2F2923152C2D1D0A1C2B05112A04140E", /* 6C */
            "052F091D1F1C200A0F290124250308000C211B2A280207192C0B142E151E2313041716102B221A2D2726110E12060D18", /* 6D */
            "27110108130D240F1E151612181C141B282003040B261A1D17072B220E0621290A02102A252C05002E2F090C2D1F1923", /* 6E */
            "2409040A2825260D2912011C171423200205132A111F220F062D161D0B2E0E15211A190C030700182F1B2710081E2C2B", /* 6F */
            "291A1D180C1F04110325260228151601000722120E0617241409272A190A212B0B2C08130D2F2E20101C1B2D1E050F23", /* 70 */
            "13032F251910182A2C2B22201E211A140429110E1D000C062D0A0927282E231C1F0D02151B17050F0724162608120B01", /* 71 */
            "1413170E0C21021A2024002B1B1D1510232716260A0B1F030D0F082F2E090705182C19252A121C042D0622290128111E", /* 72 */
            "1C26122C1D22032B232D29110E04270A1F08250024170B0D012F050F281415091A200C07101E062E192A1B1302162118", /* 73 */
            "251107101C090F160A17241D2C212E2005192314131A012B28081B061E122A2D0B000E0222181F150C2F03260D272904", /* 74 */
            "172D242E261E2C1B09270F20082913222B2A1C05160325121D280115142302061F0A2F0B1121040C1A0D19100E070018", /* 75 */
            "220309002325202D1E13190401181F2A260807212C2E100A02240E1A282B060F11290B051C12162F1D1B140C170D2715", /* 76 */
            "07110E2D2E0315241006051A19202913021B2C000C0B0F010921181C161E142F23261F220D0A1728122704252A1D082B", /* 77 */
            "042E051518191D2C0E1F1C0D0027092D232F170B1A211022031120280C25011B26061407082B0A1629020F12241E132A", /* 78 */
            "252400110E22172E061228182F200516080F1D020D2D272B2A1E2603070C1901090B0A29142C1023041A151F13211B1C", /* 79 */
            "2C21272A1A0D032D2F020F280B11011F0E1C18082E19241523172220000A0410252613140C29061D1B16050712091E2B", /* 7A */
            "1A1E001C1D212C0F01142E0A172D1B10110D0B0E19052309122B272A0626082F0C0218150324071F2022291328251604", /* 7B */
            "10190D092A12012E0304251D06142C07002B2F160F1C2702080C112D051A0E200A2923181E261721221F280B241B1513", /* 7C */
            "22042012102F1C1B012C0B150F1F110A031D160C2429192E0D142B0E170809212628131E0718052A062D1A0025230227", /* 7D */
            "2A2B1E022E27030D05231713192F2D211C0A1D1F060004291108221A0E2628120915250B1016241B181407010C2C0F20", /* 7E */
            "211424282C000F171C2D2715111B181F16060D050E2F2E1A03010B071019131E0C2904091D262223202A08252B0A1202", /* 7F */
            "02071529250C0D00281A0B1B182C031123271F1E062E21240805222D0901122617132B1C140A0E162A0F1D202F101904", /* 80 */
            "1D2F22082918050B11172D13211E122A061F2826000F1A0D2E0E19272C1520142B0301020924070A040C251C1B231016", /* 81 */
            "01222F192E0E2A14121A1F2D0D07042618280F161B17091129100A2503062C240B2702132B1D200515081C1E0C212300", /* 82 */
            "1E251B131D212D160D281205062019242E0314100726221A1C0B0F1F0A17150C2C2B2A01080227180411002F090E2923", /* 83 */
            "2D222815160A09050E1B2C072E1A1F040129001923110810250F140326062B272A0C242021131712181C1E022F0D0B1D", /* 84 */
            "172E01162D2B111F2708230018060A0C22242A1A070913190E1C12151D02102F262C031E1B0B250D0F05202804291421", /* 85 */
            "2F0E1307251F2B202919180F1E272C101226211B17082A1124062E09041514050D030C011A00221D23280B021C162D0A", /* 86 */
            "050F102D29072208180B1D02270E01132B2825200A24211E172E2C0619042603152F110D12231C09141B161A0C001F2A", /* 87 */
            "1B2E180F05230801102709241E121F07022B262D0D292F161D1704031328251C110021140A0B2C22192A0C2015061A0E", /* 88 */
            "07202B0A2D211E1305041A250B012609191C150E29030D2C1B28022408110C22122E1D2F10170F16271F2306182A1400", /* 89 */
            "17160E0006150A1C2802231E2A032211071D2418200C2B252C2F26080B1B0F101905040D27131F09012E122129142D1A", /* 8A */
            "23102A111E01040A25031C2B28122E1A0819170C2F140613290F0B2D0E001620221F261D022421180727051B150D092C", /* 8B */
            "2803001012150F1F1A1C2E14230E17181E07060B0829262C2D2025020C090A13271D05242B04222F1B160111212A190D", /* 8C */
            "182925092B0026210C1217061F241A1C051416201B1901282C270D22102D2E23152A031D080B2F0F0A1E020E13041107", /* 8D */
            "0816072E1A0C272F210A09010217100B11222326282A1B0D041C1518050F0319241F291E12062C0E131D14002B25202D", /* 8E */
            "24002A10090D230F0B17021F1E041C122E2C15031A162028250501111819221D1B0A2129130C2F2D06082614070E2B27", /* 8F */
            "1B162D05152C0E091D0826192401131F2A0B0602120A1C231E07102021002F2E04290C222B1114030D18171A27280F25", /* 90 */
            "1920061F0D171103072D1B2C140915272E0E220418100F2A28022521162F001C1D1A0A130B050C12231E290126082B24", /* 91 */
            "1E0D2615222D1425232B0C2A04241201282E082C19291827200E030B0A001F162F1C13111B061A0917070F021D052110", /* 92 */
            "271B101729191218141C110621280A231E242B0420011D02130C2E0D251F082C03070F1A05152A2600092F160E222D0B", /* 93 */
            "2C1E1B241D2F26012E19210E230717271410112822030418161A0D13202A0B25290C1206150F1F05091C0A082D02002B", /* 94 */
            "0B2A26091C0C0E02112818011B2008152D0F272C1D1A2900122F2E160D2B1710211907130604230A14241F1E25220503", /* 95 */
            "1F1E2C271B1C240E2F2A070F2E0B1128021725081223190C0A2B0318162122141001051A131520091D042D0600260D29", /* 96 */
            "1D051E0B2E130826142C202502170703272A00111C0A060E122F161910290C1A15041F24282B22182D0D1B090F012321", /* 97 */
            "29092204131D2F110517060A2B1C08072C1A0D1B0324002E022D1F262014190C1E182A10160B012112230F150E252728", /* 98 */
            "012506202728021E091C172C16132E290F1D000E212A241F2D0D11121419152B2F22100A0C03070B2608041B231A1805", /* 99 */
            "20191F110D0621172E2B1E1013032305220A2C150C1B281D1C0E2D02292A070812091604140B270F242F1A0118002625", /* 9A */
            "1A18070117020D040C2E0321201D092F25161E11081C191227000A2A2B240F0E2C261B2D15132806140B22231029051F", /* 9B */
            "012D0B0411251B0E1514062C102F1C1A1E0021021612070D28262A2E1922292003241D270F1F180A051323080C09172B", /* 9C */
            "08181D0D2814011627050B060A03120913100019071E202E0F0C041A242A250E1B2F111F292202152B212D1C23262C17", /* 9D */
            "09022D0B1006252C2E162F1C0E0C180814130427032A1D0515202324222B260D011F0A1A11002817210F29071E1B1912", /* 9E */
            "08090B272B140E0313201F2110051B042624011625221E151C061D0F170A181202072F2A2C110D0C19232E29001A282D", /* 9F */
            "172F25192229132B201B1826122E0727040D08031506021601210C1D0E050F280024111E1410090B1A2C1F2D232A0A1C", /* A0 */
            "0D1408061C1A251D00030E1116071023182C172D280F15201F02262A0A12132E2B0B24221E290C1B0121272F04091905", /* A1 */
            "2812052A2F0201150708091C1B1925241721161F271D141A22042E03231E20062918110E0F2D002B13102C0D0C0B260A", /* A2 */
            "2D0D182319131E2016252F14271C102906002B1A1B0C2E071D222C0F28260124120A1504211F2A170E020B0305110908", /* A3 */
            "211B042E091C100619181F121E0E050D010716232C262D130C2A2F1D202403020F00140A291A280B17250822272B1511", /* A4 */
            "18102C2008241F1514220225292A2B01192E0C031606212D2F002813261B1112230F0A1D2707171C040B1A1E05090D0E", /* A5 */
            "16040C07022301292A1D201228170A1A241C11001B130E1E0F152108270B222F2B0D19061F26092D18102E2C03250514", /* A6 */
            "13230D2D00090A24201A140226192F15060C17291E2A272E030B08051F221612251D1B1C2B2C0F07210E180104112810", /* A7 */
            "10042E2C1F120B2A2D2124270F2F182629251B28001E151D1611060D131C190901052022021723080C0A071A14032B0E", /* A8 */
            "09152C20021410061D272205070C241C170113082A2D0004031B182B1A25121E210A16260B232F291F2E0D0F28110E19", /* A9 */
            "0B041F270710290E0306150C0A00212D2C241405090F1D0811282F011E022B13190D2A1C2612182E251A172023161B22", /* AA */
            "120F0C220E0427282518052C1D1C021000170620032E0A23011E19211509242A1A2B0B161F0D081B2F1411262907132D", /* AB */
            "1E0A2414130116122F06082E0E2915212A222007002D1A281C0F1703250409190C052726180B1F021D2C1B11102B230D", /* AC */
            "2E162001062B1A1D24042C1713262809052D141203100C1B29112107151F080A222F1C19251E0E2A0D022723000B180F", /* AD */
            "1520050C110414162F1D1E22010A272E0E2B2C1A2825060907131823260D102A1F082921171B02242D190F0B1C120003", /* AE */
            "0F1A2315162E1D0B0024140910172A252B1E0A271F26052C1B1C0E19040C08202D1328180207120311222F0D29210106", /* AF */
            "10071C1F02202A2D2C032B1B110C2629000A230E282F041601172E250F21220B14050D08271E18191213061D1A240915", /* B0 */
            "092519022B1B132E1701152F050B281E0E2329212D161024032C0D18261C271A001D08110F1F04140712202A060C220A", /* B1 */
            "11101F09051D1E0B022A272D072600060F200E2408181B2C211303152304140D1C2F1917281A0C252B01222E120A2916", /* B2 */
            "08170E0B22021314041C26201006151B2A0F2C23112818091D242B1A00030D2D012721191E2E07291F120A2F0516250C", /* B3 */
            "180205230A20011E0D062A10040B270307241A1B150E1C250C2B291219282F09082E2C1D141F1626222D00111721130F", /* B4 */
            "191A270228141712181103262A0923130C252C0A2D2E0100041D24052B0E201B070F1508221E2F1610061C0D0B1F2129", /* B5 */
            "01132A0C2607202204171503160E191C062D241B0800122F0D1F05091E0B251828231D2E1427292C1A11210A102B0F02", /* B6 */
            "2511131F2D2E000B232A280D1C0F1B2718051E221D2C2B040724102F0917120E0C14201901261602290A0608151A2103", /* B7 */
            "1C0B002520071627292C1A03212D06111D1F191009231815041B0C2E2F0A011E2613020D2A22281708120E142B24050F", /* B8 */
            "1F13032721262D0D15162329112C2B1D0A122E0B2425170E202A0F08022F041A1B0009051C18010714100C280622191E", /* B9 */
            "2D0727291222211904261805280C14032A1A0E091E162B081B1301002C231F0A150B250F0D1C06102E242F02201D1117", /* BA */
            "0E1C141E182D0813151B050701062F0B0C27090A21242922192325111700260403162E282A0D021D1F1A102C122B0F20", /* BB */
            "12240B06200A28021018270F03000D1B130C260E1C091F292E1D08231A01112B07051722152A2D2F2C161E0419212514", /* BC */
            "0F150E0D14112C042B1619182A231D291E0B2F0C0210252E1A0009130A1F2803120627262D011B21081C202422051707", /* BD */
            "091D2F0720270B211A1F0429180A22102A05152B01260F131425032C0208061C190C1B000E1223110D172E16281E242D", /* BE */
            "0F2F070819152320122D2C03041101261D0D27020C2B0A2E1B0516182213170E091E25241C0B1F292A282106141A0010", /* BF */
            "032C0C22052B29111912071E23041D1F261A020D150B131021181708282D0914062516000E240F01272F0A1B1C2E2A20", /* C0 */
            "262D14011B240720051D111221290D0A191713220C28091A15272A0B2C022F162E061E0E0300100408181C0F232B251F", /* C1 */
            "202E2D112119250E241E061305232812292A070A00181B272C22101601081703140D042B0C150F1D091F022F0B1A1C26", /* C2 */
            "25260E1D110A1C1F191A03091B0824290B2201041702231E2D2E0514181527212C0006202F280D162B0710122A130F0C", /* C3 */
            "072812191B0B231526250C2D160A1E211A02030F2A041F102B062F0E01142E0D172C091822131100291D240527201C08", /* C4 */
            "0C14080A0209052324010B102703111D1F2F1C22132129160020281918150E07062A1A260F2C252B170D042D122E1B1E", /* C5 */
            "25031E211F060E101B0C0226231C2E240116280D07131D2D190A222A2F112920080B2C14090F040015172B271A051218", /* C6 */
            "140F2D1926221E0D170206242B092A1811032F1610291C2015271301232E000E122C1A07040A0B1F05210828251D0C1B", /* C7 */
            "0C1F2413072D1A2C0110251D020B16271C1B20222912261E230818282F0D060321192B170A14002E1504050E110F2A09", /* C8 */
            "241C1101161E0C0D2E261817091D0F13031F29151B27230A2A04051A22121410000E2F0207192B082C20060B2D252128", /* C9 */
            "22001214110B1524282F2310031A1F1E2B2E131D21180A0F0825022A1B2C010526072D061716290E2709191C0D04200C", /* CA */
            "271E1A22171124201F190D05102D092B1301041C000B2C2806292A1D230F0C2E1B1808020A0321260E161225152F1407", /* CB */
            "1B0E070212251E152D10130A14222006192F2C211D1109262B17081C0B1816272924000D232A011A2E0F031F2805040C", /* CC */
            "1C170C0525062C22072710092B190B151B1814042F1E0E08131D03240F1A2111282A231F2D0212012900162E200D0A26", /* CD */
            "1F291A1D0F0714241B182F211512060411050E27020C0B102A16282C201C2E0A1E0D0017252D2223262B190108091303", /* CE */
            "0C151C1A1E2C1D0019022B0A042625211B28101623222E07110F0D09082D2013030B1F012F18060E242A120517291427", /* CF */
            "260B13232F2D171F160F212014152A1E1C0A2500270D12041A0C07081B030E2C24222E110209100105192B292806181D", /* D0 */
            "1A201C041F2B070622112F0001181528190C171B0A16050F23290B2C031027082A2412020D1E1D0E142E092D25212613", /* D1 */
            "1F19231B010815100D1D2A13172207261A0904062C24270A0B1E05250E03200C02282B2D0F2E12112F1C212916001418", /* D2 */
            "2B02191425151B051E23002E0326102D221F290A0921071C2C1601042A0D0C1D1A080F200B240E2F1211281727180613", /* D3 */
            "2C002128241D0C2315182E22040713201F29271E191B2B0D14061226080A252D0117030F0502111C2F0E161A2A10090B", /* D4 */
            "242106200E2C052E1D1419112B1E2A2603022304270917252928000C13080D1A0F2D15011F1B0A181C2F22070B161210", /* D5 */
            "040F2010190A0211250126122E2D1E0815062A2F000D292C0C2B231F070E13181727162228051B1C090B21031D141A24", /* D6 */
            "1E2C061B210B131200220F14150C1A1C0D2607112B01241D2523160A2E200502041908032927170E2A2D28101F182F09", /* D7 */
            "1C080F0C06132D0401231427282A1017001E2603050A201119022F092518071B220D0B212E2C2B1F2412151A0E1D2916", /* D8 */
            "1A15081D2D06121E2B2A29072C2F252E030902200D0B042724280119001B180F130E0A1617111C1022051F0C21232614", /* D9 */
            "2F2C1C282A210F1316011804121509240D1E032D23070005291D251727061F191011142B0E1A1B0C0A2022082E0B0226", /* DA */
            "0E2B162A1F232C151200101705131102202E1A250C2F0406080D1D1407241B1E0319012D0A1C272109220B282618290F", /* DB */
            "1C292C0A26162F010E241D1E1306122E081A10182B220D270415000702192A2017281114051F030F2325210B2D0C1B09", /* DC */
            "060A24182D10001C26280423091D1F05211E14170D2A071B270E2B2C20111A2F251302220F0C0B1516080119032E1229", /* DD */
            "2E2219101C03212C0F2F1B28062A25111600092B1F0C2D0E0207151E140526171A0418270812012923200A131D240B0D", /* DE */
            "222B0116270E0B1921262E1D1F121A231B0A081E0328172D070F0520042F2C290D250615140C2A1309001810021C2411", /* DF */
            "06112204202A0A0C1700012908191B032615272E1F212818051D0B0E1225140D2C0F2B2F0716241E1C10131A23022D09", /* E0 */
            "08220324211A19270901000D2B292A282D0C1114260B2F1B0A25230F15051C16131E1F1D062E12170407100E18202C02", /* E1 */
            "1C1E2400072504062D0E182E1220172F220B0D190227282C1B2A29090314262B05231011010C0A131F21161D15081A0F", /* E2 */
            "26182C051911232E2115010D0B252D0E1A2722061E29091C1620130F1D042A0A2B14070302100828121B1F0C242F1700", /* E3 */
            "021A1323002521101C0B08170D2B27040A280C290119181411161E0E05241D2F2A1206222015031F1B260F2C2D07092E", /* E4 */
            "29262E13281605142B0D070E08221F0C0419202F090F1006240B15231B2D1A1218001725010A21111C021D1E272A2C03", /* E5 */
            "02051606002C17190C132B1F1D11040E0D091E152A23140A2D28181C2E26291B010312201A240F2125102F2707220B08", /* E6 */
            "1D2527290908022A10222E1E200B00261C181303282B2C1A17230C2F0D040A062D1601141F0F111921050E152412071B", /* E7 */
            "01091D2D26152C061729120D1C23210A0B10282027080C022E2A1E0311191304221B0F251A2F140016240507181F2B0E", /* E8 */
            "1D181012131C1A2B061602220F1114172D0507282319272C08151F2A1E000A251B090D2F0B030E01200C2124042E2629", /* E9 */
            "272021011A112B2A0D140C15031B2F10072305161F130B0800172D06091E2E2C180F25291204191C240A1D0E02262822", /* EA */
            "052D232708290A252A19121C2E01201B0E101A262B1E28220911181F002403061304172C141D0D072F1516020B0C0F21", /* EB */
            "1B00092F1E072D2C1A2A0F060B10211D1708261C250E0D19051511040313142B0C18022829201F122E230A2427162201", /* EC */
            "23160C2620240B282D1C18130E1B2F1E012C1D09170A12002E041522072908251F102B0611020D0F19051A272114032A", /* ED */
            "072D1F172322131202251A2F1B04292B1C0524261E110306190D280C2A270A101D140F0B2C201815081600092E210E01", /* EE */
            "0421100D2E081B0B0F0305012024160E1A06020C2D271D110A25121718262F1929232A1500281E142C1F132209072B1C", /* EF */
            "2F232A270A172E192408102D000B0C20111D1428021613121E091F250E071C262B1B180621291A0F04032215052C010D", /* F0 */
            "1F0A0B2A2E1325180D281D1C2F0F152204232714290E1E21100108111B022D1705200019262C06032B0C16121A072409", /* F1 */
            "27120C172A11202E18290822132F0B1F0226281E2C0619031A160D2524012B051C151B0A001D0723092D10140F040E21", /* F2 */
            "1E0C0D082B280B2A220F13192705231224090A1F1D18022C0610070E1417041A1611031C25291B262D2101152F2E0020", /* F3 */
            "09022C2D142F110F0C07232A212725041D0E0D05060B011E1F26100008161C2912281A1B2E2B150318220A1920172413", /* F4 */
            "18282F050016150B03230701112E1C06132D1A14272221290F2C0A0812200C171E2B1D10191B0D09022A040E241F2625", /* F5 */
            "1A112C07061E272F2814160C262B20052A0F0A241C172D1F2208122913010B09042E00251B022119230D180310150E1D", /* F6 */
            "060F07250A04131827032C051E10092E24020C12112A010B171F2F16082819142B2D1D0E20210D1C15001B291A222326", /* F7 */
            "0F261716071D100C0012020820271E0528042F2C18221A290B1B110D23211524251F0E060A2E012A14031C09192B132D", /* F8 */
            "2811191B061816231F0B271009241C2500260E0D0A052E21292F2D030F082A011E02172B13141207220C20151D041A2C", /* F9 */
            "2D2C001229232B1F261A1C0E112E0D191B1D222A0F04032824070910161E06200201080B0C210A051314152F18272517", /* FA */
            "220E262F0217251C1412272008051E072415280C10001811090B1B2D290F010A032A0621132B1923162C0D2E1F1A041D", /* FB */
            "0228170D2A192E220B04081C0A102003152123061B2D1D0E142C24291109162B12262F0F1F1A27010705251E00130C18", /* FC */
            "150B18001F1D2B2F232814072427050E161211022A012508290D1C130C2C220621091B1E0F0A2D202E041A0317191026", /* FD */
            "161B2F170C031A2E260B0D1012182C08280A2A001D2B1F0E040127201E112409050614212D0715021C23290F25132219", /* FE */
            "121921042B240F1C10050811221E1D2D15072E18281A1B2A250E201F060C2C16290D0A0B2303001726132F1427010209"  /* FF */
    ));

    private static final List<String> INPUT_RND = Collections.unmodifiableList(Arrays.asList(
            "6E14D3F63EBE468D3AA3723DD31A79F7B382A924F51CAA7FA14A56606AC5BD72AE0CFAADBF12473A73F0DDA59FF13CF2", /* 00 */
            "FCF4D6BC222C6F5A40F9D30232DF0A35F378EBD2F574E5BB0245CE76B1C4E2FD5A37DB52E8942524E42318B440EC1A32", /* 01 */
            "3318F8C3B5FB4D76CC39FD3C4799D6C21D30FA287D8A0101CC72893FFC196F6E5162BE36EB3B1CE0CDE5DB1472B639DB", /* 02 */
            "DCFA5F086DDEA49434204E9122DC4DE7469E50D9C0F391F4F08552D77A8054512E519AFDA055AB73C1CD97CD9DF47623", /* 03 */
            "F4D31E5182CA0FE2B3D3F88673FBC28DE18F2A8585F1BC049F1CD22E0819E9CD01BE6B727C256CA5874DF001D500270F", /* 04 */
            "1CCCC8808930C7443F098F6E9E5EF0CE22DAB68097865AAB66A8BD9884F500D640BC9E60085416EDBB97DA60BD379270", /* 05 */
            "9D3E7B330FAA82DB7E80C1F729AC5E02337F00E1CD6976F0925F6C215945D1C97C1281AFE5DFFDF9EE68F7D34FC82DF4", /* 06 */
            "A3F210DF39B73D5C2F3CD5EBDB58D76325C5F546D75952C6B609C094D3ECCC4D66780EA43A095A341BA10DA3DB31AB08", /* 07 */
            "F23A29633E3C74FAD384DC04F08AA3DF60C8EEE59609996A4BCBF6C02E93933C41416EA9B0CAD3EF6EACE7D316C0E8E1", /* 08 */
            "AF6EED053259B03E1D086AFE197BEE5020FC231DEF159E7282746349347B41F0CF259B8BF9637B06E4D8480B2E66480F", /* 09 */
            "6EDF03C96B28BDBD7085180E111BC38310BE1E67E41064DFDE5B8B878720F301B84492C48904A3BD1FF6823A79BF1AF1", /* 0A */
            "DF794ACE997DBEC256D10BE42AA8A9BB481A7D28FD6DC8A77CDBBA5825A76DCC39717C0417C66573444E076FC008F2CE", /* 0B */
            "4471DE71BEC1E93329637B0464F5CC885793A869D82544F2075FBCA2AF6D51185CD35E664036CF314CA682B23D7ECF27", /* 0C */
            "CEA7CCD98FD884C6830D639BFE522851F78D2CADA5C9ED397D513533A7B5A3E44AA602D1B069D9BDF78C2CC84B2C186A", /* 0D */
            "CD27BC04A624E3FC3CA72A48DEDDB5A2EF9AC085EEEC5EE65618D584112C2E26162C7275186A699908FFBD36AE9ADD3E", /* 0E */
            "91934D9446AD5943A61DEC17245F497C5FF4EB9A6127DC2C172B97F6C9DA3FCCDA92C62877DACE1649B9D98EA6661516", /* 0F */
            "3AA385E972A4328EEC4C1E4961BD31BAD76C91984834521B2A0D01C93A95B2A206C8EFE3CDBFCC476212B55141F14DAE", /* 10 */
            "37A907757BCABE760AC22CC28FBB759373F2A1035C5CEAAD2E60D675F5F313A1AD8405177FEEC3FF7F9A5F8CDC336191", /* 11 */
            "CA7589A186F58CD9C0BE87C431BF179C02DFFB3CF917BAA02EA2793C0A812A2C4325A819619221E82E53A6AC2F07BD7C", /* 12 */
            "E3D60D37FFCDA44A24437C6B2AF409966F413C985A64C0B8EE87137E95938BA99DC7E128D8BB346E130BCC8D2671CD1A", /* 13 */
            "082329FCE009DC3CB0BBAD4D3626D3B2081A5BD2F7A4097465AF790F75CBE3964537DD6D6BE3051A77E5ACC8585AF7FA", /* 14 */
            "2FDCBB5F24398B095F21DFBFEB52C2A630F97DFD1F575E42F689068109CD9133E7BEF44D9763A7F9CB524547C1BC7302", /* 15 */
            "A0F37E2095D75CBC16C3845DD381716991D1335A89C5D9E5FE1E14A8CDDA4BB9B857F41E79C7A99C361C6FC96B5BCE3C", /* 16 */
            "6C83D151D17B1B02FA3D4A315ECBDECBFF04565C1F641CA145CA69FF4D562C94ED7BA3962185C1AF4F5C36E6AE4729D9", /* 17 */
            "51C28F5C9615D08F779034CD6F7BEECE8B1ECB66F0D89D7A28319CEED08A0CC2D6F7DCF752227C30218447D7796A4448", /* 18 */
            "DDB81FDE2B641DEC4C48168DF0AB16B1847414500D6FE66C54D01924CF7EE12891DA5A0C0EFE8384E3C04EAE43CA577E", /* 19 */
            "EE8A6D2DD048A0E2C30E3DEDC2AEB8F57595C4AB28E49C1C02357D820EC9E1283E58353DD7F7A1F1BB1DD6E2B58854C5", /* 1A */
            "831BEF460B26D6FC6401AA8AEFF2F553D8449E44C5829B1B3990FEE0445FB84CE720FFFDECB945D64602A74746A064BA", /* 1B */
            "7235DCFF05CCD8667DD9476020258B119FCDC69F5ACCE7A1A03F3DBDCC06FE57091583D2C4A53A37FEF75207B6D36538", /* 1C */
            "37D3530DA9B3F89B939711A98D2C9C5CB05111E4FE1AC942ADB4AE886FEF444DB82CFC38858E39C5C14149B36492005A", /* 1D */
            "0B7803700E1D1C4B5A9664F1CD3EC46BC48D98C6361560E77291BBBA62331400FBABBC88E429CDA24CB05932B4B736E2", /* 1E */
            "89A6B976685068D62178BD10AF550FDD5967A4D03EBC353925DF82BDD78986CC56CE268E504926583C6E078AE584D8B1", /* 1F */
            "5B7C58A49B5B98859E1021FC9A0CC109F02A5EA75353472219D9A3825D8579088B8E6CCF2C04AAE40C571CC31B28D978", /* 20 */
            "17924DC7C290D9A95916FB2B4FFA5FE27BD3E1764DC3C4AE18ED13316689AC28EC177B0EBC93AE940F4D47B8BDDF11DB", /* 21 */
            "A3593E48CE7AE57C69EC5B6DE62F58A6012F214146160E14E2DA13C09FB1AC60E26E51C1B01C90C3A8DB0B3B6370C202", /* 22 */
            "DA1958D7E89443B0CC88B323E38ED2D43B59C4683132B8D0C415644AA6E2BEC441539A1924FD64AF59F4D59D51A7E76E", /* 23 */
            "DB01AEF75F8C2F86F2BD250042C839A2924F493E829C67CA00A1E1D8B43BC92B26C3BDF8BE6659B4DA5C9A2CBAAF02B2", /* 24 */
            "AA47D996D532A4DD239E778530A7394FF9DA6225D45B8988C1229EDB41AA978DEECA20DA928565C43C285566885E0B8A", /* 25 */
            "5FF62FC98EC7BDB908B5D71AA455437D970FD0B4302EBB6932E5058C7B9125F1F7F4FD00CF5B98E5F43B773A9F318085", /* 26 */
            "6401C4C97CC01BF8102EB79907D14CB4F187378026B531CE7BA5C096987C1E811E94AA1332C83BCC476CA7FAFA33638F", /* 27 */
            "DB2E664579A48317C0B547B98875B52E29C1B20D3BD007089A63C0CA17D0CC921A52981B48F7EAB8181668310BA7C38B", /* 28 */
            "AA31F2FB78F18BD007C04356D287DABFB8F2B6810C062BCE442740BA48C73A8A67D0A71F5EDAD710445D8944523508DF", /* 29 */
            "B11CCF29FFAFD1642873127FBB367D811C65E39450226E1EBBD4B05815F80B4EEB7ECC402914C564AB4CAC7CF32B0EE2", /* 2A */
            "5F666964563F2A98CF7722F7DAFF31B85634B9DB32D5EB4DFD1DEA514385D7DF539AE0A6DF60BF92828817B897296EAD", /* 2B */
            "3B9CA990B5CAB9DC36F326E787EBDE205AC2FAD02616DDBF87E34C98F9F8ACEF6CEFA1B72C5D8EFB551167A6F2B859AC", /* 2C */
            "BC1050306073EF9E5797D13B84ADA32EED3DC30053139A4FD057978FF3A3FA2BA0BF061886EC42955C04D5CF874DF634", /* 2D */
            "C9547B8E3B471C2AA78498B9071F8A77C734E0AF84E012A0DD72B3B4FEF69A1BB0B22B55530CDD0DF856455419899A17", /* 2E */
            "1AD18E44E1942898BFF89C8665C7CAEA6CCCA86A637EA33ABA61805AE7548BA9CD5BCC567E8DE158048423347120FD6B", /* 2F */
            "004C1F8164E124BDBA8A3C0D3089A50FE87AD02A3F2A34E6E08E9A7302C4A6B0B73F51BDD67B669AF3DBA3E39CC02A5A", /* 30 */
            "8C973B34E20AA770658B0D12A684E609CAEF37638F66BC4B01990E4D2D31CB9DA74A5DFD67875ADF17FB2AC65C3872C2", /* 31 */
            "A68BACC2ED63721D3FBF6CC9C3652AA67C273993FB411CEBDAC203AE77ADEE3E0AEFEC6C6081EE1977DCBFD34A363023", /* 32 */
            "4724465B929E776B060AB6ED97C9A3FF5FE066A1E763867DA8F0E478FE9091FA0BB2BAB1747028D68E32E0AAC7CCD4F2", /* 33 */
            "A814F2A5C648C4389074179A480185D4A154E9709E92DD17E4CC3239306F88C538C0F593DF4925BB77ED812D5C217DB1", /* 34 */
            "922DEB1F56B167CEA0470A395DA5C9728B88BADB75BC4105BAAA8C5EE44F6B705758EC7EBCA016E5AFCB78805957FE92", /* 35 */
            "D10535B0F5EA4577912CD91E25236E7512B32001B5D45095BEDF56D0C0812AF117945FDB157A4CB6B6C2BF5A803B6027", /* 36 */
            "F81B82C6292A68DC720C480B7160E8E08ABB7797942F07BBE2EB9B4576B7588C294843D268FFE50FB4B5BBE5341BB878", /* 37 */
            "89F81347F0F8960846312C49795786797C7ECAA3F7523CB874F1B03820FA48D4CDA2963EFF954CD38DAE7A8D76100497", /* 38 */
            "E203EB8F4EC5576D3B95D9B8C85C4D09D0B8EEC28A8AB8117904F3F8356C73C6E15981E1F97F6F3879F3C8FDB72DEBB2", /* 39 */
            "781F992AE37FC84F7F186E7B8F2EB1F3D070CE88A83C1142FA83F1F09651F4C65D5346E2D6346CD60BF813E50347413D", /* 3A */
            "4E7626F424E5E29A1C3ED317B6AEC8234C8D93DB60DC91D118DA6D2801F8E946B502623289A6DB422419C8E5B0505304", /* 3B */
            "912CC9D87057286112C696B0EF5E21E1422DF510A48CC194C160DFF8D5B5846855E00DCB27BABF3445B21FEDF9ABC67D", /* 3C */
            "352D14C465AC24199DBB985AD77599D74E8B4844FE36EC85C9F07F5C4A0A92FC391690BC90D044D10CECECB686AACCA1", /* 3D */
            "654370AEF2B447DCB3AC64E30B21F48B61034F29CEE79247E2110450B048D90D645D3EA6D576B72AA0EEAC1C809640AF", /* 3E */
            "810EC18319936D0595A983C4414F0C579D4BDC2FF16FB784CC5DFC573F500FBAE6C3F1B66359B2A5F3766043D737D8BF", /* 3F */
            "3D8B4D09A86D79FD0E371BE970F6564A7FF8F3089CBDEE8E43D8D565BB03EF449F14F6550DF62F07BAF3D74D253C8B5F", /* 40 */
            "985A596F99D88A5A4521379F019BAEB2CCAB8F2DC3198213980DB0CBDD6BFD31C01EFC604EF6EF26674E38604CCBAD13", /* 41 */
            "A4F4C3C61A30CF61C1A280C70144CF45BFDCA61AB38A6E714A141DDEE37F51201F4EEA47370AF057E823CDCB67B5D538", /* 42 */
            "FE6807CEB3C2794523659DCF7957D9C63A220B30704687B20884DAD4EE531DDAEABAEFAE552AD019EE9DD263EB57FE8E", /* 43 */
            "E86F5A70872B433E63721911EF4CF403F49FCACB1BDCFA21A93D2454CCE48C20D0EB39F668447706929F3FC9067F8595", /* 44 */
            "A1638DC3D5DA72F3420AAD9D211FE2BF38AFFB41FE81CF7BABB324BA3467150B5048F1818AB774FEF70B79B919461C42", /* 45 */
            "13FB2856EC6AD213E2493E5D8511530C25D26AA5E40DABBEB64555E1A73863420A1B6543B314FFAEF47B3D5BEBF0B7B6", /* 46 */
            "040F2CDBC9D03F027E6B37F3DCF4649AB5823BFA9742E0F2F0EE26A7A7FD9EBA504F953FCE399415D059D50A3CDBC112", /* 47 */
            "2F017CA084DEDCC90A8B063A362EFF6979F68CFA4D8B2DDC62D3E448796C49F3A35BC104092183F06AFF76E338E4E805", /* 48 */
            "2CA856DC215B02AD83780E15CD20898E7A48D67893B3F38944037B8A549773B6C84BC0962C8346A32F5CC5D7E99A45A2", /* 49 */
            "08C9923A471E76C4A22504F9FD2F37FA3150647AA343E8722237CB0474350A49B2B8663D15CF3AB70E360F6666C15B5B", /* 4A */
            "C3DA838E3345D57DF0F040675685E160EACE759EC5B5E7A6B5CE59EBB0C0AEE5A91D1A363B2AD4C3C5F1216548A706E7", /* 4B */
            "D470E0B0D54281430CC5CDF0A4C1DDAF4AB108A296B57BAB2E0444CD623AF1396A859F71109DB72A26BAB6E2021B00F8", /* 4C */
            "51C268D7B13FCA699DCD7255B97225DC070F6A84FAF8F56019BB6D3E8F0738DAEEBA73448CB15521502A0C85CDDA8B18", /* 4D */
            "572881BB3CE90518969E8E91084143694D1CD9D19408F5CC4A831F2953FFB112105ADB483D53F6B7F57C3D93A579D338", /* 4E */
            "EA4B0DA77EE7D42A3020CAFBD89829F46728CFB26CDABD4E725DB7D4D3E1781E22289A26649D00871CEB997256CAD7D9", /* 4F */
            "237BF7118094EA7B109C4A0E1BFBE027E723C10B32CEB2E9334BFF724650E94913F0269C74347C60D1E36044953CECE8", /* 50 */
            "2F3C70F7B5CE96E65B16E36A374842A1894186EA422B3BEE6D3A045FFC2F608FAA5E055EA0C3B1C7B4C6B16A79FD87D8", /* 51 */
            "F6EC7A3ABD01A2415819688B1C37C6A475A050BE62846521EE3440A1290654E98C5E8BD632E537A1404CAEF05D1001C6", /* 52 */
            "6A70DFAEC8A701DE711FAF2575FC7F221A486B45BD3B929DA1C326B238FE1B3A880972D4863FE14039772521BC13EA47", /* 53 */
            "6676161A8B123B9F5F91DBFB325087248353699E670535AFC2C04507DD21B1AEA54FFC0BDAE2A64E6106D1E1296B1AE0", /* 54 */
            "ADAAA8B72A8533B6C64E8620659E674574AC18ACCF166BF4970BAADD594D62599069A71B6A8117F46BDF8C543FACC754", /* 55 */
            "60FDEF62D399486795D08F0E8C172DABE253CCD76069D0F12AEDACA476F97FE72CBF0553A673AA628414C9EA1D656817", /* 56 */
            "D1E1F89F8B7F95CF0E038C2D4AD2F8D3791694ADB435CB0D35B6A9DC67EB279532FA5D18DF1A424B24B9790B965D40F5", /* 57 */
            "6FAF852D173003CD3C1B2600D9599542A5972C93CADDA0F4DD91428C172871CE78DBF69A5D4F752B94B98BA70F2CA806", /* 58 */
            "B50262776815CDB23D389A3B7A7CC0D29DAA3683FB70FC8CECBD15562E2D1CA8B9CAA23901EEDE3BD91689A3B5BD8223", /* 59 */
            "7C19B687504C55E9A64C802078E7F291467FE9594A40BFEB520AC4F391EC58A81019710F6054D1E5873566E16FE28B6A", /* 5A */
            "03601F9FFE871EB27F472A8C54C9AF6DEB63DE3B6D2F50EC91F152560AD3E4F4B5C4BD656C9AA25534DD692D0736AE5D", /* 5B */
            "E3E8C28E6A3B2308F8445655FB54F69733E63EB06A5342ECEA1C0FB284BEF907034CEEC822375DC15112C26CED99AA5F", /* 5C */
            "FCED5B56FAFB13184D247CFAE516851E5D7D4E30CB7C45F4963C95B48939CF4F74E71E061058FD850364C1B77FBE0AC6", /* 5D */
            "B7B3B7E5408DDB178F18579A0ED2F44311BD3C53ABA9BB02B7B38C0A4EA19B29739C64A1FB77FA1EE39BE915F0DBEBAE", /* 5E */
            "41D7382B73FD839AD725FE869027AAF4BAE4A9FE9F013E96E16C25BA42610CEE3563844227411B1587C0B3C60407EAF9", /* 5F */
            "FEC6FABCDCF44E9A440E8521E66D05E16FA35B6DCB68CCBFC55D9D62CD9564CF3998EAE1572C98DBDDD94D28835326E6", /* 60 */
            "1B4637D503E70B4EE5066ABE4B4994347405266E7A309C5A53B4D9E47A4228C036E0C1AFC9F6A43D2500BBA7CCE6333B", /* 61 */
            "12C426EDC33C20FC185516E3FF88C81ABB0AEC4A8DE3923C639C52EF60304051A39FA1F037767C4E15876AC638F98678", /* 62 */
            "747F6F4CB27933B6435E60821AE020789DA234FF0D4A191FA2466641A3484705FCC7202C3692963A570B7E792A3526D0", /* 63 */
            "B4A56F6B8806C73BF36015C3D06238F4E89B6C48CC0233A5876678168876BAB42D633BEEA7D422CC5814802C6CCA78F0", /* 64 */
            "9E7C761F7B958897CE7397CB4E861DEB2FD41074816C3D68DD74656C5CF7FB96F72D5EBFFAE344C5F90C297E87BFB706", /* 65 */
            "8F5CC81D0782AD68E4F1BEAA2DB46129636AA0A315A0EAD56CE47E756A3EF71BB4E16A2D9640E0507B2C197E4307FB14", /* 66 */
            "7C3A5DF94FBDB1A5C0E6A9F05618FAC16D33D97260093EDCB81C4F1CA5635318F3C6F4F24FC84467436D8B37653E4F92", /* 67 */
            "4FDBCCBF6F182DC4130EF59352F920EC67515443C62C3BE330CDFCBCE648D979D8A8FCB263BEA29FD5C6EDCDEC9E3E57", /* 68 */
            "9E38BD362A33B0FCC9BA098B6E74491F6D81373758BF473C54E52695D2F9C1269BF68A7382A82BA21AD1C040E2BAAF1F", /* 69 */
            "8C766B4A19048302121E25A17FBB18635F28A85E8833D9487E276E3622CACE4C04E90F3DBC2486DB66A5F0CA51A1827F", /* 6A */
            "C7A595899BFD1743CCA656AD0250D360FDC8B062C0A37417E29AC654019D2AEFAA9CD96987DB1EC0EEBED7147D7EBD4A", /* 6B */
            "E91B51FFA0F2F2BF8B64AA96C554437622F0FC719582293A54EE4D21F65A42F7CB71B4E1599DAD3475953046F35F4BA3", /* 6C */
            "6088EB3856991219849686E95B602930BE985A7416DCF8DB566E2B00B6A4CCDFC04A96F8BC9270CDBE0FD0DE76F03C76", /* 6D */
            "1483D43BCF31446A8C566589707C996BC8E78F094D6A4948F7B2F7FBEE3ED02D78C005FD912DF7FE1B9D6DEA08099AA4", /* 6E */
            "BAA96D53778A36A28C1E7E545D8829A50652F62FD50490654D0BBAA2E6A4299273CA1A643A1F3E826FC9C1515B447F0D", /* 6F */
            "082FD0D93931430FF5AB497D10C677793B1757277E948D6D4174DB9BC4BA1221EF4D14301D3535FFB9A5DC2BA79D63EF", /* 70 */
            "C7AC9717F876B7D3F6E478E564ADDC97AAC08ED71011AF89B4BDE72A764E196809B944FDCA05390B6D1080F6736041AC", /* 71 */
            "FF7FD02FF8A964FFDF424A063E6B091EA7801FF253722CE7DC7E46BA4B15A96F3C2FC583699CF25D18B26943177200EA", /* 72 */
            "B815D7D79C811FEE1535EB7607878D7B299694027E6BDC54035AAD7CB3983EEBAA94ED3A49EF1196756AE6F2D54DD59B", /* 73 */
            "6DD9FAFEA8C3ADD679E1DEC4DBC5237DDCFA31B52D2C36CCD3A937011C366AFD5965071CFA5D20C8C114DFA5EADBC3EB", /* 74 */
            "5E9F7D42DD5D10DE13B2A30A847F2909FCD38F5D35DDC8360E2B83BCE2A50DDBF786AF58B63291909A5AD1B14A4FA06D", /* 75 */
            "FE9EC09D46A74BA5E5E50D5AD4D75E513CD8A7B49DB76D2F76A6C44606D82EA17C34955E4B441260363A91A41665935C", /* 76 */
            "1A387BAA52F7CD831463BF65E13B61CBB4987C658FFE114BD31791490ABAB69D896A48A59B37A7918305FE9C9D5661FF", /* 77 */
            "16D8C340D1DF25A465554EB9AA5B1C5B4D4F619567781E325C8EE75C49489C7A0C01DFBE85917CF9EEE774CAE51B90F2", /* 78 */
            "5752DAADD31B8D045E7772C6078394317C4AAF00681F6E7044DE8315F9EDA1575BF95DE032FA1CDE649D32803243D399", /* 79 */
            "BEEC9D37E2C3BF51F21CF665F10CC7D13759407E938C064F3922F05ADFF4BB2C067BA9DC12046FB0DC1A6378D85E75E1", /* 7A */
            "99CE6651A92D8B9233A32E649354DC96C8B1E4DB90EF2865983731A3BB21018E0B78EEC1B32C74FB8C7BAD8446EC9821", /* 7B */
            "13BE20E9B3F996A69CEA6FA702FFACF6229AACB07A4A51350DFAD9EF1BC9DE6B4866186CB26FF171B0B65483FD7B2757", /* 7C */
            "B822C1DF0DD772505367DE13076D8929B24F3E427DC45136D5EC3D2D3BA11AF149DA4E0554F3E68F36DDD48608EF4FDE", /* 7D */
            "474140A4DD9C2E2E8CD9A6090B716FBC021119E7A37E467951230D2D9D016F91A77E120678A135108BCE4CB42C0EC8F3", /* 7E */
            "8A6134E21E3CA9456058CE416698C8DD51CF5C54338EA42AAFA38229FD08EAC3EB9CAF4645CA35F2D570010FB5CCD66E", /* 7F */
            "9F38FCABD0E4C940001F11FF766E8780E05144EC8CD4A11BC60B2FCE3F3A449FD4E52A635C1FE3C0B01CEC46DCDCCF7B", /* 80 */
            "A91B652F7E119F0D9A18CD9BE46B0B045FD11EE0893BC6C3EB9A68EDFA259013AA21B176266489F4D91E39198F4D638F", /* 81 */
            "9B127153A23FF92CA468EA60514AE8BF5C03B35512C8582C244D03C0524697693F89610CFEE9DA48C03E3A878DC37456", /* 82 */
            "5CD061A18365FD7BABC06B3FA933C5D1EAF3E19FE3A34D90B7A3ACC77B3BC11E55913EF33CF981053E9041B2308F7E65", /* 83 */
            "2A5C4E653E3A38B4CBC080CE9A5BD35F3358FA5DDCA2ECF2A64DB293C9F743CE9DE5B188479FBD363653637E4753CD11", /* 84 */
            "513867EE400B803942F04FA7C210E4547D707D0CF0788C833A137439BF7C098DC1C99B2DE16F8ED67125AAF5CC6365ED", /* 85 */
            "55457A44AA504B55ECA291108D66C4074BBE186DA7522A9580B73D0EB0B97DB8B9093229292512AB2FF183A8A2442A40", /* 86 */
            "D97A2321309F005E5EAB892507044AAF12DC06BBF5978A1B0E55A2DFE1A195CBF1D6096E695CB8EA0FF161F6F4714DF9", /* 87 */
            "94AEAC962A2E51F6103E5A2F195261C21D8CA4235D46BE2904B844466CB5514D85BB8CF5B683C4603F3979D53B790987", /* 88 */
            "FA0B45DA966AC2FB8BE0D401430E1AC112115236F9168AF264A0EC413A541FF5A698CEDC5E87C88638284B51A33689C0", /* 89 */
            "EC3CF18E70FBAF1911EE3CE821B6649CEAB2DEA73D5BE249934204A82ECF180A19657EA33613598878A78C1A4B112563", /* 8A */
            "F12C48C7604719C0D69E261F1E83CB3C16ACBBA6D45208DF278A9CB0E5D96389E6D8AD9B5C72D25557F59428C7962F23", /* 8B */
            "848475582D43F305098F26C04A312363370EC34F1BE0DADC557B7B076E6C2B990F129C01817FE1C3B6A9D8282B3DE732", /* 8C */
            "D6A131D6AB98261D899209564E7104B88001411382A084157E40F48385AB99B6721CF463BD040962C97932E10FE6FF88", /* 8D */
            "FC8D5ED5DD2D85892F5BBC08AE55DCA70F85E76A270B785E53376FDA4639AB02BF7DE01ADA5E02DC3CE16A48E8013609", /* 8E */
            "C74B87543E7130E8477FCCA4B80A0177031D4E8644D9D50242DC926668CAA413CA2D9B5AE4EA5ED6CE5FE69DFDF11F1D", /* 8F */
            "0D6F3D6B16D714B9845448D2A0CB277BCBBFA86F507E06BD8E9472648A2DCD882F8191AFC9B805F7E3CC637E4B19FEE3", /* 90 */
            "512E72F9CC349AFF99233DF390B778FBBDF0F53189E62EEBF25EBE4B6E0B7F56F96B0F203FEA152964CAC3FBFF7F8C14", /* 91 */
            "5B878F8D731830DCABC08A1A7DE5555F7D16FBF3ED64F8A688B1647C5E21615DF7F9A899866CAEAED4E92D6E6332D19F", /* 92 */
            "90A522499D09AF0E51A15478B7F97FED1C4B7E711B7A12CC843D326602B73DC459E7BAFDA58C00F0DC1DE6B930D99100", /* 93 */
            "5E3BB0F9A4B57CA5BC255FE7D246CEB7082AD3B5BCD1E9C14D40304A4051272384CCF7B0FE2881F5B6361449AD285BF8", /* 94 */
            "67E8C33A3BA819262ED590CBD65BD189357860BCBD50743B4FF10B29FA2C88978B632DE0635E0353AC728931A0C4E15D", /* 95 */
            "A74328F0AB7B246B9EE3D7D3BD7218B995BF28F7F48DB8A7807BDCCFC554BB4001476B123636BFFA80E2F419B2CC08B0", /* 96 */
            "D320C56D1F1102C2666ADD96EA6378DDD117AB72B7BED9CA50ADE4DB44F442608E41495513760718BAB8E718B71034BD", /* 97 */
            "D266BE8011ADB628B53DE7CC24CA4E3FE3480EBA866F67E434025E9CE77A234E7ABF9A00BC4A1218ECC46DCB0D6AFF53", /* 98 */
            "B33447DA0CE43B669072F28570241644B62B6A2D1016B0671F91EAE039398D01DECFEFB9F0F0C9EAAB7B0E65820F0671", /* 99 */
            "950BD77A87A87F663AFDF093BC4E6DC10541FC8E2D2ED9F7D1E020F0CF7A64C904A8A412E7D079E0312F24EB13EF11B5", /* 9A */
            "295E8F0DD9655EC1C7BDACAE3877E7E75313101DE97EF95FB976FA41D7F91F0FAADA477228B4A45EC785C569EC57282A", /* 9B */
            "0772EA5F66699F263AB1A2E0688FF2C16EE7AA1A689B0E80624F87B04DAC3A2DBDCA62E6EECE06D9AF4D702BBF7E139E", /* 9C */
            "76D6A54B02D13633CE92B55309886C69C6132EAA11F60EF3B17372369F32D12B40E370E4C7C5C54D8D5D7C992C5C5E7B", /* 9D */
            "C1EC121661A25B5CE2FFD23586685A622F4A0FD108A33B989340FA358BDF8958E285AD12298C00D1A522C9978AF5DD93", /* 9E */
            "374BF8DF88BB693D3CA678B10C7D67A3EF883C2A4C471B0DE0B5717B779A1F5B8CDD1A3370A38C773E2D0259D49CF160", /* 9F */
            "A39C48D001B27DE32F0457ACA30A8829C541DEFFD1D717F358B3417D427A9122433E8AE71E5DF4A3F34066514FA62C4C", /* A0 */
            "2A1D071FD5BEDB45FC140EC0B524C6F35304CB9E39E532494D19F9DA26B845B0073EE7875FBBFEB79B40BF304952ECD0", /* A1 */
            "DEEC243097DDDBC1AEFEED29DD754AD6CBE6972570338C32710B52B370818E3689C872F8EAB252EAEDB8E650646ACF75", /* A2 */
            "68279A4965A70FC69B085391B44BC29484A2E7D81DD53CC0FF14F940C55E0346B6D416B6485920FCDF5A89D9CDADA7CD", /* A3 */
            "65FFF7F2E6CF3F6D26C9385E081D9545A96177271AF9D38D8CD53845D772ED6EF764BB299B5AF407F50780EC7BD82F91", /* A4 */
            "8D41949FC84CA792D73D6E6CF47FC625A04DAE115203E638922CA22A436D9F358690F4E0DE7C2FC2F4C22F8E4F6B0E3B", /* A5 */
            "E58795EDBB13C9ACE41E59D6919E664450A10F2AED930545C6B29607A4C0157E55A203C05978C67AE468750D7A7BBCB9", /* A6 */
            "C753DFF5B09DBD728E18ED28C658BCA3D7FEE82CA58BD6578125251B780437369B3A4AFD6A5D1445173844BB34B11411", /* A7 */
            "8CFBF91C80D6118DD66A759BAC77D75FBED75E4DEEBF815C452BFC0B2D70108115DA6A5E5084389A3D5CCCF4C2B7E117", /* A8 */
            "7DAE0AD8F2D0158E7536F16350BD84FAE482B184847FDE581CF085C5E69E69F748BE157B28AB22800833CB4625963460", /* A9 */
            "8874F6C400FDA1778AF38E9D49F0373ABEB622F844F77299AC655AE4E3D2DC9B408F4D37CA0EA602906DEFF2F1B05800", /* AA */
            "F3B7F6B51B38C9A4B16D852D6AFA41ACD420073A96246253AA428CF4D26B91CACA09501AB9C14BF6948E6B0E583EE3D3", /* AB */
            "49491C258EA847E3B4261C80629FD4BA1FF8C5CD87EAE3E31126A673669C841CCA5BF2D6400D9953CABE6E5BA95017FA", /* AC */
            "0D9B3A9C8100E34A0D76CB44D02B62F7B5C8309F085A880B9C3517DCFFB1FA179F3BEFE01DDB34D21FC1BCB820974AC3", /* AD */
            "F0831E56FE3761306FBEE7B0561850EE48DC19160644B6830243D3CC9604C6DCBE817FE53E6C636E837C8E88781F3E40", /* AE */
            "5B477C40C4EC0D47AECA1C3A46554E6346BD42EC0A245EBD294906A9EFA60BE00AC51F390914B449C540CB4CCF1BF459", /* AF */
            "70ED1277BFF9323FF86A37B3B19BDA0C65A06C14E5FE52CBA12A00E84074B0B869C4E6EBD34044D7BE7C074FACA31EA6", /* B0 */
            "11F45876E8EA4CF7C2C358809DF6020E320C921C08E8DF79E20C229E0B947274EBE390E9E677B72E09B506E563A01D68", /* B1 */
            "84FB6F54B4D93B68817EF6E454BCC20A80EE99BF2A00ECCA1906829CA6F07080D79874C7146CF96ED671CCA6EB851951", /* B2 */
            "D08C750B8AB9D9AC6BD8459760667EDED5B59D70CB87915C744BEAA77063B09AFA2BA6530747A4C8200CF47CBB28A43D", /* B3 */
            "5F2E5A664EED854B8CE18412A69E08AEC126457A931906AAC0189173FD5CA4B21DBC1DFB9636AB7B220859A31556D588", /* B4 */
            "253562023883F8BF96EA05BD710F2FE0FF845AB4574DE32A70D857F50FA9E82075EE32FFA54A837861BF09065316A02C", /* B5 */
            "0B50208654627B7DE2E5B8DED47CE738171F47096352E16A23388CDFD0C6DF3328750FE762A93B6292DA26A92CD7CBAA", /* B6 */
            "13EC5FF295839297E6B4E4FD84DDC6C84BAA215B8ACF50EAE6B600FBA9E8D5903456B845EABDB79D33802868E4751870", /* B7 */
            "7AB0B9F0C942BD5441F0B92F577AA4DB3ECDD602A3779B905C3CCCD2E05A4E77B1FA04BA94E5E20CC48237E15558EEB0", /* B8 */
            "9C840203B0C89D3CF1A4F745A65FA83859E3A791EC75449AC90F90F31A16E969A068025E4A6CB8D19B10D2666360570C", /* B9 */
            "35AAAD4602165587A7FABD32841968501A8E1F7E843C1D7A8EFEF70E14ACDA23505A7B9C4AD2F2C74F3C456BFE11D8F3", /* BA */
            "0B8DA1538E52965AF941325C422C5E42E4485FD1E3ABB8270F49BB8E8197EA4815533A865E78AEABDDA02BADD730FAEF", /* BB */
            "484378F720509DA971A1927424D8DC7C961CEAAF5DA00E42AD0F6FCC349EC2B3F9158B51D367F93EF4ADA268252F26E8", /* BC */
            "D81617992D5E39938B77ECFFD068182C770307F1611539D2EE6512206CA0DA2C002AEA8DCF3EBF28BA45A7217C5CD526", /* BD */
            "8A95A81CDB0748F909E955AE891272FE883FBE4E1710418E027BD2236192D56BC2DDAAF8362B99BC8C898FC8C7BFEB3F", /* BE */
            "25140AB00678122F582BA660BEDC4B3E2E832BA548B0066A72DF43CCA093C382344B4FFB28A0E20633A8790FEC6487F9", /* BF */
            "D735B05806FAFBB7C4C3C88D71FC93C10B823ECCD7AB742459E2FCBF76C4D92D082BEF38FD0A6A0CCDBCE050B8A6A190", /* C0 */
            "A5EC9351D916698136FC738580FA0AFD9148A69E461E5593342678FEE0149E9468E413287050C81DD33E1D4098A64ACD", /* C1 */
            "69CED3D4C38A80A90C36F1E175F3604A2D2C98BFB268478EC4FC72B536A479D6ECE3ED3AEDF629A0C9F405D30F2EBB03", /* C2 */
            "4A7ADF8D554E039B53328EA523DC56BB52DE23BFA8D6724F2E4993D7DC44E4E3269CF56F8E9EAE834CC4324ED7E3C226", /* C3 */
            "FDC29BE0B2F022906914D3144A096F721EFBA0F5F8E71AC65C239020367455CF36CA8B2CC05E109463080F8D80823CDD", /* C4 */
            "A5EADAB6951792CBDA3F3F125E1487A5AFBDDC0B1FB3EC38B2FCF3FB6F2239247342FE2C14690342FB2CE563C61CEDF2", /* C5 */
            "2AA2E031C65D6A59B0DA89D1B532D14299059D9CD19623B3041772F6FA9CF49F7108F05408A4C5B00AB665960D3078A6", /* C6 */
            "E33DBDBE30DF98D30102D840FA101728A794B51EAA604D96787519D2B76D47454DE08CC1919C1DA0D2BC952087661098", /* C7 */
            "303521554C82473B39F38E8FF374B9797A25BFC8CB43FF4E68114CAEA47CC722AFDF0DD1FAC0A9CBA8C4DDE5D0077D69", /* C8 */
            "1F3BE3064AF3E4F506EF367DDDFBC5C045A24BE4471DE5DC0146458DCCC38BDC5335ED0B6C022D295CB6FF4BCA8E302A", /* C9 */
            "51DDAF63D3339689DD84588B2B6849CF22A07FD185FB80669C8DBDCD174BF0A315438AEC1B812C75FF7313DD3C2115F4", /* CA */
            "ADEA9FB2C68F9BC8134C9A00009038E9283FD752DC9C750A98C74371DA822308CA2BADFD5E788B361B54E7116B6312CE", /* CB */
            "4A1F08C276C5A9B062AD7B288257BCFC678CAA4FBF90FBF7F987C2C9D30DF669F0269F1DD2A27A56FC62ED9878B838E1", /* CC */
            "41E8A58109CBC0BBC30E7AC0CCD3D1B7ADBB44BD605DBCDD56F38F825DC30CCC6357CD3014BE98A8FF71CED275B0DC9C", /* CD */
            "AE64E6731A0EFE0A92F23FD0DD8397736F1745A6DE1B4DAEDE43EF2E4E536CA27C689A477DA394955A4144BF62BD0E95", /* CE */
            "8A08005131F6DEE49748871E1A7BEAF268B44284FBD8EEDA34BA6EAA8B07A94851D9E8C43602E06DD49D2C5DDB293B0E", /* CF */
            "8B42FBA41797020C70D1F84824119F5E50FE886D27C8DA41AD4D1A2CBADA537FBBE4982258FA6F308769053DE3EEDFF9", /* D0 */
            "002EFC51F14ACAF674A8840AE2E764235BE4739C6193A4C8A1DC2A756C98CB6891D93772A49F210B141BA0D8BD4E5455", /* D1 */
            "1920DFB69E8CB089E3D26F9108672F6060B4EAC599721E391CCB929C506C41EA81CC08FEB604F619C1C8D5DF6CBCC455", /* D2 */
            "35C7CC746DD7DC6772689E5AD86222AD02503B1D3C65996665228A8BC80D5DEC933A5F4C433795D7DE7EA854DFB0D1D3", /* D3 */
            "BC6A7D3AF01A173011EC576F1EA72503339F8EE6AAEF3C5096C9AF5BE5E7FE9FEE2C0B291DF454BC0726EB46CFF73BB3", /* D4 */
            "C231A219FDDE5180993FFD58D5BEC51D1AC58283085C702B39D778A738BC210A0B99AC2DCBB2D10F42A898BEB0F18727", /* D5 */
            "BB7893911FBF97FE7D17E56221F958E4F5942F4F9F3FB19980E83442FADC3CD98FF4E4233B958BA91EABE203B0FF9663", /* D6 */
            "934EEB3CD6B4BE2E86003806F90F6B04EDB9F66C686A9E8A46DC85A0484B1832E7A5F6FC61226C7C0BE00E959D165C00", /* D7 */
            "4F15247DA89D059A6AD0662889380A5E9A61C27640D65FA0B314237563258ED6A29BD592F80D71A3ECCA6D5CDF84A1CA", /* D8 */
            "C3E29AF0E71C2A92B81EF1381A92E07B4DA223B9383F7A21574666F3B240A55A6C4F6AEFC422F2AFBB6E1D846A7103DF", /* D9 */
            "F6A5710B6912EB36AACF4ABB8A2AA3A44C9E5339EC1B6BEE770982B9411401C9403FF56A7260D41AF8594AAAD6252394", /* DA */
            "0442FE26985CEDE85B6684C964CED824D4844B78F9BCF0598CDC000D4515D77FC6B2D103728256E90E9F407BDB94DF6E", /* DB */
            "6E1F8502C52D52E4BE900F0DCF53BE6467BA6525885AC9B34A8D0012222A136D0B6FC365D597012F5896BEFFC270F213", /* DC */
            "246AB20FF67F89E848362372F2D297E0E8D56E7C8E85B03A2D4255749D1B36FECDC346150D9984F1EB12626A103BDF66", /* DD */
            "B924E6867B7F090D28D839F52EA97CD9C22ED10B42532437529D76B451AFBDA111758144982EB300825473BC10981E28", /* DE */
            "F186B05CE29E640C74C0B7D7B67AF83C4E8FB62E2C0271BD1C9666A159EE1BB4EC4FE9CCCD5D3974F596D04CAF333859", /* DF */
            "9CAF237E57FEF048A3F90C0A5B1D87A71B8ADC2B59FCC26CCD314163188205E9C56DF11BDF425530702893F98864F438", /* E0 */
            "DD03F9DA326137A580C98E7DBF8D2C2B663113CA082640EF51AAD81580064097C7A87BF473DDD28859499CC2AC13C684", /* E1 */
            "DE95385440BC15B77D731DDC516F125366F5456133612DEEBA4BE6A8A1CC286FD196FD813D8CBF1B9AE271DF9120690A", /* E2 */
            "1D13DE6C3B642929E2EB609A73EEB7037F59C4DC0E8034DB7F3EF22A01AA9223C25A5902FCB225FF4147BD81BCBD5EA0", /* E3 */
            "A879593CC2E1F2467A06726838112E436C69C1D8EFF18BD896B56BD1688677B3323D1BCD0B9567534A6A0D50ECFF7CDB", /* E4 */
            "81BD33018085FA8D7A4FE171E2CD63C2E490D83D56CA2A61FD828EB3C4045630A0A25B91CBA54223F88D1001D4C0D85B", /* E5 */
            "BDC6857395AF5F341ECFEB99D68A43742313DC1685848BB203A9D3828D4E9D7AEB4A3D264C9ABC732940587A2A2E46C3", /* E6 */
            "DB0F0E549FA73BBE7A2C4D7FAF68DB915490CB9151EACD09109F2BD78EE122AF8526C3BAFE66FB1C7F520E68E543361E", /* E7 */
            "75CE7A8E92B0A8E5AFBBA4285B6280EF046525D22E776908A048ABEE642D156949BFBC9436DC92B26329698050DA74B0", /* E8 */
            "F29D6D3EF3861A64A530E3977C72FE02592B7DA60D8F255EA6CAC3CAB17D723D115874195F579886D18825B53C86CADC", /* E9 */
            "3CA1D3B1D4C58D549500A7560D4B4B9297D75A2B24938164B4AFEE799B8478397C510CF737FD2A8B4CFD8FCC489D1384", /* EA */
            "56592B35AED0F35D333A5A2C0EC32850A86AC3370E4E3A6664C31CC61AB2A23062221927F9CCB12DBE9D01119C9BB817", /* EB */
            "A3C148A4F198205BC75AE8CCA9C1DC9C89CE4A67DC94F244131E19FF5902FFE080DF9E27D6AB56DA08B5294471D5E90A", /* EC */
            "E9EBE45F23CF7F2E40ECFB3F4551C25C7F52FF9C8D6C28C7518C69ADD8B68A1C68F0042B004540AA4C2982FE6C7A599E", /* ED */
            "7B8F0461FF408E018834B5B2A14A977D3A6DA37BA58DDB7D52A93238E092A9078192CF830287E7159C7925ED4356FA69", /* EE */
            "96E4F0948C0E7545E71F91F64E6D3E826AC152AB10626D5B61A2A8A80A3D37DA11BA9AD6FEA81AD05682ABF6B11ED7F6", /* EF */
            "88BB59F000CB57FF9C1EC9DA89082FF68BE380BF3D5C574FE175F9AD13CE59666D704B6938FF14110C7D1712C588FBFE", /* F0 */
            "DDB10B1726417D668F498B1159F0D227D5DEF1ED6A7AEEE34D42DC52849FBD6F478B787ACA48F6F8DF3DB7CF9D3111A6", /* F1 */
            "9EA4F81015360498625BFAA2B0D3155EE7B6745C7630B2B6C57CB2B778A6DB7C272A325BC45E2A79A9487E37D59E307C", /* F2 */
            "78DA8EF664F8064F8E4C5817D58EF25C99163F9913030EA7328ACCBCC61D3B4DDD0E50CB3347775A94B5DC3F70E007F3", /* F3 */
            "8A99C207FFA2E70CADBD2A530973481326AED19D2CA2EE511DA3A4EE94AADEFCF1DB4BD8266928313B40576B8264DEFA", /* F4 */
            "B170C0778920996BBE246EE0AE69518C162B985F88F03A6A06890E8855946EE71CD8DFAE6079CEF2D166DCE600E403D2", /* F5 */
            "2AC313BB4532F0238D1F9B9495608A43235130F49D28464A0CEDE09C7582CA16851237FBF2A85BFDF0EFDFC7564D2C49", /* F6 */
            "D857DFDF47F00A14BE8A9E37AE0B311FA10A6BE264B747885DA1C498504A65E8D7B26933DE045C48CE1C012608B80EAE", /* F7 */
            "AA27ED44965202C2ABA56DB9F48BD40CDFA0F85CF2F58450BE500373BCC7D63FEC328C7976B1A88DDD73941312649BA5", /* F8 */
            "2814CF99CA31D79A4CCAE818A60199F85E4E039DB54FF01A56B814FE2038A1FA70A1AFFDC2CF4B5334C02DAE0EF535B8", /* F9 */
            "3A601C12A28E19852AB6EB653080310AAF099A089F2852CB12242A85EE640F0079E3664DCE2E838953BB1BDA000A9896", /* FA */
            "0789C427216B0A2A6BF6A8CC28E2182884B2B392544ACC9302DBDFBB34A42133A6FBEEA5098FAE41C6EB876F23B88E52", /* FB */
            "7D9EED97783A459D1419C09655248A2F96E4BEAB7AC3090AA53D280BFBAB7AAD87B0C9C0202570C7BB3084D685923F24", /* FC */
            "898C68BF0F0D6BD674720B48F15CB0EC41690B985EB014B776C0FD02748CB9EB384EF5D798C1A989B59102DFF8EAFDAA", /* FD */
            "C0A778CF0C5AA154FE60FF7A8E6FD0F63D32B251CDAE2A775F5882E8E6C02EF7A3CE20B8887B698B180DEE584F5B3E0D", /* FE */
            "089165D3322BAD3430FFDFECE79D8F2D60FE58A8F750CD58E7AE7EFA69BF1292EB99C004D44B41770CC07DFCDA13FA9E"  /* FF */
    ));

    private static final List<String> INPUT_SBX = Collections.unmodifiableList(Arrays.asList(
            "989845D6D78F01B3621D186D80B1C9681E838684E8F28710B3694B469103430A34332FED7ED999F2445F989F0677335D", /*00*/
            "8ECF527599471EA5AF8398C46F34AB1460E8FCCBCD34A120520A6EB714C798BC9C9E15FCB6DCEBFC5C1C065C59E27ADB", /*01*/
            "332CADBF50F10606B8511DB978578566C9D34FBCEEF76C34D20944EB8C76749C830931D649A23A42B7EB000E16A3181D", /*02*/
            "C351399F19362ABEEF5B82A87B6B69A46C8DF789165232DDB4043DCC68DC9BC0046D429E67A4B5E10BED30C724224596", /*03*/
            "0521198B9D7A72CC5B2A4F1E4FF3E8E2F0106052373576BBF74C59D91B44CDB9087A4BC447EF948BD7F7CE8783B556C3", /*04*/
            "221BF9B7E9B94D305064C63142E5DEFEFB7778734D2808F5FE13C6B063BC5D34D66E606EA40C9D469D8F696FBE45BE37", /*05*/
            "B539418AFEC0A92C5448E6BFA930576D27583D4C1F41F2EB894E95876769D3B2A51C7ABFB054E284A96F330B86E47830", /*06*/
            "2CA60CD0CB847E8AAC903933754C1E4304707E19BB61EC1D9068F19B1EF42B43D282F3B0D768B087F70283A4EC2F42B9", /*07*/
            "C6EB71927673B4D697145720033F10EBCF91E09D30EB1F0FF83807300BC140129D49B2DE5C8463389786BCE0F90A870D", /*08*/
            "01F54E20CA631AE3A404F97EF5F95B3BB228949EC41E56CF975B2D882536D5DB92D23B1B33792B64B8B3D69CB2991DE4", /*09*/
            "7A2676032AB89E454727026548F0CF5956EF4C4650186F793474BF318F4A52844F0C1CCECE3280FFCDC96F93CFDE683D", /*0A*/
            "62E8951224EF04D78775D83BBB034F3CC1CD56D171786B3784F3EBD48DC68035006B72E484754B1F2989742F8F7A95B5", /*0B*/
            "9666FF1E22AB8BF0C6FABB524A59BB7454261CBB805B4581C579A14D87D1185AD5208712A16C581AE400F2A09F3A6DE2", /*0C*/
            "16644C40AFFADAD140877413AED85CB35ACE06E5BD47B22C2755DB691FA0572ECEFCA4E7204BFA5859FCCF3DF714C70F", /*0D*/
            "4D30930ABDB0A0591B420FA777EBF58C129E25E904817F75D3F5AD21B73C36F92F379058BC6E8F2A026DEF053C6846BC", /*0E*/
            "8D3AF50689825F0B31D452FE3E478D95CB7DF6CC67436019924F005EA4E2E6E2F2319569A73491E9804BFCD5D918B523", /*0F*/
            "1D439FD8BB52A8D86E15A4C24B921C6001D9E7B346A9DA2F8C9BECB54328346C824ECC6D15FD6CE40378936DC712DE6A", /*10*/
            "4F3E3CCDC9995BEFCE007732B0B3C3F3ACAE98986CD9EFE935B101CEE59BD664B86FF74F0A28DB618A9C01B6E7B650D4", /*11*/
            "F7462080F29875C427F0E5B14337E22408938507DE7C4048680D0CDEA5521041ADF4D73B89AFC7AFFB456C7BEE2D8199", /*12*/
            "F1E67C389F4586F25D6C4C6C211EA4215E55939951DBBF33858503060A2147799596E58BAA019EC9D5FB3A233DDB1EBE", /*13*/
            "F3C82B68B7963CB4DEF9F7055988C7E01FBC7CF2F30C42E2A4B0376D5DC249E50101640BDE9D097758DE97470CD08449", /*14*/
            "E765B90F0DB483747EAF5F96AAFA4208299B37B61A299888F27C6DFE715624D3060589825A407689735C3726A1246A2C", /*15*/
            "32EC9848717617683A379094EEE2342AAFA2EE700E9C9F970441D52BDA14DBB00256C250071B0E7FF581E18F6396200E", /*16*/
            "FC005BBEBF4D2220DC811F95E7983A2CE674B73ACCE989FBD00E882FCBDAB62CA1EAC18F2474D6414B52C4BE1CFD7FB2", /*17*/
            "CCFC30832727FCEADD711E8D1278E3DE84955AE2F6D86EDCC6608ED7503403A2532E03516E0DA83B8715ED3F806D3F56", /*18*/
            "345C125AF9EDF1DA89AEF3C6240B5DD6D92D87385B67386D41BEACA0D6AC2295F5F0EEA5385D8B2B2B27312DF2D78368", /*19*/
            "78B6D72E09E9F4FBA5FD6C0766174996D1BD209A45E759574BCCC438884F70A0C4FA09AF13393C6CB2E321F374E1D5A8", /*1A*/
            "F93D23B5883BF7E7E04A0D390A97951CEC736AE76B37AA4663D2E7E5B5EE664BEA840CC9297B2DD60A4ED1E538BF75B7", /*1B*/
            "1EADC4CF3BD09FAF9635B18A5F3B5F28C821CFFED6EA552D8B34AEA5E1F99363A2084005185361F8983E72CDDEAFD386", /*1C*/
            "D74BCC84526E4C5DB492C01A10D97912468AB217ACE3375DAE067B12E07F2129E6221AC3E31FC4D05EE6D3B51E6F092F", /*1D*/
            "C26AD51698A38765CD039E4220050469780774C1C38A4E38109FD803234CFD3AEC77940017D5E96562F318956A00DFDE", /*1E*/
            "3AA9ED52E3D14A91E473B25AC2A68A511CB3104791DE3FB429B232431D65EFB194D3F9EB9458C84A2CA8C00A8A5E5A82", /*1F*/
            "D6CBCBD929B2316C76450AF38B277CAD3E67A409611D242513594E56D56BA1941A63AC2446DD90BA6FD6B8FD5409AB9A", /*20*/
            "E01256F90FBE93FC616575AEB2DDB5BE0647048715CC86E3B19E814B179C00EBA84DB0C09E4ECBF0E0B662C325B7D098", /*21*/
            "277C914720A1667E2595135DE98ED4FFF369C8FB60304AE71929EE4280A36EA5770EB3117A62F00B943F3C025B80E67B", /*22*/
            "247F7E2CAADFC99C3F497C3792D0B1D22BC547D49492944AE718DFBCB90F2D7EACD7D3A96BBB5B259E72DB3CBF3C389E", /*23*/
            "AD3F83C9462326122344DAC12990CDCB5FD5D974635E657B466373184660332A703A6F23C58995F19C0EE38B9D546CF7", /*24*/
            "5199E1D743D9849D838C20187936EEED39B60D12E253E03EDDA520235F224CE8D0753DBCEE1E17288224E7738578135F", /*25*/
            "5E8DA15CA04EEBB73CE7E39B7AF49E5DB4C164CABCC8F78B6FBD487D5193B00F7BB75D17C822BA4BF0CB9AB0C3F26506", /*26*/
            "185B9945D44BFB7708F3D5492AFEBF32854C5E15A19148A1DB9C17C34BB9B83F1E44E80E9542CAA1071F364E41FE8AF0", /*27*/
            "5D55AB770281CA4A2E663EE11AB51FABFD34B869F0F53DF657C3B322C766A8F51FBA6B8A763E13D4EA1822AD2D1CE805", /*28*/
            "570DBDDDDA626732AA476FEF546511E39B641293EA3957A2A165AF3A30F01157744A97F68DC9B8941366C2A56CC5F550", /*29*/
            "8FC1AC6CDE3D38F3D2EDFA714025FBF89A846EF9FF8E1703393692AB75D30FEDF1EC787EC4DEF632812C0C67B34A7164", /*2A*/
            "0A1E48EC7919291CB3709298F1F6936A8A3FC7E14A90EEAF620093AF09CC734DB195D5161C0E29965A42C5B3FFF71C32", /*2B*/
            "1BC5A2EFDDFC3DAA45F57BD045C525E85CC8882B980229B91D6B804C5881F702427D26F553E0DDC69F91E5A23AC62697", /*2C*/
            "1572853F86E425EE3BA1FB3FBC0F55F9557A02E8A8F8633093C06B59E7D74EF797893F532A7EA3C0842BC94D4DFF0860", /*2D*/
            "90605DF30750B22B9988FC0C83FC58CA44D26B6732DD1A1FED9345BF89DF37314D2A54FA3F91468568F6A812FBCB4E79", /*2E*/
            "29DE961CB143305E816ED9613D20CCA57C8F68A4AAD4278C8AE4067E933579C3C88D1EBE8850EA3CB03339E20D051F69", /*2F*/
            "C5D38C50A7867183ED1747E3BE4A9FFC1D379A95B3BA5A27582F861B4ED20965990714F1FCBAC2CA0929EA99EFD62F9C", /*30*/
            "5495DD63D6770AF6BAE5E42DA5BBA17082F26C779DF4BD49CB26DD16FBA60B8AFB6C0D7A59D33814C1790884A9B204D9", /*31*/
            "23C6C04BD289880CDBB31A0BF060913135A05735347ECC061B9DE4E66916BBBDE2B568487729149B7B9695DD6F0B0CAA", /*32*/
            "C945EE79CE68742F90A9047F5A9D98034B50BA63C127F0142C1B26B4AFCB8D7F7C864FBB3E4DE17AAE0124BCDC1A252B", /*33*/
            "BFB78A282878B9F44C06767A4CE07ECFB161977F8ACF67CE7E8155ACB2123E5BDC4610859C18FD2C533940C98BF3BA83", /*34*/
            "4E5AFA91668015D460629FB30D704A9A790DD00C1EC5D3ED18A09D41FE5790707F5292916C3A0FA6326776E4E36AF2FE", /*35*/
            "AAEA316EF116919B535043F5F62EBE7647652301483D1012D86F8CCF6ABB8F20B0CA9F7F2B039AF7E9AD199BFE0D6429", /*36*/
            "2AFB33D5AB70AC8FFD5781642206FF0F40C0E136D7BFC968452A50BEDDFFCB9F38DCCB9F83C7E45EC3C1F0098120158C", /*37*/
            "03F889369E2A2CDC948B56E7E4A5A75369153C6EE5B84363948DE00E7BB5326121BE11BDF9A196FA528B5BC2D5C1D853", /*38*/
            "BCB0DC5B2B4FF5FA8F5D861FFCA1A5222C86C46069B53AC370F941C8DF95177B5E65C508447D1A18A173DEA81985E5B6", /*39*/
            "E6DF21C75948AF4170B6D27574B8390DA7810BE08DB47C5F221AFE905227448CEB7ED040C6555F1C2A62B27F1B0247F5", /*3A*/
            "58776D95C6CDFA50CC7D7979081984BB7FD6ECC58CF6B680D7E72A926CC3CA21C751DFF9A6F956A2BF217AC6E4341608", /*3B*/
            "09F774F69724D9F71AAB33BECBDFD61EAB598B92F84C264E4CE658C7FF835E192DAD93C691EBAA76342F997765412315", /*3C*/
            "A870497BECD2526F10BBDBACD1BFBA06CD1C4D7E0DA5F30DE1A18B577837856FBB97184DDDA6F59AB4F1AF86C4A1F86C", /*3D*/
            "6625B60C2FC3DD98FBB4D606446F714FD6B2FB560C64BC82EF82FF98BC6F9DD29B2D335F31ABB205D96AB4EAF16CEB6B", /*3E*/
            "C11F2200F30C023E192EA1DC9AFF481DF9FBB04AC6E0776EB288BC7BF04BE167912CBB3A960F6635B6F857B9D748D1F8", /*3F*/
            "2DF2F0CB304C05E964EB4A21A8E650BCF80F462384A7732850F01F9EC41BA5F46CAE2DB5304673990D4A5C2253A24389", /*40*/
            "0B045ADAD01361E1A2EEAE933B851D9CE790115BC9092DFDE0FBFD3B2D3B0DD9D3F8AE478E2BCF6F86BD423E71B137F4", /*41*/
            "BBD132E1D37D439AE60966F08AEA6E97576B24FD2D21316AACA20AD8332A02D710B2C01D26EDC9D54EC4CC1CA428D755", /*42*/
            "DC5F3FE857D707546FE2297D2C220A7593B5BF91FC740E8648E0BE58A38D9118E4EB0E8164147E04C90FA6DAF5ED6204", /*43*/
            "078A811B10E87814262B54CEFA8975F0147FEDD0CB3EB40221CD5CFF966E7188FEF5B7C1F4103D98009EDF548C57F467", /*44*/
            "77AFA7FA587F2B10883CDCA25DBEF6C7C514EA22E0DA99874D705AA11CC0C2810529131A3B356A06F1E44F0D10B3E3A3", /*45*/
            "894AE52B6253E8C321EAD3F8EACA62C6BC6D39049B5F58968761A7002BB884285426AB668B87351550E1FA5D5F8D5218", /*46*/
            "FB88388F9005E61BC80F16406BA4F1A667921FACC207B5219CC2CAE15EFAA28287A8A69D7DEEAC43013BF4AA2103A85E", /*47*/
            "DA44C597F78C3FA724B97245DEEC4C4AE8A3FE32657B839E3214F819BBE11AC95D06B565A34739A83BF28C2B4711FEF9", /*48*/
            "DED6E6B80AA2334EC98FDFDE7D556627DA33535FA980A24F5E2594F63B098A7AC6ACFBE802F0DCC521EC10DC5D5AF7CA", /*49*/
            "9D7E2DB303C49B2237A507E060C488349580EB102BAB9BC143B93E1540D488F2561944F4993C36676A948D1B9C01D9AB", /*4A*/
            "3E6E1ACEFC54AA93E7C9F1CACAE163DF890B3E536D66D0158E6EC725C5AF94E1FA802B03AF179C1778443524359B7CA4", /*4B*/
            "198903903849093CF7DEAB9773B9CE359C405828A055DCC0D1D9E81D4264BABEDA4FDA684D363F7E66A32E61B6A68C36", /*4C*/
            "EF5384A184A818BCEEAC84821DA3A6DAC40231435A267140EBEBD2815B3DADD6F81FA2D48045185BB16077717258C362", /*4D*/
            "40CA6ADE7F25A4F8D49A252F7F95F473BB48B5C975959A137983137C12826A01B6DF475CC0AD16077E09489827CD2AD6", /*4E*/
            "6A0FE786BA74C4D228ECC85531C344807DC932A87E06CF8E3EB6B7B22ECA14E609A43707DCCE64E26EA412707B32FA7D", /*4F*/
            "FA94243DDB269C57BD3E8F7C0FFB83B7B7E03A20DB44923D2ABF51DA18745C524EB6716AB23BE00C17B5E04BED60C11F", /*50*/
            "E53C4BEBCCF69A850CF483D23354E67DF7E650A3991B2C29F171703E05451CF66B35456203380C213AE7BB00332CCF4D", /*51*/
            "70271D342391D34BD6415EF29CB664A7800665400B9E3BA4EA057E94D9E8C597A72BFF3C39FC8E8A5F137897050EBD43", /*52*/
            "26C7974361D6967604CCD0A981A737DCB85B8C2F33D11E6CFC803F4413E46F9367DA9D74D8E3BBCFBB90AE4F893F282D", /*53*/
            "4552E444187E4F794161BEEA9FBC94195D60226D0898BE609D905F97379DECC484E061271B129F2933C29EF8D3C034AC", /*54*/
            "82793BEDCDC2F317EC3ACBD1305FEF612888DD39F2F1051E08F279340207B11A3D2F48E09B8E4293C8AF50D0D8AEB771", /*55*/
            "A182D4C6EB28BBBFB7917D150B527D097A115C7D592D8FD164ACA378EC582F99367B7D06355AD7CC54371C6C7F848E42", /*56*/
            "0E83583B816BAE2412A6FECF6DB23C15F5EE28F8193AA769CDF19070078483BB7DCFDC72A9F65E503F0BEBD113BC9F1A", /*57*/
            "C77DD87104880CAD1DCAEE22E25E3D481B4DA9FCF77506B6ADA36936F6ADC1E3B79F738DAC61126A631E2B795764C426", /*58*/
            "92CCFBE90E9FD5B5782494E4E19BEB579D0135A19C797AAA81C78A5AFCAA1E9B2099D4AACA8FFEA70C4788F7EAD85407", /*59*/
            "88D5AACA9603B370B9FF4ED8DDDCA9CECC058FAA7F3BA0421433EF5C9EB0B33788143EF7FAC493E0568863E3B0BDB38D", /*5A*/
            "563B92EE8EE11673745A31B51CA95EBD4596AFD7DA9FF458E5E138FCE668191C23E5FA5443CBD81B8FD9682CE18AF03C", /*5B*/
            "6401C18CB2DEB0FD82B55838C76627864994F89FB56C1652E4CA54293E17AFD0FD905971DA667C606BC327CF76DFB091", /*5C*/
            "748015628FC7354F98798E8E894D67380376A13CD0C0B7F8BBDF3A3CAB961D46C1F9C4317023E37220875AD44A1BB8FB", /*5D*/
            "C049B73AB6D4122D02F8B8AF6AC9DD8E1620CE55C07F8C99401E7C8934FBF03C4BE3CD44EC31373904640366829CFD9F", /*5E*/
            "B32B4719954070C866C0489FDA2F0E873872D72EAE8DD8F9B78E18FD1631E8238991FDCFD60960A308694A0320E6EDD8", /*5F*/
            "6E75277FF6EAB1B25E891B90E57D728165A6E8B0CE5C787D8319ED078362F9ABCF2124283D825DBEA6236A4A0104218A", /*60*/
            "39068E1740412021D31C4B30386AD18243794BEB950821FAFFDA2ED2C95527C6D7F125F2FFD410BF880C4C6309767380", /*61*/
            "C42929FD80CE6FC94EFE170E993AE9E9878EC31679715F71BE121464A6D906DAB4C9579B2D274E49DC7E84DB4CFB8528", /*62*/
            "53BD68673FFE3433C5208844CF6726C952B9774D1872825917C93002D710F2B73CE4BE860B4CADC18B597035ADEBC5EA", /*63*/
            "5061BED4E74A6CC577439D43BF810D7B422BF466AB88E1DA0E3CAB73BF948CEC283C6CA82EF53119F4DF1B8149D286CE", /*64*/
            "A642663CAC6CCD484312241785D120C1D8FC012592B6E20530CB1668A16A9E26A93BDDF3C9B5FB3470309FFF8EF636DD", /*65*/
            "5A32A4AEA16D76947F112F5855CC688BA553BE3FD3E1EB85A8E89C3F725E3053E9CE91190DBE4DB6BC148A8537061772", /*66*/
            "D9E58024DFA06035D9D9E9FF117E893D7BA99C2C2F62F9C678D7DE177EDE61A433D449AE664159B8400AC7D78873E734", /*67*/
            "6F90B2419C31955B9B08553AED87FA4B8BADFA6456F99E43F527A0050E2E2CF02BC552F02C7354D7555402119A9E66B1", /*68*/
            "E14DBA1A4FCB53275F59405336713016BF27C690D9FC6A6605B39EA8EB1F7790A47867CA74A88290B3D4B3A118A8823F", /*69*/
            "364EB189B31B2D84717268C5DCD2142FE9F3C1FA4CB91D91232C68AD70CF31CDA0DB305E37D6C68E263279F22C3B9CC7", /*6A*/
            "2ED2F157EEADAD58917BD1BC4926D837EE5499CF8613022E0684E18EF19E4269C5B9CAB860B31C5249FA4572528E490C", /*6B*/
            "3FFAA96F2C12C1815CDAF2085EC23E4266AA07ABA632D211DC3E3611A7ABA7DF4A38690A874383FB7DF5C660E61DEE8B", /*6C*/
            "21FDB5300BBA46BAFA02C367AC766AD91A6FF3429015E60B09A7C38F45FE68754362ADD134B82166FE0D94B70BC2EA51", /*6D*/
            "E9EEC86BA97941A195B27A72B780FE1F30E2A2D6CA865D644296229F5780072D3EAB5CADA8716D9DD8DAD741C2CEEF93", /*6E*/
            "7192591DE0A6C07B03992268715D07A1E3CBB1C4ADD3F51CAF664C6CFAC4C6BF51C407EC922FE7367AD2A40CDD9D2EA6", /*6F*/
            "63477D98B020BC5FBCAA7E3EC55896F74A09162A006E2FD2980BC595480DC4682C41E0C26AC5F31DA8AC73A767973A4F", /*70*/
            "287A2C3964307387EBC835C986ABDB787657901C7D7A51E0A98B47F7BAA56C915A504684E8E6BEB4EFCF2D07CCAD6070", /*71*/
            "AC407A73B40D1D6BE931F660061BF8ACA1F892E6B6FD0323C1FE7D5111E660DE4CC8EB49E025C5F3ABD72C5F1DE780FA", /*72*/
            "679E70DCA29E9931486FEABD96E9062522B0638D03BBAF26E8351D3377268E080EAA5BDCFD881545D44D513697519421", /*73*/
            "CF58FEF5C38DD0B1863B01A3A11F2B64B0B4D2A73F6D9CCDE650B69D5AF2209E5FED770C81B76B30E1B0AB2E4E40194E", /*74*/
            "499373BBC22C0EF5A8D7E1A02DDE35B0F2435129541FCE00207AF976A9F723EA60E85A4E4A9087AEFA7D147A45156984", /*75*/
            "808C062275C62F02DF22214BDFF7A888718B67185F4930BABD52839AFD5C0CF3D4D6A86B8CA572B3A05B6D537EB97473", /*76*/
            "EBA20ED355C114AE6954342EA2F8BCEE3D525DF68996CD7C53AAB58C84912AC59AE99922F16B6FCE06D10A57291EAE6F", /*77*/
            "02364465825D8C995A5650EDB300D9A88EEADB1B81F00F76997B2FC19B3AE4C8FF8802AC4EAA8D694710299D5ADADAC0", /*78*/
            "111342FEC1DC927D58A7B4CD58F24105FF46918FE668FA5A7F97758A4A08D02F8A8CB8932205B6782756F3C4846E3977", /*79*/
            "DF71F4324104427FE246A7D570A00C448F1AABD5CFCE35F0A73BDC1419D8150C29BC8FEA3CD02C629B3AE46B62EC5CDA", /*7A*/
            "46489BE0D9F4517A4629B58F8CEEFD020E98186849CA7D51821134EFCD23D43EF39385B298F1C33128A128B8870F72D2", /*7B*/
            "17F6784CADA71B8BD1B8ED48BD75D25602AB1EF720C1C5183A6D09E0B65DB58E69408B80E59B1FE6DDC8BF5E0E265F0B", /*7C*/
            "D473097415E0F68EC10AF0731B3E78F49EE9425E1385197F6DA88FC426016422453FE3E94257EE53D69216AB9938CBB8", /*7D*/
            "FF3426B0331F8A5A06E1515BC3319D3A0BC36F1EFA3FC2A396951B5FF74D8630EECDB9F89D1643AD9628A564FC46E941", /*7E*/
            "D32D75DF266F55A28BD6599CA3EDE010EF687361F56AE892316C5D1C8A725A8F418B829CE9B04CA595EF3E19738B0639", /*7F*/
            "7378CF7D1A7C63C02C9408D768AE5455D73972056A9BC817A01D4A710CA716961BA7C9A3D2BC4A7D832E437D792EFB27", /*80*/
            "37097BCC39F0541F36BA36999140924598BB690821BCDDEF28D5D7726EF5AEF111DD1641A2ACCC73314F6B4604D5A2F3", /*81*/
            "8618F369540277E60DC1ECC32FCE31D18DDA05C7D101FCAEC2BB352465F1B759F60F56CD0FD2F89F1A20CAD27A9A30A9", /*82*/
            "8B2462B2738562D9F221A3579414D3D4752C14FF574EEAFF4A245B2CE90E894C5B7F3601C7B4A62F1F680FF4C5273EBA", /*83*/
            "F0B9642F56EE48C2F6DC385C2E3C8E1AE4CF13370A73A3A52FD6CB7AEA6D0E1D76858A75C2A085E3461BC188503614BD", /*84*/
            "C88F5CE6E5216B4401C30E8150213646B96681F58BB24F9B54217F013F90AA0B4024F032ADD18916CB84E87EAE37D6B4", /*85*/
            "68AA13AF68070F518D69262627CBB85FFA1330EC9A971C2238E51A2EC1A235BAF4DED6A66FE97FAAB5199169771F90E1", /*86*/
            "765D0A967B585618BE630941018ADA5BC756A83D62D512311E4DC10820A9968BAAFD0F3EFB993EDB3CF00E27C8E8B17F", /*87*/
            "2FB4A3AAEA9532AC7C340C4FC1FD99C853CC1AF0315779B133B5C885D249BEB40DC23CEE7FB119636CD5815568933DAD", /*88*/
            "D00365295DE53740E838CFDB1368654C079CB6E458B0BA045162BAD108DB56551C47A3731F152F0F6D80D025B77BDCD1", /*89*/
            "149F7231B93250F9797CA202A746D7290C007DBFE30B93B8070F3C54008BE017B918F4D94F56C0CDE55DDAB4CBC89DA2", /*8A*/
            "CD1A2E04E11DF20EC2E66EA4D049E1E64D1E33D93ECBF17749471948943875C26459E25BF2BF44BB997FF8E7B921B9F6", /*8B*/
            "955E9C08D8594B2EADDD2BC0071D03C03B24BC570142234DC367B0849233B427668F9AEFC144DFE519484D175C082990", /*8C*/
            "25C0BBA7C08BE1533E4F19A64DE3F27C105A3F652946A5CC9B16A2671A020A5CD8B4F202459A5A13389D8F7C36EAD41B", /*8D*/
            "726CEFABA5D5E24715BF152CB45AC2C23CF0CA4982241BD0009177138298D924124CEAA4E421D14D427CB6AEE549C02A", /*8E*/
            "D2A351D1637B4ECB0A743AE5FDB00971E5A10948857649623B2EDA9C8EB28180EF733588B824BD4FEEE8A3620274DD92", /*8F*/
            "7B85A04F5EB1D8D3A09D53FC39352E4D61308413279D3EC5DEEDCDF92CEAE3718E5E00460E9397C2E2AB0D83B856A9CB", /*90*/
            "7CE1774AC734A22A38677FCB5745AD5C6DC4DA001D935ED3FA15E60F900097C707A1D89A365F5122FFB482AC0AEF227E", /*91*/
            "F2AC87A6AE557BFF304D0B12C81C6BC42EF4B9752C9A5B0E6E7F4D47BE5BF11127D06E6F10DB116845FF258C7C67A517", /*92*/
            "4ADB829C1790CE6A17B12C5F9D53CB2ED05C215C78059589CF022763C6F6F6A82EB81BFBEDFE208D24260740BA33AA74", /*93*/
            "F4D84F2349F7D43DF09C67E29799135A23DDDC1425ACD4BD671785A94406C35D3AD99E368ADAB1FEEBCD20C0395B5E33", /*94*/
            "DD370160E85E47E2F518621D8DDA23A268312CDD936BDF39B58C8726B07063B5375BBC97D08AD5D91B05A73B22653C8E", /*95*/
            "BABA14E3C4C5E09FFCCF426E237C45B6A317D4B170A6ED1BBCBC0D83350ADD668C3DEDB354264859AA433B28909F7D09", /*96*/
            "B2CE1C6D0029696E84D3BFDFC9F5F9EF1738C282DFA047C7A3DE9B8641507BACB51AFEE29F5C082E101AEC2A1F7C057A", /*97*/
            "FEB128703CAF0D39C34B85B0258B289372DCA0AFFEED70844E8611E78B85E786815D4ED0006F794E79DBF5B22FBE4C1E", /*98*/
            "F89A07FC4E72FE1D11AD9AD915132F0EE0FA5FB235EC4CF3AA8942C0ADBE467CABA541300569CD806117F73234A7F3DF", /*99*/
            "0808B010379C49697A0C9CFDA67208B9F60345DFBE405C56DAF752913692FF0D0BA9763DCDC0FC3E57BA4EBBA8D363AE", /*9A*/
            "996F7F4D51A57A9ECA973DEC530C3B2D4ED8274426E50D24D5EEA4CBEE7153B3301084252FC3B388E3B8F94C31A03511", /*9B*/
            "5FF1A5E2C51089C692362E25052D19BF33FF36DE68A2B9DFC8F649B13C7DD706E0E7530D792D53703795158242828B57", /*9C*/
            "20A1F7584D0BE592F3FB0303C6427F4ED4822634F18B540A65DDBD8D22CD6B5844B1E698148DAE5CF9B10BC59BF52D2E", /*9D*/
            "47A00DC236CA2E4C655878B2EFB73F30EAED7B58B2C628505F23335574E059151954F15D5F1CBFA9746EF1A6DFCC8F9B", /*9E*/
            "814CEB8183BBDE56CB55C2635CEF2CB55B354183B91488E1564508377F737CA7CA132295824FB93FBE65A0FBFABB9E8F", /*9F*/
            "877B0BFF4A463989EAE4A82A3708F36E37E3838C228CC3E66194A532DE159CCF1D15D9AB522055DA4CBBFF5608137EF1", /*A0*/
            "2BDD46821CE6DC3F4F80AA74FFB40B8A920ED5E32454FEBE3D32D3B8ACE51333793E061F486347EF39711EB1037591CD", /*A1*/
            "1256B31FE4AA7919496BFDDDFE8DC66BBAD7710FA784015CA53AD950CEB37F515CD83477B93F70818D06FEDE91CAC6ED", /*A2*/
            "E80CF2A835CF5AB91E1664625B0ADC6594C29FB9C5944BAB6022890AB404CE74E1287F92DBBD81262E537DF515FC673B", /*A3*/
            "35E0DF14602D6E823913BD6FFB4F1284DDD1C5AE6FAEE7AD9FD325CDB3E78B50351E622F5E85789EFCE292BD12559A5C", /*A4*/
            "3B4F024ED5F3D2430953060AF7AF249DCA89FF4EC8BE14C8801015E4F8B1384A1523587C4030458C1176AC066D4BDBE6", /*A5*/
            "6B620FF8676ACB3ABF1E91761F8FCA1B882EE56B11E633D97D423BBDB818958732588DA0EB675C4CC0C087E62EF94166", /*A6*/
            "EDFE257C6BC8EC2505E33F1CB1415A8D831B7A85662AB05E03E3B8535C391B7372C0326CF37FA91171508ECC51AC0E01", /*A7*/
            "4BBC634621937FA48A1FB6B73F841872DB1855A28833FBA7F3C604747689293218A038C8F006013393F9DDFA66E30F40", /*A8*/
            "38F467A0DCEB9D1E14012710AFDB0189B604B3DB474A2B53151F66DB79A41242BE36E1DA85814F10729F89EFEBDCE120", /*A9*/
            "6D6B18545FBD19962BB0416947D7C017BDC71D5076E280B71CFCF3A3613EF4059F42EF2E6DAE68B9E640A9C1B46B1A61", /*AA*/
            "60B3AFC83A971C42203FB3EB560D56FA968540ED4FBD84C4240843CACF2972489312BA4C979427377FDCD93198C7FC58", /*AB*/
            "425079EA14B55C88BBA44546DB5190D0749FB47A9777399AA6EAA86B9D19EE03166639C5CB07228690CC2FBFE0590735", /*AC*/
            "653511074CECFDD5B6CE60C817D359620522758E2EC38EA90C4A78E824F8489DCB0D04DFB537C1B0F6A2AA760F2A40AF", /*AD*/
            "9AF0A887858359364B9E1214E69352FD324B7FB51489A44B36460E6532FC45AF6811F62BF8650D3A4197C8F6D24CBC65", /*AE*/
            "9C2FC3996C38A5BD1F2673D3E304C4002678E6B48EA8E561B94828DDE286DF6B7EF71F90E7520B91DA16C36ED447A178", /*AF*/
            "69C4DAC1BC8AF8677B10C51688E8611864E52F76B7FB0B93445D744A2A47A32B8F74DBD8198BD02464BC2A04B5A95B24", /*B0*/
            "B4E4C2BC5A9B7D66A7A09551D724ED678C410871AF1997E52B99C2E2E4BFA66DC9D17B5A505E3012D0C55F453B52CEE3", /*B1*/
            "B902539AFB51BFCAA6C5936B9E6146BAAA2FDE30B0690C836AEFB9499875FA09E5D52A83236DDE0916E95465F4616F63", /*B2*/
            "CED4EA7AB806F9A8B207D79E32967AAA13DBC0F1E9E8BB950AC5A9A74D0B4A726FCB881E56CDF495699AE916A7F103E8", /*B3*/
            "D191CD187C5C900F3528E2ADD528E50AD5AFF054DD033635250146EDF97C08D83B8AE7574CF7237B926B6434F8870AB3", /*B4*/
            "F697BF66F865E3EDFF2CB94EAB103307D349C9883B7DD116665A64A6047851E0865A238E3A0A2840F307B196AB53C981", /*B5*/
            "B1B855AC08FDCCA3292D00A535070F049F8CA6A05C0F465BCE77F093CAE37D6263035176727A8CF48E11FB52E9C9D294", /*B6*/
            "130BA6FB4201BA5CE38ECA77646297B273082997A5AA41F1DF0CFCFA4F7A4F45A381C7DB21B9E5F5C43DFDD6A2441138", /*B7*/
            "D857D2A316DAA6F1B04C141B28388201816E525A3865DBD45B7D632A062F62EEBAC1794561E1260AA3B234ED6B7F0BBB", /*B8*/
            "0DEF0059F4AE8E602240A927BA6CF7F6977B96D24ED7C05591FA53F86FBDD13826026DFD8F784175CA8EB0DFBBB4C8A7", /*B9*/
            "E331AE5D8D1757DFC7846BE9CDD540F2860C179444B77467376AAA2D9F3FC70E6DF2197BAB9ED97965347113A0F04A4B", /*BA*/
            "FD05E3D269B603782A82493DF333602091D4A5C39E4804BC7339A64ED4CE6DAD0FFED12C279207DF2FA63D51F39251C1", /*BB*/
            "8AEDD05F6EA4FFE81CDF0501527B2990BE23342753F3C1FEC9AF6099219905E978251D294B6AA082A77ABDCB963598FF", /*BC*/
            "7DE25425FA9D3B37596DDE843A6DE4586EAC15C0F46F96F4D492D4827AEBA9A658A3B13904779B08ED755D1EA694F6F2", /*BD*/
            "A2113EE70CFF45719C5EA500D4CDEC63B336BD0E96A3158FD9BA2C6FDC25EB1EDF4B2E21F5987A9C18EA3833F0F8777C", /*BE*/
            "52E9FC569B3CBD6D3419E091F2021B7FA94282413C9918FC59A6F6ECB1D5E57D61E67CE368A91DF9CF8D47745EB0974C", /*BF*/
            "44234D9378DD108D55BC3C7093C722D700CA9EF4234BCB479A58E3F5318EF82559169B64EA956E0D67E5E2BA44D1BBD0", /*C0*/
            "AB8137F1015BD613D5D86D9AAD1543D36A1219967CFAA97469FF6A1056E95F78CC8317A2FE64ABDE23D3A149CDF427CF", /*C1*/
            "30179A5570F2971A42CBF4240C11EA83DCA7431F12366207FD54C0790D533BA9039B0A70CF72EDDDC63C550FE8E9EC3A", /*C2*/
            "0010E2B6872B7C01D0F7D4AA267F2DD8ED9ABB7BBAA1D67A1251EABA4C1AF5CEE764A037E259848F77A9176A56958D31", /*C3*/
            "A3E7DB8E4537800A6B9B99B809D48B3E0DA4F26F73DCAD652ED0E9669A2C65DCAFBB4D67D4F2CE5F25637CD375D9A4E0", /*C4*/
            "6CDAD161052E369052A8A62904C8B99FB57EE35DEF1CE9B35DC11CB62F32D20047F670871A1A5003C761497869C47675", /*C5*/
            "418B348577BCABAB8C4E61473C8274FB586276DC8F0E209C47A9621FAA79543B52A20BE11E700AD8DB8C965A925F9616", /*C6*/
            "9BA45F111E3E21112D789736D21A86AE59FDCDC607D2E4A07A9A655DE3FDBC927187A961E67C04B2D2D0D243480732EE", /*C7*/
            "BEF9165EA65613EB7D5C5DBAE850FCC5CE3CEF21BF0069D772E2238BF29758B8229C08107BE2A4540E8ABE20A5EE5785", /*C8*/
            "A476699E5B9A82860EC232D661E4B39811F53BD33D8F4D7377D4985B5913C01331EE01B709EAEC551482D868C64E4448", /*C9*/
            "A5D994BA7AF9C3B0B105F5DA95BDAE5225EBD362F9EF64544F64F2C69C1C04A3DE55CF14D104627C51AA1F59AF9048EB", /*CA*/
            "108EF615A45A6523A3A3ACFB0E44D0A96F2A0AADD5AFA62AA2E9F5D6861192448D715E4ABB0898C8F2B94458648953C5", /*CB*/
            "B8A7CA8D3257A326FE77CD50CE7A4B6C3A4FE9511B7034F7CA209F4060B7C91639AFA5425B1900C4ACF4CBD9BCCFB487", /*CC*/
            "55A86E213D1E68DDA9F24D7BD912B47EA0F7CCA96EC9D72BC4A40FA2D14E55D18B76A1BA75B6B7D2DFDD863711FA10C8", /*CD*/
            "5C868D273E442828A10B70E8EC2B16949999D680DCB3133C1FDC0B523A7E3CA1A6EFA79632DF2E9205D82330C1631247", /*CE*/
            "E4AECE266A0FC50D13DB5BAB98099CB1C33A38C2B80A7B3BFBAD57D31561DEEF9E600518B70234ACC538B7F0267D7913", /*CF*/
            "79D78F51A89400B6E1528B568E9FC5CC34B17979FB4F686BB8DBB480D3C9BD39174327D755A3490E4D3613753250A702", /*D0*/
            "DBD05E3748E7B6BB4D2311FAB886876F21448A78105D72E875B71E6E548C678DC039801578A7BCA0ECC758D855729903", /*D1*/
            "3DE3E89493185D499FD5448584C670A0AD1DFD7C5D12B1EE8FD171EE7388FB5FBD323ADD69F875CB0FAEB515E23061C2", /*D2*/
            "1FA52F02251AB5E080A28C04B95621E76271D1EA36C2CA09C075CC3D62AEFEFC6A9D501C1697B4F6222A605BACD424CC", /*D3*/
            "EA6D6BB1D1B3C846F92F8D4AC0A28133A2E4668B870D619471F49A6A990C2676D1458699B4ECA5277C514B1A610C6B4A", /*D4*/
            "840717C5EF1C6472F139AF11826EDF5E09B78E0AED2F09459572FA96CC512554750AECC7D92C33BDCE5726F107624D54", /*D5*/
            "EE634A78FF3F0816188AEB59EB2A4D8FD23262335558530874C4840D818FF33DF770C3A7BDCC8A00A4357BC8D117A014", /*D6*/
            "5920367E2DC96A05AEC79B356364A277194E2B6C641775C2F94B97C57DDD7A6AAE174C785D13E6ECA5229B42FD86BF10", /*D7*/
            "9FBE8864A3DBD1DB8598EF8390CF5199EB3ECBBAA3A4DE780F2D100C53414149735C5F799300252D150352A3587E005A", /*D8*/
            "312A2AC374FBF004C4BE5A0F7E79B6A3C6970F4FB42B25B57C8FCF1A49BA9F83F9682CCCDFE41B013D4159892AA4313E", /*D9*/
            "4C6850A9F567DB089AD0DD28C49E773FE1DE0E31A251AC7E762B7260641DABAADD5375D311D8A7D136C60939D61602FD", /*DA*/
            "E28735C49115DF616A6AB73C8F69B00B7729F92D52D600CA01567AD5D0404D0457CC280F9A8677B55BB7E68D2B66B6C6", /*DB*/
            "0F33EC335C33A7C1F486BA091E4B9BDB2D1995267AD0F6419EFD7661E82D2E9849619694513D8844C26C7E944BBAF9FC", /*DC*/
            "97BB10F01BA9EE344460C7A1F448B74750C6AC6A41C7E3362DF831F366594BD480C3E952A5F3866B60BFEE9060694F12", /*DD*/
            "EC59E9764B8E9480727A969DE0397654FC0A5B03EC8211E48DD86FC20F1ED8FF7A6A7E4BCCFF71E81D85BA924F8CB222", /*DE*/
            "F59D43B48C69C609AB96BC4C465C05796B3BE48A2A239D3A1A3096A4A05AB2E755F363A190C8FFDC75931A29705C6EA5", /*DF*/
            "AE741BF27E11D7C70BEF718619A8802B2A51D8D80904FD0CB6B4F4B9D88A016065B0DE8C867624C3CC9B67E1464FAD1C", /*E0*/
            "93288B6A06CCA1CFC01BC4F972916CEA0FA81B060587A8ECEC5F39F0299FDAFE0A67B656D3C232E7D33104383E429B5B", /*E1*/
            "1CC2900B4466ED629D7E102B7CC1BD40A8BE49CD17639172F49805DCA2EDEAFDDBFF20B1F7487448AD46B9AF943E58B0", /*E2*/
            "06DC612DB535BE7C68F62AF60263C1854F5F00727B1AD91ACCB86C20BDB45B89C3489CFF5811657189A556146E8FCAEF", /*E3*/
            "94B2D9F4940ACFEC165F65872BF115230AD02D0BE7DF0AA8ABCEC9F43843506E620B8C0962E5F1574325323A1431A6BF", /*E4*/
            "61C360352EE2C2DED7C6E8F1F89CA3493675AE45774D6D44F6032BF2DB543ACA25048138084AAFB7FD4966A99398E045", /*E5*/
            "CB0AC92AED14E76307B72DCC6E9AAAAF48F6AAB84250858D7BCF9135C2B6ED102A7C6560EF80D247D199111D1AC388E7", /*E6*/
            "8CCDD6426F0E11B867BD37784E23ACE5F1259DF34BB1444C8807BB093D63394F96579834065BF2FD76839C8E303D01C4", /*E7*/
            "041D1F9BFDD3E4A6D8E9C134B6770050DFFE9B24E125B88A0D878DD055C869FAE3342955639F52BCA22D6148CE19E2C9", /*E8*/
            "A9F35753F0F84475B532A0EE008CE79EE2F954860FFFC470BA53B14F1087CF561327D22DBE2EEFD3305E85CEDA817019", /*E9*/
            "BD2E6F7212228D07F8F11C23D8AC6DDD2FBA0C1A9FFE90F2D64499AE014287CCB3FB741373FA3B3DAF5505CAD071E4D5", /*EA*/
            "83BFB8A41F08248C3D93CC6A650EA026C2F1F1CED21181D83F31E2EA85059A07CD1B8ED2AE49F9234FFDD52123DDFFEC", /*EB*/
            "B60EE0A2345F3ECD4AD128B46C5B4E4141E170813AEED59D021CD6286BC5E9E43F9A12590C1DE820E8778BEEDBA5C2D3", /*EC*/
            "439CD3E47DF56DA0003D63C7412CAF923FDFA70DFD1007EAE3EC21DF03EFB99A0CBFF52A7CCA02519A5A4108A3E0AFDC", /*ED*/
            "B754BC3ECFE34029CFFCB0E6F93253F5319D03BDE48352015C378262AEEC28AEB2E2AAE6B3C14083BD581DEB7D83596D", /*EE*/
            "9EAB050E13BF98E59E33E7D4D6C0F036157C6D59065622B0B03FF775F5D699F8148E6A3365D77BEBF84C8080AAAB4BD7", /*EF*/
            "7FB5DEDB312F2352758D875E873D8C91904559EFD859C7B2E2AE5E1EF4247E36461D43042883D36D48A09DE80023CCA1", /*F0*/
            "5BC93AB947003A9763D2C9F7764E470C7E1F899BA4AD7ECB3CAB40276D46DCC150A6BD63BAC669EE35E0AD50C943AC6E", /*F1*/
            "3C15B4A59A42B8006C7F8A54D3012AE1FEEC8D11432C8A9F6B786739C020A47798B3CEB441F4F7EA1E70659A9591F152", /*F2*/
            "A084C7AD1D9258156D0D80809B741713F4BF4A021C2E66DEC78A02B397A878FB6EE1AFCB572AD41E3EFE909E284D0DE5", /*F3*/
            "CA19F8F7E6641FA9321AAD4D14AD9AE4A66A441D7220ABD66CC84FF1A85FA0D5BFC7B4B971B2055AE7047F449EAA2B95", /*F4*/
            "0C16FD49E239E9CE9330238BA4738F39636C803B5EC43CC95A43B2E37C487647BC69217DF660676EBA98468A3F709359", /*F5*/
            "B04140018A8785D0E5E0F866B529C8D5AE4ADFEED4E4C63F26766104ED2B1FDD4879E4D5125106AB91CAD4FE172B1BA0", /*F6*/
            "1A6904C0BEB70B0357C43089A0BAD5C34CA5F53E4045B3DB863D29454767E25EE892BF35A0961EA45D7B5EFCB1298925", /*F7*/
            "A7FF9D8811608FFE73E86A1916D63211DE5DE2BEB1228D6FE95724C9959A3F4085304A891DFBDAB11CCE53F940795D46", /*F8*/
            "7E9BC69D53D8B7552F0EFF0D51AA6F7A24E7614BC738F8ACF073D00B286CAC1B245FF820BFE72AC74A0875917810CD9D", /*F9*/
            "75389E0D6D09C738DA8589F467E71ACDC06348B7EB31AED5EE5E56E9EFA1C84EC2C6C8B6B13303022DA7DCE9C0397B00", /*FA*/
            "911C3D13C875EF6456CDCEB6CC43B29B5187ADA6835AFF900B49CEBB273082CBF098FC3FC3E8A197DEEE3F10BD5DA376", /*FB*/
            "48148605653A81E451255C8C341838F118162AC8286050A6BF28FBFBC877CC85D90055FE01CF7DED12BE6EECCA25920A", /*FC*/
            "D52208E58BAC27958E9F3B92181673EC70B84EDA74162EBF557EE57F39D03D1490BD8326D50B57748574A2018DB83B44", /*FD*/
            "AF671E0992615E4D0F6846BB62837BB8A43DA39C393C2A32165C12AAC3F3BFB6FC946643E19CA256B91DCD18F6E55588", /*FE*/
            "85966CBD7271EA3B33766988699402B4205E2EA502CD8B981140D177F37BFC1FED72C6E5258C925D8C12F61F43882CE9"  /*FF*/
    ));

    private static final List<String> INPUT_TST = Collections.unmodifiableList(Arrays.asList(
            "4A078BE656B29464389C7109F0F408B0AD9C92BDB436036373AC50BC993C21F7348B9EE20980F77F4D2BED490EBC9D2E", /*00*/
            "F841FD28F975A827ED5755CDBAE2A0F0FB4DD8A02C8410040FEC8B42590EEF70FF0F504AE44F381BAF9959960376C9E9", /*01*/
            "601F8D7E319448A63E32860064912F344DD71552AC70613119396EB4BBCBAC9595E6F1806C127F83B07A7D449AFA8371", /*02*/
            "EFACBDE18D6AB7294574D2F7688649D96D74AEA685F6D4343876BB9C257CEDB5A4E60F8E014296FB5E46C17AC679E7A8", /*03*/
            "DA238BFF842BDA78D30F1FA2A12DFA7D6AC5F1B028BB41FB1FFA85D36321FE08A60064CD28CBB0F573EA616BFE08C272", /*04*/
            "92393181E1413D83DB9B83522083FC6AEECEE278AACF53E1F24F7A39360575141F711E762CD1BB439638F711ECD91C15", /*05*/
            "0B99271A6F344052FF380136F528BCF3602AAB733D5986B257EC9165AA6150A1792554B19E7ED911BEB6486DC09D85A2", /*06*/
            "4CA31A25692CFC7C718A9DD1E05DF8BEAAE538669FE7110F7263EFC0E3CE108ECDA0C3D9854642AAC756DFC4C6D4228E", /*07*/
            "9BF938C833E258443EDEFD510A622DEDED721C0145D5C9D0F98815740A239DD8505BFF7B2D0E6A1B0A4DF2C5F67B4128", /*08*/
            "873744979EFA032F78D066973013FD442F9197E7D412B9AAEFE5CBB174BE63C0CC02A6E030E61475EB894AE12D165751", /*09*/
            "567F3254FD4F58C966F5119927F12F0B593E5ACF900288F1C59DCCC0E477C34A00F4BF6B71F2F908FD1AF29C27888797", /*0A*/
            "0E732BB38DC5359DA7C97B63D52C5EE12073337DE74E16954840D6E3630EC35BD2FFEAD65DFD115C1B581EEF1B32DD2A", /*0B*/
            "8E7EC736DFF6C79C55AE1299055D9D61D3262E84EECC563347296B89BF812FDCC630D262CEAC4CBE211F579EB6EC1CB2", /*0C*/
            "34DE19511BE9A284D8B9B4C76DD6934FC0B5EBC20D6EB7E39B5413B3398D349F806B13367A11F4B5D4C3D0CE6F8EA3A7", /*0D*/
            "0FF2A2F0AC6F43AF48CE8A380B1AB134CE6A3176CEC813B2AE6C3B7AB6DAEFD1BE99E7F3C9E61A1B3E631D077C273513", /*0E*/
            "0919F02A721C4ADA9879C3BE6CEC98FB6EA0D453C20DB4BD9B287ED6252B9825CF66A065A209AFC8D5A22ADF2E457C6E"  /*0F*/
    ));

    // =========================================================================
    // INITIALIZATION
    // =========================================================================

    private static final List<String> ENCODER;

    static {
        final String[] escape = new String[256];
        for (int i = 0; i < escape.length; ++i) {
            final String escapeSequence;
            switch (i) {
            case 0x0A:
                escapeSequence = "\\n";
                break;
            case 0x0D:
                escapeSequence = "\\r";
                break;
            case 0x22:
                escapeSequence = "\\\"";
                break;
            case 0x5C:
                escapeSequence = "\\\\";
                break;
            default:
                if ((i <= 0x20) || (i >= 0x7F)) {
                    escapeSequence = String.format("\\u%04X", i);
                }
                else {
                    escapeSequence = String.valueOf((char) i);
                }
            }
            escape[i] = escapeSequence;
        }
        ENCODER = Collections.unmodifiableList(Arrays.asList(escape));
    }

    // =========================================================================
    // UTILITIES
    // =========================================================================

    private static byte[] decodeHexStr(final String hexStr) {
        final int len = hexStr.length() / 2;
        final byte[] result = new byte[len];
        int offset = 0;
        for (int i = 0; i < len; ++i) {
            final String current = hexStr.substring(offset, (offset += 2));
            result[i] = (byte) Integer.parseUnsignedInt(current, 16);
        }
        return result;
    }

    private static String bytesToString(final byte[] data) {
        final StringBuilder sb = new StringBuilder();
        for (final byte b : data) {
            sb.append(ENCODER.get(b & 0xFF));
        }
        return sb.toString();
    }

    private static Map<String,List<String>> getInputs() {
        final Map<String,List<String>> inputs = new LinkedHashMap<String,List<String>>();
        inputs.put("TABLE_INI", INPUT_INI);
        inputs.put("TABLE_XOR", INPUT_XOR);
        inputs.put("TABLE_MIX", INPUT_MIX);
        inputs.put("TABLE_RND", INPUT_RND);
        inputs.put("TABLE_SBX", INPUT_SBX);
        inputs.put("TABLE_TST", INPUT_TST);
        return Collections.unmodifiableMap(inputs);
    }
    
    // =========================================================================
    // MAIN FUNCTION
    // =========================================================================

    public static void main(String[] args) throws Exception {
        final Map<String,List<String>> INPUTS = getInputs();
        for (final String input : INPUTS.keySet()) {
            final List<String> rowData = INPUTS.get(input);
            System.out.printf("private static final List<ByteString> %s = buildTable(\n", input);
            int index = 0;
            boolean first = true;
            for (final String hexStr : rowData) {
                final byte[] bytes = decodeHexStr(hexStr);
                if (!first) {
                    System.out.println(",");
                }
                else {
                    first = false;
                }
                if (index <= 255) {
                    System.out.printf("\t/*%02X*/ \"%s\"", index++, bytesToString(bytes));
                }
                else {
                    System.out.printf("\t/*ZZ*/ \"%s\"", bytesToString(bytes));
                }
            }
            System.out.println("\n);\n");
        }
    }

}
