package org.maachang.dbm.engine ;

import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;

import org.maachang.util.FileUtil;

/**
 * Random-File-I/O処理.
 * 
 * @version 2008/01/15
 * @author masahito suzuki
 * @since MaachangDBM 1.00
 */
public class RandIO {
    
    /**
     * 基本バッファ長.
     */
    private static final int DEFAULT_BUFFER = 512 ;
    
    /**
     * オープンファイル名.
     */
    private String name = null ;
    
    /**
     * ファイルアクセス.
     */
    private RandomAccessFile rfp = null ;
    
    /**
     * ファイルチャネル.
     */
    private FileChannel channel = null ;
    
    /**
     * ByteBuffer.
     */
    private ByteBuffer buffer = null ;
    
    /**
     * ファイルサイズ.
     */
    private int length = -1 ;
    
    /**
     * コンストラクタ.
     */
    private RandIO() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 指定条件を設定して既存ファイルを読み込みます.
     * <BR>
     * @param buffer 対象のバッファサイズを設定します.
     * @param name 読み込み対象のファイル名を設定します.
     * @exception Exception 例外.
     */
    public RandIO( int buffer,String name ) throws Exception {
        if( name == null || ( name = name.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( FileUtil.isFileExists( name ) == false ) {
            throw new IOException( "指定ファイル名[" + name + "]は存在しません" ) ;
        }
        if( buffer <= DEFAULT_BUFFER ) {
            buffer = DEFAULT_BUFFER ;
        }
        this.rfp = new RandomAccessFile( name,"rwd" ) ;
        this.channel = this.rfp.getChannel() ;
        this.name = name ;
        this.buffer = ByteBuffer.allocateDirect( buffer ) ;
        this.length = ( int )this.rfp.length() ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 指定条件を設定して新しいファイルを生成します.
     * <BR>
     * @param buffer 対象のバッファサイズを設定します.
     * @param size 対象のファイルサイズを設定します.
     * @param name 対象のファイル名を設定します.
     * @exception Exception 例外.
     */
    public RandIO( int buffer,long size,String name )
        throws Exception {
        if( size <= 0L ||
            name == null || ( name = name.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.rfp = new RandomAccessFile( name,"rwd" ) ;
        if( FileUtil.isFileExists( name ) == true ) {
            this.rfp.setLength( 0L ) ;
        }
        if( buffer <= DEFAULT_BUFFER ) {
            buffer = DEFAULT_BUFFER ;
        }
        this.rfp.setLength( size ) ;
        this.channel = this.rfp.getChannel() ;
        this.name = name ;
        this.buffer = ByteBuffer.allocateDirect( buffer ) ;
        this.length = ( int )this.rfp.length() ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public synchronized void destroy() {
        if( rfp != null ) {
            try {
                channel.close() ;
            } catch( Exception e ) {
            }
            try {
                rfp.close() ;
            } catch( Exception e ) {
            }
        }
        rfp = null ;
        channel = null ;
        name = null ;
        buffer = null ;
        length = -1 ;
    }
    
    /**
     * 空き容量を増やす.
     * <BR><BR>
     * 新しい空き容量を増やします.
     * <BR>
     * @param size 新しく増やすサイズを設定します.
     * @exception Exception 例外.
     */
    public synchronized void expansion( int size ) throws Exception {
        if( size <= 0 ) {
            return ;
        }
        int newLen = this.length + size ;
        this.rfp.setLength( newLen ) ;
        this.length = newLen ;
    }
    
    /**
     * データ読み込み.
     * <BR><BR>
     * 指定データを読み込みます.
     * <BR>
     * @param out 受け取り対象のバイナリを設定します.
     * @param seek 読み込み開始位置を設定します.
     * @return int 読み込みデータ長が返されます.
     * @exception Exception 例外.
     */
    public synchronized int read( byte[] out,long seek )
        throws Exception {
        return read( out,seek,0,out.length ) ;
    }
    
    /**
     * データ読み込み.
     * <BR><BR>
     * 指定データを読み込みます.
     * <BR>
     * @param out 受け取り対象のバイナリを設定します.
     * @param seek 読み込み開始位置を設定します.
     * @param offset 対象のオフセット値を設定します.
     * @param length 読み込み長さを設定します.
     * @return int 読み込みデータ長が返されます.
     * @exception Exception 例外.
     */
    public synchronized int read( byte[] out,long seek,int offset,int length )
        throws Exception {
        channel.position( seek ) ;
        int ret = 0 ;
        for( ;; ) {
            if( length <= 0 ) {
                break ;
            }
            buffer.clear() ;
            if( length < buffer.capacity() ) {
                buffer.limit( length ) ;
            }
            int len = channel.read( buffer ) ;
            if( len <= 0 ) {
                break ;
            }
            buffer.flip() ;
            buffer.get( out,offset,buffer.limit() ) ;
            offset += len ;
            length -= len ;
            ret += len ;
        }
        return ret ;
    }
    
    /**
     * データ書き込み.
     * <BR><BR>
     * 指定データを書き込みます.
     * <BR>
     * @param in 書き込み対象のバイナリを設定します.
     * @param seek 書き込み開始位置を設定します.
     * @exception Exception 例外.
     */
    public synchronized void write( byte[] in,long seek )
        throws Exception {
        write( in,seek,0,in.length ) ;
    }
    
    /**
     * データ書き込み.
     * <BR><BR>
     * 指定データを書き込みます.
     * <BR>
     * @param in 書き込み対象のバイナリを設定します.
     * @param seek 書き込み開始位置を設定します.
     * @param offset 対象のオフセット値を設定します.
     * @param length 書き込み対象のデータ長が返されます.
     * @exception Exception 例外.
     */
    public synchronized void write( byte[] in,long seek,int offset,int length )
        throws Exception {
        channel.position( seek ) ;
        if( ( int )seek + length >= this.length ) {
            if( ( int )seek >= this.length ) {
                return ;
            }
            length = this.length - ( int )seek ;
        }
        for( ;; ) {
            if( length <= 0 ) {
                break ;
            }
            buffer.clear() ;
            int writeLen = buffer.capacity() ;
            if( length < buffer.capacity() ) {
                writeLen = length ;
            }
            buffer.put( in,offset,writeLen ) ;
            buffer.flip() ;
            channel.write( buffer ) ;
            offset += writeLen ;
            length -= writeLen ;
        }
    }
    
    /**
     * ファイル名を取得.
     * <BR><BR>
     * ファイル名が返されます.
     * <BR>
     * @return String ファイル名が返されます.
     */
    public synchronized String getName() {
        return name ;
    }
    
    /**
     * ファイル名長を取得.
     * <BR><BR>
     * 現在のファイル名長が返されます.
     * <BR>
     * @return long 現在のファイル名長が返されます.
     * @exception Exception 例外.
     */
    public synchronized long getLength()
        throws Exception {
        return length ;
    }
}
