package org.maachang.dbm.engine ;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;

/**
 * Value管理オブジェクト.
 * 
 * @version 2008/01/17
 * @author masahito suzuki
 * @since MaachangDBM 1.00
 */
public class MValue {
    
    /**
     * セクター管理オブジェクト.
     */
    private MSctArray sectors = null ;
    
    /**
     * 同期オブジェクト.
     */
    private final Object sync = new Object() ;
    
    /**
     * コンストラクタ.
     */
    private MValue() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を指定してオブジェクトを生成します.
     * <BR>
     * @param sectors 対象のセクターオブジェクトを設定します.
     * @exception Exception 例外.
     */
    public MValue( MSctArray sectors ) throws Exception {
        if( sectors == null || sectors.isUse() == false ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.sectors = sectors ;
    }
    
    /**
     * デストラクタ.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        synchronized( sync ) {
            if( sectors != null ) {
                sectors.destroy() ;
            }
            sectors = null ;
        }
    }
    
    /**
     * オブジェクト更新.
     * <BR><BR>
     * オブジェクトを更新します.
     * <BR>
     * @exception Exception 例外.
     */
    public void flush() throws Exception {
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            sectors.flush() ;
        }
    }
    
    /**
     * 新しいデータを追加.
     * <BR><BR>
     * 新しいデータを追加します.
     * <BR>
     * @param binary 対象のバイナリを設定します.
     * @return int[] 追加された先頭の[0:ファイルNo,1:ファイルポジション]が返されます.
     * @exception Exception 例外.
     */
    public int[] put( byte[] binary ) throws Exception {
        if( binary == null || binary.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        MSctArray ary = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            ary = sectors ;
        }
        // 書き込みセクターがない場合は、作成.
        int binLen = binary.length ;
        int secLen = sectorLength( binLen ) ;
        if( secLen * 2 >= ary.useSector() ) {
            ary.add() ;
        }
        MSector sector = null ;
        ArrayList<ValueSector> cache = new ArrayList<ValueSector>() ;
        int secCnt = secLen ;
        // ポジション予約.
        for( int i = 0 ; i < secLen ; i ++,secCnt -- ) {
            if( sector == null ) {
                sector = getUseObject( ary,secCnt ) ;
            }
            ValueSector value = new ValueSector() ;
            sector = addSector( ary,sector,value,secCnt ) ;
            cache.add( value ) ;
            ary.removeUseSector() ;
        }
        // 予約ポジションに対して、データ出力.
        int[] ret = new int[ 2 ] ;
        int len = cache.size() ;
        int offset = 0 ;
        ValueSector bef = null ;
        for( int i = 0 ; i < len ; i ++ ) {
            ValueSector now = cache.get( i ) ;
            if( bef == null ) {
                ret[ 0 ] = now.getFileNo() ;
                ret[ 1 ] = now.getPosition() ;
            }
            int oneLen = ( binLen >= MSector.ONE_DATA ) ? MSector.ONE_DATA : binLen ;
            now.write( binary,0,offset,oneLen ) ;
            offset += oneLen ;
            binLen -= oneLen ;
            if( bef != null ) {
                now.setBeforeFileNo( bef.getFileNo() ) ;
                now.setBeforeFilePos( bef.getPosition() ) ;
                bef.setNextFileNo( now.getFileNo() ) ;
                bef.setNextFilePos( now.getPosition() ) ;
                sector = ary.get( bef.getFileNo() ) ;
                sector.put( bef.getPosition(),bef ) ;
            }
            bef = now ;
            now = null ;
        }
        // 一番最後の領域を出力.
        sector = ary.get( bef.getFileNo() ) ;
        sector.put( bef.getPosition(),bef ) ;
        return ret ;
    }
    
    /**
     * 指定データを削除.
     * <BR><BR>
     * 指定されたデータを削除します.
     * <BR>
     * @param fileNo 対象のファイルNoを設定します.
     * @param filePos 対象のファイル項番を設定します.
     * @exception Exception 例外.
     */
    public void remove( int fileNo,int filePos ) throws Exception {
        MSctArray ary = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            ary = sectors ;
        }
        MSector sector = null ;
        for( ;; ) {
            if( sector == null ) {
                sector = ary.get( fileNo ) ;
                if( sector == null ) {
                    throw new IOException( "指定セクター[fileNo:" + fileNo +
                        " filePos:" + filePos + "]は存在しません" ) ;
                }
            }
            ValueSector val = sector.remove( filePos ) ;
            if( val == null || val.getLength() <= 0 ) {
                throw new IOException( "指定条件(fileNo:" + fileNo + " filePos:" +
                    filePos + ")は不正な条件です" ) ;
            }
            ary.addUseSector() ;
            fileNo = val.getNextFileNo() ;
            filePos = val.getNextFilePos() ;
            if( fileNo <= -1 || filePos <= -1 ) {
                break ;
            }
            if( val.getFileNo() != fileNo ) {
                sector = null ;
            }
        }
    }
    
    /**
     * 指定データを取得.
     * <BR><BR>
     * 指定されたデータを取得します.
     * <BR>
     * @param fileNo 対象のファイルNoを設定します.
     * @param filePos 対象のファイル項番を設定します.
     * @return byte[] 対象のデータが返されます.
     * @exception Exception 例外.
     */
    public byte[] get( int fileNo,int filePos ) throws Exception {
        MSctArray ary = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            ary = sectors ;
        }
        MSector sector = null ;
        ByteArrayOutputStream bo = new ByteArrayOutputStream() ;
        for( ;; ) {
            if( sector == null ) {
                sector = ary.get( fileNo ) ;
                if( sector == null ) {
                    throw new IOException( "指定セクター[fileNo:" + fileNo +
                        " filePos:" + filePos + "]は存在しません" ) ;
                }
            }
            ValueSector val = sector.get( filePos ) ;
            if( val == null || val.getLength() <= 0 ) {
                throw new IOException( "指定条件(fileNo:" + fileNo + " filePos:" +
                    filePos + ")は不正な条件です" ) ;
            }
            byte[] b = val.read() ;
            if( b == null ) {
                throw new IOException( "指定条件(fileNo:" + fileNo + " filePos:" +
                    filePos + ")には、データが存在しません" ) ;
            }
            bo.write( b,0,b.length ) ;
            fileNo = val.getNextFileNo() ;
            filePos = val.getNextFilePos() ;
            if( fileNo <= -1 || filePos <= -1 ) {
                break ;
            }
            if( val.getFileNo() != fileNo ) {
                sector = null ;
            }
        }
        bo.flush() ;
        byte[] ret = bo.toByteArray() ;
        bo.close() ;
        bo = null ;
        return ret ;
    }
    
    /**
     * セクター管理オブジェクトを取得.
     * <BR><BR>
     * セクター管理オブジェクトを取得します.
     * <BR>
     * @return MSctArray セクター管理オブジェクトが返されます.
     */
    public MSctArray getArray() {
        MSctArray ret = null ;
        synchronized( sync ) {
            if( check() == false ) {
                ret = null ;
            }
            else {
                ret = sectors ;
            }
        }
        return ret ;
    }
    
    /**
     * このオブジェクトが有効かチェック.
     * <BR><BR>
     * このオブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        boolean ret = false ;
        synchronized( sync ) {
            ret = check() ;
        }
        return ret ;
    }
    
    private boolean check() {
        if( sectors == null ) {
            return false ;
        }
        return true ;
    }
    
    private static final MSector getUseObject( MSctArray ary,int secLen ) throws Exception {
        MSector ret = null ;
        for( ;; ) {
            ret = ary.getMaxUseObject( secLen ) ;
            if( ret == null ) {
                ary.add() ;
            }
            else {
                break ;
            }
        }
        return ret ;
    }
    
    private static final MSector addSector( MSctArray ary,MSector sector,ValueSector value,int secLen )
        throws Exception {
        for( ;; ) {
            if( sector.add( value ) == true ) {
                break ;
            }
            sector = getUseObject( ary,secLen ) ;
        }
        return sector ;
    }
    
    private static final int SECTOR = 0x00001fff ;
    private static final int MASK = ~SECTOR ;
    private static final int SHIFT = 13 ;
    
    private static final int sectorLength( int len ) {
        return ( ( len & MASK ) >> SHIFT ) + ( ( ( len & SECTOR ) != 0 ) ? 1 : 0 ) ;
    }
}
