package org.maachang.dbm.engine ;

import java.io.IOException;

import org.maachang.util.ConvertParam;
import org.maachang.util.FileUtil;

/**
 * Key管理.
 * 
 * @version 2008/01/16
 * @author masahito suzuki
 * @since MaachangDBM 1.00
 */
public class MKey {
    
    /**
     * 1Keyサイズ.
     */
    protected static final int KEY_LENGTH = 528 ;
    
    /**
     * キー長最大値.
     */
    protected static final int MAX_KEY_LENGTH = 512 ;
    
    /**
     * 最大データ件数.
     */
    private static final int MAX_LENGTH = 4250000 ;
    
    /**
     * 要素ファイル数最大値.
     */
    protected static final int MAX_VALUE_FILE_SIZE = 128 ;
    
    /**
     * キーデータオフセット値.
     */
    private static final int OFFSET = 531272 ;
    
    /**
     * データ初期値.
     */
    private static final int DEFAULT_SIZE = 64 ;
    
    /**
     * Hash管理情報.
     */
    private MHash hash = null ;
    
    /**
     * キー管理用ファイル.
     */
    private RandIO fp = null ;
    
    /**
     * 空き情報管理用.
     */
    private Flag space = null ;
    
    /**
     * 同期オブジェクト.
     */
    private final Object sync = new Object() ;
    
    /**
     * コンストラクタ.
     */
    private MKey() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ファイル名を設定してオブジェクトを生成します.
     * <BR>
     * @param hash Hash管理オブジェクトを設定します.
     * @param filename 対象のファイル名を設定します.
     * @exception Exception 例外.
     */
    public MKey( MHash hash,String filename ) throws Exception {
        if( hash == null || hash.isUse() == false ||
            filename == null || ( filename = filename.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        filename = FileUtil.getFullPath( filename ) ;
        boolean isFile = FileUtil.isFileExists( filename ) ;
        RandIO fp = null ;
        Flag space = null ;
        if( isFile == false ) {
            fp = new RandIO( KEY_LENGTH,OFFSET + ( KEY_LENGTH * DEFAULT_SIZE ),filename ) ;
            space = initSpace() ;
            saveSpace( fp,space ) ;
        }
        else {
            fp = new RandIO( KEY_LENGTH,filename ) ;
            space = loadSpace( fp ) ;
        }
        this.space = space ;
        this.hash = hash ;
        this.fp = fp ;
    }
    
    /**
     * デストラクタ.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        synchronized( sync ) {
            if( this.hash != null ) {
                this.hash.destroy() ;
            }
            if( this.fp != null ) {
                try {
                    saveSpace( this.fp,this.space ) ;
                } catch( Exception e ) {
                }
                this.fp.destroy() ;
            }
            this.hash = null ;
            this.fp = null ;
            this.space = null ;
        }
    }
    
    /**
     * オブジェクト更新.
     * <BR><BR>
     * オブジェクトを更新します.
     * <BR>
     * @exception Exception 例外.
     */
    public void flush() throws Exception {
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            this.hash.flush() ;
            saveSpace( this.fp,this.space ) ;
        }
    }
    
    private static final Flag initSpace() throws Exception {
        return new Flag( MAX_LENGTH ) ;
    }
    
    private static final Flag loadSpace( RandIO fp ) throws Exception {
        byte[] b = new byte[ OFFSET ] ;
        fp.read( b,0L ) ;
        return Flag.load( b ) ;
    }
    
    private static final void saveSpace( RandIO fp,Flag space ) throws Exception {
        fp.write( space.save(),0L ) ;
    }
    
    private static final int nowMaxLength( RandIO fp ) throws Exception {
        return ( ( int )( fp.getLength() & 0x00000000ffffffffL ) - OFFSET ) / KEY_LENGTH ;
    }
    
    private final void expansion() throws Exception {
        int now = nowMaxLength( fp ) ;
        if( now <= space.size() ) {
            if( now >= MAX_LENGTH ) {
                throw new IOException( "格納数は最大件数("+MAX_LENGTH+
                    ")を越しているため、追加できません" ) ;
            }
            int addSpace = now ;
            if( addSpace + now >= MAX_LENGTH ) {
                addSpace = MAX_LENGTH - now ;
            }
            fp.expansion( addSpace * KEY_LENGTH ) ;
        }
    }
    
    /**
     * データ設定.
     * <BR><BR>
     * 指定内容のデータを設定します.
     * <BR>
     * @param code Hash値を設定します.
     * @param key 対象のKeyを設定します.
     * @param fileNo 対象のファイルNoを設定します.
     * @param filePos 対象のファイルポジションを設定します.
     * @return int[] 設定前に存在した[0:ファイルNo,1:ファイルポジション]が返されます.<BR>
     *               [null]の場合、情報は存在しません.
     * @exception Exception 例外.
     */
    public int[] put( int code,byte[] key,int fileNo,int filePos )
        throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        expansion() ;
        MKeyChild ch = getMKeyChild( true,code,key ) ;
        if( ch == null || equalsKey( key,ch ) == false ) {
            addByChild( code,ch,key,fileNo,filePos ) ;
            return null ;
        }
        int befFileNo = ch.getValueFileNo() ;
        int befFilePos = ch.getValueFilePos() ;
        ch.setValueFileNo( fileNo ) ;
        ch.setValueFilePos( filePos ) ;
        saveByChild( ch.getPosition(),ch ) ;
        return new int[]{ befFileNo,befFilePos } ;
    }
    
    /**
     * データ削除.
     * <BR><BR>
     * 指定内容のデータを削除します.
     * <BR>
     * @param code Hash値を設定します.
     * @param key 対象のKeyを設定します.
     * @return int[] 削除された[0:ファイルNo,1:ファイルポジション]が返されます.<BR>
     *               [null]の場合、情報は存在しません.
     * @exception Exception 例外.
     */
    public int[] remove( int code,byte[] key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        MKeyChild ch = getMKeyChild( false,code,key ) ;
        if( ch == null ) {
            return null ;
        }
        else {
            removeByChild( code,ch ) ;
        }
        return new int[]{
            ch.getValueFileNo(),ch.getValueFilePos() } ;
    }
    
    /**
     * データ取得.
     * <BR><BR>
     * 指定内容のデータを取得します.
     * <BR>
     * @param code Hash値を設定します.
     * @param key 対象のKeyを設定します.
     * @return int[] [0:ファイルNo,1:ファイルポジション]が返されます.<BR>
     *               [null]の場合、情報は存在しません.
     * @exception Exception 例外.
     */
    public int[] get( int code,byte[] key ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        MKeyChild ch = getMKeyChild( false,code,key ) ;
        if( ch == null ) {
            return null ;
        }
        return new int[]{
            ch.getValueFileNo(),ch.getValueFilePos() } ;
    }
    
    private MKeyChild getMKeyChild( boolean mode,int code,byte[] key ) throws Exception {
        MKeyChild ch = null ;
        if( mode == true ) {
            MKeyChild bef = null ;
            for( ;; ) {
                ch = nextChild( bef,code ) ;
                if( ch == null || equalsKey( key,ch ) == true ) {
                    break ;
                }
                bef = ch ;
            }
            if( ch == null ) {
                ch = bef ;
            }
        }
        else {
            for( ;; ) {
                ch = nextChild( ch,code ) ;
                if( ch == null || equalsKey( key,ch ) == true ) {
                    break ;
                }
            }
        }
        return ch ;
    }
    
    /**
     * 指定項番のキー内容を取得.
     * <BR><BR>
     * 指定項番のキー内容を取得します.
     * <BR>
     * @param next 取得対象のキー内容を設定します.<BR>
     *             [null]を設定した場合、初期位置から、キー内容を取得します.
     * @return NextKey 次キー内容が返されます.<BR>
     *                 [null]の場合、情報は存在しません.
     * @exception Exception 例外.
     */
    public NextKey nextKey( NextKey next ) throws Exception {
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            if( next == null ) {
                next = new NextKey() ;
            }
            int nextPos = space.useNextPos( next.getPos() ) ;
            if( nextPos <= -1 ) {
                next = null ;
            }
            else {
                MKeyChild ch = posByChild( nextPos ) ;
                if( ch == null ) {
                    next = null ;
                }
                else {
                    next = new NextKey() ;
                    next.setKey( ch.getKey() ) ;
                    next.setPos( nextPos ) ;
                }
            }
        }
        return next ;
    }
    
    /**
     * ファイル名を取得.
     * <BR><BR>
     * ファイル名を取得します.
     * <BR>
     * @return String ファイル名が返されます.
     */
    public String getFileName() {
        String ret = null ;
        synchronized( sync ) {
            if( check() == false ) {
                ret = null ;
            }
            else {
                ret = this.fp.getName() ;
            }
        }
        return ret ;
    }
    
    /**
     * 現在のデータ数を取得.
     * <BR><BR>
     * 現在のデータ数が返されます.
     * <BR>
     * @return int 現在のデータ数が返されます.
     */
    public int size() {
        int ret = 0 ;
        synchronized( sync ) {
            if( check() == false ) {
                ret = -1 ;
            }
            else {
                ret = space.size() ;
            }
        }
        return ret ;
    }
    
    /**
     * このオブジェクトが有効かチェック.
     * <BR><BR>
     * このオブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        boolean ret = false ;
        synchronized( sync ) {
            ret = check() ;
        }
        return ret ;
    }
    
    private boolean check() {
        if( hash == null || hash.isUse() == false ||
            fp == null || space == null ) {
            return false ;
        }
        return true ;
    }
    
    private static final int getPosByAddress( int pos ) {
        return OFFSET + ( pos * KEY_LENGTH ) ;
    }
    
    private final MKeyChild posByChild( int pos ) throws Exception {
        if( pos <= -1 || pos >= MHash.MAX_HASH_SIZE ) {
            return null ;
        }
        synchronized( sync ) {
            if( space.getPos( pos ) == false ) {
                return null ;
            }
        }
        byte[] b = new byte[ KEY_LENGTH ] ;
        long addr = ( long )getPosByAddress( pos ) ;
        synchronized( sync ) {
            fp.read( b,addr ) ;
        }
        MKeyChild ret = MKeyChild.load( b ) ;
        ret.setPosition( pos ) ;
        return ret ;
    }
    
    private final MKeyChild firstChild( int code ) throws Exception {
        int p = hash.get( code ) ;
        if( p <= -1 ) {
            return null ;
        }
        MKeyChild ret = posByChild( p ) ;
        if( ret == null ) {
            return null ;
        }
        if( ret.isUseKey() == false ) {
            throw new IOException(
                "不正なHash構成が存在します([firstChild],pos:" +
                getPosByAddress( p ) + ")" ) ;
        }
        return ret ;
    }
    
    private final MKeyChild nextChild( MKeyChild ch,int code ) throws Exception {
        if( ch == null || ch.isUseKey() == false ) {
            return firstChild( code ) ;
        }
        int p = ch.getNextHashPos() ;
        if( p <= -1 ) {
            return null ;
        }
        MKeyChild ret = posByChild( p ) ;
        if( ret == null ) {
            return null ;
        }
        if( ret.isUseKey() == false || ret.getNextHashPos() == p ) {
            int type ;
            if( ret.isUseKey() == false ) {
                type = 1 ;
            }
            else {
                type = 2 ;
            }
            throw new IOException(
                "不正なHash構成が存在します([nextChild],pos:" +
                getPosByAddress( p ) + " type:" + type + ")" ) ;
        }
        return ret ;
    }
    
    private final boolean equalsKey( byte[] key,MKeyChild ch ) throws Exception {
        if( ch == null ) {
            return false ;
        }
        int len = key.length ;
        if( len == ch.getKeyLength() ) {
            byte[] dest = ch.getKey() ;
            for( int i = 0 ; i < len ; i ++ ) {
                if( key[ i ] != dest[ i ] ) {
                    return false ;
                }
            }
            return true ;
        }
        return false ;
    }
    
    private final void saveByChild( int pos,MKeyChild ch ) throws Exception {
        if( ch == null ) {
            throw new IOException( "不正な要素["+getPosByAddress( pos )+
                "]保存を行おうとしました" ) ;
        }
        byte[] b = ch.save() ;
        long addr = ( long )getPosByAddress( pos ) ;
        synchronized( sync ) {
            fp.write( b,addr ) ;
        }
    }
    
    private final void removeByChild( int code,MKeyChild ch ) throws Exception {
        synchronized( sync ) {
            Integer hashCodePos = null ;
            MKeyChild nextChild = posByChild( ch.getNextHashPos() ) ;
            MKeyChild beforeChild = posByChild( ch.getBeforeHashPos() ) ;
            if( nextChild != null || beforeChild != null ) {
                if( nextChild != null ) {
                    if( beforeChild == null ) {
                        nextChild.setBeforeHashPos( -1 ) ;
                        hashCodePos = new Integer( nextChild.getPosition() ) ;
                    }
                    else {
                        nextChild.setBeforeHashPos( beforeChild.getPosition() ) ;
                    }
                }
                if( beforeChild != null ) {
                    if( nextChild == null ) {
                        beforeChild.setNextHashPos( -1 ) ;
                    }
                    else {
                        beforeChild.setNextHashPos( nextChild.getPosition() ) ;
                    }
                }
            }
            else {
                hashCodePos = new Integer( -1 ) ;
            }
            ch.setUseKey( false ) ;
            saveByChild( ch.getPosition(),ch ) ;
            if( nextChild != null ) {
                saveByChild( nextChild.getPosition(),nextChild ) ;
            }
            if( beforeChild != null ) {
                saveByChild( beforeChild.getPosition(),beforeChild ) ;
            }
            if( hashCodePos != null ) {
                if( hashCodePos == -1 ) {
                    hash.remove( code ) ;
                }
                else {
                    hash.put( code,hashCodePos.intValue() ) ;
                }
            }
            space.removePos( ch.getPosition() ) ;
        }
    }
    
    private final void addByChild( int code,MKeyChild ch,byte[] key,int fileNo,int filePos )
        throws Exception {
        MKeyChild newChild = new MKeyChild( -1,-1,fileNo,filePos,key ) ;
        synchronized( sync ) {
            int usePos = space.usePos( 0 ) ;
            if( usePos <= -1 ) {
                throw new IOException( "空き条件は存在しません" ) ;
            }
            newChild.setPosition( usePos ) ;
            if( ch != null ) {
                ch.setNextHashPos( usePos ) ;
                newChild.setBeforeHashPos( ch.getPosition() ) ;
                saveByChild( ch.getPosition(),ch ) ;
            }
            else {
                hash.put( code,newChild.getPosition() ) ;
            }
            saveByChild( newChild.getPosition(),newChild ) ;
            space.setPos( usePos ) ;
        }
    }
    
}

/**
 * 1key情報.
 */
class MKeyChild {
    private boolean useKey = true ;
    private int beforeHashPos = -1 ;
    private int nextHashPos = -1 ;
    private int valueFileNo = -1 ;
    private int valueFilePos = -1 ;
    private int keyLength = -1 ;
    private byte[] key = null ;
    private int position = -1 ;
    
    public MKeyChild() {
        
    }
    
    public MKeyChild( int beforeHashPos,int nextHashPos,int valueFileNo,int valueFilePos,byte[] key ) {
        this.useKey = true ;
        this.beforeHashPos = beforeHashPos ;
        this.nextHashPos = nextHashPos ;
        this.valueFileNo = valueFileNo ;
        this.valueFilePos = valueFilePos ;
        this.key = key ;
        this.keyLength = key.length ;
    }
    
    public void setUseKey( boolean useKey ) {
        this.useKey = useKey ;
    }
    
    public boolean isUseKey() {
        return this.useKey ;
    }
    
    public void setBeforeHashPos( int beforeHashPos ) {
        this.beforeHashPos = beforeHashPos ;
    }
    
    public int getBeforeHashPos() {
        return this.beforeHashPos ;
    }
    
    public void setNextHashPos( int nextHashPos ) {
        this.nextHashPos = nextHashPos ;
    }
    
    public int getNextHashPos() {
        return this.nextHashPos ;
    }
    
    public void setValueFileNo( int valueFileNo ) {
        this.valueFileNo = valueFileNo ;
    }
    
    public int getValueFileNo() {
        return this.valueFileNo ;
    }
    
    public void setValueFilePos( int valueFilePos ) {
        this.valueFilePos = valueFilePos ;
    }
    
    public int getValueFilePos() {
        return this.valueFilePos ;
    }
    
    public int getKeyLength() {
        return this.keyLength ;
    }
    
    public void setKey( byte[] key ) {
        if( key == null || key.length <= 0 ) {
            return ;
        }
        this.key = key ;
        this.keyLength = key.length ;
    }
    
    public byte[] getKey() {
        return this.key ;
    }
    
    public void setPosition( int position ) {
        this.position = position ;
    }
    
    public int getPosition() {
        return position ;
    }
    
    public byte[] save() throws Exception {
        if( this.key == null || this.key.length <= 0 ) {
            throw new IOException( "Key情報は不正です" ) ;
        }
        byte[] ret = new byte[ 16 + this.keyLength ] ;
        int pnt = 0 ;
        ConvertParam.convertBoolean( ret,pnt,this.useKey ) ;
        pnt += 1 ;
        ConvertParam.convertInt( ret,pnt,this.beforeHashPos ) ;
        pnt += 4 ;
        ConvertParam.convertInt( ret,pnt,this.nextHashPos ) ;
        pnt += 4 ;
        ConvertParam.convertByte( ret,pnt,( byte )( this.valueFileNo & 0x000000ff ) ) ;
        pnt += 1 ;
        ConvertParam.convertInt( ret,pnt,this.valueFilePos ) ;
        pnt += 4 ;
        ConvertParam.convertShort( ret,pnt,( short )( keyLength & 0x0000ffff ) ) ;
        pnt += 2 ;
        System.arraycopy( this.key,0,ret,pnt,keyLength ) ;
        return ret ;
    }
    
    public static final MKeyChild load( byte[] binary ) throws Exception {
        if( binary == null || binary.length != MKey.KEY_LENGTH ) {
            throw new IOException( "指定バイナリは不正です" ) ;
        }
        int pnt = 0 ;
        MKeyChild ret = new MKeyChild() ;
        ret.useKey = ConvertParam.convertBoolean( pnt,binary ) ;
        pnt += 1 ;
        ret.beforeHashPos = ConvertParam.convertInt( pnt,binary ) ;
        pnt += 4 ;
        ret.nextHashPos = ConvertParam.convertInt( pnt,binary ) ;
        pnt += 4 ;
        ret.valueFileNo = ( int )( ConvertParam.convertByte( pnt,binary ) & 0x000000ff ) ;
        ret.valueFileNo = ( ( ret.valueFileNo & 0x00000080 ) != 0 ) ? -1 : ret.valueFileNo ;
        pnt += 1 ;
        ret.valueFilePos = ConvertParam.convertInt( pnt,binary ) ;
        pnt += 4 ;
        ret.keyLength = ( int )( ConvertParam.convertShort( pnt,binary ) & 0x0000ffff ) ;
        pnt += 2 ;
        ret.key = new byte[ ret.keyLength ] ;
        System.arraycopy( binary,pnt,ret.key,0,ret.keyLength ) ;
        return ret ;
    }
}
