package org.maachang.dbm.engine ;

import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileChannel.MapMode;
import java.util.Arrays;

import org.maachang.util.ConvertParam;
import org.maachang.util.FileUtil;

/**
 * Hash管理.
 * 
 * @version 2008/01/16
 * @author masahito suzuki
 * @since MaachangDBM 1.00
 */
public class MHash {
    
    /**
     * Hashサイズ.
     */
    public static final int MAX_HASH_SIZE = 65536 ;
    
    /**
     * ファイル.
     */
    private RandomAccessFile fp = null ;
    
    /**
     * mapFile.
     */
    private MappedByteBuffer map = null ;
    
    /**
     * ファイル名.
     */
    private String filename = null ;
    
    /**
     * データ確保用.
     */
    private final byte[] tmp = new byte[ 4 ] ;
    
    /**
     * コンストラクタ.
     */
    private MHash() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ファイル名を設定して、オブジェクトを生成します.
     * <BR>
     * @param filename 対象のファイル名を設定します.
     * @exception Exception 例外.
     */
    public MHash( String filename ) throws Exception {
        if( filename == null || ( filename = filename.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "ファイル名は不正です" ) ;
        }
        filename = FileUtil.getFullPath( filename ) ;
        boolean isFile = FileUtil.isFileExists( filename ) ;
        RandomAccessFile fp = new RandomAccessFile( filename,"rwd" ) ;
        if( isFile == false ) {
            initFile( fp ) ;
        }
        FileChannel channel = fp.getChannel();
        MappedByteBuffer map = channel.map( MapMode.READ_WRITE,0,fp.length() );
        this.fp = fp ;
        this.map = map ;
        this.filename = filename ;
    }
    
    /**
     * デストラクタ.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public synchronized void destroy() {
        if( this.map != null ) {
            try {
                this.map.force() ;
            } catch( Exception e ) {
            }
        }
        if( this.fp != null ) {
            try {
                this.fp.close() ;
            } catch( Exception e ) {
            }
        }
        this.fp = null ;
        this.map = null ;
        this.filename = null ;
    }
    
    /**
     * 強制書き込み.
     * <BR>
     * @exception Exception 例外.
     */
    public synchronized void flush() throws Exception {
        if( this.map != null ) {
            this.map.force() ;
        }
    }
    
    /**
     * Hashポジションを設定.
     * <BR>
     * @param hash 対象のHashコードを設定します.
     * @param pos 対象のポジション値を設定します.
     * @exception Exception 例外.
     */
    public synchronized void put( int hash,int pos ) throws Exception {
        if( check() == false ) {
            throw new IOException( "既に破棄されています" ) ;
        }
        if( hash <= -1 || hash >= MAX_HASH_SIZE ) {
            throw new IllegalArgumentException( "指定Hashコード("+hash+")は範囲外です" ) ;
        }
        this.map.position( hash * 4 ) ;
        this.map.put( ConvertParam.convertInt( pos ) ) ;
    }
    
    /**
     * Hashポジションを削除.
     * <BR>
     * @param hash 対象のHashコードを設定します.
     * @exception Exception 例外.
     */
    public synchronized void remove( int hash ) throws Exception {
        this.put( hash,-1 ) ;
    }
    
    /**
     * Hashポジションを取得.
     * <BR>
     * @param hash 対象のHashコードを設定します.
     * @return int 対象のポジション値が返されます.<BR>
     *              [-1]の場合、無効です.
     * @exception Exception 例外.
     */
    public synchronized int get( int hash ) throws Exception {
        if( check() == false ) {
            throw new IOException( "既に破棄されています" ) ;
        }
        if( hash <= -1 || hash >= MAX_HASH_SIZE ) {
            throw new IllegalArgumentException( "指定Hashコード("+hash+")は範囲外です" ) ;
        }
        this.map.position( hash * 4 ) ;
        byte[] b = tmp ;
        this.map.get( b ) ;
        return ConvertParam.convertInt( 0,b ) ;
    }
    
    /**
     * ファイル名を取得.
     * <BR>
     * @return String ファイル名が返されます.
     */
    public synchronized String getFileName() {
        if( check() == false ) {
            return null ;
        }
        return this.filename ;
    }
    
    /**
     * このオブジェクトが有効かチェック.
     * <BR>
     * @return boolean [true]の場合、有効です.
     */
    public synchronized boolean isUse() {
        return check() ;
    }
    
    private boolean check() {
        if( this.fp == null || this.map == null ) {
            return false ;
        }
        return true ;
    }
    
    /**
     * ファイル初期化.
     */
    private static final void initFile( RandomAccessFile fp ) throws Exception {
        byte[] b = new byte[ MAX_HASH_SIZE * 4 ] ;
        Arrays.fill( b,( byte )0xff ) ;
        fp.write( b ) ;
    }
}
