package org.maachang.dbm.engine ;

import java.io.IOException;
import java.util.Enumeration;

/**
 * MaachangDbmEngine.
 * 
 * @version 2008/01/18
 * @author masahito suzuki
 * @since MaachangDBM 1.00
 */
public class MDbmEngine {
    
    /**
     * Key管理.
     */
    private MKey key = null ;
    
    /**
     * 要素管理.
     */
    private MValue value = null ;
    
    /**
     * 基本同期.
     */
    private final Object sync = new Object() ;
    
    /**
     * Hash同期.
     */
    private Object[] hashSync = null ;
    
    /**
     * コンストラクタ.
     */
    private MDbmEngine() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを生成します.
     * <BR>
     * @param key 対象のキー管理オブジェクトを設定します.
     * @param value 対象の要素管理オブジェクトを設定します.
     * @exception Exception 例外.
     */
    public MDbmEngine( MKey key,MValue value ) throws Exception {
        if( key == null || key.isUse() == false ||
            value == null || value.isUse() == false ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.key = key ;
        this.value = value ;
        int len = MHash.MAX_HASH_SIZE ;
        this.hashSync = new Object[ len ] ;
        for( int i = 0 ; i < len ; i ++ ) {
            this.hashSync[ i ] = new Object() ;
        }
    }
    
    /**
     * デストラクタ.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        synchronized( sync ) {
            if( key != null ) {
                key.destroy() ;
            }
            if( value != null ) {
                value.destroy() ;
            }
            this.key = null ;
            this.value = null ;
        }
    }
    
    /**
     * 強制書き込み.
     * <BR>
     * @exception Exception 例外.
     */
    public void flush() throws Exception {
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "既に破棄されています" ) ;
            }
            key.flush() ;
            value.flush() ;
        }
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( byte[] key,byte[] value ) throws Exception {
        if( key == null || key.length <= 0 ||
            value == null || value.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( key.length > MKey.MAX_KEY_LENGTH ) {
            throw new IllegalArgumentException( "最大キー長["+MKey.MAX_KEY_LENGTH+
                "]を越しています" ) ;
        }
        MKey k = null ;
        MValue v = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "既に破棄されています" ) ;
            }
            k = this.key ;
            v = this.value ;
        }
        int code = FnvHash.fnv16a( key ) ;
        int[] putPos = null ;
        int[] befPos = null ;
        synchronized( hashSync[ code ] ) {
            putPos = v.put( value ) ;
            befPos = k.put( code,key,putPos[ 0 ],putPos[ 1 ] ) ;
            if( befPos != null ) {
                v.remove( befPos[ 0 ],befPos[ 1 ] ) ;
            }
        }
    }
    
    /**
     * 情報を削除.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @exception Exception 例外.
     */
    public void remove( byte[] key ) throws Exception {
        if( key == null || key.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( key.length > MKey.MAX_KEY_LENGTH ) {
            throw new IllegalArgumentException( "最大キー長["+MKey.MAX_KEY_LENGTH+
                "]を越しています" ) ;
        }
        MKey k = null ;
        MValue v = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "既に破棄されています" ) ;
            }
            k = this.key ;
            v = this.value ;
        }
        int code = FnvHash.fnv16a( key ) ;
        int[] valPos = null ;
        synchronized( hashSync[ code ] ) {
            valPos = k.remove( code,key ) ;
            if( valPos != null ) {
                v.remove( valPos[ 0 ],valPos[ 1 ] ) ;
            }
        }
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return byte[] 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public byte[] get( byte[] key ) throws Exception {
        if( key == null || key.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( key.length > MKey.MAX_KEY_LENGTH ) {
            throw new IllegalArgumentException( "最大キー長["+MKey.MAX_KEY_LENGTH+
                "]を越しています" ) ;
        }
        MKey k = null ;
        MValue v = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "既に破棄されています" ) ;
            }
            k = this.key ;
            v = this.value ;
        }
        int code = FnvHash.fnv16a( key ) ;
        int[] valPos = null ;
        byte[] ret = null ;
        synchronized( hashSync[ code ] ) {
            valPos = k.get( code,key ) ;
            if( valPos != null ) {
                ret = v.get( valPos[ 0 ],valPos[ 1 ] ) ;
            }
        }
        return ret ;
    }
    
    /**
     * キー情報一覧を取得.
     * <BR>
     * @param nextKey 次のキー位置を保持するオブジェクトを設定します.<BR>
     *                [null]を設定した場合、初めから取得します.
     * @return NextKey 次のキー情報を格納したオブジェクトが返されます.<BR>
     *                 [null]の場合、それ以上キー情報は存在しません.
     * @exception Exception 例外.
     */
    protected NextKey next( NextKey nextKey ) throws Exception {
        NextKey ret = null ;
        synchronized( sync ) {
            if( check() == false ) {
                throw new IOException( "既に破棄されています" ) ;
            }
            ret = key.nextKey( nextKey ) ;
        }
        return ret ;
    }
    
    /**
     * キー内容を列挙.
     * <BR><BR>
     * @return  Enumeration<byte[]> 列挙オブジェクトが返されます.
     */
    public Enumeration<byte[]> elements() {
        return new MDbmEnumeration( this ) ;
    }
    
    /**
     * 格納情報数を取得.
     * <BR>
     * @return int 格納情報数が返されます.<BR>
     *             [-1]が返された場合、オブジェクトは既に破棄されています.
     */
    public int size() {
        int ret = -1 ;
        synchronized( sync ) {
            if( check() == true ) {
                ret = key.size() ;
            }
        }
        return ret ;
    }
    
    /**
     * キー管理オブジェクトを取得.
     * <BR>
     * @return MKey キー管理オブジェクトを取得します.
     */
    public MKey getMKey() {
        MKey ret = null ;
        synchronized( sync ) {
            if( check() == true ) {
                ret = this.key ;
            }
        }
        return ret ;
    }
    
    /**
     * データ管理オブジェクトを取得.
     * <BR>
     * @return MValue データ管理オブジェクトを取得します.
     */
    public MValue getMValue() {
        MValue ret = null ;
        synchronized( sync ) {
            if( check() == true ) {
                ret = this.value ;
            }
        }
        return ret ;
    }
    
    /**
     * このオブジェクトが有効かチェック.
     * <BR>
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        boolean ret = false ;
        synchronized( sync ) {
            ret = check() ;
        }
        return ret ;
    }
    
    private boolean check() {
        if( key == null || key.isUse() == false ||
            value == null || value.isUse() == false ) {
            return false ;
        }
        return true ;
    }
}
