package org.maachang.dbm.engine ;

import java.io.IOException;
import java.util.Arrays;

import org.maachang.util.ConvertParam;

/**
 * フラグ情報.
 * 
 * @version 2008/01/15
 * @author masahito suzuki
 * @since MaachangDBM 1.00
 */
public class Flag {
    
    /**
     * 残りのファイルスペースマスクを判別する条件.
     */
    protected static final int ETC_MASK = 0x0000003f ;
    
    /**
     * ファイルスペースマスク.
     */
    protected static final int FSS_MASK = ~ETC_MASK ;
    
    /**
     * ビット係数.
     */
    protected static final int RIGHT_SHIFT = 6 ;
    
    /**
     * 有効ファイルスペースサイズ.
     */
    private int use = 0 ;
    
    /**
     * 最大ファイルスペースサイズ.
     */
    private int max = 0 ;
    
    /**
     * 最大ファイルスペースポジション値.
     */
    private int maxPos = 0 ;
    
    /**
     * ファイルスペース位置管理.
     */
    private long[] flags = null ;
    
    /**
     * コンストラクタ.
     */
    private Flag() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 最大ファイルスペース長を設定してオブジェクトを生成します.
     * <BR>
     * @param size 対象の最大ファイルスペース長を設定します.
     * @exception Exception 例外.
     */
    public Flag( int size ) throws Exception {
        if( size <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.use = 0 ;
        this.max = ( ( size & FSS_MASK ) >> RIGHT_SHIFT ) + ( ( ( size & ETC_MASK ) != 0 ) ? 1 : 0 ) ;
        this.maxPos = this.max * 64 ;
        this.flags = new long[ this.max ] ;
        Arrays.fill( this.flags,0L ) ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public synchronized void clear() {
        use = 0 ;
        max = 0 ;
        maxPos = 0 ;
        flags = null ;
    }
    
    /**
     * 領域を拡張.
     * <BR><BR>
     * 領域を拡張します.
     * <BR>
     * @param size 拡張する領域を設定します.
     * @exception Exception 例外.
     */
    public synchronized void expansion( int size )
        throws Exception {
        if( size <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        size += maxPos ;
        int mx = ( ( size & FSS_MASK ) >> RIGHT_SHIFT ) + ( ( ( size & ETC_MASK ) != 0 ) ? 1 : 0 ) ;
        long[] pos = new long[ mx ] ;
        Arrays.fill( pos,0L ) ;
        System.arraycopy( this.flags,0,pos,0,this.max ) ;
        this.max = mx ;
        this.maxPos = mx * 64 ;
        this.flags = pos ;
    }
    
    /**
     * 指定位置のポジションを利用中に設定.
     * <BR><BR>
     * 指定位置のポジションを利用中に設定します.
     * <BR>
     * @param pos 対象のポジションを設定します.
     * @exception Exception 例外.
     */
    public synchronized void setPos( int pos ) throws Exception {
        if( pos < 0 || pos >= maxPos ) {
            throw new IllegalArgumentException( "指定ポジションは不正です" ) ;
        }
        int thisPos = ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ;
        int innerPos = ( pos & ETC_MASK ) ;
        if( ( ( ( 1L << ( long )innerPos ) ) & flags[ thisPos ] ) != 0 ) {
            throw new IOException( "指定項番["+pos+"]は既にONです" ) ;
        }
        flags[ thisPos ] = ( 1L << ( long )innerPos ) | flags[ thisPos ] ;
        use ++ ;
    }
    
    /**
     * 指定位置のポジションを空ける.
     * <BR><BR>
     * 指定位置のポジションを空けます.
     * <BR>
     * @param pos 対象のポジションを設定します.
     * @exception Exception 例外.
     */
    public synchronized void removePos( int pos ) throws Exception {
        if( pos < 0 || pos >= maxPos ) {
            throw new IllegalArgumentException( "指定ポジションは不正です" ) ;
        }
        int thisPos = ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ;
        int innerPos = ( pos & ETC_MASK ) ;
        if( ( ( ( 1L << ( long )innerPos ) ) & flags[ thisPos ] ) == 0 ) {
            throw new IOException( "指定項番["+pos+"]は既にOFFです" ) ;
        }
        flags[ thisPos ] = ( ~( 1L << ( long )innerPos ) ) & flags[ thisPos ] ;
        use -- ;
    }
    
    /**
     * 指定位置のポジションが空いているか取得.
     * <BR><BR>
     * 指定位置のポジションが空いているか取得します.
     * <BR>
     * @param pos 対象のポジションを設定します.
     * @return boolean [false]の場合は、空いています.
     * @exception Exception 例外.
     */
    public synchronized boolean getPos( int pos ) throws Exception {
        if( pos < 0 || pos >= maxPos ) {
            throw new IllegalArgumentException( "指定ポジション["+pos+"]は不正です" ) ;
        }
        return targetPos( pos ) ;
    }
    
    private boolean targetPos( int pos ) {
        int thisPos = ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ;
        int innerPos = ( pos & ETC_MASK ) ;
        return ( ( ( 1L << ( long )innerPos ) ) & flags[ thisPos ] ) != 0 ;
    }
    
    /**
     * 指定ポジションの次の有効な項番を取得.
     * <BR><BR>
     * 指定ポジションの次の有効な項番を取得します.
     * <BR>
     * @param pos 対象のポジションを設定します.
     * @return int 次に有効なポジションが返されます.
     * @exception Exception 例外.
     */
    public synchronized int useNextPos( int pos ) throws Exception {
        if( pos < -1 || pos >= maxPos ) {
            throw new IllegalArgumentException( "指定ポジションは不正です" ) ;
        }
        pos ++ ;
        for( ;; ) {
            if( pos >= maxPos ) {
                return -1 ;
            }
            if( targetPos( pos ) == true ) {
                return pos ;
            }
            pos ++ ;
        }
    }
    
    /**
     * 空いているポジション位置を取得.
     * <BR><BR>
     * 空いているポジション位置を取得します.
     * <BR>
     * @param pos 検索開始ポジションを設定します.
     * @return int 空いているポジション位置が返されます.<BR>
     *             [-1]が返された場合、空き位置は存在しません.
     * @exception Exception 例外.
     */
    public synchronized int usePos( int pos ) throws Exception {
        if( pos < 0 ) {
            throw new IllegalArgumentException( "指定ポジションは不正です" ) ;
        }
        int len = max ;
        for( int i = ( ( pos & FSS_MASK ) >> RIGHT_SHIFT ) ; i < len ; i ++ ) {
            if( flags[ i ] != 0xffffffffffffffffL ) {
                long c = flags[ i ] ;
                for( long j = 0 ; j < 64L ; j += 1L ) {
                    if( ( c & ( 1L << j ) ) == 0L ) {
                        return ( i * 64 ) + ( int )j ;
                    }
                }
            }
        }
        return -1 ;
    }
    
    /**
     * 空いているポジション位置を予約して取得.
     * <BR><BR>
     * 空いているポジション位置を予約して取得します.
     * <BR>
     * @param pos 検索開始ポジションを設定します.
     * @return int 空いているポジション位置が返されます.<BR>
     *             [-1]が返された場合、空き位置は存在しません.
     * @exception Exception 例外.
     */
    public synchronized int usePosBySet( int pos ) throws Exception {
        int u = usePos( pos ) ;
        if( u == -1 ) {
            return -1 ;
        }
        setPos( u ) ;
        return u ;
    }
    
    /**
     * 現在有効件数を取得.
     * <BR><BR>
     * 現在の有効件数を取得します.
     * <BR>
     * @return int 現在の有効件数が返されます.
     */
    public synchronized int size() {
        return use ;
    }
    
    /**
     * 現在の最大有効件数を取得.
     * <BR><BR>
     * 現在の最大有効件数が返されます.
     * <BR>
     * @return int 現在の最大有効件数が返されます.
     */
    public synchronized int maxSize() {
        return maxPos ;
    }
    
    /**
     * 現在データをバイナリに出力.
     * <BR><BR>
     * 現在のデータをバイナリに出力します.
     * <BR>
     * @return byte[] 現在のデータをバイナリに出力します.
     * @exception Exception 例外.
     */
    public synchronized byte[] save() throws Exception {
        int flagLen = flags.length ;
        int len = 4 + 4 + 4 + 4 + ( 8 * flagLen ) ;
        byte[] ret = new byte[ len ] ;
        int pnt = 0 ;
        ConvertParam.convertInt( ret,pnt,len ) ;
        pnt += 4 ;
        ConvertParam.convertInt( ret,pnt,this.use ) ;
        pnt += 4 ;
        ConvertParam.convertInt( ret,pnt,this.max ) ;
        pnt += 4 ;
        ConvertParam.convertInt( ret,pnt,this.maxPos ) ;
        pnt += 4 ;
        for( int i = 0 ; i < flagLen ; i ++ ) {
            ConvertParam.convertLong( ret,pnt,this.flags[ i ] ) ;
            pnt += 8 ;
        }
        return ret ;
    }
    
    /**
     * 指定バイナリをフラグオブジェクトにロード.
     * <BR><BR>
     * 指定バイナリフラグオブジェクトにロードします.
     * <BR>
     * @param binary 対象のバイナリを設定します.
     * @return Flag 対象のフラグ情報が返されます.
     * @exception Exception 例外.
     */
    public static final Flag load( byte[] binary ) throws Exception {
        if( binary == null ) {
            throw new IllegalArgumentException( "指定情報は不正です" ) ;
        }
        Flag ret = new Flag() ;
        int pnt = 4 ;
        ret.use = ConvertParam.convertInt( pnt,binary ) ;
        pnt += 4 ;
        ret.max = ConvertParam.convertInt( pnt,binary ) ;
        pnt += 4 ;
        ret.maxPos = ConvertParam.convertInt( pnt,binary ) ;
        pnt += 4 ;
        int len = ret.max ;
        ret.flags = new long[ len ] ;
        for( int i = 0 ; i < len ; i ++ ) {
            ret.flags[ i ] = ConvertParam.convertLong( pnt,binary ) ;
            pnt += 8 ;
        }
        return ret ;
    }

}
