package org.maachang.dbm ;

import java.util.Enumeration;

import org.maachang.dbm.engine.MDbmEngine;
import org.maachang.dbm.engine.MDbmThread;
import org.maachang.dbm.engine.MHash;
import org.maachang.dbm.engine.MKey;
import org.maachang.dbm.engine.MSctArray;
import org.maachang.dbm.engine.MValue;
import org.maachang.util.FileUtil;

/**
 * MaachangDbm.
 * 
 * @version 2008/01/18
 * @author masahito suzuki
 * @since MaachangDBM 1.00
 */
public class MaachangDbm {
    
    /**
     * Hash基本管理ファイル名.
     */
    private static final String BASE_FILE = "base.hash" ;
    
    /**
     * Key管理ファイル名.
     */
    private static final String KEY_FILE = "key.hash" ;
    
    /**
     * DbmEngine.
     */
    private MDbmEngine engine = null ;
    
    /**
     * 監視スレッド.
     */
    private MDbmThread thread = null ;
    
    /**
     * データ展開用ディレクトリ.
     */
    private String directory = null ;
    
    /**
     * コンストラクタ.
     */
    private MaachangDbm() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 指定条件から、MaachangDbmを生成します.
     * <BR>
     * @param directory MaachangDbm展開先のディレクトリ名を設定します.
     * @exception Exception 例外.
     */
    public MaachangDbm( String directory ) throws Exception {
        if( directory == null || ( directory = directory.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        directory = FileUtil.getFullPath( directory ) ;
        if( directory.endsWith( "\\" ) || directory.endsWith( "/" ) ) {
            directory = directory.substring( 0,directory.length() - 1 ) ;
        }
        if( FileUtil.isDirExists( directory ) == false ) {
            FileUtil.mkdirs( directory ) ;
        }
        MHash hash = new MHash( new StringBuilder().
            append( directory ).append( FileUtil.FILE_SPACE ).append( BASE_FILE ).toString() ) ;
        MKey key = new MKey( hash,new StringBuilder().
            append( directory ).append( FileUtil.FILE_SPACE ).append( KEY_FILE ).toString() ) ;
        MValue value = new MValue( new MSctArray( directory ) ) ;
        MDbmEngine engine = new MDbmEngine( key,value ) ;
        MDbmThread thread = new MDbmThread( engine ) ;
        this.engine = engine ;
        this.thread = thread ;
        this.directory = directory ;
    }
    
    /**
     * デストラクタ.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        if( thread != null ) {
            thread.destroy() ;
        }
        thread = null ;
        if( engine != null ) {
            engine.destroy() ;
        }
        engine = null ;
        directory = null ;
    }
    
    /**
     * 強制書き込み.
     * <BR>
     * @exception Exception 例外.
     */
    public void flush() throws Exception {
        if( isUse() == false ) {
            return ;
        }
        if( thread != null ) {
            thread.destroyTiming() ;
        }
        if( engine != null ) {
            engine.flush() ;
        }
    }
    
    /**
     * 情報を設定.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @param value 対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( byte[] key,byte[] value ) throws Exception {
        if( isUse() == false ) {
            return ;
        }
        engine.put( key,value ) ;
        thread.setTiming() ;
    }
    
    /**
     * 情報を削除.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @exception Exception 例外.
     */
    public void remove( byte[] key ) throws Exception {
        if( isUse() == false ) {
            return ;
        }
        engine.remove( key ) ;
        thread.setTiming() ;
    }
    
    /**
     * 情報を取得.
     * <BR>
     * @param key 対象のキー情報を設定します.
     * @return byte[] 対象の情報が返されます.
     * @exception Exception 例外.
     */
    public byte[] get( byte[] key ) throws Exception {
        if( isUse() == false ) {
            return null ;
        }
        return engine.get( key ) ;
    }
    
    /**
     * キー内容を列挙.
     * <BR><BR>
     * @return  Enumeration<byte[]> 列挙オブジェクトが返されます.
     */
    public Enumeration<byte[]> elements() {
        if( isUse() == false ) {
            return null ;
        }
        return engine.elements() ;
    }
    
    /**
     * 格納情報数を取得.
     * <BR>
     * @return int 格納情報数が返されます.<BR>
     *             [-1]が返された場合、オブジェクトは既に破棄されています.
     */
    public int size() {
        if( isUse() == false ) {
            return -1 ;
        }
        return engine.size() ;
    }
    
    /**
     * MaachangDbm展開ディレクトリを取得.
     * <BR>
     * @return String MaachangDbm展開ディレクトリ名が返されます.
     */
    public String getDirectory() {
        if( isUse() == false ) {
            return null ;
        }
        return directory ;
    }
    
    /**
     * MaachangDbmエンジンを取得.
     * <BR>
     * @return MDbmEngine MaachangDbmエンジンが返されます.
     */
    public MDbmEngine getEngine() {
        if( isUse() == false ) {
            return null ;
        }
        return engine ;
    }
    
    /**
     * このオブジェクトが有効かチェック.
     * <BR>
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        if( engine == null ) {
            return false ;
        }
        return engine.isUse() ;
    }
}
