/*
 * @(#)SequenceLong.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.index.core.element ;

import java.io.ByteArrayOutputStream;

import org.maachang.util.ConvertBinary;

/**
 * ID発行オブジェクト.
 *
 * @version 2007/05/19
 * @author  masahito suzuki
 * @since   MaachangIndex 1.00
 */
public class SequenceLong {
    
    /**
     * バイナリ長.
     */
    public static final int BINARY_LENGTH = 24 ;
    
    /**
     * 取得IDなし.
     */
    public static final long NOT_ID = 0x8000000000000000L ;
    
    /**
     * MAX-ID.
     */
    private long maxId = 0L ;
    
    /**
     * 開始値.
     */
    private long startId = 0L ;
    
    /**
     * 管理ID.
     */
    private long id = 0L ;
    
    /**
     * 同期オブジェクト.
     */
    private Object sync = null ;
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 任意の最大値を用いて情報を生成します.
     */
    public SequenceLong()
    {
        this.sync = new Object() ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        try{
            this.clear() ;
        }catch( Exception t ){
        }
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     */
    public void create()
    {
        try{
            this.create( 0,Long.MAX_VALUE ) ;
        }catch( Exception e ){
        }
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param start 対象の開始値を設定します.
     * @param max 管理を行うIDの幅(MAX値)を設定します.
     * @exception InputException 入力例外.
     */
    public final void create( long start,long max )
        throws Exception
    {
        
        if( start < 0L || max <= 1L ||
            start >= ( max - 1L ) || max > Long.MAX_VALUE ){
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        this.clear() ;
        
        try{
            synchronized( this.sync ){
                this.maxId = max ;
                this.startId = start ;
                this.id = start ;
            }
        }catch( NullPointerException nul ){
        }
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        try{
            synchronized( this.sync ){
                this.id = this.startId ;
            }
        }catch( NullPointerException nul ){
            this.id = this.startId ;
        }
    }
    
    /**
     * 一意なID項番を取得.
     * <BR><BR>
     * 一意なID項番を取得します.
     * <BR>
     * @return long 一意なID項番が返されます.
     */
    public final long getId()
    {
        long ret ;
        
        try{
            synchronized( this.sync ){
                
                ret = this.id ;
                
                if( ret >= this.maxId ){
                    this.id = this.startId ;
                }
                else{
                    this.id = ret + 1L ;
                }
                
            }
        }catch( NullPointerException nul ){
            ret = SequenceLong.NOT_ID ;
        }
        
        return ret ;
    }
    
    /**
     * 次に発行するID情報を設定.
     * <BR><BR>
     * 次に発行するID情報を設定します.
     * <BR>
     * @param id 次に発行されるID位置を設定します.
     */
    public final void setNextId( long id )
    {
        try{
            synchronized( this.sync ){
                this.id = ( id <= this.startId || id >= this.maxId ) ?
                    ( ( id <= this.startId ) ? this.startId : this.maxId ) : id ;
            }
        }catch( NullPointerException nul ){
        }
        
    }
    
    /**
     * 設定ID開始番号の取得.
     * <BR><BR>
     * 設定されているID開始番号を取得します.
     * <BR>
     * @return long 設定されているID開始番号が返されます.
     */
    public final long getStartId()
    {
        long ret ;
        
        try{
            synchronized( this.sync ){
                ret = this.startId ;
            }
        }catch( NullPointerException nul ){
            ret = SequenceLong.NOT_ID ;
        }
        
        return ret ;
    }
    
    /**
     * 設定ID幅(MAX値)を取得します.
     * <BR><BR>
     * 設定されているID幅(MAX値)を取得します.
     * <BR>
     * @return long 設定されているID幅(MAX値)が返されます.
     */
    public final long getMaxID()
    {
        long ret ;
        
        try{
            synchronized( this.sync ){
                ret = this.maxId ;
            }
        }catch( NullPointerException nul ){
            ret = SequenceLong.NOT_ID ;
        }
        
        return ret ;
    }
    
    /**
     * オブジェクト内容をバイナリ化.
     * <BR><BR>
     * オブジェクト内容をバイナリ化します.
     * <BR>
     * @return byte[] バイナリ情報が返されます.
     * @exception Exception 例外.
     */
    public byte[] toBinary() throws Exception {
        ByteArrayOutputStream bo = new ByteArrayOutputStream() ;
        bo.write( ConvertBinary.convertLong( maxId ) ) ;
        bo.write( ConvertBinary.convertLong( startId ) ) ;
        bo.write( ConvertBinary.convertLong( id ) ) ;
        return bo.toByteArray() ;
    }
    
    /**
     * 対象バイナリからオブジェクトを生成.
     * <BR><BR>
     * 対象バイナリからオブジェクトを生成します.
     * <BR>
     * @param bin 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public void toObject( byte[] bin ) throws Exception {
        maxId = ConvertBinary.convertLong( 0,bin ) ;
        startId = ConvertBinary.convertLong( 8,bin ) ;
        id = ConvertBinary.convertLong( 16,bin ) ;
    }

}
