/*
 * @(#)SearchHistoryCache.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.index.core.element ;

import org.maachang.util.FileUtil;

/**
 * SearchHistoryCache.
 *
 * @version 2007/06/28
 * @author  masahito suzuki
 * @since   MaachangIndex 1.00
 */
public class SearchHistoryCache {
    
    /**
     * ヒストリサイズ : 最小値.
     */
    private static final int MIN_HISTRORY_SIZE = 64 ;
    
    /**
     * ヒストリサイズ : 最大値.
     */
    private static final int MAX_HISTRORY_SIZE = 32768 ;
    
    /**
     * キー名格納用.
     */
    private CacheMap<String,SearchValue> map = null ;
    
    /**
     * ヒストリ格納用ディレクトリ.
     */
    private String historyDir = null ;
    
    /**
     * コンストラクタ.
     */
    private SearchHistoryCache() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 最大ヒストリ容量を設定してオブジェクトを生成.
     * <BR>
     * @param max 最大ヒストリ容量を設定します.
     * @param indexDir 対象のヒストリディレクトリ名を設定します.
     */
    public SearchHistoryCache( int max,String historyDir ) {
        if( max <= MIN_HISTRORY_SIZE ) {
            max = MIN_HISTRORY_SIZE ;
        }
        else if( max >= MAX_HISTRORY_SIZE ) {
            max = MAX_HISTRORY_SIZE ;
        }
        this.historyDir = historyDir ;
        this.map =  new CacheMap<String,SearchValue>( max ) ;
    }
    
    /**
     * 全てのHistroyファイルを削除.
     * <BR><BR>
     * 全てのHistoryファイルを削除します.
     * <BR>
     * @param dir 対象のディレクトリ名を設定します.
     */
    public static final void removeHistorySerializable( String dir ) {
        if( dir.endsWith( "/" ) == true || dir.endsWith( "\\" ) == true ) {
            dir = dir.substring( 0,dir.length()-1 ) ;
        }
        if( FileUtil.isDirExists( dir ) == true ) {
            FileUtil.removeFile( dir ) ;
            try {
                FileUtil.mkdirs( dir ) ;
            } catch( Exception e ) {
            }
        }
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 情報を削除します.
     */
    public void clear() {
        map.clear() ;
    }
    
    /**
     * キャッシュ情報を全て出力.
     * <BR><BR>
     * キャッシュ情報を全て出力します.
     */
    public void flush() {
        int len = this.map.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            SearchValue bean = this.map.remove() ;
            if( bean.isSerialize() == false ) {
                bean.sort() ;
                bean.putSerialize( this.historyDir ) ;
            }
        }
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 情報を追加します.
     * <BR>
     * @param bean 対象のSearchBean情報を設定します.
     * @exception IllegalArgumentException 入力例外.
     */
    public void add( SearchValue bean ) throws IllegalArgumentException {
        if( bean == null || bean.getSearchString() == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( this.map.isKey( bean.getSearchString() ) == true ) {
            return ;
        }
        SearchValue s = this.map.put( bean.getSearchString(),bean ) ;
        if( s != null && s.isSerialize() == false ) {
            s.sort() ;
            s.putSerialize( this.historyDir ) ;
        }
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象キーコードに対する情報を取得します.
     * <BR>
     * @param string 対象のキーコードを設定します.
     * @return SearchValue 情報が返されます.
     */
    public SearchValue get( String string ) {
        if( string == null || ( string = string.trim() ).length() <= 0 ) {
            return null ;
        }
        SearchValue ret = this.map.get( string ) ;
        if( ret == null ) {
            ret = SearchValue.getSerializable( this.historyDir,string ) ;
            if( ret != null ) {
                this.add( ret ) ;
            }
        }
        return ret ;
    }
    
    /**
     * ヒストリディレクトリ名を取得.
     * <BR><BR>
     * ヒストリディレクトリ名を取得します.
     * <BR>
     * @return String ヒストリディレクトリ名が返されます.
     */
    public String getHistoryDirectory() {
        return this.historyDir ;
    }
    
    /**
     * 最大ヒストリ数を取得.
     * <BR><BR>
     * 最大ヒストリ数を取得します.
     * <BR>
     * @return int 最大ヒストリ数が返されます.
     */
    public int getMaxHistory() {
        return this.map.getMaxLength() ;
    }
}

