/*
 * @(#)SearchBaseCache.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.index.core.element ;


/**
 * SearchBaseCache.
 *
 * @version 2007/06/28
 * @author  masahito suzuki
 * @since   MaachangIndex 1.00
 */
public class SearchBaseCache {
    
    /**
     * キャッシュサイズ : 最小値.
     */
    private static final int MIN_CACHE_SIZE = 0 ;
    
    /**
     * キャッシュサイズ : 最大値.
     */
    private static final int MAX_CACHE_SIZE = 4096 ;
    
    /**
     * キー名格納用.
     */
    private CacheMap<Long,IndexBase> map = null ;
    
    /**
     * IndexBase格納用ディレクトリ.
     */
    private String baseDir = null ;
    
    /**
     * コンストラクタ.
     */
    private SearchBaseCache() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 最大キャッシュ容量を設定してオブジェクトを生成.
     * <BR>
     * @param maxCache 最大キャッシュ容量を設定します.
     * @param baseDir 対象のIndexBaseディレクトリ名を設定します.
     */
    public SearchBaseCache( int maxCache,String baseDir ) {
        if( maxCache <= MIN_CACHE_SIZE ) {
            maxCache = MIN_CACHE_SIZE ;
        }
        else if( maxCache >= MAX_CACHE_SIZE ) {
            maxCache = MAX_CACHE_SIZE ;
        }
        this.baseDir = baseDir ;
        if( maxCache > 0 ) {
            this.map =  new CacheMap<Long,IndexBase>( maxCache ) ;
        }
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 情報を削除します.
     */
    public void clear() {
        if( map != null ) {
            map.clear() ;
        }
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象キーコードに対する情報を取得します.
     * <BR>
     * @param key 対象のキーコードを設定します.
     * @return IndexChip 情報が返されます.
     */
    public IndexBase get( Long id ) {
        return get( id,true,true ) ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象キーコードに対する情報を取得します.
     * <BR>
     * @param key 対象のキーコードを設定します.
     * @param part 部分判別用情報を取得する場合には[true]を設定します.
     * @param value 内容情報を取得する場合には[true]を設定します.
     * @return IndexChip 情報が返されます.
     */
    public IndexBase get( Long id,boolean part,boolean value ) {
        if( id == null ) {
            return null ;
        }
        IndexBase ret = null ;
        if( map != null ) {
            ret = this.map.get( id ) ;
            if( ret == null ) {
                ret = IndexBase.getSerializable( this.baseDir,id,part,value ) ;
                if( ret != null ) {
                    this.map.put( ret.getId(),ret ) ;
                }
            }
        }
        else {
            ret = IndexBase.getSerializable( this.baseDir,id,part,value ) ;
        }
        return ret ;
    }
    
    /**
     * IndexBaseディレクトリ名を取得.
     * <BR><BR>
     * IndexBaseディレクトリ名を取得します.
     * <BR>
     * @return String IndexBaseディレクトリ名が返されます.
     */
    public String getIndexBaseDirectory() {
        return this.baseDir ;
    }
    
    /**
     * 最大キャッシュ容量を取得.
     * <BR><BR>
     * 最大キャッシュ容量を取得します.
     * <BR>
     * @return int 最大キャッシュ容量が返されます.
     */
    public int getMaxCache() {
        if( this.map != null ) {
            return this.map.getMaxLength() ;
        }
        else {
            return 0 ;
        }
    }
    
    /**
     * 非キャッシュ条件チェック.
     * <BR><BR>
     * 非キャッシュ条件であるかチェックします.
     * <BR>
     * @return boolean 非キャッシュ条件であるか返されます.<BR>
     *                 [true]の場合は、非キャッシュ状態です.
     */
    public boolean isNoCache() {
        return ( this.map != null ) ? true : false ;
    }
}

