/*
 * @(#)IndexKeyCache.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.index.core.element ;

/**
 * IndexKeyCache.
 *
 * @version 2007/06/28
 * @author  masahito suzuki
 * @since   MaachangIndex 1.00
 */
public class IndexKeyCache {
    
    /**
     * キャッシュサイズ : 最小値.
     */
    private static final int MIN_CACHE_SIZE = 128 ;
    
    /**
     * キャッシュサイズ : 最大値.
     */
    private static final int MAX_CACHE_SIZE = 2048 ;
    
    /**
     * キャッシュ管理.
     */
    private CacheMap<String,IndexKey> map = null ;
    
    /**
     * IndexBase格納用ディレクトリ.
     */
    private String baseDir = null ;
    
    /**
     * 生成モード.
     */
    private boolean mode = false ;
    
    /**
     * コンストラクタ.
     */
    private IndexKeyCache() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 最大キャッシュ容量を設定してオブジェクトを生成.
     * <BR>
     * @param mode インデックス生成時の場合は[true]を設定します.
     * @param maxCache 最大キャッシュ容量を設定します.
     * @param baseDir 対象のIndexBaseディレクトリ名を設定します.
     */
    public IndexKeyCache( boolean mode,int maxCache,String baseDir ) {
        if( maxCache <= MIN_CACHE_SIZE ) {
            maxCache = MIN_CACHE_SIZE ;
        }
        else if( maxCache >= MAX_CACHE_SIZE ) {
            maxCache = MAX_CACHE_SIZE ;
        }
        this.mode = mode ;
        this.baseDir = baseDir ;
        this.map =  new CacheMap<String,IndexKey>( maxCache ) ;
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 情報を削除します.
     */
    public void clear() {
        map.clear() ;
    }
    
    /**
     * 格納中のキー内容をファイル出力.
     * <BR><BR>
     * 格納中のキー内容をファイルに出力します.
     */
    public void flush() {
        int len ;
        if( map != null && ( len = this.map.size() ) > 0 ) {
            for( int i = 0 ; i < len ; i ++ ) {
                IndexKey key = this.map.remove() ;
                if( key != null ) {
                    if( key.size() <= 0 ) {
                        IndexKey.removeSerializable( baseDir,key.getKey(),
                            key.getNumber() ) ;
                    }
                    else {
                        try {
                            key.putSerializable( baseDir ) ;
                        } catch( Exception e ) {
                        }
                    }
                }
            }
        }
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象条件の内容を取得します.
     * <BR>
     * @param key 対象のキーコードを設定します.
     * @return IndexKey 情報が返されます.
     */
    public IndexKey get( String key,int number ) {
        String keyCode = new StringBuilder().
            append( number ).append( "_" ).append( key ).toString() ;
        IndexKey ret = this.map.get( keyCode ) ;
        if( ret == null ) {
            try {
                ret = IndexKey.getSerializable( this.baseDir,key,number ) ;
                if( ret != null ) {
                    IndexKey k = this.map.put( keyCode,ret ) ;
                    if( k != null && mode == true ) {
                        k.putSerializable( baseDir ) ;
                    }
                }
                else {
                    ret = new IndexKey( key,number ) ;
                    IndexKey k = this.map.put( keyCode,ret ) ;
                    if( k != null && mode == true ) {
                        k.putSerializable( baseDir ) ;
                    }
                }
            } catch( Exception e ) {
                ret = null ;
            }
        }
        return ret ;
    }
    
    /**
     * 最大キャッシュ容量を取得.
     * <BR><BR>
     * 最大キャッシュ容量を取得します.
     * <BR>
     * @return int 最大キャッシュ容量が返されます.
     */
    public int getMaxCache() {
        return this.map.getMaxLength() ;
    }
    
}

