/*
 * @(#)IndexChipPos.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.index.core.element ;

import java.io.BufferedOutputStream;

import org.maachang.util.ConvertBinary;
import org.maachang.util.IntArray;
import org.maachang.util.ReadBinary;

/**
 * Indexポジション.
 *
 * @version 2007/06/28
 * @author  masahito suzuki
 * @since   MaachangIndex 1.00
 */
public class IndexChipPos {
    
    /**
     * バイナリ長.
     */
    public static final int BINARY_LENGTH = 20 ;
    
    /**
     * 次ポジションN-Gram:Long値.
     */
    private long nextNGram = 0L ;
    
    /**
     * ポジション開始値.
     */
    private int startPos = -1 ;
    
    /**
     * ポジション終了値.
     */
    private int endPos = -1 ;
    
    /**
     * ポジション位置.
     */
    private IntArray list = null ;
    
    /**
     * ポジション位置(Read).
     */
    private int[] readList = null ;
    
    /**
     * コンストラクタ.
     */
    protected IndexChipPos() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを生成します.
     * <BR>
     * @param nextNGram 次ポジションN-Gramを設定します.
     * @param position ポジション情報を設定します.
     */
    public IndexChipPos( char[] nextNGram ) {
        this.nextNGram = convertNextNGram( nextNGram ) ;
        this.list = new IntArray() ;
    }
    
    /**
     * 次ポジションN-Gramをlong値で取得.
     * <BR><BR>
     * 次ポジションN-Gramをlong値で取得します.
     * <BR>
     * @return long 次ポジションN-Gramをlong値で取得します.
     */
    public long getNextNGram() {
        return nextNGram ;
    }
    
    /**
     * ポジション情報を設定.
     * <BR><BR>
     * ポジション情報を設定します.
     * <BR>
     * @param pos 対象のポジションを設定します.
     */
    public void setPosition( int pos ) {
        if( list != null ) {
            if( startPos == -1 || startPos >= pos ) {
                startPos = pos ;
            }
            if( endPos == -1 || endPos <= pos ) {
                endPos = pos ;
            }
            list.add( pos ) ;
        }
    }
    
    /**
     * ポジション情報を取得.
     * <BR><BR>
     * ポジション情報を取得します.
     * <BR>
     * @param no 対象の項番を設定します.
     * @return int ポジション情報が返されます.
     */
    public int getPosition( int no ) {
        if( list != null ) {
            if( no < 0 || no >= list.size() ) {
                return -1 ;
            }
            return list.get( no ) ;
        }
        else if( readList != null ) {
            if( no < 0 || no >= readList.length ) {
                return -1 ;
            }
            return readList[ no ] ;
        }
        return 0 ;
    }
    
    /**
     * ポジション数を取得.
     * <BR><BR>
     * ポジション数を取得します.
     * <BR>
     * @return int ポジション数が返されます.
     */
    public int size() {
        if( list != null ) {
            return list.size() ;
        }
        else if( readList != null ) {
            return readList.length ;
        }
        return 0 ;
    }
    
    /**
     * ポジション範囲をチェック.
     * <BR><BR>
     * 指定ポジションの範囲をチェックします.
     * <BR>
     * @param nowPos 対象の指定ポジションを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、指定ポジションは範囲内です.
     */
    public boolean isBitweenPos( int nowPos ) {
        return ( startPos <= nowPos && endPos >= nowPos ) ? true : false ;
    }
    
    /**
     * 次NGram値を数値変換.
     * <BR><BR>
     * 次NGram値を数値変換します.
     * <BR>
     * @param ngram 対象のNGram値を設定します.
     * @return long 数値化された情報が返されます.
     */
    public static final long convertNextNGram( char[] ngram ) {
        int len ;
        long ret = 0L ;
        if( ngram != null && ( len = ngram.length ) > 0 ) {
            long pos = 0L ;
            for( int i = 0 ; i < len ; i ++ ) {
                ret |= ( long )( ngram[ i ] & 0x0000ffff ) << pos ;
                pos += 16 ;
            }
        }
        return ret ;
    }
    
    /**
     * 指定オブジェクトを文字列変換.
     * <BR><BR>
     * 指定オブジェクトを文字列変換します.
     * <BR>
     * @return String 文字列変換された情報が返されます.
     */
    public String toString() {
        StringBuilder buf =  new StringBuilder().
            append( " startPos:" ).
            append( startPos ).
            append( " endPos:" ).
            append( endPos ).
            append( " pos-len:" ).
            append( list.size() ) ;
        if( list != null ) {
            int len = list.size() ;
            for( int i = 0 ; i < len ; i ++ ) {
                buf.append( " pos[" ).
                    append( i ).
                    append( "]:" ).
                    append( list.get( i ) ) ;
            }
        }
        else if( readList != null ) {
            int len = readList.length ;
            for( int i = 0 ; i < len ; i ++ ) {
                buf.append( " pos[" ).
                    append( i ).
                    append( "]:" ).
                    append( readList[ i ] ) ;
            }
        }
        return buf.toString() ;
    }
    
    /**
     * オブジェクトをファイル化.
     * <BR><BR>
     * オブジェクトをファイル化します.
     * <BR>
     * @param mode 対象モードを設定します.<BR>
     *             [true]を設定した場合、Insert処理を表します.<BR>
     *             [false]を設定した場合、Search処理を表します.
     * @param bo ファイルオブジェクトを設定します.
     * @exceptino Exception 例外.
     */
    protected void toFile( boolean mode,BufferedOutputStream bo )
        throws Exception {
        int len ;
        bo.write( ConvertBinary.convertLong( nextNGram ) ) ;
        bo.write( ConvertBinary.convertInt( startPos ) ) ;
        bo.write( ConvertBinary.convertInt( endPos ) ) ;
        if( list != null ) {
            len = list.size() ;
            if( len > 0 ) {
                bo.write( ConvertBinary.convertInt( len ) ) ;
                for( int i = 0 ; i < len ; i ++ ) {
                    bo.write( ConvertBinary.convertInt( list.get( i ) ) ) ;
                }
            }
            else {
                bo.write( ConvertBinary.convertInt( 0 ) ) ;
            }
        }
        else if( readList != null ) {
            len = readList.length ;
            if( len > 0 ) {
                bo.write( ConvertBinary.convertInt( len ) ) ;
                for( int i = 0 ; i < len ; i ++ ) {
                    bo.write( ConvertBinary.convertInt( readList[ i ] ) ) ;
                }
            }
            else {
                bo.write( ConvertBinary.convertInt( 0 ) ) ;
            }
        }
        else {
            bo.write( ConvertBinary.convertInt( 0 ) ) ;
        }
    }
    
    /**
     * ファイルからオブジェクトに変換.
     * <BR><BR>
     * ファイルからオブジェクトに変換します.
     * <BR>
     * @param mode 対象モードを設定します.<BR>
     *             [true]を設定した場合、Insert処理を表します.<BR>
     *             [false]を設定した場合、Search処理を表します.
     * @param r ファイルオブジェクトを設定します.
     * @exceptino Exception 例外.
     */
    protected void toObject( boolean mode,ReadBinary r )
        throws Exception {
        nextNGram = ConvertBinary.convertLong( 0,r.getBinary( 8 ) ) ;
        startPos = ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ;
        endPos = ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ;
        int len = ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ;
        if( mode == true ) {
            if( len > 0 ) {
                list = new IntArray( len ) ;
                readList = null ;
                for( int i = 0 ; i < len ; i ++ ) {
                    list.add( ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ) ;
                }
            }
            else {
                list = new IntArray() ;
                readList = null ;
            }
        }
        else {
            if( len > 0 ) {
                list = null ;
                readList = new int[ len ] ;
                for( int i = 0 ; i < len ; i ++ ) {
                    readList[ i ] = ( ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ) ;
                }
            }
            else {
                list = null ;
                readList = new int[ 0 ] ;
            }
        }
    }
}

