/*
 * @(#)IndexChipById.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.index.core.element ;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;

import org.maachang.util.ConvertBinary;
import org.maachang.util.ObjectArray;
import org.maachang.util.ReadBinary;

/**
 * IndexChip内のID情報.
 *
 * @version 2007/06/28
 * @author  masahito suzuki
 * @since   MaachangIndex 1.00
 */
public class IndexChipById {
    /**
     * IndexBase-Id.
     */
    private Long id = null ;
    
    /**
     * 得点.
     */
    private int score = 0 ;
    
    /**
     * 得点係数.
     */
    private int scoreCoefficient = 0 ;
    
    /**
     * ポジション管理情報.
     */
    private ObjectArray list = null ;
    
    /**
     * ポジション管理情報(read).
     */
    private IndexChipPos[] readList = null ;
    
    /**
     * コンストラクタ.
     */
    protected IndexChipById() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * インデックス子要素を生成します.
     * <BR>
     * @param id 対象のIndexBaseBean-Idを設定します.
     * @param scoreCoefficient 得点係数を設定します.
     * @exception IllegalArgumentException 入力例外.
     */
    public IndexChipById( Long id,int scoreCoefficient )
        throws IllegalArgumentException {
        if( id == null || id.longValue() <= 0L ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.id = id ;
        this.scoreCoefficient = scoreCoefficient ;
        this.score = 1 ;
        this.list = new ObjectArray() ;
    }
    
    /**
     * IndexBase-Idを取得.
     * <BR><BR>
     * IndexBase-Idを取得します.
     * <BR>
     * @return Long ID情報が返されます.
     */
    public Long getId() {
        return this.id ;
    }
    
    /**
     * 得点係数を取得.
     * <BR><BR>
     * 得点係数を取得します.
     * <BR>
     * @return int 得点係数が返されます.
     */
    public int getScoreCoefficient() {
        return this.scoreCoefficient ;
    }
    
    /**
     * 得点を取得.
     * <BR><BR>
     * 得点を取得します.
     * <BR>
     * @return int 得点が返されます.
     */
    public int getScore() {
        return this.score ;
    }
    
    /**
     * 得点を１インクリメント.
     * <BR><BR>
     * 得点を１インクリメントします.
     */
    public void addScore() {
        this.score ++ ;
    }
    
    /**
     * ポジション情報を追加.
     * <BR><BR>
     * ポジション情報を追加します.
     * <BR>
     * @param pos 対象のポジション情報を設定します.
     */
    public void addPos( IndexChipPos pos ) {
        if( list != null ) {
            if( pos != null ) {
                list.add( pos ) ;
            }
        }
    }
    
    /**
     * ポジション情報を取得.
     * <BR><BR>
     * ポジション情報を取得します.
     * <BR>
     * @param no 対象の項番を設定します.
     * @return IndexChipPos ポジション情報が返されます.
     */
    public IndexChipPos getPos( int no ) {
        if( list != null ) {
            if( no < 0 || no >= list.size() ) {
                return null ;
            }
            return ( IndexChipPos )list.get( no ) ;
        }
        else if( readList != null ) {
            if( no < 0 || no >= readList.length ) {
                return null ;
            }
            return readList[ no ] ;
        }
        return null ;
    }
    
    /**
     * ポジション情報を取得.
     * <BR><BR>
     * ポジション情報を次NGram情報で取得します.
     * <BR>
     * @param ngram 対象のNGram情報を設定します.
     * @return IndexChipPos ポジション情報が返されます.
     */
    public IndexChipPos getPos( char[] ngram ) {
        if( ngram == null ) {
            return null ;
        }
        return this.getPos( IndexChipPos.convertNextNGram( ngram ) ) ;
    }
    
    /**
     * ポジション情報を取得.
     * <BR><BR>
     * ポジション情報を次NGram情報で取得します.
     * <BR>
     * @param n 対象のNGram情報を設定します.
     * @return IndexChipPos ポジション情報が返されます.
     */
    public IndexChipPos getPos( long n ) {
        if( list != null ) {
            int len = list.size() ;
            for( int i = 0 ; i < len ; i ++ ) {
                IndexChipPos pos = ( IndexChipPos )list.get( i ) ;
                if( n == pos.getNextNGram() ) {
                    return pos ;
                }
            }
        }
        else if( readList != null ) {
            int len = readList.length ;
            for( int i = 0 ; i < len ; i ++ ) {
                IndexChipPos pos = readList[ i ] ;
                if( n == pos.getNextNGram() ) {
                    return pos ;
                }
            }
        }
        return null ;
    }
    
    /**
     * ポジション情報数を取得.
     * <BR><BR>
     * ポジション情報数が返されます.
     * <BR>
     * @return int ポジション情報数が返されます.
     */
    public int size() {
        if( list != null ) {
            return list.size() ;
        }
        else if( readList != null ) {
            return readList.length ;
        }
        return 0 ;
    }
    
    /**
     * ソート条件で利用.
     * <BR><BR>
     * ソート条件で利用します.
     * <BR>
     * @param o 対象のオブジェクト(IndexChild)を設定します.
     * @return int 結果が返されます.
     */
    public int compareTo( Object o ) {
        if( o == null || ( o instanceof IndexChipById ) == false ) {
            return 0 ;
        }
        
        return this.score - ( ( IndexChipById )o ).score ;
    }
    
    /**
     * 指定オブジェクトを文字列変換.
     * <BR><BR>
     * 指定オブジェクトを文字列変換します.
     * <BR>
     * @return String 文字列変換された情報が返されます.
     */
    public String toString() {
        StringBuilder buf =  new StringBuilder().
            append( "id:" ).
            append( id ).
            append( " score:" ).
            append( score ).
            append( " scoreCoefficient:" ).
            append( scoreCoefficient ) ;
        if( list != null ) {
            int len = list.size() ;
            buf.append( " pos:" ).append( "(" ).append( len ).append( "): " ) ;
            for( int i = 0 ; i < len ; i ++ ) {
                if( i != 0 ) { buf.append( "," ) ; }
                buf.append( list.get( i ) ) ;
            }
        }
        else if( readList != null ) {
            int len = readList.length ;
            buf.append( " pos:" ).append( "(" ).append( len ).append( "): " ) ;
            for( int i = 0 ; i < len ; i ++ ) {
                if( i != 0 ) { buf.append( "," ) ; }
                buf.append( readList[ i ] ) ;
            }
        }
        return buf.toString() ;
    }
    
    /**
     * オブジェクトをファイル化.
     * <BR><BR>
     * オブジェクトをファイル化します.
     * <BR>
     * @param mode 対象モードを設定します.<BR>
     *             [true]を設定した場合、Insert処理を表します.<BR>
     *             [false]を設定した場合、Search処理を表します.
     * @param bo ファイルオブジェクトを設定します.
     * @exceptino Exception 例外.
     */
    protected void toFile( boolean mode,BufferedOutputStream bo )
        throws Exception {
        bo.write( ConvertBinary.convertLong( id.longValue() ) ) ;
        bo.write( ConvertBinary.convertInt( score ) ) ;
        bo.write( ConvertBinary.convertInt( scoreCoefficient ) ) ;
        if( list != null ) {
            int len = list.size() ;
            if( len > 0 ) {
                bo.write( ConvertBinary.convertInt( len ) ) ;
                for( int i = 0 ; i < len ; i ++ ) {
                    ( ( IndexChipPos )list.get( i ) ).toFile( mode,bo ) ;
                }
            }
            else {
                bo.write( ConvertBinary.convertInt( 0 ) ) ;
            }
        }
        else if( readList != null ) {
            int len = readList.length ;
            if( len > 0 ) {
                bo.write( ConvertBinary.convertInt( len ) ) ;
                for( int i = 0 ; i < len ; i ++ ) {
                    readList[ i ].toFile( mode,bo ) ;
                }
            }
            else {
                bo.write( ConvertBinary.convertInt( 0 ) ) ;
            }
        }
        else {
            bo.write( ConvertBinary.convertInt( 0 ) ) ;
        }
    }
    
    /**
     * ファイルからオブジェクトに変換.
     * <BR><BR>
     * ファイルからオブジェクトに変換します.
     * <BR>
     * @param mode 対象モードを設定します.<BR>
     *             [true]を設定した場合、Insert処理を表します.<BR>
     *             [false]を設定した場合、Search処理を表します.
     * @param bi ファイルオブジェクトを設定します.
     * @exceptino Exception 例外.
     */
    protected void toObject( boolean mode,BufferedInputStream bi )
        throws Exception {
        ReadBinary r = new ReadBinary( bi ) ;
        id = new Long( ConvertBinary.convertLong( 0,r.getBinary( 8 ) ) ) ;
        score = ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ;
        scoreCoefficient = ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ;
        int len = ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ;
        if( mode == true ) {
            if( len > 0 ) {
                this.list = new ObjectArray( len ) ;
                this.readList = null ;
                for( int i = 0 ; i < len ; i ++ ) {
                    IndexChipPos pos = new IndexChipPos() ;
                    pos.toObject( mode,r ) ;
                    list.add( pos ) ;
                }
            }
            else {
                this.list = new ObjectArray() ;
                this.readList = null ;
            }
        }
        else {
            if( len > 0 ) {
                this.list = null ;
                this.readList = new IndexChipPos[ len ] ;
                for( int i = 0 ; i < len ; i ++ ) {
                    IndexChipPos pos = new IndexChipPos() ;
                    pos.toObject( mode,r ) ;
                    readList[ i ] = pos ;
                }
            }
            else {
                this.list = null ;
                this.readList = null ;
            }
        }
    }
}

