/*
 * @(#)IndexChip.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.index.core.element ;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.HashMap;
import java.util.Set;

import org.maachang.util.ConvertBinary;
import org.maachang.util.FileUtil;
import org.maachang.util.ReadBinary;

/**
 * IndexChip.
 *
 * @version 2007/06/28
 * @author  masahito suzuki
 * @since   MaachangIndex 1.00
 */
public class IndexChip {
    
    /**
     * インデックス文字情報.
     */
    private String indexCode = null ;
    
    /**
     * インデックスMap管理.
     */
    private HashMap<Long,IndexChipById> map = null ;
    
    /**
     * 更新フラグ.
     */
    private boolean updateFlag = false ;
    
    /**
     * コンストラクタ.
     */
    private IndexChip() {
        updateFlag = false ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * インデックス化文字列を設定して、オブジェクトを生成します.
     * <BR>
     * @param indexCode 対象のインデックスコードを設定します.
     * @exception IllegalArgumentException 入力例外.
     */
    public IndexChip( String indexCode )
        throws IllegalArgumentException {
        if( indexCode == null || indexCode.length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.indexCode = indexCode ;
        this.map = new HashMap<Long,IndexChipById>() ;
        updateFlag = false ;
    }
    
    /**
     * シリアライズ化.
     * <BR><BR>
     * シリアライズ化します.
     * <BR>
     * @param mode 対象モードを設定します.<BR>
     *             [true]を設定した場合、Insert処理を表します.<BR>
     *             [false]を設定した場合、Search処理を表します.
     * @param indexDir 対象のディレクトリ名を設定します.
     * @param bean 対象のIndexBeanを設定します.
     */
    public void putSerialize( boolean mode,String indexDir ) {
        BufferedOutputStream o = null ;
        String out = indexDir ;
        if( out.endsWith( "/" ) == true || out.endsWith( "\\" ) == true ) {
            out = out.substring( 0,out.length()-1 ) ;
        }
        out = new StringBuilder().append( out ).append( FileUtil.FILE_SPACE ).
            append( this.indexCode ).toString() ;
        try{
            o = new BufferedOutputStream( new FileOutputStream( out ) ) ;
            toFile( mode,o ) ;
            updateFlag = false ;
        }catch( Exception e ){
            e.printStackTrace() ;
        }finally{
            try{
                o.close() ;
            }catch( Exception e ){
            }
            o = null ;
        }
    }
    
    /**
     * シリアライズから情報を取得.
     * <BR><BR>
     * シリアライズされた情報から、IndexChipを取得します.
     * <BR>
     * @param mode 対象モードを設定します.<BR>
     *             [true]を設定した場合、Insert処理を表します.<BR>
     *             [false]を設定した場合、Search処理を表します.
     * @param indexDir 対象のIディレクトリ名を設定します.
     * @param key 対象のn-gramキー名を設定します.
     * @return IndexChip シリアライズから復元したIndexBeanオブジェクトが返されます.
     */
    public static final IndexChip getSerializable( boolean mode,String indexDir,String key ) {
        BufferedInputStream in = null ;
        String out = indexDir ;
        if( out.endsWith( "/" ) == true || out.endsWith( "\\" ) == true ) {
            out = out.substring( 0,out.length()-1 ) ;
        }
        out = new StringBuilder().append( out ).append( FileUtil.FILE_SPACE ).
            append( key ).toString() ;
        if( FileUtil.isFileExists( out ) == false ) {
            return null ;
        }
        try{
            in = new BufferedInputStream(
                new FileInputStream( out )
            ) ;
            IndexChip ch = new IndexChip() ;
            ch.toObject( mode,in ) ;
            return ch ;
        }catch( Exception e ){
            e.printStackTrace() ;
        }finally{
            try{
                in.close() ;
            }catch( Exception e ){
            }
            in = null ;
        }
        return null ;
    }
    
    /**
     * 指定IndexChipファイルを削除.
     * <BR><BR>
     * 指定IndexChipファイルを削除します.
     * <BR>
     * @param indexDir 対象のディレクトリ名を設定します.
     * @param key 対象のn-gramキー名を設定します.
     */
    public static final void removeSerializable( String indexDir,String key ) {
        if( indexDir.endsWith( "/" ) == true || indexDir.endsWith( "\\" ) == true ) {
            indexDir = indexDir.substring( 0,indexDir.length()-1 ) ;
        }
        indexDir = new StringBuilder().append( indexDir ).append( FileUtil.FILE_SPACE ).
            append( key ).toString() ;
        if( FileUtil.isFileExists( indexDir ) == true ) {
            FileUtil.removeFile( indexDir ) ;
        }
    }
    
    /**
     * 指定したIndexBaseBeanの内容をこのオブジェクトに追加.
     * <BR><BR>
     * 指定したIndexBaseBeanの内容をこのオブジェクトに追加します.
     * <BR>
     * @param indexBaseBean 追加対象のIndexBaseBeanを設定します.
     * @param scoreCoefficient 得点係数を設定します.
     * @param nextNGram 対象の次N-Gram情報を設定します.
     * @param position 対象の追加ポジションを設定します.
     * @exceptino IllegalArgumentException 入力例外.
     */
    public void addBase( IndexBase indexBaseBean,int scoreCoefficient,char[] nextNGram,int position )
        throws IllegalArgumentException {
        if( indexBaseBean == null || position < 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        IndexChipById ch = null ;
        ch = map.get( indexBaseBean.getId() ) ;
        if( ch != null ) {
            ch.addScore() ;
            IndexChipPos pos = ch.getPos( nextNGram ) ;
            if( pos != null ) {
                pos.setPosition( position ) ;
            }
            else {
                pos = new IndexChipPos( nextNGram ) ;
                pos.setPosition( position ) ;
                ch.addPos( pos ) ;
            }
        }
        else {
            ch = new IndexChipById( indexBaseBean.getId(),scoreCoefficient ) ;
            IndexChipPos pos = new IndexChipPos( nextNGram ) ;
            pos.setPosition( position ) ;
            ch.addPos( pos ) ;
            ch.addScore() ;
            map.put( indexBaseBean.getId(),ch ) ;
        }
        updateFlag = true ;
    }
    
    /**
     * 指定したID情報を削除.
     * <BR><BR>
     * 指定したID情報を削除します.
     * <BR>
     * @param id 削除対象のIDを設定します.
     */
    public void removeBase( Long id ) {
        map.remove( id ) ;
        updateFlag = true ;
    }
    
    /**
     * 指定IDの内容を取得.
     * <BR><BR>
     * 指定IDの内容を取得します.
     * <BR>
     * @param id 対象のIDを設定します.
     * @return IndexChidById 対象情報が返されます.
     */
    public IndexChipById get( Long id ) {
        return map.get( id ) ;
    }
    
    /**
     * 管理しているIndexBaseBean数を取得.
     * <BR><BR>
     * 管理しているIndexBaseBean数を取得します.
     * <BR>
     * @return int 管理しているIndexBaseBean数が返されます.
     */
    public int size() {
        return map.size() ;
    }
    
    /**
     * ID一覧を取得.
     * <BR><BR>
     * ID一覧を取得します.
     * <BR>
     * @return Long[] ID一覧が返されます.
     */
    public Long[] getKeys() {
        Set set = map.keySet() ;
        if( set != null ) {
            int len ;
            Object[] objs = set.toArray() ;
            if( objs != null && ( len = objs.length ) > 0 ) {
                Long[] ret = new Long[ len ] ;
                System.arraycopy( objs,0,ret,0,len ) ;
                return ret ;
            }
        }
        return null ;
    }
    
    /**
     * データ一覧を取得.
     * <BR><BR>
     * データ一覧を取得します.
     * <BR>
     * @return IndexChipById[] 情報一覧が返されます.
     */
    public IndexChipById[] getEntrys() {
        Set set = map.entrySet() ;
        if( set != null ) {
            int len ;
            Object[] objs = set.toArray() ;
            if( objs != null && ( len = objs.length ) > 0 ) {
                IndexChipById[] ret = new IndexChipById[ len ] ;
                System.arraycopy( objs,0,ret,0,len ) ;
                return ret ;
            }
        }
        return null ;
    }
    
    /**
     * IndexCodeを取得.
     * <BR><BR>
     * IndexCodeを取得します.
     * <BR>
     * @return String IndexCodeが返されます.
     */
    public String getIndexCode() {
        return indexCode ;
    }
    
    /**
     * 更新フラグ取得.
     * <BR><BR>
     * 更新フラグを取得します.
     * <BR>
     * @return boolean 更新フラグが返されます.<BR>
     *                 [true]が返された場合、更新されています.<BR>
     *                 [false]が返された場合、更新されていません.
     */
    public boolean isUpdate() {
        return updateFlag ;
    }
    
    /**
     * オブジェクトをファイル化.
     */
    private void toFile( boolean mode,BufferedOutputStream bo )
        throws Exception {
        byte[] bin = null ;
        bin = ConvertBinary.convertString( indexCode ) ;
        bo.write( ConvertBinary.convertInt( bin.length ) ) ;
        bo.write( bin ) ;bin = null ;
        int len = map.size() ;
        bo.write( ConvertBinary.convertInt( len ) ) ;
        if( len > 0 ) {
            Object[] objs = map.keySet().toArray() ;
            for( int i = 0 ; i < len ; i ++ ) {
                map.get( ( Long )objs[ i ] ).toFile( mode,bo ) ;
            }
        }
        else {
            bo.write( ConvertBinary.convertInt( 0 ) ) ;
        }
        bo.flush() ;
    }
    
    /**
     * ファイルからオブジェクトに変換.
     */
    private void toObject( boolean mode,BufferedInputStream bi )
        throws Exception {
        int len ;
        ReadBinary r = new ReadBinary( bi ) ;
        len = ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ;
        indexCode = ConvertBinary.convertString( 0,len,r.getBinary( len ) ) ;
        len = ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ;
        if( len > 0 ) {
            this.map = new HashMap<Long,IndexChipById>( len+1 ) ;
            for( int i = 0 ; i < len ; i ++ ) {
                IndexChipById c = new IndexChipById() ;
                c.toObject( mode,bi ) ;
                map.put( c.getId(),c ) ;
            }
        }
        else {
            this.map = new HashMap<Long,IndexChipById>() ;
        }
    }
    
}
