/*
 * @(#)IndexBase.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.index.core.element ;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.ArrayList;

import org.maachang.index.core.IndexUtil;
import org.maachang.util.ConvertBinary;
import org.maachang.util.FileUtil;
import org.maachang.util.ReadBinary;
import org.maachang.util.StringUtil;

/**
 * IndexBase.
 *
 * @version 2007/06/28
 * @author  masahito suzuki
 * @since   MaachangIndex 1.00
 */
public class IndexBase {
    
    /**
     * インデックスバージョン.
     */
    private long indexVersion = 0L ;
    
    /**
     * ID.
     */
    private Long id = null ;
    
    /**
     * この情報を表す名前.
     */
    private String name = null ;
    
    /**
     * 内容.
     */
    private String value = null ;
    
    /**
     * 要素.
     */
    private String element = null ;
    
    /**
     * 部分判別用.
     */
    private String part = null ;
    
    /**
     * キー内容.
     */
    private String[] keys = null ;
    
    /**
     * 削除フラグ.
     */
    private boolean deleteFlag = false ;
    
    /**
     * 非更新モード.
     */
    private boolean notUpdateFlag = false ;
    
    /**
     * コンストラクタ.
     */
    private IndexBase() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を設定します.
     * <BR>
     * @param name この情報を表す一意名を設定します.
     * @param value 対象の内容を設定します.
     * @param element この情報を表す要素情報を設定します.
     * @param keys 対象のキー内容を設定します.
     * @exception IllegalArgumentException 入力例外.
     */
    public IndexBase( String name,String value,String element,String[] keys )
        throws IllegalArgumentException {
        
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ||
            value == null || value.length() <= 0 ) {
            throw new IllegalArgumentException( "必須項目が存在しません" ) ;
        }
        
        this.name = name ;
        this.value = value ;
        this.element = element ;
        this.part = partCode( value ) ;
        if( keys == null || keys.length <= 0 ) {
            this.keys = null ;
        }
        else {
            this.keys = keys ;
        }
        deleteFlag = false ;
    }
    
    /**
     * シリアライズ化.
     * <BR><BR>
     * IndexBaseBeanをシリアライズ化します.
     * <BR>
     * @param dir 対象のディレクトリを設定します.
     * @param bean 対象のIndexBaseBeanを設定します.
     */
    public void putSerialize( String dir ) {
        if( notUpdateFlag == true ) {
            return ;
        }
        BufferedOutputStream o = null ;
        String out = dir ;
        if( out.endsWith( "/" ) == true || out.endsWith( "\\" ) == true ) {
            out = out.substring( 0,out.length()-1 ) ;
        }
        out = new StringBuilder().append( out ).append( FileUtil.FILE_SPACE ).
            append( this.id ).toString() ;
        try{
            o = new BufferedOutputStream( new FileOutputStream( out ) ) ;
            toFile( o ) ;
        }catch( Exception e ){
        }finally{
            try{
                o.close() ;
            }catch( Exception e ){
            }
            o = null ;
        }
    }
    
    /**
     * シリアライズから情報を取得.
     * <BR><BR>
     * シリアライズされた情報から、IndexBaseBeanを取得します.
     * <BR>
     * @param dir 対象のディレクトリ名を設定します.
     * @param id 対象のIDを設定します.
     * @param part 部分判別用情報を取得する場合には[true]を設定します.
     * @param value 内容情報を取得する場合には[true]を設定します.
     * @return IndexChip シリアライズから復元したIndexBeanオブジェクトが返されます.
     */
    public static final IndexBase getSerializable( String dir,Long id,boolean part,boolean value ) {
        BufferedInputStream in = null ;
        String out = dir ;
        if( out.endsWith( "/" ) == true || out.endsWith( "\\" ) == true ) {
            out = out.substring( 0,out.length()-1 ) ;
        }
        out = new StringBuilder().append( out ).append( FileUtil.FILE_SPACE ).
            append( id ).toString() ;
        if( FileUtil.isFileExists( out ) == false ) {
            return null ;
        }
        try{
            in = new BufferedInputStream(
                new FileInputStream( out )
            ) ;
            IndexBase ret = new IndexBase() ;
            ret.toObject( in,part,value ) ;
            return ret ;
        }catch( Exception e ){
        }finally{
            try{
                in.close() ;
            }catch( Exception e ){
            }
            in = null ;
        }
        return null ;
    }
    
    /**
     * 指定IndexBaseBeanファイルを削除.
     * <BR><BR>
     * 指定IndexBaseBeanファイルを削除します.
     * <BR>
     * @param dir 対象のディレクトリ名を設定します.
     * @param id 対象のIDを設定します.
     */
    public static final void removeSerializable( String dir,Long id ) {
        if( dir.endsWith( "/" ) == true || dir.endsWith( "\\" ) == true ) {
            dir = dir.substring( 0,dir.length()-1 ) ;
        }
        String file = new StringBuilder().append( dir ).append( FileUtil.FILE_SPACE ).
            append( id ).toString() ;
        if( FileUtil.isFileExists( file ) == true ) {
            FileUtil.removeFile( file ) ;
        }
    }
    
    /**
     * IDを設定.
     * <BR><BR>
     * IDを設定します.
     * <BR>
     * @param id IDをセットします.
     */
    public void setId( long id ) {
        this.id = new Long( id ) ;
    }
    
    /**
     * IDを設定.
     * <BR><BR>
     * IDを設定します.
     * <BR>
     * @param id IDをセットします.
     */
    public void setId( Long id ) {
        this.id = id ;
    }
    
    /**
     * インデックスバージョンを設定.
     * <BR><BR>
     * インデックスバージョンを設定します.
     * <BR>
     * @param indexVersion インデックスバージョンを設定します.
     */
    public void setIndexVersion( long indexVersion ) {
        this.indexVersion = indexVersion ;
    }
    
    /**
     * インデックスバージョンを取得.
     * <BR><BR>
     * インデックスバージョンを取得します.
     * <BR>
     * @return long インデックスバージョンが返されます.
     */
    public long getIndexVersion() {
        return this.indexVersion ;
    }
    
    /**
     * IDを取得.
     * <BR><BR>
     * IDを取得します.
     * <BR>
     * @return Long IDが返されます.
     */
    public Long getId() {
        return this.id ;
    }
    
    /**
     * この情報を表す一意名を取得.
     * <BR><BR>
     * この情報を表す一意名を取得します.
     * <BR>
     * @return String この情報を表す一意名が返されます.
     */
    public String getName() {
        return this.name ;
    }
    
    /**
     * 内容を取得.
     * <BR><BR>
     * 内容を取得します.
     * <BR>
     * @return String 内容が返されます.
     */
    public String getValue() {
        return this.value ;
    }
    
    /**
     * 部分判別用データを取得.
     * <BR><BR>
     * 部分判別用データを取得します.
     * <BR>
     * @return String 部分判別用データが返されます.
     */
    public String getPart() {
        return this.part ;
    }
    
    /**
     * 要素情報を取得.
     * <BR><BR>
     * 要素情報を取得します.
     * <BR>
     * @return String 要素情報が返されます.
     */
    public String getElement() {
        return this.element ;
    }
    
    /**
     * キー内容群を取得.
     * <BR><BR>
     * キー内容群を取得します.
     * <BR>
     * @return String キー内容が返されます.
     */
    public String[] getKeys() {
        return this.keys ;
    }
    
    /**
     * １つのキー内容を取得.
     * <BR><BR>
     * １つのキー内容を取得します.
     * <BR>
     * @param no 取得項番を設定します.
     * @return String キー内容が返されます.<BR>
     *                [null]が返された場合、情報は存在しません.
     */
    public String getKey( int no ) {
        if( no < 0 || this.keys == null || this.keys.length <= no ) {
            return null ;
        }
        return this.keys[ no ] ;
    }
    
    /**
     * キー内容数を取得.
     * <BR><BR>
     * キー内容数を取得します.
     * <BR>
     * @return int キー内容数が返されます.
     */
    public int getKeySize() {
        return ( this.keys != null ) ? this.keys.length : 0 ;
    }
    
    /**
     * オブジェクトを文字列化.
     * <BR><BR>
     * オブジェクトを文字列化します.
     * <BR>
     * @return String オブジェクトを文字列化します.
     */
    
    /**
     * 削除処理.
     * <BR><BR>
     * データを削除します.
     */
    public void delete() {
        deleteFlag = true ;
    }
    
    /**
     * 削除条件を取得.
     * <BR><BR>
     * 削除条件を取得します.
     * <BR>
     * @return boolean 削除条件が返されます.
     */
    public boolean isDelete() {
        return deleteFlag ;
    }
    
    /**
     * 非更新フラグを取得.
     * <BR><BR>
     * 非更新フラグを取得します.
     * <BR>
     * @return boolean 非更新フラグが返されます.
     */
    public boolean isNoUpdateFlag() {
        return notUpdateFlag ;
    }
    
    /**
     * 部分判別用データ作成処理.
     */
    private static final String partCode( String string ) {
        if( string == null || ( string = string.trim() ).length() <= 0 ) {
            return "" ;
        }
        string = IndexUtil.cutTag( string ) ;
        string = IndexUtil.invalidityString( string ) ;
        ArrayList<String> lst = StringUtil.cutString( string,IndexUtil.CUT_CODE+IndexUtil.PARSE_CODE ) ;
        if( lst == null && lst.size() <= 0 ) {
            return "" ;
        }
        int len = lst.size() ;
        StringBuilder buf = new StringBuilder( string.length() ) ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( lst.get( i ) == null || lst.get( i ).length() <= 0 ) {
                continue ;
            }
            buf.append( lst.get( i ) ) ;
            buf.append( " " ) ;
        }
        return buf.toString() ;
    }
    
    /**
     * オブジェクトをファイル化.
     */
    private void toFile( BufferedOutputStream bo )
        throws Exception {
        byte[] bin = null ;
        bo.write( ConvertBinary.convertLong( indexVersion ) ) ;
        bo.write( ConvertBinary.convertLong( id.longValue() ) ) ;
        bin = ConvertBinary.convertString( name ) ;
        bo.write( ConvertBinary.convertInt( bin.length ) ) ;
        bo.write( bin ) ;bin = null ;
        bin = ConvertBinary.convertString( value ) ;
        bo.write( ConvertBinary.convertInt( bin.length ) ) ;
        bo.write( bin ) ;bin = null ;
        if( element != null && element.length() > 0 ) {
            bin = ConvertBinary.convertString( element ) ;
            bo.write( ConvertBinary.convertInt( bin.length ) ) ;
            bo.write( bin ) ;bin = null ;
        }
        else {
            bo.write( ConvertBinary.convertInt( 0 ) ) ;
        }
        bin = ConvertBinary.convertString( part ) ;
        bo.write( ConvertBinary.convertInt( bin.length ) ) ;
        bo.write( bin ) ;bin = null ;
        if( keys != null && keys.length > 0 ) {
            int len = keys.length ;
            bo.write( ConvertBinary.convertInt( len ) ) ;
            for( int i = 0 ; i < len ; i ++ ) {
                if( keys[ i ] != null && keys[ i ].length() > 0 ) {
                    bin = ConvertBinary.convertString( keys[ i ] ) ;
                    bo.write( ConvertBinary.convertInt( bin.length ) ) ;
                    bo.write( bin ) ;bin = null ;
                }
                else {
                    bo.write( ConvertBinary.convertInt( 0 ) ) ;
                }
            }
        }
        else {
            bo.write( ConvertBinary.convertInt( 0 ) ) ;
        }
        bo.flush() ;
    }
    
    /**
     * ファイルからオブジェクトに変換.
     */
    private void toObject( BufferedInputStream bi,boolean partFlag,boolean valueFlag )
        throws Exception {
        int len ;
        ReadBinary r = new ReadBinary( bi ) ;
        indexVersion = ConvertBinary.convertLong( 0,r.getBinary( 8 ) ) ;
        id = new Long( ConvertBinary.convertLong( 0,r.getBinary( 8 ) ) ) ;
        len = ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ;
        name = ConvertBinary.convertString( 0,len,r.getBinary( len ) ) ;
        len = ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ;
        if( valueFlag == false ) {
            r.getBinary( len ) ;
            value = null ;
        }
        else {
            value = ConvertBinary.convertString( 0,len,r.getBinary( len ) ) ;
        }
        len = ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ;
        if( len > 0 ) {
            element = ConvertBinary.convertString( 0,len,r.getBinary( len ) ) ;
        }
        else {
            element = null ;
        }
        len = ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ;
        if( partFlag == false ) {
            r.getBinary( len ) ;
            part = null ;
        }
        else {
            part = ConvertBinary.convertString( 0,len,r.getBinary( len ) ) ;
        }
        len = ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ;
        if( len > 0 ) {
            int lenI = len ;
            keys = new String[ lenI ] ;
            for( int i = 0 ; i < len ; i ++ ) {
                len = ConvertBinary.convertInt( 0,r.getBinary( 4 ) ) ;
                if( len > 0 ) {
                    keys[ i ] = ConvertBinary.convertString( 0,len,r.getBinary( len ) ) ;
                }
                else {
                    keys[ i ] = null ;
                }
            }
        }
        else {
            keys = null ;
        }
        if( partFlag == false || valueFlag == false ) {
            notUpdateFlag = true ;
        }
    }
}

