/*
 * @(#)CacheMap.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.index.core.element ;

import java.util.HashMap;
import java.util.Set;

/**
 * キャッシュMap情報.
 *
 * @version 2007/06/28
 * @author  masahito suzuki
 * @since   MaachangIndex 1.00
 */
public class CacheMap<K,V> {
    
    /**
     * MIN:最大値.
     */
    private static final int MIN_LENGTH = 8 ;
    
    /**
     * データマップ.
     */
    private HashMap<K,CacheElement<K,V>> map = null ;
    
    /**
     * キャッシュ管理.
     */
    protected CacheElement<K,V> cache = null ;
    
    /**
     * キャッシュエンドポイント.
     */
    protected CacheElement<K,V> endPoint = null ;
    
    /**
     * キャッシュサイズ.
     */
    private int length = 0 ;
    
    /**
     * キャッシュ最大値.
     */
    private int max = 0 ;
    
    /**
     * コンストラクタ.
     */
    private CacheMap() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * キャッシュ最大値を設定してオブジェクトを生成します.
     * <BR>
     * @param max 対象の最大キャッシュ値を設定します.
     */
    public CacheMap( int max ) {
        if( max <= MIN_LENGTH ) {
            max = MIN_LENGTH ;
        }
        this.max = max ;
        this.map = new HashMap<K,CacheElement<K,V>>( max+2 ) ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public synchronized void clear() {
        this.map = new HashMap<K,CacheElement<K,V>>( max+2 ) ;
        this.cache = null ;
        this.endPoint = null ;
        this.length = 0 ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 情報を追加します.
     * <BR>
     * @param key 対象のKeyを設定します.
     * @param value 対象のValueを設定します.
     * @return V キャッシュサイズを越した場合の情報が返されます.
     */
    public synchronized V put( K key,V value ) {
        CacheElement<K,V> ins = null ;
        V ret = null ;
        if( cache == null ) {
            ins = new CacheElement<K,V>( this ) ;
            cache = ins ;
            endPoint = ins ;
            ins.setKey( key ) ;
            ins.setValue( value ) ;
            length = 1 ;
        }
        else {
            if( ( ins = map.get( key ) ) != null ) {
                if( ins.before != null || ins.next != null ) {
                    ins.setKey( key ) ;
                    ins.setValue( value ) ;
                    ins.remove() ;
                    ins.before = endPoint ;
                    endPoint.next = ins ;
                    ins.next = null ;
                    endPoint = ins ;
                }
                return null ;
            }
            else {
                ins = new CacheElement<K,V>( this ) ;
                ins.setKey( key ) ;
                ins.setValue( value ) ;
                ins.before = endPoint ;
                endPoint.next = ins ;
                ins.next = null ;
                endPoint = ins ;
                length ++ ;
                if( max < length ) {
                    K rk = cache.getKey() ;
                    map.remove( rk ) ;
                    ret = cache.getValue() ;
                    cache = cache.next ;
                    cache.before = null ;
                    length -- ;
                }
            }
        }
        map.put( key,ins ) ;
        return ret ;
    }
    
    /**
     * キャッシュ削除.
     * <BR><BR>
     * 対象キャッシュ情報を削除します.
     * <BR>
     * @param key 対象のキー名を設定します.
     * @return V 削除された内容が返されます.
     */
    public synchronized V remove( K key ) {
        CacheElement<K,V> ce = null ;
        if( ( ce = map.remove( key ) ) != null ) {
            ce.remove() ;
            length -- ;
            return ce.getValue() ;
        }
        return null ;
    }
    
    /**
     * キャッシュを削除して取得.
     * <BR><BR>
     * キャッシュ情報を削除して取得します.
     * <BR>
     * @return V 先頭の情報が返されます.
     */
    public synchronized V remove() {
        if( cache == null ) {
            return null ;
        }
        K key = cache.getKey() ;
        V ret = cache.getValue() ;
        cache.remove() ;
        if( key != null ) {
            map.remove( key ) ;
        }
        return ret ;
    }
    
    /**
     * キャッシュ取得.
     * <BR><BR>
     * 対象キャッシュ情報を取得します.
     * <BR>
     * @param key 対象のキー名を設定します.
     * @return V 取得された内容が返されます.
     */
    public synchronized V get( K key ) {
        CacheElement<K,V> ce = null ;
        if( ( ce = map.get( key ) ) != null ) {
            ce.remove() ;
            if( cache == null ) {
            	cache = ce ;
            	endPoint = ce ;
            }
            else {
	            ce.before = endPoint ;
	            endPoint.next = ce ;
	            ce.next = null ;
	            endPoint = ce ;
            }
            return ce.getValue() ;
        }
        return null ;
    }
    
    /**
     * データサイズを取得.
     * <BR><BR>
     * データサイズを取得します.
     * <BR>
     * @return int データサイズが返されます.
     */
    public synchronized int size() {
        return length ;
    }
    
    /**
     * Key一覧を取得.
     * <BR><BR>
     * Key一覧が返されます.
     * <BR>
     * @return Object[] Key一覧が返されます.
     */
    public synchronized Object[] getKeys() {
        Set set = this.map.keySet() ;
        if( set != null ) {
            return set.toArray() ;
        }
        return null ;
    }
    
    /**
     * 最大管理サイズを取得.
     * <BR><BR>
     * 最大管理サイズを取得します.
     * <BR>
     * @return int 最大データサイズが返されます.
     */
    public synchronized int getMaxLength() {
        return max ;
    }
    
    /**
     * 指定キー情報存在チェック.
     * <BR><BR>
     * 指定キー情報が存在するかチェックします.
     * <BR>
     * @param key 対象のキーを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、存在します.
     */
    public synchronized boolean isKey( K key ) {
        return ( map.get( key ) == null ) ? false : true ;
    }
    
    /**
     * 文字列化.
     * <BR><BR>
     * 文字列化させます.
     * <BR>
     * @return String 文字列情報が返されます.
     */
    public synchronized String toString() {
        CacheElement<K,V> c = cache ;
        StringBuilder buf = new StringBuilder() ;
        buf.append( "{" ) ;
        for( int i = 0 ; i < length ; i ++ ) {
            if( c == null ) {
                break ;
            }
            buf.append( " [" ).append( i ).append( "]:" ) ;
            buf.append( c ) ;
            c = c.next ;
        }
        buf.append( " }" ) ;
        return buf.toString() ;
    }
}

/**
 * CacheMap要素.
 */
class CacheElement<K,V> {
    protected CacheMap<K,V> parent = null ;
    protected CacheElement<K,V> before = null ;
    protected CacheElement<K,V> next = null ;
    protected K key = null ;
    protected V value = null ;
    private CacheElement() { }
    protected CacheElement( CacheMap<K,V> parent ) {
        this.parent = parent ;
    }
    protected void setKey( K key ) {
        this.key = key ;
    }
    protected K getKey() {
        return this.key ;
    }
    protected void setValue( V value ) {
        this.value = value ;
    }
    protected V getValue() {
        return this.value ;
    }
    protected void remove() {
        if( before == null ) {
            if( next != null ) {
                next.before = null ;
                parent.cache = next ;
            }
            else {
                parent.cache = null ;
                parent.endPoint = null ;
            }
        }
        else {
            if( next == null ) {
                if( before != null ) {
                    before.next = null ;
                    parent.endPoint = before ;
                }
            }
            else {
                before.next = next ;
                next.before = before ;
            }
        }
        before = null ;
        next = null ;
    }
    public String toString() {
        return new StringBuilder().append( "key:" ).
        append( key ).append( " value:" ).append( value ).toString() ;
    }
}

