/*
 * @(#)SearchIndex.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.index.core ;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;

import org.maachang.index.MaachangResult;
import org.maachang.index.SubKey;
import org.maachang.index.core.element.ChipCache;
import org.maachang.index.core.element.IndexChip;
import org.maachang.index.core.element.IndexChipById;
import org.maachang.index.core.element.IndexChipCache;
import org.maachang.index.core.element.IndexChipPos;
import org.maachang.index.core.element.IndexKey;
import org.maachang.index.core.element.IndexKeyCache;
import org.maachang.index.core.element.IndexVersion;
import org.maachang.index.core.element.SearchBaseCache;
import org.maachang.index.core.element.SearchHistoryCache;
import org.maachang.index.core.element.SearchValue;
import org.maachang.util.FileUtil;

/**
 * インデックス検索.
 *
 * @version 2007/06/28
 * @author  masahito suzuki
 * @since   MaachangIndex 1.00
 */
public class SearchIndex {
    
    /**
     * N-Gram係数.
     */
    private int ngram = -1 ;
    
    /**
     * キャッシュ格納用.
     */
    private IndexChipCache cache = null ;
    
    /**
     * ヒストリ格納用.
     */
    private SearchHistoryCache history = null ;
    
    /**
     * IndexBase格納用.
     */
    private SearchBaseCache baseCache = null ;
    
    /**
     * IndexKey格納用.
     */
    private IndexKeyCache keyCache = null ;
    
    /**
     * インデックスバージョン.
     */
    private IndexVersion indexVersion = null ;
    
    /**
     * IndexChipキャッシュ.
     */
    private ChipCache chipCache = null ;
    
    /**
     * 検索結果ヒント情報表示最大長.
     */
    private int partLength = IndexDef.SEARCH_PART_LENGTH ;
    
    /**
     * 実行モード.
     */
    private boolean mode = false ;
    
    /**
     * コンストラクタ.
     */
    private SearchIndex() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * インデックス検索用オブジェクトを生成します.
     * <BR>
     * @param n インデックス幅を設定します.
     * @param cache 最大キャッシュサイズを設定します.
     * @param baseCache 最大IndexBaseキャッシュサイズを設定します.
     * @param history 最大ヒストリサイズを設定します.
     * @param keyCache 最大IndexKeyキャッシュサイズを設定します.
     * @param baseDir インデックス出力用ディレクトリを設定します.
     * @exception Exception 例外.
     */
    public SearchIndex( int n,int cache,int baseCache,int history,int keyCache,String baseDir )
        throws Exception {
        this( n,cache,baseCache,history,keyCache,baseDir,IndexDef.SEARCH_PART_LENGTH ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * インデックス検索用オブジェクトを生成します.
     * <BR>
     * @param n インデックス幅を設定します.
     * @param cache 最大キャッシュサイズを設定します.
     * @param baseCache 最大IndexBaseキャッシュサイズを設定します.
     * @param history 最大ヒストリサイズを設定します.
     * @param keyCache 最大IndexKeyキャッシュサイズを設定します.
     * @param baseDir インデックス出力用ディレクトリを設定します.
     * @param partLength 検索結果のヒント表示文字列長を設定します.
     * @exception Exception 例外.
     */
    public SearchIndex( int n,int cache,int baseCache,int history,int keyCache,String baseDir,int partLength )
        throws Exception {
        
        if( baseDir == null || ( baseDir = baseDir.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        if( n <= IndexDef.MIN_INDEX_N_GRAM ) {
            n = IndexDef.MIN_INDEX_N_GRAM ;
        }
        else if( n >= IndexDef.MAX_INDEX_N_GRAM ) {
            n = IndexDef.MAX_INDEX_N_GRAM ;
        }
        
        if( baseDir.endsWith( "/" ) == true || baseDir.endsWith( "\\" ) == true ) {
            baseDir = baseDir.substring( 0,baseDir.length() - 1 ) ;
        }
        if( partLength <= 0 ) {
            partLength = IndexDef.SEARCH_PART_LENGTH ;
        }
        String dataDir = baseDir + FileUtil.FILE_SPACE + IndexDef.BASE_DIR ;
        String indexDir = baseDir + FileUtil.FILE_SPACE + IndexDef.INDEX_DIR ;
        String historyDir = baseDir + FileUtil.FILE_SPACE + IndexDef.HISTORY_DIR ;
        String keyDir = baseDir + FileUtil.FILE_SPACE + IndexDef.KEY_DIR ;
        String nameDir = baseDir + FileUtil.FILE_SPACE + IndexDef.NAME_DIR ;
        
        if( FileUtil.isDirExists( baseDir ) == false ) {
            throw new IllegalArgumentException( "基本ディレクトリ[" + baseDir + "]は存在しません" ) ;
        }
        if( FileUtil.isDirExists( dataDir ) == false ) {
            throw new IllegalArgumentException( "IndexBaseディレクトリ[" + dataDir + "]は存在しません" ) ;
        }
        if( FileUtil.isDirExists( indexDir ) == false ) {
            throw new IllegalArgumentException( "インデックスディレクトリ[" + indexDir + "]は存在しません" ) ;
        }
        if( FileUtil.isDirExists( historyDir ) == false ) {
            throw new IllegalArgumentException( "ヒストリディレクトリ[" + historyDir + "]は存在しません" ) ;
        }
        if( FileUtil.isDirExists( keyDir ) == false ) {
            throw new IllegalArgumentException( "キー格納ディレクトリ[" + keyDir + "]は存在しません" ) ;
        }
        if( FileUtil.isDirExists( nameDir ) == false ) {
            throw new IllegalArgumentException( "一意名格納ディレクトリ[" + nameDir + "]は存在しません" ) ;
        }
        this.indexVersion = IndexVersion.getIndexVersion( baseDir ) ;
        if( this.indexVersion == null ) {
            throw new IllegalArgumentException( "インデックス情報は存在しません" ) ;
        }
        
        this.ngram = n ;
        
        this.cache = new IndexChipCache( false,cache,indexDir ) ;
        if( history > 0 ) {
            this.history = new SearchHistoryCache( history,historyDir ) ;
        }
        this.baseCache = new SearchBaseCache( baseCache,dataDir ) ;
        this.keyCache = new IndexKeyCache( false,keyCache,keyDir ) ;
        this.chipCache = new ChipCache( indexDir ) ;
        this.chipCache.create() ;
        this.partLength = partLength ;
        this.mode = false ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * InsertIndexで更新処理や削除処理を同一プロセス上で行う場合に
     * このコンストラクタを利用してください.
     * <BR>
     * @param insertIndex 対象のInsertIndexオブジェクトを設定します.
     * @param baseCache 最大IndexBaseキャッシュサイズを設定します.
     * @param history 最大ヒストリサイズを設定します.
     * @exception Exception 例外.
     */
    public SearchIndex( InsertIndex insertIndex,int baseCache,int history )
        throws Exception {
        this( insertIndex,baseCache,history,IndexDef.SEARCH_PART_LENGTH ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * InsertIndexで更新処理や削除処理を同一プロセス上で行う場合に
     * このコンストラクタを利用してください.
     * <BR>
     * @param insertIndex 対象のInsertIndexオブジェクトを設定します.
     * @param baseCache 最大IndexBaseキャッシュサイズを設定します.
     * @param history 最大ヒストリサイズを設定します.
     * @param partLength 検索結果のヒント表示文字列長を設定します.
     * @exception Exception 例外.
     */
    public SearchIndex( InsertIndex insertIndex,int baseCache,int history,int partLength )
        throws Exception {
        if( insertIndex == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        String baseDir = insertIndex.baseDir ;
        if( baseDir.endsWith( "/" ) == true || baseDir.endsWith( "\\" ) == true ) {
            baseDir = baseDir.substring( 0,baseDir.length() - 1 ) ;
        }
        if( partLength <= 0 ) {
            partLength = IndexDef.SEARCH_PART_LENGTH ;
        }
        String dataDir = baseDir + FileUtil.FILE_SPACE + IndexDef.BASE_DIR ;
        String historyDir = baseDir + FileUtil.FILE_SPACE + IndexDef.HISTORY_DIR ;
        if( FileUtil.isDirExists( dataDir ) == false ) {
            throw new IllegalArgumentException( "IndexBaseディレクトリ[" + dataDir + "]は存在しません" ) ;
        }
        if( FileUtil.isDirExists( historyDir ) == false ) {
            throw new IllegalArgumentException( "ヒストリディレクトリ[" + historyDir + "]は存在しません" ) ;
        }
        this.cache = insertIndex.cache ;
        if( history > 0 ) {
            this.history = new SearchHistoryCache( history,historyDir ) ;
        }
        this.baseCache = new SearchBaseCache( baseCache,dataDir ) ;
        this.keyCache = insertIndex.keyCache ;
        insertIndex.searchIndex = this ;
        this.chipCache = new ChipCache( this.cache.getIndexDirectory() ) ;
        this.chipCache.create() ;
        this.partLength = partLength ;
        this.mode = true ;
    }
    
    /**
     * オブジェクト解放.
     * <BR><BR>
     * オブジェクトを解放した場合の処理を実装.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        ngram = -1 ;
        cache = null ;
        history = null ;
        baseCache = null ;
        indexVersion = null ;
        keyCache = null ;
        partLength = 0 ;
        mode = false ;
    }
    
    /**
     * 情報更新.
     * <BR><BR>
     * 情報を更新します.
     */
    public void flush() {
        if( history != null ) {
            this.history.flush() ;
        }
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * キャッシュ内容をクリアします.
     */
    public void clear() {
        cache.clear() ;
        baseCache.clear() ;
        keyCache.clear() ;
        if( history != null ) {
            history.clear() ;
        }
        mode = false ;
    }
    
    /**
     * InsertIndex削除処理時の処理.
     * <BR><BR>
     * InsertIndexと連動している場合にremoveメソッドが正常実行されたら、
     * 呼び出されます.
     */
    protected void clearByInsertIndexRemove() {
        if( mode == true ) {
            if( history != null ) {
                history.clear() ;
                SearchHistoryCache.removeHistorySerializable( history.getHistoryDirectory() ) ;
            }
            baseCache.clear() ;
            chipCache.create() ;
        }
    }
    
    /**
     * 検索バージョンを取得.
     * <BR><BR>
     * 現在有効な検索バージョンを取得します.
     * <BR>
     * @return IndexVersion 検索バージョンが返されます.
     */
    public IndexVersion getIndexVersion() {
        return this.indexVersion ;
    }
    
    /**
     * 検索結果のヒント表示文字列長を取得.
     * <BR><BR>
     * 検索結果のヒント表示文字列長を取得します.
     * <BR>
     * @return int 検索結果のヒント表示文字列長が返されます.
     */
    public int getPartLength() {
        return this.partLength ;
    }
    
    /**
     * 新しいキーワードで検索.
     * <BR><BR>
     * 新しいキーワードで検索処理を行う場合に利用します.
     * <BR>
     * @param word 対象の検索キーワードを設定します.
     * @param subKeys 対象のSubKey群を設定します.
     * @return MaachangResult 検索結果が返されます.
     * @exception Exception 例外.
     */
    public MaachangResult searchByNewWord( String word,SubKey[] subkeys )
        throws Exception {
        return searchByNewWord( word,MaachangResultImpl.DEF_LENGTH,subkeys ) ;
    }
    
    /**
     * 新しいキーワードで検索.
     * <BR><BR>
     * 新しいキーワードで検索処理を行う場合に利用します.
     * <BR>
     * @param word 対象の検索キーワードを設定します.
     * @param length １回で表示するデータ件数を設定します.
     * @param subKeys 対象のSubKey群を設定します.
     * @return MaachangResult 検索結果が返されます.
     * @exception Exception 例外.
     */
    public MaachangResult searchByNewWord( String word,int length,SubKey[] subkeys )
        throws Exception {
        if( word == null || ( word = word.trim() ).length() <= 0 ) {
            return null ;
        }
        MaachangResultImpl ret = new MaachangResultImpl( length,partLength ) ;
        ret.create( word,subkeys ) ;
        if( searchTo( ret ) == 0 ) {
            ret.setMaxResult( 0 ) ;
            return ret ;
        }
        return ret ;
    }
    
    /**
     * 前回キーワード条件で、次ページの内容を取得.
     * <BR><BR>
     * 前回キーワード条件で、次ページの内容を取得します.
     * <BR>
     * @param result 前回の検索条件を設定します.
     * @return boolean 処理結果が返されます.<BR>
     *                 [true]が返された場合、次ページの情報が取得できました.
     * @exception Exception 例外.
     */
    public boolean nextSearch( MaachangResult result )
        throws Exception {
        if( result == null ) {
            return false ;
        }
        if( ( ( MaachangResultImpl )result ).next() == false ) {
            ( ( MaachangResultImpl )result ).cleanResult() ;
            return false ;
        }
        if( searchTo( result ) == 0 ) {
            ( ( MaachangResultImpl )result ).cleanResult() ;
            return false ;
        }
        return true ;
    }
    
    /**
     * 前回キーワード条件で、前ページの内容を取得.
     * <BR><BR>
     * 前回キーワード条件で、前ページの内容を取得します.
     * <BR>
     * @param result 前回の検索条件を設定します.
     * @return boolean 処理結果が返されます.<BR>
     *                 [true]が返された場合、前ページの情報が取得できました.
     * @exception Exception 例外.
     */
    public boolean beforeSearch( MaachangResult result )
        throws Exception {
        if( result == null ) {
            return false ;
        }
        if( ( ( MaachangResultImpl )result ).before() == false ) {
            ( ( MaachangResultImpl )result ).cleanResult() ;
            return false ;
        }
        if( searchTo( result ) == 0 ) {
            ( ( MaachangResultImpl )result ).cleanResult() ;
            return false ;
        }
        return true ;
    }
    
    /**
     * 前回キーワード条件で、指定ページの内容を取得.
     * <BR><BR>
     * 前回キーワード条件で、指定ページの内容を取得します.
     * <BR>
     * @param result 前回の検索条件を設定します.
     * @param pageNo 指定ページNoを設定します.
     * @return boolean 処理結果が返されます.<BR>
     *                 [true]が返された場合、指定ページの情報が取得できました.
     * @exception Exception 例外.
     */
    public boolean pageNoSearch( MaachangResult result,int pageNo )
        throws Exception {
        if( result == null ) {
            return false ;
        }
        if( ( ( MaachangResultImpl )result ).change( pageNo ) == false ) {
            ( ( MaachangResultImpl )result ).cleanResult() ;
            return false ;
        }
        if( searchTo( result ) == 0 ) {
            ( ( MaachangResultImpl )result ).cleanResult() ;
            return false ;
        }
        return true ;
    }
    
    /**
     * 検索モードを取得.
     * <BR><BR>
     * 検索モードを取得します.
     * <BR>
     * @return boolean 検索モードが返されます.<BR>
     *                 [true]が返された場合、Insert+Searchモードです.<BR>
     *                 [false]が返された場合、Searchモードです.
     */
    public boolean isMode() {
        return mode ;
    }
    
    /**
     * 区分された検索ワードを結合.
     */
    private String unitingSearchString( String[] search,SubKey[] subkeys ) {
        StringBuilder buf = new StringBuilder() ;
        int i,len ;
        len = search.length ;
        for( i = 0 ; i < len ; i ++ ) {
            if( i != 0 ) {
                buf.append( " " ) ;
            }
            buf.append( search[ i ] ) ;
        }
        if( subkeys != null && ( len = subkeys.length ) > 0 ) {
            Arrays.sort( subkeys ) ;
            buf.append( "@#{" ) ;
            for( i = 0 ; i < len ; i ++ ) {
                buf.append( subkeys[ i ].toString() ) ;
            }
            buf.append( "}#@" ) ;
        }
        return buf.toString() ;
    }
    
    /**
     * 条件から、検索結果を指定件数分取得.
     */
    private int resultData( MaachangResultImpl result,SearchValue bean,int offset,int length )
        throws Exception {
        if( bean == null || bean.size() <= offset ) {
            return 0 ;
        }
        if( length <= 0 ) {
            return 0 ;
        }
        if( bean.size() <= offset + length ) {
            length = bean.size() - offset ;
        }
        
        int i,j ;
        int len = offset + length ;
        result.setMaxResult( bean.size() ) ;
        result.cleanResult() ;
        for( i = offset,j = 0 ; i < len ; i ++,j ++ ) {
            result.setResult( new ResultChildImpl( i+1,bean.getId( i ),bean.getScore( i ),baseCache,result ),j ) ;
        }
        return len ;
    }
    
    /**
     * 検索処理.
     */
    private int searchTo( MaachangResult r )
        throws Exception {
        if( r == null ) {
            return 0 ;
        }
        MaachangResultImpl result = ( MaachangResultImpl )r ;
        String string = result.getSearchWord() ;
        SubKey[] subkeys = result.getSubKeys() ;
        int offset = result.getOffset() ;
        int length = result.getViewLength() ;
        String[] cutSearch = IndexUtil.cutSearchString( string ) ;
        int len ;
        if( cutSearch == null || ( len = cutSearch.length ) <= 0 ) {
            return 0 ;
        }
        if( offset <= 0 ) {
            offset = 0 ;
        }
        string = unitingSearchString( cutSearch,subkeys ) ;
        SearchValue bean = null ;
        if( history != null ) {
            bean = history.get( string ) ;
        }
        if( bean == null ) {
            HashMap<String,IndexChip> useNGram = new HashMap<String,IndexChip>() ;
            SearchByIdToScore res = getMinSearchId( cutSearch,useNGram ) ;
            if( res == null || res.size() <= 0 ) {
                return 0 ;
            }
            if( subkeys != null && subkeys.length > 0 ) {
                Long[] id = res.getId() ;
                int lenJ = subkeys.length ;
                int lenK = id.length ;
                for( int j = 0 ; j < lenJ ; j ++ ) {
                    IndexKey key = keyCache.get( subkeys[ j ].getKeyword(),subkeys[ j ].getNumber() ) ;
                    if( key == null || key.size() <= 0 ) {
                        return 0 ;
                    }
                    for( int k = 0 ; k < lenK ; k ++ ) {
                        if( id[ k ] != null ) {
                            if( key.isId( id[ k ] ) == false ) {
                                id[ k ] = null ;
                            }
                        }
                    }
                }
                res.smart() ;
                lenJ = res.size() ;
            }
            if( res == null || res.size() <= 0 ) {
                return 0 ;
            }
            len = cutSearch.length ;
            for( int i = 0 ; i < len ; i ++ ) {
                if( cutSearch[ i ].length() < this.ngram ) {
                    if( smalNGramSearch( res,useNGram,cutSearch[ i ] ) == false ) {
                        return 0 ;
                    }
                }
                else {
                    if( fullNGramSearch( res,useNGram,cutSearch[ i ] ) == false ) {
                        return 0 ;
                    }
                }
                res.smart() ;
            }
            useNGram.clear() ;
            useNGram = null ;
            bean = convertSearchValue( string,res ) ;
            if( bean == null ) {
                return 0 ;
            }
        }
        return this.resultData( result,bean,offset,length ) ;
    }
    
    /**
     * 対象情報をSearchValueに変換.
     */
    private SearchValue convertSearchValue( String words,SearchByIdToScore res ) {
        Long[] id = res.getId() ;
        int[] scores = res.getScore() ;
        if( id == null || id.length <= 0 ) {
            return null ;
        }
        SearchValue ret = new SearchValue( words ) ;
        int len = id.length ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( id != null ) {
                ret.add( id[ i ],scores[ i ] ) ;
            }
        }
        if( ret.size() <= 0 ) {
            return null ;
        }
        ret.sort() ;
        if( history != null ) {
            history.add( ret ) ;
        }
        return ret ;
    }
    
    /**
     * 一番小さい検索ID群を取得.
     */
    private SearchByIdToScore getMinSearchId( String[] cutSearch,HashMap<String,IndexChip> useNGram ) {
        int size = -1 ;
        Long[] smallKeys = null ;
        IndexChip fullChip = null ;
        int len = cutSearch.length ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( cutSearch[ i ].length() < this.ngram ) {
                String[] code = chipCache.getIndexChip( IndexUtil.getNGramByNumber( cutSearch[ i ],cutSearch[ i ].length() ) ) ;
                int lenJ ;
                if( code == null || ( lenJ = code.length ) <= 0 ) {
                    return null ;
                }
                for( int j = 0 ; j < lenJ ; j ++ ) {
                    if( useNGram.get( code[ i ] ) == null ) {
                        IndexChip cp = this.cache.get( code[ j ] ) ;
                        if( cp.size() <= 0 ) {
                            return null ;
                        }
                        if( size == -1 || cp.size() > size ) {
                            if( smallKeys == null ) {
                                smallKeys = cp.getKeys() ;
                                size = cp.size() ;
                            }
                            else {
                                smallKeys = margeKeys( smallKeys,cp.getKeys() ) ;
                                if( smallKeys == null ) {
                                    size = 0 ;
                                }
                                else {
                                    size = smallKeys.length ;
                                }
                            }
                            //useNGram.put( code[ i ],cp ) ;
                            code[ i ] = null ;
                        }
                    }
                }
            }
            else {
                int lenJ = IndexUtil.getNGramByStringLength( cutSearch[ i ],ngram ) ;
                if( lenJ <= 0 ) {
                    return null ;
                }
                String[] ngramStrings = new String[ lenJ ] ;
                for( int j = 0 ; j < lenJ ; j ++ ) {
                    ngramStrings[ j ] = IndexUtil.getNGramCode( cutSearch[ i ],ngram,j ) ;
                    ngramStrings[ j ] = IndexUtil.getNGramByNumber( ngramStrings[ j ],ngram ) ;
                    if( useNGram.get( ngramStrings[ j ] ) == null ) {
                        IndexChip cp = this.cache.get( ngramStrings[ j ] ) ;
                        if( cp == null || cp.size() <= 0 ) {
                            return null ;
                        }
                        if( size == -1 || cp.size() < size ) {
                            fullChip = cp ;
                            size = cp.size() ;
                        }
                        useNGram.put( ngramStrings[ j ],cp ) ;
                        ngramStrings[ j ] = null ;
                    }
                }
            }
        }
        if( fullChip != null ) {
            return ( fullChip == null ) ? null : new SearchByIdToScore( fullChip.getKeys(),new int[ fullChip.size() ] ) ;
        }
        else if( smallKeys != null && smallKeys.length > 0 ) {
            return ( smallKeys == null ) ? null : new SearchByIdToScore( smallKeys,new int[ smallKeys.length ] ) ;
        }
        return null ;
    }
    
    /**
     * キーコードマージ.
     */
    private Long[] margeKeys( Long[] src,Long[] dest ) {
        if( src == null || src.length <= 0 ) {
            return dest ;
        }
        if( dest == null || dest.length <= 0 ) {
            return src ;
        }
        HashSet<Long> set = new HashSet<Long>() ;
        int len = src.length ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( src[ i ] != null ) {
                set.add( src[ i ] ) ;
            }
        }
        len = dest.length ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( dest[ i ] != null ) {
                set.add( dest[ i ] ) ;
            }
        }
        len = set.size() ;
        if( len > 0 ) {
            Long[] ret = new Long[ len ] ;
            System.arraycopy( set.toArray(),0,ret,0,len ) ;
            return ret ;
        }
        return null ;
    }
    
    /**
     * n-gram係数より少ない文字列で検索.
     */
    private boolean smalNGramSearch( SearchByIdToScore res,HashMap<String,IndexChip> useNGram,String word )
        throws Exception {
        Long[] id = res.getId() ;
        int[] scores = res.getScore() ;
        String[] ngram = chipCache.getIndexChip( IndexUtil.getNGramByNumber( word,word.length() ) ) ;
        if( ngram != null && ngram.length > 0 ) {
            int i ;
            int len = ngram.length ;
            boolean[] flags = new boolean[ id.length ] ;
            for( i = 0 ; i < len ; i ++ ) {
                IndexChip cp = null ;
                if( ( cp = useNGram.get( ngram[ i ] ) ) == null ) {
                    cp = this.cache.get( ngram[ i ] ) ;
                }
                ngram[ i ] = null ;
                if( cp != null ) {
                    int lenJ = id.length ;
                    for( int j = 0 ; j < lenJ ; j ++ ) {
                        IndexChipById ci = null ;
                        if( id[ j ] != null ) {
                            if( ( ci = cp.get( id[ j ] ) ) == null ) {
                                continue ;
                            }
                        }
                        else {
                            continue ;
                        }
                        int sc = ci.getScore() ;
                        if( ci.getScore() >= ci.getScoreCoefficient() ) {
                            sc = 0 ;
                        }
                        if( scores[ j ] <= 0 ) {
                            scores[ j ] = sc ;
                        }
                        else {
                            scores[ j ] = ( sc + scores[ j ] ) / 2 ;
                        }
                        flags[ j ] = true ;
                    }
                }
                else {
                    return false ;
                }
            }
            len = id.length ;
            for( i = 0 ; i < len ; i ++ ) {
                if( flags[ i ] == false ) {
                    id[ i ] = null ;
                }
            }
            return true ;
        }
        return false ;
    }
    
    /**
     * n-gram係数より大きい文字列で検索.
     */
    private boolean fullNGramSearch( SearchByIdToScore res,HashMap<String,IndexChip> useNGram,String word )
        throws Exception {
        Long[] id = res.getId() ;
        int[] scores = res.getScore() ;
        int i ;
        int len = IndexUtil.getNGramByStringLength( word,ngram ) ;
        String[] ngramStrings = new String[ len ] ;
        for( i = 0 ; i < len ; i ++ ) {
            ngramStrings[ i ] = IndexUtil.getNGramCode( word,ngram,i ) ;
        }
        IndexChip ch = null ;
        ch = useNGram.get( IndexUtil.getNGramByNumber( ngramStrings[ 0 ],ngram ) ) ;
        if( checkFirst( id,ch ) == false ) {
            return false ;
        }
        for( i = 0 ; i < len ; i ++ ) {
            ch = nextNGram( id,scores,useNGram,ch,ngramStrings,ngram,i ) ;
            if( ch == null ) {
                break ;
            }
        }
        return true ;
    }
    
    /**
     * 最初の処理条件を取得.
     */
    private boolean checkFirst( Long[] id,IndexChip ch ) {
        int len = id.length ;
        boolean ret = false ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( id[ i ] != null ) {
                if( ch.get( id[ i ] ) == null ) {
                    id[ i ] = null ;
                }
                else {
                    ret = true ;
                }
            }
        }
        return ret ;
    }
    
    /**
     * 現NGramと次NGramのポジションが一致するかチェック.
     */
    private boolean checkNowByNextPos( int no,IndexChipById now,IndexChipById next ) {
        IndexChipPos pos = now.getPos( no ) ;
        int len = pos.size() ;
        int lenJ = next.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            int position = pos.getPosition( i ) + 1 ;
            for( int j = 0 ; j < lenJ ; j ++ ) {
                IndexChipPos nextPos = next.getPos( j ) ;
                if( nextPos.isBitweenPos( position ) == true ) {
                    int lenK = nextPos.size() ;
                    for( int k = 0 ; k < lenK ; k ++ ) {
                        if( position == nextPos.getPosition( k ) ) {
                            return true ;
                        }
                    }
                }
            }
        }
        return false ;
    }
    
    /**
     * 次NGram情報処理.
     */
    private IndexChip nextNGram( Long[] id,int[] scores,HashMap<String,IndexChip> useNGram,IndexChip nowChip,String[] ngramStrings,int ngram,int nowNo ) {
        int nextNo = nowNo + 1 ;
        if( nowNo < 0 || ngramStrings.length <= nextNo ) {
            if( nowNo >= 0 && ngramStrings.length <= nextNo ) {
                int len = id.length ;
                for( int i = 0 ; i < len ; i ++ ) {
                    if( id[ i ] == null ) {
                        continue ;
                    }
                    IndexChipById now = nowChip.get( id[ i ] ) ;
                    int sc = now.getScore() ;
                    if( now.getScore() >= now.getScoreCoefficient() ) {
                        sc = 0 ;
                    }
                    if( scores[ i ] <= 0 ) {
                        scores[ i ] = sc ;
                    }
                    else {
                        scores[ i ] = ( sc + scores[ i ] ) / 2 ;
                    }
                }
            }
            return null ;
        }
        int len = id.length ;
        long nextNgram = IndexChipPos.convertNextNGram( ngramStrings[ nextNo ].toCharArray() ) ;
        IndexChip ch = useNGram.get( IndexUtil.getNGramByNumber( ngramStrings[ nextNo ],ngram ) ) ;
        for( int i = 0 ; i < len ; i ++ ) {
            if( id[ i ] == null ) {
                continue ;
            }
            IndexChipById now = nowChip.get( id[ i ] ) ;
            int lenJ = now.size() ;
            int no = -1 ;
            for( int j = 0 ; j < lenJ ; j ++ ) {
                IndexChipPos pos = now.getPos( j ) ;
                if( nextNgram == pos.getNextNGram() ) {
                    no = j ;
                    break ;
                }
            }
            if( no != -1 && checkNowByNextPos( no,now,ch.get( id[ i ] ) ) == true ) {
                int sc = now.getScore() ;
                if( now.getScore() >= now.getScoreCoefficient() ) {
                    sc = 0 ;
                }
                if( scores[ i ] <= 0 ) {
                    scores[ i ] = sc ;
                }
                else {
                    scores[ i ] = ( sc + scores[ i ] ) / 2 ;
                }
            }
            else {
                id[ i ] = null ;
            }
            
        }
        return ch ;
    }
}

/**
 * IDと得点を表す.
 */
class SearchByIdToScore {
    Long[] id = null ;
    int[] score = null ;
    private SearchByIdToScore() {
    }
    public SearchByIdToScore( Long[] id,int[] score ) {
        this.id = id ;
        this.score = score ;
    }
    public Long[] getId() {
        return this.id ;
    }
    public int[] getScore() {
        return this.score ;
    }
    public int size() {
        return ( id != null ) ? id.length : 0 ;
    }
    public void smart() {
        if( id == null && id.length <= 0 ) {
            this.id = null ;
            this.score = null ;
            return ;
        }
        int i ;
        int len = this.id.length ;
        int cnt = 0 ;
        for( i = 0 ; i < len ; i ++ ) {
            if( this.id[ i ] != null ) {
                cnt ++ ;
            }
        }
        if( cnt <= 0 ) {
            this.id = null ;
            this.score = null ;
        }
        else {
            Long[] tid = new Long[ cnt ] ;
            int[] tscore = new int[ cnt ] ;
            for( i = 0,cnt = 0 ; i < len ; i ++ ) {
                if( this.id[ i ] != null ) {
                    tid[ cnt ] = this.id[ i ] ;
                    tscore[ cnt ] = this.score[ i ] ;
                    cnt ++ ;
                }
            }
            this.id = tid ;
            this.score = tscore ;
        }
    }
    public static final Long[] smart( Long[] id ) {
        if( id == null && id.length <= 0 ) {
            return null ;
        }
        int i ;
        int len = id.length ;
        int cnt = 0 ;
        for( i = 0 ; i < len ; i ++ ) {
            if( id[ i ] != null ) {
                cnt ++ ;
            }
        }
        if( cnt <= 0 ) {
            id = null ;
        }
        else {
            Long[] tid = new Long[ cnt ] ;
            for( i = 0,cnt = 0 ; i < len ; i ++ ) {
                if( id[ i ] != null ) {
                    tid[ cnt ] = id[ i ] ;
                    cnt ++ ;
                }
            }
            id = tid ;
        }
        return id ;
    }
    public String toString() {
        StringBuilder buf = new StringBuilder() ;
        buf.append( " size:" ).
            append( this.size() ) ;
        int len = this.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            buf.append( " [" ).append( i ).append( "]:" ).
                append( " id:" ).append( id[ i ] ).
                append( " score:" ).append( score[ i ] ) ;
        }
        return buf.toString() ;
    }
}
