/*
 * @(#)InsertIndex.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.index.core ;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.ArrayList;

import org.maachang.index.core.element.IndexBase;
import org.maachang.index.core.element.IndexChip;
import org.maachang.index.core.element.IndexChipCache;
import org.maachang.index.core.element.IndexKey;
import org.maachang.index.core.element.IndexKeyCache;
import org.maachang.index.core.element.IndexName;
import org.maachang.index.core.element.IndexNameCache;
import org.maachang.index.core.element.IndexVersion;
import org.maachang.index.core.element.SequenceLong;
import org.maachang.util.FileUtil;
import org.maachang.util.StringUtil;

/**
 * インデックスを追加する.
 *
 * @version 2007/06/28
 * @author  masahito suzuki
 * @since   MaachangIndex 1.00
 */
public class InsertIndex {
    
    /**
     * シーケンスファイル名.
     */
    private static final String SEQ_NAME = "sequence" ;
    
    /**
     * N-Gram係数.
     */
    protected int ngram = -1 ;
    
    /**
     * ベースインデックスディレクトリ.
     */
    protected String baseDir = null ;
    
    /**
     * IndexBase格納ディレクトリ.
     */
    private String indexBaseDir = null ;
    
    /**
     * ID管理.
     */
    private SequenceLong sequence = null ;
    
    /**
     * キャッシュ格納用.
     */
    protected IndexChipCache cache = null ;
    
    /**
     * インデックスバージョン.
     */
    protected IndexVersion indexVersion = null ;
    
    /**
     * IndexKey格納用.
     */
    protected IndexKeyCache keyCache = null ;
    
    /**
     * IndexName格納用.
     */
    protected IndexNameCache nameCache = null ;
    
    /**
     * 検索結合用.
     */
    protected SearchIndex searchIndex = null ;
    
    /**
     * コンストラクタ.
     */
    private InsertIndex() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定して、インデックス注入オブジェクトを生成します.
     * <BR>
     * @param n インデックス幅を設定します.
     * @param cache 最大キャッシュサイズを設定します.
     * @param keyCache 最大IndexKeyキャッシュサイズを設定します.
     * @param nameCache 最大IndexNameキャッシュサイズを設定します.
     * @param baseDir インデックス出力用ディレクトリを設定します.
     * @exception Exception 例外.
     */
    public InsertIndex( int n,int cache,int keyCache,int nameCache,String baseDir )
        throws Exception {
        
        if( baseDir == null || ( baseDir = baseDir.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        if( n <= IndexDef.MIN_INDEX_N_GRAM ) {
            n = IndexDef.MIN_INDEX_N_GRAM ;
        }
        else if( n >= IndexDef.MAX_INDEX_N_GRAM ) {
            n = IndexDef.MAX_INDEX_N_GRAM ;
        }
        
        if( baseDir.endsWith( "/" ) == true || baseDir.endsWith( "\\" ) == true ) {
            baseDir = baseDir.substring( 0,baseDir.length() - 1 ) ;
        }
        String dataDir = baseDir + FileUtil.FILE_SPACE + IndexDef.BASE_DIR ;
        String indexDir = baseDir + FileUtil.FILE_SPACE + IndexDef.INDEX_DIR ;
        String historyDir = baseDir + FileUtil.FILE_SPACE + IndexDef.HISTORY_DIR ;
        String keyDir = baseDir + FileUtil.FILE_SPACE + IndexDef.KEY_DIR ;
        String nameDir = baseDir + FileUtil.FILE_SPACE + IndexDef.NAME_DIR ;
        
        if( FileUtil.isFileExists( baseDir ) == false ) {
            FileUtil.mkdirs( baseDir ) ;
        }
        if( FileUtil.isFileExists( dataDir ) == false ) {
            FileUtil.mkdirs( dataDir ) ;
        }
        if( FileUtil.isFileExists( indexDir ) == false ) {
            FileUtil.mkdirs( indexDir ) ;
        }
        if( FileUtil.isFileExists( historyDir ) == false ) {
            FileUtil.mkdirs( historyDir ) ;
        }
        if( FileUtil.isFileExists( keyDir ) == false ) {
            FileUtil.mkdirs( keyDir ) ;
        }
        if( FileUtil.isFileExists( nameDir ) == false ) {
            FileUtil.mkdirs( nameDir ) ;
        }
        
        this.ngram = n ;
        this.baseDir = baseDir ;
        this.indexBaseDir = dataDir ;
        
        this.cache = new IndexChipCache( true,cache,indexDir ) ;
        this.keyCache = new IndexKeyCache( true,keyCache,keyDir ) ;
        this.nameCache = new IndexNameCache( nameCache,nameDir ) ;
        
        this.sequence = getSequence( baseDir ) ;
        if( this.sequence == null ) {
            this.sequence = new SequenceLong() ;
            this.sequence.create( 1L,Long.MAX_VALUE ) ;
            putSequence( baseDir,sequence ) ;
        }
        this.indexVersion = IndexVersion.getIndexVersion( this.baseDir ) ;
        if( this.indexVersion == null ) {
            this.indexVersion = new IndexVersion() ;
            this.indexVersion.putIndexVersion( this.baseDir ) ;
        }
        
    }
    
    /**
     * オブジェクト解放.
     * <BR><BR>
     * オブジェクトを解放した場合の処理を実装.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        ngram = -1 ;
        baseDir = null ;
        indexBaseDir = null ;
        cache = null ;
        keyCache = null ;
        nameCache = null ;
        sequence = null ;
        indexVersion = null ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public void clear() {
        this.flush() ;
        this.cache.clear() ;
        this.keyCache.clear() ;
        this.nameCache.clear() ;
    }
    
    /**
     * インデックスを全て出力.
     * <BR><BR>
     * インデックスを全て出力します.
     */
    public void flush() {
        this.cache.flush() ;
        this.keyCache.flush() ;
        this.nameCache.flush() ;
        this.indexVersion.putIndexVersion( this.baseDir ) ;
        putSequence( baseDir,sequence ) ;
    }
    
    /**
     * 管理バージョンを１つ上げる.
     * <BR><BR>
     * 管理バージョンを１つ上げます.
     */
    public void addVersion() {
        this.indexVersion.update() ;
    }
    
    /**
     * 検索バージョンを取得.
     * <BR><BR>
     * 現在有効な検索バージョンを取得します.
     * <BR>
     * @return IndexVersion 検索バージョンが返されます.
     */
    public IndexVersion getIndexVersion() {
        return this.indexVersion ;
    }
    
    /**
     * インデックス作成用情報をセット.
     * <BR><BR>
     * インデックス作成用情報をセットします.
     * <BR>
     * @param base 追加対象の情報を設定します.
     * @exception Exception 例外.
     */
    public void put( IndexBase base ) throws Exception {
        if( base == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        boolean putFlag = false ;
        ArrayList lst = parseCode( base.getValue(),this.ngram ) ;
        if( lst == null || lst.size() <= 0 ) {
            return ;
        }
        boolean newIndexFlag = false ;
        IndexName name = this.nameCache.get( base.getName() ) ;
        if( name != null ) {
            IndexBase before = IndexBase.getSerializable( this.indexBaseDir,name.getId(),true,true ) ;
            if( before != null ) {
                this.remove( before.getName() ) ;
                base.setId( before.getId() ) ;
                newIndexFlag = true ;
            }
        }
        if( newIndexFlag == false ) {
            base.setId( sequence.getId() ) ;
        }
        int partLen = ( base.getPart() != null && base.getPart().length() > 0 ) ?
            base.getPart().length() : -1 ;
        int sccff = ( partLen == -1 ) ? -1 : ( int )( ( partLen * ( IndexDef.SCORE_COEFFICIENT * ( double )ngram ) ) + 1.0f ) ;
        int len = lst.size() ;
        int cnt = 0 ;
        for( int i = 0 ; i < len ; i ++ ) {
            String code = ( String )lst.get( i ) ;
            if( code == null || ( code = code.trim() ).length() < ngram ) {
                cnt ++ ;
                continue ;
            }
            int lenJ = IndexUtil.getNGramByStringLength( code,ngram ) ;
            if( lenJ > 0 ) {
                putFlag = true ;
                int j ;
                String[] indexStrings = new String[ lenJ ] ;
                for( j = 0 ; j < lenJ ; j ++ ) {
                    indexStrings[ j ] = IndexUtil.getNGramCode( code,ngram,j ).toLowerCase() ;
                }
                for( j = 0 ; j < lenJ ; j ++ ) {
                    String indexString = indexStrings[ j ] ;
                    indexString = IndexUtil.getNGramByNumber( indexString,ngram ) ;
                    IndexChip b = this.cache.get( indexString ) ;
                    if( b == null ) {
                        b = new IndexChip( indexString ) ;
                        this.cache.add( b ) ;
                    }
                    if( lenJ <= j+1 ) {
                        b.addBase( base,sccff,null,cnt ) ;
                    }
                    else {
                        b.addBase( base,sccff,indexStrings[ j+1 ].toCharArray(),cnt ) ;
                    }
                    cnt ++ ;
                }
            }
            cnt ++ ;
        }
        if( putFlag == true ) {
            base.setIndexVersion( this.indexVersion.getVersion() ) ;
            base.putSerialize( indexBaseDir ) ;
            this.nameCache.add( base.getName(),base.getId(),partLen ) ;
            String[] keys = base.getKeys() ;
            if( keys != null && ( len = keys.length ) > 0 ) {
                for( int i = 0 ; i < len ; i ++ ) {
                    IndexKey key = keyCache.get( keys[ i ],i ) ;
                    if( key != null ) {
                        key.add( base.getId() ) ;
                    }
                }
            }
        }
    }
    
    /**
     * インデックス作成用情報を削除.
     * <BR><BR>
     * インデックス作成用情報を削除します.
     * <BR>
     * @param name 削除対象名を設定します.
     * @exception Exception 例外.
     */
    public void remove( String name ) throws Exception {
        IndexName indexName = this.nameCache.get( name ) ;
        if( indexName != null ) {
            IndexBase base = IndexBase.getSerializable( this.indexBaseDir,indexName.getId(),true,true ) ;
            if( base != null ) {
                IndexBase.removeSerializable( this.indexBaseDir,indexName.getId() ) ;
                base.delete() ;
            }
            ArrayList lst = parseCode( base.getValue(),this.ngram ) ;
            if( lst == null || lst.size() <= 0 ) {
                return ;
            }
            int len = lst.size() ;
            for( int i = 0 ; i < len ; i ++ ) {
                String code = ( String )lst.get( i ) ;
                if( code == null || ( code = code.trim() ).length() < ngram ) {
                    continue ;
                }
                int lenJ = IndexUtil.getNGramByStringLength( code,ngram ) ;
                if( lenJ > 0 ) {
                    for( int j = 0 ; j < lenJ ; j ++ ) {
                        String indexString = IndexUtil.getNGramCode( code,ngram,j ).toLowerCase() ;
                        indexString = IndexUtil.getNGramByNumber( indexString,ngram ) ;
                        IndexChip b = this.cache.get( indexString ) ;
                        if( b != null ) {
                            b.removeBase( indexName.getId() ) ;
                        }
                    }
                }
            }
            this.nameCache.remove( name ) ;
            String[] keys = base.getKeys() ;
            if( keys != null && ( len = keys.length ) > 0 ) {
                for( int i = 0 ; i < len ; i ++ ) {
                    IndexKey key = keyCache.get( keys[ i ],i ) ;
                    if( key != null ) {
                        key.remove( base.getId() ) ;
                    }
                }
            }
            if( this.searchIndex != null ) {
                this.searchIndex.clearByInsertIndexRemove() ;
            }
        }
    }
    
    /**
     * 対象名に対するＩＤを取得.
     * <BR><BR>
     * 対象名に対するＩＤを取得します.
     * <BR>
     * @param name 対象名を設定します.
     * @return Long IDが返されます.
     */
    public Long getNameById( String name ) {
        IndexName indexName = this.nameCache.get( name ) ;
        if( indexName != null ) {
            return indexName.getId() ;
        }
        return null ;
    }
    
    /**
     * 対象IDに対するIndexBaseを取得.
     * <BR><BR>
     * 対象IDに対するIndexBaseを取得します.
     * <BR>
     * @param id 対象のIDを設定します.
     * @return IndexBase 対象のIndexBaseが返されます.
     */
    public IndexBase getIndexBase( Long id ) {
        return IndexBase.getSerializable( this.indexBaseDir,id,true,true ) ;
    }
    
    /**
     * 検索ワードを分解して情報セット.
     */
    private static final ArrayList parseCode( String code,int ngram ) {
        code = IndexUtil.cutTag( code ) ;
        code = IndexUtil.invalidityString( code ) ;
        ArrayList<String> lst = StringUtil.cutString( code,IndexUtil.CUT_CODE+IndexUtil.PARSE_CODE ) ;
        if( lst == null || lst.size() <= 0 ) {
            return null ;
        }
        int len = lst.size() ;
        for( int i = len-1 ; i >= 0 ; i -- ) {
            String one = ( String )lst.get( i ) ;
            if( one.length() < ngram ) {
                lst.remove( i ) ;
                if( lst.size() == 0 ) {
                    lst.add( one ) ;
                    break ;
                }
                if( i == 0 ) {
                    lst.set( 0,one + ( String )lst.get( i ) ) ;
                }
                else {
                    int p = i-1 ;
                    for( int j = i-1 ; j >= 0 ; j -- ) {
                        if( ( ( String )lst.get( j ) ).trim().length() > 0 ) {
                            p = j ;
                            break ;
                        }
                    }
                    lst.set( p,( String )lst.get( p ) + one ) ;
                }
            }
        }
        return lst ;
    }
    
    /**
     * シリアライズ化.
     */
    private static final void putSequence( String dir,SequenceLong seq ) {
        FileOutputStream o = null ;
        String out = dir ;
        if( out.endsWith( "/" ) == true || out.endsWith( "\\" ) == true ) {
            out = out.substring( 0,out.length()-1 ) ;
        }
        out = new StringBuilder().append( out ).append( FileUtil.FILE_SPACE ).
            append( SEQ_NAME ).toString() ;
        try{
            o = new FileOutputStream( out ) ;
            o.write( seq.toBinary(),0,SequenceLong.BINARY_LENGTH ) ;
        }catch( Exception e ){
        }finally{
            try{
                o.close() ;
            }catch( Exception e ){
            }
            o = null ;
        }
    }
    
    /**
     * シリアライズから情報を取得.
     */
    private static final SequenceLong getSequence( String dir ) {
        FileInputStream in = null ;
        String out = dir ;
        if( out.endsWith( "/" ) == true || out.endsWith( "\\" ) == true ) {
            out = out.substring( 0,out.length()-1 ) ;
        }
        out = new StringBuilder().append( out ).append( FileUtil.FILE_SPACE ).
            append( SEQ_NAME ).toString() ;
        if( FileUtil.isFileExists( out ) == false ) {
            return null ;
        }
        try{
            in =new FileInputStream( out ) ;
            SequenceLong ret = new SequenceLong() ;
            byte[] bin = new byte[ SequenceLong.BINARY_LENGTH ] ;
            in.read( bin ) ;
            ret.toObject( bin ) ;
            return ret ;
        }catch( Exception e ){
        }finally{
            try{
                in.close() ;
            }catch( Exception e ){
            }
            in = null ;
        }
        return null ;
    }

}

