/*
 * @(#)IndexUtil.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.index.core ;

import java.util.ArrayList;

import org.maachang.util.StringUtil;

/**
 * IndexUtil.
 *
 * @version 2007/06/28
 * @author  masahito suzuki
 * @since   MaachangIndex 1.00
 */
public class IndexUtil {
    
    /**
     * カットコード.
     */
    public static final String CUT_CODE = " 　\t" ;
    
    /**
     * 区切りコード.
     */
    public static final String PARSE_CODE = "(){}[]、。\"\'【】《》〔〕()｢｣「」『』｛｝［］" ;
    
    /**
     * 不要コード.
     */
    public static final String NOT_CODE = "\r\n*/,.･・#$%&*+-/:;<=>?@\\￥!！?？^`|~━┗┛┏┓⇒↑→←↓⇔" ;
    
    /**
     * Digest.
     */
    public static final String DIGEST = "SHA1" ;
    
    /**
     * キャラクターセット.
     */
    public static final String CHARSET = "UTF8" ;
    
    
    /**
     * n-gramに対する文字列分の処理件数を取得.
     * <BR><BR>
     * n-gramに対する文字列分の処理件数を取得します.
     * <BR>
     * @param string 対象の文字列を設定します.
     * @param ngram 対象のn-gram値を設定します.
     * @return int 処理件数が返されます.
     */
    public static int getNGramByStringLength( String string,int ngram ) {
        if( string == null || string.length() <= 0 ) {
            return 0 ;
        }
        return string.length() - ( ngram - 1 ) ;
    }
    
    /**
     * 対象の文字コード列をn-gramの条件で取得.
     * <BR><BR>
     * 対象の文字コード列をn-gramの条件で取得します.
     * <BR>
     * @param code 対象の文字コード列を設定します.
     * @param ngram 対象のngram値を設定します.
     * @param offset ngram開始値のポイントを設定します.
     * @return String n-gramで区切られた情報が返されます.
     */
    public static String getNGramCode( String code,int ngram,int offset ) {
        return code.substring( offset,offset + ngram ) ;
    }
    
    /**
     * n-gramで取得した文字列を数値変換.
     * <BR><BR>
     * n-gramで取得した文字列を数値変換します.
     * <BR>
     * @param indexString 対象のn-gramキー値となる文字列を設定します.
     * @param ngram 対象のn-gram値を設定します.
     * @return String 数値化された情報が返されます.
     */
    public static String getNGramByNumber( String indexString,int ngram ) {
        StringBuilder buf = new StringBuilder() ;
        for( int i = 0 ; i < ngram ; i ++ ) {
            char c = indexString.charAt( i ) ;
            String x = Long.toHexString( ( int )( c & 0x0000ffff ) ) ;
            int ln = x.length() ;
            switch( ln ) {
                case 1 : buf.append( "XXX" ).append( x ) ; break ;
                case 2 : buf.append( "XX" ).append( x ) ; break ;
                case 3 : buf.append( "X" ).append( x ) ; break ;
                case 4 : buf.append( x ) ; break ;
            }
        }
        return buf.toString() ;
    }
    
    /**
     * 対象文字列を数値化.
     * <BR><BR>
     * 対象文字列を数値化します.
     * <BR>
     * @param string 対象の文字列を設定します.
     * @return String 文字列を数値化した情報が返されます.
     */
    public static String getStringByNumber( String string ) {
        StringBuilder buf = new StringBuilder() ;
        int len = string.length() ;
        for( int i = 0 ; i < len ; i ++ ) {
            String x = Integer.toHexString( ( ( ( int )string.charAt( i ) ) & 0x0000ffff ) ) ;
            int ln = x.length() ;
            switch( ln ) {
                case 1 : buf.append( "XXX" ).append( x ) ; break ;
                case 2 : buf.append( "XX" ).append( x ) ; break ;
                case 3 : buf.append( "X" ).append( x ) ; break ;
                case 4 : buf.append( x ) ; break ;
            }
        }
        return buf.toString() ;
    }
    
    /**
     * 特殊文字.
     */
    private static final String[] SPECIAL_CODE = { "&quot;", "&lt;","&gt;",
        "&nbsp;","&raquo","&copy;" } ;
    
    /**
     * 特殊文字変換.
     * <BR><BR>
     * 特殊文字変換を行います.
     * <BR>
     * @param code 対象の文字列を設定します.
     * @return String 変換された文字列が返されます.
     */
    public static final String convertSpecial( String code ) {
        for( int i = 0 ; i < SPECIAL_CODE.length ; i ++ ) {
            code = StringUtil.changeString( code,SPECIAL_CODE[ i ]," " ) ;
        }
        return code ;
    }
    
    /**
     * 無効文字をカット.
     * <BR><BR>
     * 無効文字をカットします.
     * <BR>
     * @param string 対象の文字列を設定します.
     * @return String 無効文字が外された文字列が返されます.
     */
    public static final String invalidityString( String string ) {
        int len = IndexUtil.NOT_CODE.length() ;
        for( int i = 0 ; i < len ; i ++ ) {
            String s = IndexUtil.NOT_CODE.substring( i,i+1 ) ;
            string = StringUtil.changeString( string,s," " ) ;
        }
        return string ;
    }
    
    /**
     * HTMLタグ内容を外す.
     * <BR><BR>
     * HTMLタグ形式の内容を除外した内容を取得します.
     * <BR>
     * @param code 変換元の情報を設定します.
     * @return String HTMLタグ内容が外された内容が返されます.
     */
    public static final String cutTag( String code ) {
        StringBuilder buf = new StringBuilder() ;
        int ed,bf ;
        int len = code.length() ;
        for( bf = 0 ;; ) {
            int p = code.indexOf( "<",bf ) ;
            if( p != -1 ) {
                if( p + 4 < len && code.charAt( p ) == '<' && code.charAt( p+1 ) == '!' &&
                    code.charAt( p+2 ) == '-' && code.charAt( p+3 ) == '-' ) {
                    ed = code.indexOf( "-->",p ) ;
                    if( ed != -1 ) {
                        buf.append( code.substring( bf,p ) ) ;
                        buf.append( " " ) ;
                        bf = ed+3 ;
                    }
                }
                else if( ( ed = code.indexOf( ">",p ) ) != -1 ) {
                    buf.append( code.substring( bf,p ) ) ;
                    buf.append( " " ) ;
                    bf = ed+1 ;
                }
                else {
                    buf.append( code.substring( bf ) ) ;
                    break ;
                }
            }
            else {
                buf.append( code.substring( bf ) ) ;
                break ;
            }
        }
        return IndexUtil.convertSpecial( buf.toString() ) ;
    }
    
    /**
     * 検索ワードに対して有効ワードを区分して取得.
     * <BR><BR>
     * 検索ワードに対して有効ワードを区分して取得します.
     * <BR>
     * @param string 検索ワードを設定します.
     * @return String[] 有効検索文字列が返されます.
     */
    public static final String[] cutSearchString( String string ) {
        string = IndexUtil.invalidityString( string ) ;
        ArrayList lst = StringUtil.cutString( string,IndexUtil.CUT_CODE+IndexUtil.PARSE_CODE ) ;
        int len ;
        if( lst != null && ( len = lst.size() ) > 0 ) {
            String[] ret = new String[ len ] ;
            for( int i = 0 ; i < len ; i ++ ) {
                ret[ i ] = ( ( String )lst.get( i ) ).toLowerCase() ;
            }
            return ret ;
        }
        return null ;
    }
}

