/*
 * @(#)MaachangSearch.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.index ;

import org.maachang.index.core.SearchIndex;

/**
 * 検索処理.
 *
 * @version 2007/07/08
 * @author  masahito suzuki
 * @since   MaachangIndex 1.00
 */
public class MaachangSearch {
    
    /**
     * 検索コア処理.
     */
    private SearchIndex searchIndex = null ;
    
    /**
     * コンストラクタ.
     */
    private MaachangSearch() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * インデックス検索用オブジェクトを生成します.
     * <BR>
     * @param n インデックス幅を設定します.
     * @param cache 最大キャッシュサイズを設定します.
     * @param baseCache 最大IndexBaseキャッシュサイズを設定します.
     * @param history 最大ヒストリサイズを設定します.
     * @param keyCache 最大IndexKeyキャッシュサイズを設定します.
     * @param baseDir インデックス出力用ディレクトリを設定します.
     * @exception Exception 例外.
     */
    public MaachangSearch( int n,int cache,int baseCache,int history,int keyCache,String baseDir )
        throws Exception {
        searchIndex = new SearchIndex( n,cache,baseCache,history,keyCache,baseDir ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * インデックス検索用オブジェクトを生成します.
     * <BR>
     * @param n インデックス幅を設定します.
     * @param cache 最大キャッシュサイズを設定します.
     * @param baseCache 最大IndexBaseキャッシュサイズを設定します.
     * @param history 最大ヒストリサイズを設定します.
     * @param keyCache 最大IndexKeyキャッシュサイズを設定します.
     * @param baseDir インデックス出力用ディレクトリを設定します.
     * @param partLength 検索結果のヒント表示文字列長を設定します.
     * @exception Exception 例外.
     */
    public MaachangSearch( int n,int cache,int baseCache,int history,int keyCache,String baseDir,int partLength )
        throws Exception {
        searchIndex = new SearchIndex( n,cache,baseCache,history,keyCache,baseDir,partLength ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * インデックス検索用オブジェクトを生成します.
     * <BR>
     * @param n インデックス幅を設定します.
     * @param cache 最大キャッシュサイズを設定します.
     * @param baseCache 最大IndexBaseキャッシュサイズを設定します.
     * @param history 最大ヒストリサイズを設定します.
     * @param keyCache 最大IndexKeyキャッシュサイズを設定します.
     * @param baseDir インデックス出力用ディレクトリを設定します.
     * @exception Exception 例外.
     */
    public MaachangSearch( MaachangIndex index,int baseCache,int history )
        throws Exception {
        searchIndex = new SearchIndex( index.getInsertIndex(),baseCache,history ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * インデックス検索用オブジェクトを生成します.
     * <BR>
     * @param n インデックス幅を設定します.
     * @param cache 最大キャッシュサイズを設定します.
     * @param baseCache 最大IndexBaseキャッシュサイズを設定します.
     * @param history 最大ヒストリサイズを設定します.
     * @param keyCache 最大IndexKeyキャッシュサイズを設定します.
     * @param baseDir インデックス出力用ディレクトリを設定します.
     * @param partLength 検索結果のヒント表示文字列長を設定します.
     * @exception Exception 例外.
     */
    public MaachangSearch( MaachangIndex index,int baseCache,int history,int partLength )
        throws Exception {
        searchIndex = new SearchIndex( index.getInsertIndex(),baseCache,history,partLength ) ;
    }
    
    /**
     * オブジェクト解放.
     * <BR><BR>
     * オブジェクトを解放した場合の処理を実装.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.flush() ;
    }
    
    /**
     * 情報書き込み.
     * <BR><BR>
     * 検索結果のHistoryを書き込みます.
     */
    public void flush() {
        if( searchIndex != null ) {
            searchIndex.flush() ;
        }
    }
    
    /**
     * 検索バージョンを取得.
     * <BR><BR>
     * 現在有効な検索バージョンを取得します.
     * <BR>
     * @return String 検索バージョンが返されます.
     */
    public String getIndexVersion() {
        return String.valueOf( searchIndex.getIndexVersion().getVersion() ) ;
    }
    
    /**
     * 検索結果のヒント表示文字列長を取得.
     * <BR><BR>
     * 検索結果のヒント表示文字列長を取得します.
     * <BR>
     * @return int 検索結果のヒント表示文字列長が返されます.
     */
    public int getPartLength() {
        return searchIndex.getPartLength() ;
    }
    
    /**
     * 新しいキーワードで検索.
     * <BR><BR>
     * 新しいキーワードで検索処理を行う場合に利用します.
     * <BR>
     * @param word 対象の検索キーワードを設定します.
     * @return MaachangResult 検索結果が返されます.
     * @exception Exception 例外.
     */
    public MaachangResult searchByNewWord( String word )
        throws Exception {
        return searchIndex.searchByNewWord( word,null ) ;
    }
    
    /**
     * 新しいキーワードで検索.
     * <BR><BR>
     * 新しいキーワードで検索処理を行う場合に利用します.
     * <BR>
     * @param word 対象の検索キーワードを設定します.
     * @param length １回で表示するデータ件数を設定します.
     * @return MaachangResult 検索結果が返されます.
     * @exception Exception 例外.
     */
    public MaachangResult searchByNewWord( String word,int length )
        throws Exception {
        return searchIndex.searchByNewWord( word,length,null ) ;
    }
    
    /**
     * 新しいキーワードで検索.
     * <BR><BR>
     * 新しいキーワードで検索処理を行う場合に利用します.
     * <BR>
     * @param word 対象の検索キーワードを設定します.
     * @param subKeys 対象のSubKey群を設定します.
     * @return MaachangResult 検索結果が返されます.
     * @exception Exception 例外.
     */
    public MaachangResult searchByNewWord( String word,SubKey... subkeys )
        throws Exception {
        return searchIndex.searchByNewWord( word,subkeys ) ;
    }
    
    /**
     * 新しいキーワードで検索.
     * <BR><BR>
     * 新しいキーワードで検索処理を行う場合に利用します.
     * <BR>
     * @param word 対象の検索キーワードを設定します.
     * @param length １回で表示するデータ件数を設定します.
     * @param subKeys 対象のSubKey群を設定します.
     * @return MaachangResult 検索結果が返されます.
     * @exception Exception 例外.
     */
    public MaachangResult searchByNewWord( String word,int length,SubKey... subkeys )
        throws Exception {
        return searchIndex.searchByNewWord( word,length,subkeys ) ;
    }
    
    /**
     * 前回キーワード条件で、次ページの内容を取得.
     * <BR><BR>
     * 前回キーワード条件で、次ページの内容を取得します.
     * <BR>
     * @param result 前回の検索条件を設定します.
     * @return boolean 処理結果が返されます.<BR>
     *                 [true]が返された場合、次ページの情報が取得できました.
     * @exception Exception 例外.
     */
    public boolean nextSearch( MaachangResult result )
        throws Exception {
        return searchIndex.nextSearch( result ) ;
    }
    
    /**
     * 前回キーワード条件で、前ページの内容を取得.
     * <BR><BR>
     * 前回キーワード条件で、前ページの内容を取得します.
     * <BR>
     * @param result 前回の検索条件を設定します.
     * @return boolean 処理結果が返されます.<BR>
     *                 [true]が返された場合、前ページの情報が取得できました.
     * @exception Exception 例外.
     */
    public boolean beforeSearch( MaachangResult result )
        throws Exception {
        return searchIndex.beforeSearch( result ) ;
    }
    
    /**
     * 前回キーワード条件で、指定ページの内容を取得.
     * <BR><BR>
     * 前回キーワード条件で、指定ページの内容を取得します.
     * <BR>
     * @param result 前回の検索条件を設定します.
     * @param pageNo 指定ページNoを設定します.
     * @return boolean 処理結果が返されます.<BR>
     *                 [true]が返された場合、指定ページの情報が取得できました.
     * @exception Exception 例外.
     */
    public boolean pageNoSearch( MaachangResult result,int pageNo )
        throws Exception {
        return searchIndex.pageNoSearch( result,pageNo ) ;
    }
    
    /**
     * 検索モードを取得.
     * <BR><BR>
     * 検索モードを取得します.
     * <BR>
     * @return boolean 検索モードが返されます.<BR>
     *                 [true]が返された場合、Insert+Searchモードです.<BR>
     *                 [false]が返された場合、Searchモードです.
     */
    public boolean isMode() {
        return searchIndex.isMode() ;
    }
}

