/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import java.util.Deque;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSInputStream;
import org.apache.hadoop.hdfs.DFSUtilClient;
import org.apache.hadoop.hdfs.protocol.ClientDatanodeProtocol;
import org.apache.hadoop.hdfs.protocol.DatanodeInfo;
import org.apache.hadoop.hdfs.protocol.DatanodeLocalInfo;
import org.apache.hadoop.util.Daemon;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DeadNodeDetector
extends Daemon {
    public static final Logger LOG = LoggerFactory.getLogger(DeadNodeDetector.class);
    private static final long ERROR_SLEEP_MS = 5000L;
    private final long idleSleepMs;
    private String name;
    private Configuration conf;
    private final Map<String, DatanodeInfo> deadNodes;
    private final Map<DFSInputStream, HashSet<DatanodeInfo>> suspectAndDeadNodes;
    private Map<String, DatanodeInfo> probeInProg = new ConcurrentHashMap<String, DatanodeInfo>();
    private long deadNodeDetectInterval = 0L;
    private long suspectNodeDetectInterval = 0L;
    private long probeConnectionTimeoutMs;
    private UniqueQueue<DatanodeInfo> deadNodesProbeQueue;
    private UniqueQueue<DatanodeInfo> suspectNodesProbeQueue;
    private ExecutorService probeDeadNodesThreadPool;
    private ExecutorService probeSuspectNodesThreadPool;
    private Thread probeDeadNodesSchedulerThr;
    private Thread probeSuspectNodesSchedulerThr;
    private ExecutorService rpcThreadPool;
    private int socketTimeout;
    private static volatile boolean disabledProbeThreadForTest = false;
    private State state;

    public DeadNodeDetector(String name, Configuration conf) {
        this.conf = new Configuration(conf);
        this.deadNodes = new ConcurrentHashMap<String, DatanodeInfo>();
        this.suspectAndDeadNodes = new ConcurrentHashMap<DFSInputStream, HashSet<DatanodeInfo>>();
        this.name = name;
        this.deadNodeDetectInterval = conf.getLong("dfs.client.deadnode.detection.probe.deadnode.interval.ms", 60000L);
        this.suspectNodeDetectInterval = conf.getLong("dfs.client.deadnode.detection.probe.suspectnode.interval.ms", 300L);
        this.socketTimeout = conf.getInt("dfs.client.socket-timeout", 60000);
        this.probeConnectionTimeoutMs = conf.getLong("dfs.client.deadnode.detection.probe.connection.timeout.ms", 20000L);
        this.deadNodesProbeQueue = new UniqueQueue();
        this.suspectNodesProbeQueue = new UniqueQueue();
        this.idleSleepMs = conf.getLong("dfs.client.deadnode.detection.idle.sleep.ms", 10000L);
        int deadNodeDetectDeadThreads = conf.getInt("dfs.client.deadnode.detection.probe.deadnode.threads", 10);
        int suspectNodeDetectDeadThreads = conf.getInt("dfs.client.deadnode.detection.probe.suspectnode.threads", 10);
        int rpcThreads = conf.getInt("dfs.client.deadnode.detection.rpc.threads", 20);
        this.probeDeadNodesThreadPool = Executors.newFixedThreadPool(deadNodeDetectDeadThreads, new Daemon.DaemonFactory());
        this.probeSuspectNodesThreadPool = Executors.newFixedThreadPool(suspectNodeDetectDeadThreads, new Daemon.DaemonFactory());
        this.rpcThreadPool = Executors.newFixedThreadPool(rpcThreads, new Daemon.DaemonFactory());
        if (!disabledProbeThreadForTest) {
            this.startProbeScheduler();
        }
        LOG.info("Start dead node detector for DFSClient {}.", (Object)this.name);
        this.state = State.INIT;
    }

    @Override
    public void run() {
        block8: while (!Thread.currentThread().isInterrupted()) {
            this.clearAndGetDetectedDeadNodes();
            LOG.debug("Current detector state {}, the detected nodes: {}.", (Object)this.state, this.deadNodes.values());
            switch (this.state) {
                case INIT: {
                    this.init();
                    continue block8;
                }
                case CHECK_DEAD: {
                    this.checkDeadNodes();
                    continue block8;
                }
                case IDLE: {
                    this.idle();
                    continue block8;
                }
                case ERROR: {
                    try {
                        Thread.sleep(5000L);
                    }
                    catch (InterruptedException e) {
                        LOG.debug("Got interrupted while DeadNodeDetector is error.", (Throwable)e);
                        Thread.currentThread().interrupt();
                    }
                    return;
                }
            }
        }
    }

    public void shutdown() {
        DeadNodeDetector.threadShutDown(this);
        DeadNodeDetector.threadShutDown(this.probeDeadNodesSchedulerThr);
        DeadNodeDetector.threadShutDown(this.probeSuspectNodesSchedulerThr);
        this.probeDeadNodesThreadPool.shutdown();
        this.probeSuspectNodesThreadPool.shutdown();
        this.rpcThreadPool.shutdown();
    }

    private static void threadShutDown(Thread thread) {
        if (thread != null && thread.isAlive()) {
            thread.interrupt();
            try {
                thread.join();
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
    }

    @VisibleForTesting
    boolean isThreadsShutdown() {
        return !this.isAlive() && !this.probeDeadNodesSchedulerThr.isAlive() && !this.probeSuspectNodesSchedulerThr.isAlive() && this.probeDeadNodesThreadPool.isShutdown() && this.probeSuspectNodesThreadPool.isShutdown() && this.rpcThreadPool.isShutdown();
    }

    @VisibleForTesting
    static void setDisabledProbeThreadForTest(boolean disabledProbeThreadForTest) {
        DeadNodeDetector.disabledProbeThreadForTest = disabledProbeThreadForTest;
    }

    @VisibleForTesting
    void startProbeScheduler() {
        this.probeDeadNodesSchedulerThr = new Thread(new ProbeScheduler(this, ProbeType.CHECK_DEAD));
        this.probeDeadNodesSchedulerThr.setDaemon(true);
        this.probeDeadNodesSchedulerThr.start();
        this.probeSuspectNodesSchedulerThr = new Thread(new ProbeScheduler(this, ProbeType.CHECK_SUSPECT));
        this.probeSuspectNodesSchedulerThr.setDaemon(true);
        this.probeSuspectNodesSchedulerThr.start();
    }

    private void scheduleProbe(ProbeType type) {
        block4: {
            DatanodeInfo datanodeInfo;
            block3: {
                LOG.debug("Schedule probe datanode for probe type: {}.", (Object)type);
                datanodeInfo = null;
                if (type != ProbeType.CHECK_DEAD) break block3;
                while ((datanodeInfo = this.deadNodesProbeQueue.poll()) != null) {
                    if (this.probeInProg.containsKey(datanodeInfo.getDatanodeUuid())) {
                        LOG.debug("The datanode {} is already contained in probe queue, skip to add it.", (Object)datanodeInfo);
                        continue;
                    }
                    this.probeInProg.put(datanodeInfo.getDatanodeUuid(), datanodeInfo);
                    Probe probe = new Probe(this, datanodeInfo, ProbeType.CHECK_DEAD);
                    this.probeDeadNodesThreadPool.execute(probe);
                }
                break block4;
            }
            if (type != ProbeType.CHECK_SUSPECT) break block4;
            while ((datanodeInfo = this.suspectNodesProbeQueue.poll()) != null) {
                if (this.probeInProg.containsKey(datanodeInfo.getDatanodeUuid())) continue;
                this.probeInProg.put(datanodeInfo.getDatanodeUuid(), datanodeInfo);
                Probe probe = new Probe(this, datanodeInfo, ProbeType.CHECK_SUSPECT);
                this.probeSuspectNodesThreadPool.execute(probe);
            }
        }
    }

    private void probeCallBack(Probe probe, boolean success) {
        LOG.debug("Probe datanode: {} result: {}, type: {}", new Object[]{probe.getDatanodeInfo(), success, probe.getType()});
        this.probeInProg.remove(probe.getDatanodeInfo().getDatanodeUuid());
        if (success) {
            if (probe.getType() == ProbeType.CHECK_DEAD) {
                LOG.info("Remove the node out from dead node list: {}.", (Object)probe.getDatanodeInfo());
                this.removeDeadNode(probe.getDatanodeInfo());
            } else if (probe.getType() == ProbeType.CHECK_SUSPECT) {
                LOG.debug("Remove the node out from suspect node list: {}.", (Object)probe.getDatanodeInfo());
                this.removeNodeFromDeadNodeDetector(probe.getDatanodeInfo());
            }
        } else if (probe.getType() == ProbeType.CHECK_SUSPECT) {
            LOG.warn("Probe failed, add suspect node to dead node list: {}.", (Object)probe.getDatanodeInfo());
            this.addToDead(probe.getDatanodeInfo());
        }
    }

    private void checkDeadNodes() {
        Set<DatanodeInfo> datanodeInfos = this.clearAndGetDetectedDeadNodes();
        for (DatanodeInfo datanodeInfo : datanodeInfos) {
            if (!this.deadNodesProbeQueue.offer(datanodeInfo)) {
                LOG.debug("Skip to add dead node {} to check since the node is already in the probe queue.", (Object)datanodeInfo);
                continue;
            }
            LOG.debug("Add dead node to check: {}.", (Object)datanodeInfo);
        }
        this.state = State.IDLE;
    }

    private void idle() {
        try {
            Thread.sleep(this.idleSleepMs);
        }
        catch (InterruptedException e) {
            LOG.debug("Got interrupted while DeadNodeDetector is idle.", (Throwable)e);
            Thread.currentThread().interrupt();
        }
        this.state = State.CHECK_DEAD;
    }

    private void init() {
        this.state = State.CHECK_DEAD;
    }

    private void addToDead(DatanodeInfo datanodeInfo) {
        this.deadNodes.put(datanodeInfo.getDatanodeUuid(), datanodeInfo);
    }

    public boolean isDeadNode(DatanodeInfo datanodeInfo) {
        return this.deadNodes.containsKey(datanodeInfo.getDatanodeUuid());
    }

    private void removeFromDead(DatanodeInfo datanodeInfo) {
        this.deadNodes.remove(datanodeInfo.getDatanodeUuid());
    }

    public UniqueQueue<DatanodeInfo> getDeadNodesProbeQueue() {
        return this.deadNodesProbeQueue;
    }

    public UniqueQueue<DatanodeInfo> getSuspectNodesProbeQueue() {
        return this.suspectNodesProbeQueue;
    }

    @VisibleForTesting
    void setSuspectQueue(UniqueQueue<DatanodeInfo> queue) {
        this.suspectNodesProbeQueue = queue;
    }

    @VisibleForTesting
    void setDeadQueue(UniqueQueue<DatanodeInfo> queue) {
        this.deadNodesProbeQueue = queue;
    }

    public synchronized void addNodeToDetect(DFSInputStream dfsInputStream, DatanodeInfo datanodeInfo) {
        HashSet<DatanodeInfo> datanodeInfos = this.suspectAndDeadNodes.get(dfsInputStream);
        if (datanodeInfos == null) {
            datanodeInfos = new HashSet();
            datanodeInfos.add(datanodeInfo);
            this.suspectAndDeadNodes.putIfAbsent(dfsInputStream, datanodeInfos);
        } else {
            datanodeInfos.add(datanodeInfo);
        }
        LOG.debug("Add datanode {} to suspectAndDeadNodes.", (Object)datanodeInfo);
        this.addSuspectNodeToDetect(datanodeInfo);
    }

    private boolean addSuspectNodeToDetect(DatanodeInfo datanodeInfo) {
        return this.suspectNodesProbeQueue.offer(datanodeInfo);
    }

    public synchronized Set<DatanodeInfo> clearAndGetDetectedDeadNodes() {
        HashSet<DatanodeInfo> newDeadNodes = new HashSet<DatanodeInfo>();
        for (HashSet<DatanodeInfo> datanodeInfos : this.suspectAndDeadNodes.values()) {
            newDeadNodes.addAll(datanodeInfos);
        }
        for (DatanodeInfo datanodeInfo : this.deadNodes.values()) {
            if (newDeadNodes.contains(datanodeInfo)) continue;
            this.deadNodes.remove(datanodeInfo.getDatanodeUuid());
        }
        return new HashSet<DatanodeInfo>(this.deadNodes.values());
    }

    public synchronized void removeNodeFromDeadNodeDetector(DFSInputStream dfsInputStream, DatanodeInfo datanodeInfo) {
        Set datanodeInfos = this.suspectAndDeadNodes.get(dfsInputStream);
        if (datanodeInfos != null) {
            datanodeInfos.remove(datanodeInfo);
            dfsInputStream.removeFromLocalDeadNodes(datanodeInfo);
            if (datanodeInfos.isEmpty()) {
                this.suspectAndDeadNodes.remove(dfsInputStream);
            }
        }
    }

    private synchronized void removeNodeFromDeadNodeDetector(DatanodeInfo datanodeInfo) {
        for (Map.Entry<DFSInputStream, HashSet<DatanodeInfo>> entry : this.suspectAndDeadNodes.entrySet()) {
            Set datanodeInfos = entry.getValue();
            if (!datanodeInfos.remove(datanodeInfo)) continue;
            DFSInputStream dfsInputStream = entry.getKey();
            dfsInputStream.removeFromLocalDeadNodes(datanodeInfo);
            if (!datanodeInfos.isEmpty()) continue;
            this.suspectAndDeadNodes.remove(dfsInputStream);
        }
    }

    private void removeDeadNode(DatanodeInfo datanodeInfo) {
        this.removeNodeFromDeadNodeDetector(datanodeInfo);
        this.removeFromDead(datanodeInfo);
    }

    private static void probeSleep(long time) {
        try {
            Thread.sleep(time);
        }
        catch (InterruptedException e) {
            LOG.debug("Got interrupted while probe is scheduling.", (Throwable)e);
            Thread.currentThread().interrupt();
            return;
        }
    }

    static class ProbeScheduler
    implements Runnable {
        private DeadNodeDetector deadNodeDetector;
        private ProbeType type;

        ProbeScheduler(DeadNodeDetector deadNodeDetector, ProbeType type) {
            this.deadNodeDetector = deadNodeDetector;
            this.type = type;
        }

        @Override
        public void run() {
            while (!Thread.currentThread().isInterrupted()) {
                this.deadNodeDetector.scheduleProbe(this.type);
                if (this.type == ProbeType.CHECK_SUSPECT) {
                    DeadNodeDetector.probeSleep(this.deadNodeDetector.suspectNodeDetectInterval);
                    continue;
                }
                DeadNodeDetector.probeSleep(this.deadNodeDetector.deadNodeDetectInterval);
            }
        }
    }

    class Probe
    implements Runnable {
        private DeadNodeDetector deadNodeDetector;
        private DatanodeInfo datanodeInfo;
        private ProbeType type;

        Probe(DeadNodeDetector deadNodeDetector, DatanodeInfo datanodeInfo, ProbeType type) {
            this.deadNodeDetector = deadNodeDetector;
            this.datanodeInfo = datanodeInfo;
            this.type = type;
        }

        public DatanodeInfo getDatanodeInfo() {
            return this.datanodeInfo;
        }

        public ProbeType getType() {
            return this.type;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            LOG.debug("Check node: {}, type: {}.", (Object)this.datanodeInfo, (Object)this.type);
            try {
                final ClientDatanodeProtocol proxy = DFSUtilClient.createClientDatanodeProtocolProxy(this.datanodeInfo, this.deadNodeDetector.conf, DeadNodeDetector.this.socketTimeout, true);
                Future future = DeadNodeDetector.this.rpcThreadPool.submit(new Callable(){

                    public DatanodeLocalInfo call() throws Exception {
                        return proxy.getDatanodeInfo();
                    }
                });
                try {
                    future.get(DeadNodeDetector.this.probeConnectionTimeoutMs, TimeUnit.MILLISECONDS);
                }
                catch (TimeoutException e) {
                    LOG.error("Probe failed, datanode: {}, type: {}.", new Object[]{this.datanodeInfo, this.type, e});
                    this.deadNodeDetector.probeCallBack(this, false);
                    return;
                }
                finally {
                    future.cancel(true);
                }
                this.deadNodeDetector.probeCallBack(this, true);
                return;
            }
            catch (Exception e) {
                LOG.error("Probe failed, datanode: {}, type: {}.", new Object[]{this.datanodeInfo, this.type, e});
                this.deadNodeDetector.probeCallBack(this, false);
                return;
            }
        }
    }

    static class UniqueQueue<T> {
        private Deque<T> queue = new LinkedList<T>();
        private Set<T> set = new HashSet<T>();

        UniqueQueue() {
        }

        synchronized boolean offer(T dn) {
            if (this.set.add(dn)) {
                this.queue.addLast(dn);
                return true;
            }
            return false;
        }

        synchronized T poll() {
            T dn = this.queue.pollFirst();
            this.set.remove(dn);
            return dn;
        }

        synchronized int size() {
            return this.set.size();
        }
    }

    private static enum State {
        INIT,
        CHECK_DEAD,
        IDLE,
        ERROR;

    }

    private static enum ProbeType {
        CHECK_DEAD,
        CHECK_SUSPECT;

    }
}

