/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.sstable;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.SSTableId;
import org.apache.cassandra.io.sstable.SSTableIdFactory;
import org.apache.cassandra.io.sstable.format.SSTableFormat;
import org.apache.cassandra.io.sstable.format.Version;
import org.apache.cassandra.io.sstable.metadata.IMetadataSerializer;
import org.apache.cassandra.io.sstable.metadata.MetadataSerializer;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.utils.Pair;
import org.apache.cassandra.utils.TimeUUID;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Descriptor {
    private static final Logger logger = LoggerFactory.getLogger(Descriptor.class);
    static final Pattern SSTABLE_DIR_PATTERN = Pattern.compile(".*/(?<keyspace>\\w+)/(?<tableName>\\w+)-(?<tableId>[0-9a-f]{32})/(backups/|snapshots/(?<tag>[^/]+)/)?(\\.(?<indexName>[\\w-]+)/)?(?<component>[\\w-\\+]+)\\.(?<ext>[\\w]+)$");
    static final Pattern LEGACY_SSTABLE_DIR_PATTERN = Pattern.compile(".*/(?<keyspace>\\w+)/(?<tableName>\\w+)/(backups/|snapshots/(?<tag>[^/]+)/)?(\\.(?<indexName>[\\w-]+)/)?(?<component>[\\w-]+)\\.(?<ext>[\\w]+)$");
    private static final String LEGACY_TMP_REGEX_STR = "^((.*)\\-(.*)\\-)?tmp(link)?\\-((?:l|k).)\\-(\\d)*\\-(.*)$";
    private static final Pattern LEGACY_TMP_REGEX = Pattern.compile("^((.*)\\-(.*)\\-)?tmp(link)?\\-((?:l|k).)\\-(\\d)*\\-(.*)$");
    public static final String EXTENSION = ".db";
    public static String TMP_EXT = ".tmp";
    public static final char FILENAME_SEPARATOR = '-';
    private static final Splitter filenameSplitter = Splitter.on('-');
    public final File directory;
    public final Version version;
    public final String ksname;
    public final String cfname;
    public final SSTableId id;
    private final int hashCode;
    private final String prefix;
    private final File baseFile;

    @VisibleForTesting
    public Descriptor(File directory, String ksname, String cfname, SSTableId id) {
        this(DatabaseDescriptor.getSelectedSSTableFormat().getLatestVersion(), directory, ksname, cfname, id);
    }

    public Descriptor(File directory, String ksname, String cfname, SSTableId id, SSTableFormat<?, ?> format) {
        this(format.getLatestVersion(), directory, ksname, cfname, id);
    }

    @VisibleForTesting
    public Descriptor(String version, File directory, String ksname, String cfname, SSTableId id, SSTableFormat<?, ?> format) {
        this(format.getVersion(version), directory, ksname, cfname, id);
    }

    public Descriptor(Version version, File directory, String ksname, String cfname, SSTableId id) {
        Preconditions.checkNotNull(version);
        Preconditions.checkNotNull(directory);
        Preconditions.checkNotNull(ksname);
        Preconditions.checkNotNull(cfname);
        this.version = version;
        this.directory = directory.toCanonical();
        this.ksname = ksname;
        this.cfname = cfname;
        this.id = id;
        StringBuilder buf = new StringBuilder();
        this.appendFileName(buf);
        this.prefix = buf.toString();
        this.baseFile = new File(directory.toPath().resolve(this.prefix));
        this.hashCode = Objects.hashCode(version, id, ksname, cfname);
    }

    private String tmpFilenameFor(Component component) {
        return this.fileFor(component) + TMP_EXT;
    }

    public File tmpFileFor(Component component) {
        return new File(this.directory.toPath().resolve(this.tmpFilenameFor(component)));
    }

    private String tmpFilenameForStreaming(Component component) {
        return String.format("%s.%s%s", this.filenameFor(component), TimeUUID.Generator.nextTimeUUID(), TMP_EXT);
    }

    public File tmpFileForStreaming(Component component) {
        return new File(this.directory.toPath().resolve(this.tmpFilenameForStreaming(component)));
    }

    private String filenameFor(Component component) {
        return this.prefix + "-" + component.name();
    }

    public File fileFor(Component component) {
        return new File(this.directory.toPath().resolve(this.filenameFor(component)));
    }

    public File baseFile() {
        return this.baseFile;
    }

    private void appendFileName(StringBuilder buff) {
        buff.append(this.version).append('-');
        buff.append(this.id.toString());
        buff.append('-').append(this.version.format.name());
    }

    public String relativeFilenameFor(Component component) {
        StringBuilder buff = new StringBuilder();
        if (Directories.isSecondaryIndexFolder(this.directory)) {
            buff.append(this.directory.name()).append(File.pathSeparator());
        }
        this.appendFileName(buff);
        buff.append('-').append(component.name());
        return buff.toString();
    }

    public SSTableFormat<?, ?> getFormat() {
        return this.version.format;
    }

    public List<File> getTemporaryFiles() {
        File[] tmpFiles = this.directory.tryList((dir, name) -> name.endsWith(TMP_EXT));
        ArrayList<File> ret = new ArrayList<File>(tmpFiles.length);
        for (File tmpFile : tmpFiles) {
            ret.add(tmpFile);
        }
        return ret;
    }

    public Set<Component> getComponents(Set<Component> mandatory, Set<Component> optional) {
        ImmutableSet.Builder builder = ImmutableSet.builder();
        builder.addAll(mandatory);
        for (Component component : optional) {
            if (!this.fileFor(component).exists()) continue;
            builder.add(component);
        }
        return builder.build();
    }

    public static boolean isValidFile(File file) {
        String filename = file.name();
        return filename.endsWith(EXTENSION) && !LEGACY_TMP_REGEX.matcher(filename).matches();
    }

    public static Descriptor fromFile(File file) {
        return (Descriptor)Descriptor.fromFileWithComponent((File)file).left;
    }

    public static Component componentFromFile(File file) {
        String name = file.name();
        List<String> tokens = Descriptor.filenameTokens(name);
        return Component.parse(tokens.get(3), Descriptor.formatFromName(name, tokens));
    }

    private static SSTableFormat<?, ?> formatFromName(String fileName, List<String> tokens) {
        String formatString = tokens.get(2);
        SSTableFormat<?, ?> format = DatabaseDescriptor.getSSTableFormats().get(formatString);
        if (format == null) {
            throw Descriptor.invalidSSTable(fileName, "unknown 'format' part (%s)", formatString);
        }
        return format;
    }

    public static Pair<Descriptor, Component> fromFileWithComponent(File file) {
        return Descriptor.fromFileWithComponent(file, true);
    }

    public static Pair<Descriptor, Component> fromFileWithComponent(File file, boolean validateDirs) {
        if (!file.isAbsolute()) {
            file = file.toAbsolute();
        }
        SSTableInfo info = Descriptor.validateAndExtractInfo(file);
        String name = file.name();
        String keyspaceName = "";
        String tableName = "";
        Matcher sstableDirMatcher = SSTABLE_DIR_PATTERN.matcher(file.toString());
        if (!sstableDirMatcher.find(0)) {
            sstableDirMatcher = LEGACY_SSTABLE_DIR_PATTERN.matcher(file.toString());
        }
        if (sstableDirMatcher.find(0)) {
            keyspaceName = sstableDirMatcher.group("keyspace");
            tableName = sstableDirMatcher.group("tableName");
            String indexName = sstableDirMatcher.group("indexName");
            if (indexName != null) {
                tableName = String.format("%s.%s", tableName, indexName);
            }
        } else if (validateDirs) {
            logger.debug("Could not extract keyspace/table info from sstable directory {}", (Object)file.toString());
            throw Descriptor.invalidSSTable(name, String.format("cannot extract keyspace and table name from %s; make sure the sstable is in the proper sub-directories", file), new Object[0]);
        }
        return Pair.create(new Descriptor(info.version, Descriptor.parentOf(name, file), keyspaceName, tableName, info.id), info.component);
    }

    public static Pair<Descriptor, Component> fromFileWithComponent(File file, String keyspace, String table) {
        if (null == keyspace || null == table) {
            return Descriptor.fromFileWithComponent(file);
        }
        SSTableInfo info = Descriptor.validateAndExtractInfo(file);
        return Pair.create(new Descriptor(info.version, Descriptor.parentOf(file.name(), file), keyspace, table, info.id), info.component);
    }

    private static List<String> filenameTokens(String name) {
        List<String> tokens = filenameSplitter.splitToList(name);
        int size = tokens.size();
        if (size != 4) {
            if (size == 5 || size == 6) {
                throw new IllegalArgumentException(String.format("%s is of version %s which is now unsupported and cannot be read.", name, tokens.get(size - 3)));
            }
            throw new IllegalArgumentException(String.format("Invalid sstable file %s: the name doesn't look like a supported sstable file name", name));
        }
        return tokens;
    }

    private static SSTableInfo validateAndExtractInfo(File file) {
        SSTableId id;
        String name = file.name();
        List<String> tokens = Descriptor.filenameTokens(name);
        String versionString = tokens.get(0);
        if (!Version.validate(versionString)) {
            throw Descriptor.invalidSSTable(name, "invalid version %s", versionString);
        }
        try {
            id = SSTableIdFactory.instance.fromString(tokens.get(1));
        }
        catch (RuntimeException e) {
            throw Descriptor.invalidSSTable(name, "the 'id' part (%s) of the name doesn't parse as a valid unique identifier", tokens.get(1));
        }
        SSTableFormat<?, ?> format = Descriptor.formatFromName(name, tokens);
        Component component = Component.parse(tokens.get(3), format);
        Version version = format.getVersion(versionString);
        if (!version.isCompatible()) {
            throw Descriptor.invalidSSTable(name, "incompatible sstable version (%s); you should have run upgradesstables before upgrading", versionString);
        }
        return new SSTableInfo(version, id, component);
    }

    private static File parentOf(String name, File file) {
        File parent = file.parent();
        if (parent == null) {
            throw Descriptor.invalidSSTable(name, "cannot extract keyspace and table name; make sure the sstable is in the proper sub-directories", new Object[0]);
        }
        return parent;
    }

    private static IllegalArgumentException invalidSSTable(String name, String msgFormat, Object ... parameters) {
        throw new IllegalArgumentException(String.format("Invalid sstable file " + name + ": " + msgFormat, parameters));
    }

    public IMetadataSerializer getMetadataSerializer() {
        return new MetadataSerializer();
    }

    public boolean isCompatible() {
        return this.version.isCompatible();
    }

    public Set<Component> discoverComponents() {
        HashSet<Component> components = Sets.newHashSetWithExpectedSize(Component.Type.all.size());
        for (Component component : Component.getSingletonsFor(this.version.format)) {
            if (!this.fileFor(component).exists()) continue;
            components.add(component);
        }
        return components;
    }

    public String toString() {
        return this.baseFile().absolutePath();
    }

    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        if (!(o instanceof Descriptor)) {
            return false;
        }
        Descriptor that = (Descriptor)o;
        if (this.hashCode != that.hashCode) {
            return false;
        }
        return that.directory.equals(this.directory) && that.id.equals(this.id) && that.ksname.equals(this.ksname) && that.cfname.equals(this.cfname) && that.version.equals(this.version);
    }

    public int hashCode() {
        return this.hashCode;
    }

    private static class SSTableInfo {
        final Version version;
        final SSTableId id;
        final Component component;

        SSTableInfo(Version version, SSTableId id, Component component) {
            this.version = version;
            this.id = id;
            this.component = component;
        }
    }
}

