/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.spark.utils;

import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.common.util.concurrent.Uninterruptibles;
import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.util.Collection;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.cassandra.analytics.stats.Stats;
import org.apache.cassandra.spark.data.FileType;
import org.apache.cassandra.spark.data.SSTable;
import org.apache.cassandra.spark.utils.ByteBufferUtils;
import org.apache.cassandra.spark.utils.RandomUtils;
import org.apache.cassandra.spark.utils.streaming.BufferingInputStream;
import org.apache.cassandra.spark.utils.streaming.CassandraFileSource;
import org.apache.cassandra.spark.utils.streaming.StreamBuffer;
import org.apache.cassandra.spark.utils.streaming.StreamConsumer;
import org.apache.commons.lang.mutable.MutableInt;
import org.assertj.core.api.Assertions;
import org.jetbrains.annotations.Nullable;
import org.junit.jupiter.api.Test;

public class BufferingInputStreamTests {
    private static final ScheduledExecutorService SCHEDULER = Executors.newScheduledThreadPool(1);
    private static final ExecutorService EXECUTOR = Executors.newFixedThreadPool(4, new ThreadFactoryBuilder().setNameFormat("sstable-tests-%d").setDaemon(true).build());
    static final int DEFAULT_CHUNK_SIZE = 8192;
    static final Stats STATS = Stats.DoNothingStats.INSTANCE;

    @Test
    public void testMockedClient() throws IOException {
        this.runMockedTest(1, 1, 8192L);
        this.runMockedTest(1, 5, 40960L);
        this.runMockedTest(10, 10, 0x600004L);
        this.runMockedTest(20, 1024, 33554400L);
        this.runMockedTest(10, 10, 81920L);
    }

    private static CassandraFileSource<SSTable> buildSource(final long size, final Long maxBufferSize, final Long requestChunkSize, final SSTableRequest request, final Duration duration) {
        return new CassandraFileSource<SSTable>(){

            public void request(long start, long end, StreamConsumer consumer) {
                request.request(start, end, consumer);
            }

            public SSTable cassandraFile() {
                return null;
            }

            public FileType fileType() {
                return null;
            }

            public long size() {
                return size;
            }

            public long maxBufferSize() {
                return maxBufferSize != null ? maxBufferSize : 0x600004L;
            }

            public long chunkBufferSize() {
                return requestChunkSize != null ? requestChunkSize : 0x3FFFFCL;
            }

            public Duration timeout() {
                return duration;
            }
        };
    }

    private void runMockedTest(int numRequests, int chunksPerRequest, long maxBufferSize) throws IOException {
        long requestChunkSize = 8192L * (long)chunksPerRequest;
        long fileSize = requestChunkSize * (long)numRequests;
        AtomicInteger requestCount = new AtomicInteger(0);
        CassandraFileSource<SSTable> mockedClient = BufferingInputStreamTests.buildSource(fileSize, maxBufferSize, requestChunkSize, (start, end, consumer) -> {
            requestCount.incrementAndGet();
            BufferingInputStreamTests.writeBuffers(consumer, BufferingInputStreamTests.randomBuffers(chunksPerRequest));
        }, null);
        BufferingInputStream is = new BufferingInputStream(mockedClient, STATS.bufferingInputStreamStats());
        BufferingInputStreamTests.readStreamFully((BufferingInputStream<SSTable>)is);
        Assertions.assertThat((int)requestCount.get()).isEqualTo(numRequests);
        Assertions.assertThat((long)is.bytesBuffered()).isEqualTo(0L);
        Assertions.assertThat((long)is.bytesWritten()).isEqualTo(fileSize);
        Assertions.assertThat((long)is.bytesRead()).isEqualTo(fileSize);
    }

    @Test
    public void testFailure() {
        int chunksPerRequest = 10;
        int numRequests = 10;
        long length = 0x3FFFFCL * (long)chunksPerRequest * (long)numRequests;
        AtomicInteger count = new AtomicInteger(0);
        CassandraFileSource<SSTable> source = BufferingInputStreamTests.buildSource(length, 0x600004L, 0x3FFFFCL, (start, end, consumer) -> {
            if (count.incrementAndGet() > numRequests / 2) {
                EXECUTOR.submit(() -> consumer.onError((Throwable)new RuntimeException("Something bad happened...")));
            } else {
                BufferingInputStreamTests.writeBuffers(consumer, BufferingInputStreamTests.randomBuffers(chunksPerRequest));
            }
        }, null);
        Assertions.assertThatThrownBy(() -> BufferingInputStreamTests.readStreamFully((BufferingInputStream<SSTable>)new BufferingInputStream(source, STATS.bufferingInputStreamStats()))).isInstanceOf(IOException.class);
    }

    @Test
    public void testTimeout() {
        long now = System.nanoTime();
        Assertions.assertThat((long)BufferingInputStream.timeoutLeftNanos((Duration)Duration.ofMillis(1000L), (long)now, (long)(now - Duration.ofMillis(900L).toNanos()))).isEqualTo(Duration.ofMillis(100L).toNanos());
        Assertions.assertThat((long)BufferingInputStream.timeoutLeftNanos((Duration)Duration.ofMillis(1000L), (long)now, (long)(now - Duration.ofMillis(1500L).toNanos()))).isEqualTo(Duration.ofMillis(-500L).toNanos());
        Assertions.assertThat((long)BufferingInputStream.timeoutLeftNanos((Duration)Duration.ofMillis(1000L), (long)now, (long)(now - Duration.ofMillis(5L).toNanos()))).isEqualTo(Duration.ofMillis(995L).toNanos());
        Assertions.assertThat((long)BufferingInputStream.timeoutLeftNanos((Duration)Duration.ofMillis(1000L), (long)now, (long)(now - Duration.ofMillis(0L).toNanos()))).isEqualTo(Duration.ofMillis(1000L).toNanos());
        Assertions.assertThat((long)BufferingInputStream.timeoutLeftNanos((Duration)Duration.ofMillis(1000L), (long)now, (long)(now + Duration.ofMillis(500L).toNanos()))).isEqualTo(Duration.ofMillis(1000L).toNanos());
        Assertions.assertThat((long)BufferingInputStream.timeoutLeftNanos((Duration)Duration.ofMillis(60000L), (long)now, (long)(now - Duration.ofMillis(25000L).toNanos()))).isEqualTo(Duration.ofMillis(35000L).toNanos());
        Assertions.assertThat((long)BufferingInputStream.timeoutLeftNanos((Duration)Duration.ofMillis(60000L), (long)now, (long)(now - Duration.ofMillis(65000L).toNanos()))).isEqualTo(Duration.ofMillis(-5000L).toNanos());
        Assertions.assertThat((long)BufferingInputStream.timeoutLeftNanos((Duration)Duration.ofMillis(60000L), (long)now, (long)(now - Duration.ofMillis(60000L).toNanos()))).isEqualTo(Duration.ofMillis(0L).toNanos());
    }

    @Test
    public void testTimeoutShouldAccountForActivityTime() {
        int chunksPerRequest = 10;
        int numRequests = 10;
        long length = 0x3FFFFCL * (long)chunksPerRequest * (long)numRequests;
        AtomicInteger count = new AtomicInteger(0);
        Duration timeout = Duration.ofMillis(1000L);
        long startTime = System.nanoTime();
        long sleepTimeInMillis = 100L;
        CassandraFileSource<SSTable> source = BufferingInputStreamTests.buildSource(length, 0x600004L, 0x3FFFFCL, (start, end, consumer) -> {
            if (count.incrementAndGet() == 1) {
                EXECUTOR.submit(() -> {
                    Uninterruptibles.sleepUninterruptibly((long)sleepTimeInMillis, (TimeUnit)TimeUnit.MILLISECONDS);
                    BufferingInputStreamTests.writeBuffers(consumer, BufferingInputStreamTests.randomBuffers(chunksPerRequest));
                });
            }
        }, timeout);
        BufferingInputStream inputStream = new BufferingInputStream(source, STATS.bufferingInputStreamStats());
        try {
            BufferingInputStreamTests.readStreamFully((BufferingInputStream<SSTable>)inputStream);
            Assertions.fail((String)"Should not reach here, should throw TimeoutException");
        }
        catch (IOException exception) {
            Assertions.assertThat((Throwable)exception.getCause()).isInstanceOf(TimeoutException.class);
        }
        long readAndTimeoutTotal = TimeUnit.NANOSECONDS.toMillis(inputStream.timeBlockedNanos()) + timeout.toMillis();
        Duration clientTimeoutTotal = Duration.ofNanos(System.nanoTime() - startTime);
        Assertions.assertThat((long)clientTimeoutTotal.toMillis()).isGreaterThanOrEqualTo(readAndTimeoutTotal).describedAs("Timeout didn't account for activity time. Took %dms should have taken at least %dms", new Object[]{clientTimeoutTotal.toMillis(), readAndTimeoutTotal});
    }

    @Test
    public void testSkipOnInit() throws IOException {
        final int size = 0x1400000;
        final int chunkSize = 1024;
        int numChunks = 16;
        final MutableInt bytesRead = new MutableInt(0);
        final MutableInt count = new MutableInt(0);
        CassandraFileSource<SSTable> source = new CassandraFileSource<SSTable>(){

            public void request(long start, long end, StreamConsumer consumer) {
                Assertions.assertThat((long)start).isNotEqualTo(0L);
                int length = (int)(end - start + 1L);
                consumer.onRead(BufferingInputStreamTests.randomBuffer(length));
                bytesRead.add(length);
                count.increment();
                consumer.onEnd();
            }

            public long chunkBufferSize() {
                return chunkSize;
            }

            public SSTable cassandraFile() {
                return null;
            }

            public FileType fileType() {
                return FileType.INDEX;
            }

            public long size() {
                return size;
            }

            @Nullable
            public Duration timeout() {
                return Duration.ofSeconds(5L);
            }
        };
        int bytesToRead = chunkSize * numChunks;
        long skipAhead = size - bytesToRead + 1;
        try (BufferingInputStream stream = new BufferingInputStream((CassandraFileSource)source, STATS.bufferingInputStreamStats());){
            ByteBufferUtils.skipFully((InputStream)stream, (long)skipAhead);
            BufferingInputStreamTests.readStreamFully((BufferingInputStream<SSTable>)stream);
        }
        Assertions.assertThat((int)bytesRead.intValue()).isEqualTo(bytesToRead);
        Assertions.assertThat((int)count.intValue()).isEqualTo(numChunks);
    }

    @Test
    public void testSkipToEnd() throws IOException {
        CassandraFileSource<SSTable> source = new CassandraFileSource<SSTable>(){

            public void request(long start, long end, StreamConsumer consumer) {
                consumer.onRead(BufferingInputStreamTests.randomBuffer((int)(end - start + 1L)));
                consumer.onEnd();
            }

            public SSTable cassandraFile() {
                return null;
            }

            public FileType fileType() {
                return FileType.INDEX;
            }

            public long size() {
                return 0x1400000L;
            }

            @Nullable
            public Duration timeout() {
                return Duration.ofSeconds(5L);
            }
        };
        try (BufferingInputStream stream = new BufferingInputStream((CassandraFileSource)source, STATS.bufferingInputStreamStats());){
            ByteBufferUtils.skipFully((InputStream)stream, (long)0x1400000L);
            BufferingInputStreamTests.readStreamFully((BufferingInputStream<SSTable>)stream);
        }
    }

    private static ImmutableList<StreamBuffer> randomBuffers(int count) {
        return ImmutableList.copyOf((Collection)IntStream.range(0, count).mapToObj(buffer -> BufferingInputStreamTests.randomBuffer()).collect(Collectors.toList()));
    }

    private static StreamBuffer randomBuffer() {
        return BufferingInputStreamTests.randomBuffer(8192);
    }

    private static StreamBuffer randomBuffer(int size) {
        return StreamBuffer.wrap((byte[])RandomUtils.randomBytes((int)size));
    }

    private static void readStreamFully(BufferingInputStream<SSTable> inputStream) throws IOException {
        try (BufferingInputStream<SSTable> in = inputStream;){
            while (in.read() >= 0) {
            }
        }
    }

    private static void writeBuffers(StreamConsumer consumer, ImmutableList<StreamBuffer> buffers) {
        if (buffers.isEmpty()) {
            consumer.onEnd();
            return;
        }
        SCHEDULER.schedule(() -> EXECUTOR.submit(() -> {
            consumer.onRead((StreamBuffer)buffers.get(0));
            BufferingInputStreamTests.writeBuffers(consumer, (ImmutableList<StreamBuffer>)buffers.subList(1, buffers.size()));
        }), (long)RandomUtils.RANDOM.nextInt(50), TimeUnit.MICROSECONDS);
    }

    private static interface SSTableRequest {
        public void request(long var1, long var3, StreamConsumer var5);
    }
}

