/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.cql3;

import com.google.common.base.Objects;
import java.util.Calendar;
import java.util.Locale;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.cassandra.cql3.statements.RequestValidations;
import org.apache.cassandra.serializers.MarshalException;

public final class Duration {
    public static final long NANOS_PER_MICRO = 1000L;
    public static final long NANOS_PER_MILLI = 1000000L;
    public static final long NANOS_PER_SECOND = 1000000000L;
    public static final long NANOS_PER_MINUTE = 60000000000L;
    public static final long NANOS_PER_HOUR = 3600000000000L;
    public static final int DAYS_PER_WEEK = 7;
    public static final int MONTHS_PER_YEAR = 12;
    private static final Pattern STANDARD_PATTERN = Pattern.compile("\\G(\\d+)(y|Y|mo|MO|mO|Mo|w|W|d|D|h|H|s|S|ms|MS|mS|Ms|us|US|uS|Us|\u00b5s|\u00b5S|ns|NS|nS|Ns|m|M)");
    private static final Pattern ISO8601_PATTERN = Pattern.compile("P((\\d+)Y)?((\\d+)M)?((\\d+)D)?(T((\\d+)H)?((\\d+)M)?((\\d+)S)?)?");
    private static final Pattern ISO8601_WEEK_PATTERN = Pattern.compile("P(\\d+)W");
    private static final Pattern ISO8601_ALTERNATIVE_PATTERN = Pattern.compile("P(\\d{4})-(\\d{2})-(\\d{2})T(\\d{2}):(\\d{2}):(\\d{2})");
    private final int months;
    private final int days;
    private final long nanoseconds;

    private Duration(int months, int days, long nanoseconds) {
        assert (months >= 0 && days >= 0 && nanoseconds >= 0L || months <= 0 && days <= 0 && nanoseconds <= 0L);
        this.months = months;
        this.days = days;
        this.nanoseconds = nanoseconds;
    }

    public static Duration newInstance(int months, int days, long nanoseconds) {
        return new Duration(months, days, nanoseconds);
    }

    public static Duration from(String input) {
        String source;
        boolean isNegative = input.startsWith("-");
        String string = source = isNegative ? input.substring(1) : input;
        if (source.startsWith("P")) {
            if (source.endsWith("W")) {
                return Duration.parseIso8601WeekFormat(isNegative, source);
            }
            if (source.contains("-")) {
                return Duration.parseIso8601AlternativeFormat(isNegative, source);
            }
            return Duration.parseIso8601Format(isNegative, source);
        }
        return Duration.parseStandardFormat(isNegative, source);
    }

    private static Duration parseIso8601Format(boolean isNegative, String source) {
        Matcher matcher = ISO8601_PATTERN.matcher(source);
        if (!matcher.matches()) {
            throw RequestValidations.invalidRequest("Unable to convert '%s' to a duration", source);
        }
        Builder builder = new Builder(isNegative);
        if (matcher.group(1) != null) {
            builder.addYears(Duration.groupAsLong(matcher, 2));
        }
        if (matcher.group(3) != null) {
            builder.addMonths(Duration.groupAsLong(matcher, 4));
        }
        if (matcher.group(5) != null) {
            builder.addDays(Duration.groupAsLong(matcher, 6));
        }
        if (matcher.group(7) != null) {
            if (matcher.group(8) != null) {
                builder.addHours(Duration.groupAsLong(matcher, 9));
            }
            if (matcher.group(10) != null) {
                builder.addMinutes(Duration.groupAsLong(matcher, 11));
            }
            if (matcher.group(12) != null) {
                builder.addSeconds(Duration.groupAsLong(matcher, 13));
            }
        }
        return builder.build();
    }

    private static Duration parseIso8601AlternativeFormat(boolean isNegative, String source) {
        Matcher matcher = ISO8601_ALTERNATIVE_PATTERN.matcher(source);
        if (!matcher.matches()) {
            throw RequestValidations.invalidRequest("Unable to convert '%s' to a duration", source);
        }
        return new Builder(isNegative).addYears(Duration.groupAsLong(matcher, 1)).addMonths(Duration.groupAsLong(matcher, 2)).addDays(Duration.groupAsLong(matcher, 3)).addHours(Duration.groupAsLong(matcher, 4)).addMinutes(Duration.groupAsLong(matcher, 5)).addSeconds(Duration.groupAsLong(matcher, 6)).build();
    }

    private static Duration parseIso8601WeekFormat(boolean isNegative, String source) {
        Matcher matcher = ISO8601_WEEK_PATTERN.matcher(source);
        if (!matcher.matches()) {
            throw RequestValidations.invalidRequest("Unable to convert '%s' to a duration", source);
        }
        return new Builder(isNegative).addWeeks(Duration.groupAsLong(matcher, 1)).build();
    }

    private static Duration parseStandardFormat(boolean isNegative, String source) {
        Matcher matcher = STANDARD_PATTERN.matcher(source);
        if (!matcher.find()) {
            throw RequestValidations.invalidRequest("Unable to convert '%s' to a duration", source);
        }
        Builder builder = new Builder(isNegative);
        boolean done = false;
        do {
            long number = Duration.groupAsLong(matcher, 1);
            String symbol = matcher.group(2);
            Duration.add(builder, number, symbol);
            boolean bl = done = matcher.end() == source.length();
        } while (matcher.find());
        if (!done) {
            throw RequestValidations.invalidRequest("Unable to convert '%s' to a duration", source);
        }
        return builder.build();
    }

    private static long groupAsLong(Matcher matcher, int group) {
        return Long.parseLong(matcher.group(group));
    }

    private static Builder add(Builder builder, long number, String symbol) {
        switch (symbol.toLowerCase()) {
            case "y": {
                return builder.addYears(number);
            }
            case "mo": {
                return builder.addMonths(number);
            }
            case "w": {
                return builder.addWeeks(number);
            }
            case "d": {
                return builder.addDays(number);
            }
            case "h": {
                return builder.addHours(number);
            }
            case "m": {
                return builder.addMinutes(number);
            }
            case "s": {
                return builder.addSeconds(number);
            }
            case "ms": {
                return builder.addMillis(number);
            }
            case "us": 
            case "\u00b5s": {
                return builder.addMicros(number);
            }
            case "ns": {
                return builder.addNanos(number);
            }
        }
        throw new MarshalException(String.format("Unknown duration symbol '%s'", symbol));
    }

    public int getMonths() {
        return this.months;
    }

    public int getDays() {
        return this.days;
    }

    public long getNanoseconds() {
        return this.nanoseconds;
    }

    public long addTo(long timeInMillis) {
        return Duration.add(timeInMillis, this.months, this.days, this.nanoseconds);
    }

    public long substractFrom(long timeInMillis) {
        return Duration.add(timeInMillis, -this.months, -this.days, -this.nanoseconds);
    }

    private static long add(long timeInMillis, int months, int days, long nanoseconds) {
        if (months == 0) {
            long durationInMillis = (long)days * 86400000L + nanoseconds / 1000000L;
            return timeInMillis + durationInMillis;
        }
        Calendar calendar = Calendar.getInstance(TimeZone.getTimeZone("UTC"), Locale.US);
        calendar.setTimeInMillis(timeInMillis);
        calendar.add(2, months);
        calendar.add(5, days);
        calendar.add(14, (int)(nanoseconds / 1000000L));
        return calendar.getTimeInMillis();
    }

    public int hashCode() {
        return Objects.hashCode((Object[])new Object[]{this.days, this.months, this.nanoseconds});
    }

    public boolean equals(Object obj) {
        if (!(obj instanceof Duration)) {
            return false;
        }
        Duration other = (Duration)obj;
        return this.days == other.days && this.months == other.months && this.nanoseconds == other.nanoseconds;
    }

    public String toString() {
        StringBuilder builder = new StringBuilder();
        if (this.months < 0 || this.days < 0 || this.nanoseconds < 0L) {
            builder.append('-');
        }
        long remainder = Duration.append(builder, Math.abs(this.months), 12L, "y");
        Duration.append(builder, remainder, 1L, "mo");
        Duration.append(builder, Math.abs(this.days), 1L, "d");
        if (this.nanoseconds != 0L) {
            remainder = Duration.append(builder, Math.abs(this.nanoseconds), 3600000000000L, "h");
            remainder = Duration.append(builder, remainder, 60000000000L, "m");
            remainder = Duration.append(builder, remainder, 1000000000L, "s");
            remainder = Duration.append(builder, remainder, 1000000L, "ms");
            remainder = Duration.append(builder, remainder, 1000L, "us");
            Duration.append(builder, remainder, 1L, "ns");
        }
        return builder.toString();
    }

    public boolean hasDayPrecision() {
        return this.getNanoseconds() == 0L;
    }

    public boolean hasMillisecondPrecision() {
        return this.getNanoseconds() % 1000000L == 0L;
    }

    private static long append(StringBuilder builder, long dividend, long divisor, String unit) {
        if (dividend == 0L || dividend < divisor) {
            return dividend;
        }
        builder.append(dividend / divisor).append(unit);
        return dividend % divisor;
    }

    private static class Builder {
        private final boolean isNegative;
        private int months;
        private int days;
        private long nanoseconds;
        private int currentUnitIndex;

        public Builder(boolean isNegative) {
            this.isNegative = isNegative;
        }

        public Builder addYears(long numberOfYears) {
            this.validateOrder(1);
            this.validateMonths(numberOfYears, 12);
            this.months = (int)((long)this.months + numberOfYears * 12L);
            return this;
        }

        public Builder addMonths(long numberOfMonths) {
            this.validateOrder(2);
            this.validateMonths(numberOfMonths, 1);
            this.months = (int)((long)this.months + numberOfMonths);
            return this;
        }

        public Builder addWeeks(long numberOfWeeks) {
            this.validateOrder(3);
            this.validateDays(numberOfWeeks, 7);
            this.days = (int)((long)this.days + numberOfWeeks * 7L);
            return this;
        }

        public Builder addDays(long numberOfDays) {
            this.validateOrder(4);
            this.validateDays(numberOfDays, 1);
            this.days = (int)((long)this.days + numberOfDays);
            return this;
        }

        public Builder addHours(long numberOfHours) {
            this.validateOrder(5);
            this.validateNanos(numberOfHours, 3600000000000L);
            this.nanoseconds += numberOfHours * 3600000000000L;
            return this;
        }

        public Builder addMinutes(long numberOfMinutes) {
            this.validateOrder(6);
            this.validateNanos(numberOfMinutes, 60000000000L);
            this.nanoseconds += numberOfMinutes * 60000000000L;
            return this;
        }

        public Builder addSeconds(long numberOfSeconds) {
            this.validateOrder(7);
            this.validateNanos(numberOfSeconds, 1000000000L);
            this.nanoseconds += numberOfSeconds * 1000000000L;
            return this;
        }

        public Builder addMillis(long numberOfMillis) {
            this.validateOrder(8);
            this.validateNanos(numberOfMillis, 1000000L);
            this.nanoseconds += numberOfMillis * 1000000L;
            return this;
        }

        public Builder addMicros(long numberOfMicros) {
            this.validateOrder(9);
            this.validateNanos(numberOfMicros, 1000L);
            this.nanoseconds += numberOfMicros * 1000L;
            return this;
        }

        public Builder addNanos(long numberOfNanos) {
            this.validateOrder(10);
            this.validateNanos(numberOfNanos, 1L);
            this.nanoseconds += numberOfNanos;
            return this;
        }

        private void validateMonths(long units, int monthsPerUnit) {
            this.validate(units, (Integer.MAX_VALUE - this.months) / monthsPerUnit, "months");
        }

        private void validateDays(long units, int daysPerUnit) {
            this.validate(units, (Integer.MAX_VALUE - this.days) / daysPerUnit, "days");
        }

        private void validateNanos(long units, long nanosPerUnit) {
            this.validate(units, (Long.MAX_VALUE - this.nanoseconds) / nanosPerUnit, "nanoseconds");
        }

        private void validate(long units, long limit, String unitName) {
            RequestValidations.checkTrue(units <= limit, "Invalid duration. The total number of %s must be less or equal to %s", unitName, Integer.MAX_VALUE);
        }

        private void validateOrder(int unitIndex) {
            if (unitIndex == this.currentUnitIndex) {
                throw RequestValidations.invalidRequest("Invalid duration. The %s are specified multiple times", this.getUnitName(unitIndex));
            }
            if (unitIndex <= this.currentUnitIndex) {
                throw RequestValidations.invalidRequest("Invalid duration. The %s should be after %s", this.getUnitName(this.currentUnitIndex), this.getUnitName(unitIndex));
            }
            this.currentUnitIndex = unitIndex;
        }

        private String getUnitName(int unitIndex) {
            switch (unitIndex) {
                case 1: {
                    return "years";
                }
                case 2: {
                    return "months";
                }
                case 3: {
                    return "weeks";
                }
                case 4: {
                    return "days";
                }
                case 5: {
                    return "hours";
                }
                case 6: {
                    return "minutes";
                }
                case 7: {
                    return "seconds";
                }
                case 8: {
                    return "milliseconds";
                }
                case 9: {
                    return "microseconds";
                }
                case 10: {
                    return "nanoseconds";
                }
            }
            throw new AssertionError((Object)("unknown unit index: " + unitIndex));
        }

        public Duration build() {
            return this.isNegative ? new Duration(-this.months, -this.days, -this.nanoseconds) : new Duration(this.months, this.days, this.nanoseconds);
        }
    }
}

