"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseVisitor = void 0;
const ignore = (_n, _st, _cb) => { };
/**
 * @see https://github.com/swc-project/swc/blob/main/packages/core/src/Visitor.ts
 */
class BaseVisitor {
    ArrayExpression(n, st, cb) {
        for (const el of n.elements) {
            if (el) {
                cb(el.expression, st);
            }
        }
    }
    ArrayPattern(n, st, cb) {
        for (const el of n.elements) {
            if (el) {
                cb(el, st);
            }
        }
    }
    ArrowFunctionExpression(n, st, cb) {
        cb(n.body, st);
        if (n.typeParameters) {
            cb(n.typeParameters, st);
        }
    }
    AssignmentExpression(n, st, cb) {
        cb(n.left, st);
        cb(n.right, st);
    }
    AssignmentPattern(n, st, cb) {
        cb(n.left, st);
        cb(n.right, st);
        if (n.typeAnnotation) {
            cb(n.typeAnnotation, st);
        }
    }
    AssignmentPatternProperty(n, st, cb) {
        cb(n.key, st);
        if (n.value) {
            cb(n.value, st);
        }
    }
    AssignmentProperty(n, st, cb) {
        cb(n.value, st);
    }
    AwaitExpression(n, st, cb) {
        cb(n.argument, st);
    }
    BigIntLiteral = ignore;
    BinaryExpression(n, st, cb) {
        cb(n.left, st);
        cb(n.right, st);
    }
    BlockStatement(n, st, cb) {
        for (const stmt of n.stmts) {
            cb(stmt, st);
        }
    }
    BooleanLiteral = ignore;
    BreakStatement(n, st, cb) {
        if (n.label) {
            cb(n.label, st);
        }
    }
    CallExpression(n, st, cb) {
        cb(n.callee, st);
        for (const arg of n.arguments) {
            cb(arg.expression, st);
        }
        if (n.typeArguments) {
            cb(n.typeArguments, st);
        }
    }
    CatchClause(n, st, cb) {
        if (n.param) {
            cb(n.param, st);
        }
        cb(n.body, st);
    }
    ClassDeclaration(n, st, cb) {
        for (const decorator of n.decorators ?? []) {
            cb(decorator, st);
        }
        cb(n.identifier, st);
        for (const implement of n.implements) {
            cb(implement, st);
        }
        if (n.superClass) {
            cb(n.superClass, st);
        }
        if (n.superTypeParams) {
            cb(n.superTypeParams, st);
        }
        if (n.typeParams) {
            cb(n.typeParams, st);
        }
        for (const member of n.body) {
            cb(member, st);
        }
    }
    ClassExpression(n, st, cb) {
        for (const decorator of n.decorators ?? []) {
            cb(decorator, st);
        }
        if (n.identifier) {
            cb(n.identifier, st);
        }
        for (const implement of n.implements) {
            cb(implement, st);
        }
        if (n.superClass) {
            cb(n.superClass, st);
        }
        if (n.superTypeParams) {
            cb(n.superTypeParams, st);
        }
        if (n.typeParams) {
            cb(n.typeParams, st);
        }
        for (const member of n.body) {
            cb(member, st);
        }
    }
    ClassMethod(n, st, cb) {
        cb(n.key, st);
        for (const decorator of n.function.decorators ?? []) {
            cb(decorator, st);
        }
        for (const param of n.function.params) {
            cb(param, st);
        }
        if (n.function.returnType) {
            cb(n.function.returnType, st);
        }
        if (n.function.typeParameters) {
            cb(n.function.typeParameters, st);
        }
        if (n.function.body) {
            cb(n.function.body, st);
        }
    }
    ClassProperty(n, st, cb) {
        for (const decorator of n.decorators ?? []) {
            cb(decorator, st);
        }
        cb(n.key, st);
        if (n.typeAnnotation) {
            cb(n.typeAnnotation, st);
        }
        if (n.value) {
            cb(n.value, st);
        }
    }
    Computed(n, st, cb) {
        cb(n.expression, st);
    }
    ConditionalExpression(n, st, cb) {
        cb(n.test, st);
        cb(n.consequent, st);
        cb(n.alternate, st);
    }
    Constructor(n, st, cb) {
        cb(n.key, st);
        for (const param of n.params) {
            cb(param, st);
        }
        if (n.body) {
            cb(n.body, st);
        }
    }
    ContinueStatement(n, st, cb) {
        if (n.label) {
            cb(n.label, st);
        }
    }
    DebuggerStatement = ignore;
    Decorator(n, st, cb) {
        cb(n.expression, st);
    }
    DoWhileStatement(n, st, cb) {
        cb(n.body, st);
        cb(n.test, st);
    }
    EmptyStatement = ignore;
    ExportAllDeclaration(n, st, cb) {
        cb(n.source, st);
        // @ts-expect-error -- asserts is not typed in ExportAllDeclaration
        // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
        n.asserts = n.asserts ?? n.with;
        if (n.asserts) {
            cb(n.asserts, st);
        }
    }
    ExportDeclaration(n, st, cb) {
        cb(n.declaration, st);
    }
    ExportDefaultDeclaration(n, st, cb) {
        cb(n.decl, st);
    }
    ExportDefaultExpression(n, st, cb) {
        cb(n.expression, st);
    }
    ExportDefaultSpecifier(n, st, cb) {
        cb(n.exported, st);
    }
    ExportNamedDeclaration(n, st, cb) {
        for (const specifier of n.specifiers) {
            cb(specifier, st);
        }
        if (n.source) {
            cb(n.source, st);
        }
        // @ts-expect-error -- asserts is not typed in ExportAllDeclaration
        // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
        n.asserts = n.asserts ?? n.with;
        if (n.asserts) {
            cb(n.asserts, st);
        }
    }
    ExportNamespaceSpecifier(n, st, cb) {
        cb(n.name, st);
    }
    ExportSpecifier(n, st, cb) {
        if (n.exported) {
            cb(n.exported, st);
        }
        cb(n.orig, st);
    }
    ExpressionStatement(n, st, cb) {
        cb(n.expression, st);
    }
    ForInStatement(n, st, cb) {
        cb(n.left, st);
        cb(n.right, st);
        cb(n.body, st);
    }
    ForOfStatement(n, st, cb) {
        cb(n.left, st);
        cb(n.right, st);
        cb(n.body, st);
    }
    ForStatement(n, st, cb) {
        if (n.init) {
            cb(n.init, st);
        }
        if (n.test) {
            cb(n.test, st);
        }
        if (n.update) {
            cb(n.update, st);
        }
        cb(n.body, st);
    }
    FunctionDeclaration(n, st, cb) {
        for (const decorator of n.decorators ?? []) {
            cb(decorator, st);
        }
        cb(n.identifier, st);
        for (const param of n.params) {
            cb(param, st);
        }
        if (n.returnType) {
            cb(n.returnType, st);
        }
        if (n.typeParameters) {
            cb(n.typeParameters, st);
        }
        if (n.body) {
            cb(n.body, st);
        }
    }
    FunctionExpression(n, st, cb) {
        for (const decorator of n.decorators ?? []) {
            cb(decorator, st);
        }
        if (n.identifier) {
            cb(n.identifier, st);
        }
        for (const param of n.params) {
            cb(param, st);
        }
        if (n.returnType) {
            cb(n.returnType, st);
        }
        if (n.typeParameters) {
            cb(n.typeParameters, st);
        }
        if (n.body) {
            cb(n.body, st);
        }
    }
    GetterProperty(n, st, cb) {
        cb(n.key, st);
        if (n.typeAnnotation) {
            cb(n.typeAnnotation, st);
        }
        if (n.body) {
            cb(n.body, st);
        }
    }
    Identifier(n, st, cb) {
        if ('typeAnnotation' in n && n.typeAnnotation) {
            cb(n.typeAnnotation, st);
        }
    }
    IfStatement(n, st, cb) {
        cb(n.test, st);
        cb(n.consequent, st);
        if (n.alternate) {
            cb(n.alternate, st);
        }
    }
    Import = ignore;
    ImportDeclaration(n, st, cb) {
        for (const specifier of n.specifiers) {
            cb(specifier, st);
        }
        cb(n.source, st);
        // @ts-expect-error -- asserts is not typed in ExportAllDeclaration
        // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
        n.asserts = n.asserts ?? n.with;
        if (n.asserts) {
            cb(n.asserts, st);
        }
    }
    ImportDefaultSpecifier(n, st, cb) {
        cb(n.local, st);
    }
    ImportNamespaceSpecifier(n, st, cb) {
        cb(n.local, st);
    }
    ImportSpecifier(n, st, cb) {
        if (n.imported) {
            cb(n.imported, st);
        }
        cb(n.local, st);
    }
    Invalid = ignore;
    JSXAttribute(n, st, cb) {
        cb(n.name, st);
        if (n.value) {
            cb(n.value, st);
        }
    }
    JSXClosingElement(n, st, cb) {
        cb(n.name, st);
    }
    JSXClosingFragment = ignore;
    JSXElement(n, st, cb) {
        cb(n.opening, st);
        for (const child of n.children) {
            cb(child, st);
        }
        if (n.closing) {
            cb(n.closing, st);
        }
    }
    JSXEmptyExpression = ignore;
    JSXExpressionContainer(n, st, cb) {
        cb(n.expression, st);
    }
    JSXFragment(n, st, cb) {
        cb(n.opening, st);
        for (const child of n.children) {
            cb(child, st);
        }
        cb(n.closing, st);
    }
    JSXMemberExpression(n, st, cb) {
        cb(n.property, st);
        cb(n.object, st);
    }
    JSXNamespacedName(n, st, cb) {
        cb(n.namespace, st);
        cb(n.name, st);
    }
    JSXOpeningElement(n, st, cb) {
        cb(n.name, st);
        for (const attr of n.attributes) {
            cb(attr, st);
        }
        if (n.typeArguments) {
            cb(n.typeArguments, st);
        }
    }
    JSXOpeningFragment = ignore;
    JSXSpreadChild(n, st, cb) {
        cb(n.expression, st);
    }
    JSXText = ignore;
    KeyValuePatternProperty(n, st, cb) {
        cb(n.key, st);
        cb(n.value, st);
    }
    KeyValueProperty(n, st, cb) {
        cb(n.key, st);
        cb(n.value, st);
    }
    LabeledStatement(n, st, cb) {
        cb(n.label, st);
        cb(n.body, st);
    }
    MemberExpression(n, st, cb) {
        cb(n.object, st);
        cb(n.property, st);
    }
    MetaProperty = ignore;
    MethodProperty(n, st, cb) {
        for (const decorator of n.decorators ?? []) {
            cb(decorator, st);
        }
        cb(n.key, st);
        for (const param of n.params) {
            cb(param, st);
        }
        if (n.returnType) {
            cb(n.returnType, st);
        }
        if (n.typeParameters) {
            cb(n.typeParameters, st);
        }
        if (n.body) {
            cb(n.body, st);
        }
    }
    Module(n, st, cb) {
        for (const stmt of n.body) {
            cb(stmt, st);
        }
    }
    NewExpression(n, st, cb) {
        cb(n.callee, st);
        if (n.arguments) {
            for (const arg of n.arguments) {
                cb(arg.expression, st);
            }
        }
        if (n.typeArguments) {
            cb(n.typeArguments, st);
        }
    }
    NullLiteral = ignore;
    NumericLiteral = ignore;
    ObjectExpression(n, st, cb) {
        for (const property of n.properties) {
            cb(property, st);
        }
    }
    ObjectPattern(n, st, cb) {
        for (const property of n.properties) {
            cb(property, st);
        }
        if (n.typeAnnotation) {
            cb(n.typeAnnotation, st);
        }
    }
    OptionalChainingExpression(n, st, cb) {
        cb(n.base, st);
    }
    Parameter(n, st, cb) {
        for (const decorator of n.decorators ?? []) {
            cb(decorator, st);
        }
        cb(n.pat, st);
    }
    ParenthesisExpression(n, st, cb) {
        cb(n.expression, st);
    }
    PrivateMethod(n, st, cb) {
        cb(n.key, st);
        for (const decorator of n.function.decorators ?? []) {
            cb(decorator, st);
        }
        for (const param of n.function.params) {
            cb(param, st);
        }
        if (n.function.returnType) {
            cb(n.function.returnType, st);
        }
        if (n.function.typeParameters) {
            cb(n.function.typeParameters, st);
        }
        if (n.function.body) {
            cb(n.function.body, st);
        }
    }
    PrivateName = ignore;
    PrivateProperty(n, st, cb) {
        for (const decorator of n.decorators ?? []) {
            cb(decorator, st);
        }
        cb(n.key, st);
        if (n.typeAnnotation) {
            cb(n.typeAnnotation, st);
        }
        if (n.value) {
            cb(n.value, st);
        }
    }
    RegExpLiteral = ignore;
    RestElement(n, st, cb) {
        cb(n.argument, st);
        if (n.typeAnnotation) {
            cb(n.typeAnnotation, st);
        }
    }
    ReturnStatement(n, st, cb) {
        if (n.argument) {
            cb(n.argument, st);
        }
    }
    Script(n, st, cb) {
        for (const stmt of n.body) {
            cb(stmt, st);
        }
    }
    SequenceExpression(n, st, cb) {
        for (const expression of n.expressions) {
            cb(expression, st);
        }
    }
    SetterProperty(n, st, cb) {
        cb(n.key, st);
        cb(n.param, st);
        if (n.body) {
            cb(n.body, st);
        }
    }
    SpreadElement(n, st, cb) {
        cb(n.arguments, st);
    }
    StaticBlock(n, st, cb) {
        cb(n.body, st);
    }
    StringLiteral = ignore;
    Super = ignore;
    SuperPropExpression(n, st, cb) {
        cb(n.obj, st);
        cb(n.property, st);
    }
    SwitchCase(n, st, cb) {
        if (n.test) {
            cb(n.test, st);
        }
        for (const consequent of n.consequent) {
            cb(consequent, st);
        }
    }
    SwitchStatement(n, st, cb) {
        cb(n.discriminant, st);
        for (const cases of n.cases) {
            cb(cases, st);
        }
    }
    TaggedTemplateExpression(n, st, cb) {
        cb(n.tag, st);
        cb(n.template, st);
        if (n.typeParameters) {
            cb(n.typeParameters, st);
        }
    }
    TemplateElement = ignore;
    TemplateLiteral(n, st, cb) {
        for (const quasis of n.quasis) {
            cb(quasis, st);
        }
        if ('expressions' in n) {
            for (const expressions of n.expressions) {
                cb(expressions, st);
            }
        }
        if ('types' in n) {
            for (const types of n.types) {
                cb(types, st);
            }
        }
    }
    ThisExpression = ignore;
    ThrowStatement(n, st, cb) {
        cb(n.argument, st);
    }
    TryStatement(n, st, cb) {
        cb(n.block, st);
        if (n.handler) {
            cb(n.handler, st);
        }
        if (n.finalizer) {
            cb(n.finalizer, st);
        }
    }
    TsArrayType(n, st, cb) {
        cb(n.elemType, st);
    }
    TsExpressionWithTypeArguments(n, st, cb) {
        cb(n.expression, st);
        if (n.typeArguments) {
            cb(n.typeArguments, st);
        }
    }
    TsInterfaceDeclaration(n, st, cb) {
        cb(n.id, st);
        cb(n.body, st);
        for (const ext of n.extends) {
            cb(ext, st);
        }
        if (n.typeParams) {
            cb(n.typeParams, st);
        }
    }
    TsInterfaceBody(n, st, cb) {
        for (const ele of n.body) {
            cb(ele, st);
        }
    }
    TsKeywordType = ignore;
    TsPropertySignature(n, st, cb) {
        cb(n.key, st);
        if (n.typeAnnotation) {
            cb(n.typeAnnotation, st);
        }
    }
    TsAsExpression(n, st, cb) {
        cb(n.expression, st);
        cb(n.typeAnnotation, st);
    }
    TsCallSignatureDeclaration(n, st, cb) {
        for (const param of n.params) {
            cb(param, st);
        }
        if (n.typeParams) {
            cb(n.typeParams, st);
        }
        if (n.typeAnnotation) {
            cb(n.typeAnnotation, st);
        }
    }
    TsConditionalType(n, st, cb) {
        cb(n.checkType, st);
        cb(n.extendsType, st);
        cb(n.trueType, st);
        cb(n.falseType, st);
    }
    TsConstAssertion(n, st, cb) {
        cb(n.expression, st);
    }
    TsConstructorType(n, st, cb) {
        for (const param of n.params) {
            cb(param, st);
        }
        cb(n.typeAnnotation, st);
        if (n.typeParams) {
            cb(n.typeParams, st);
        }
    }
    TsConstructSignatureDeclaration(n, st, cb) {
        for (const param of n.params) {
            cb(param, st);
        }
        if (n.typeParams) {
            cb(n.typeParams, st);
        }
        if (n.typeAnnotation) {
            cb(n.typeAnnotation, st);
        }
    }
    TsEnumDeclaration(n, st, cb) {
        cb(n.id, st);
        for (const member of n.members) {
            cb(member, st);
        }
    }
    TsEnumMember(n, st, cb) {
        cb(n.id, st);
        if (n.init) {
            cb(n.init, st);
        }
    }
    TsExportAssignment(n, st, cb) {
        cb(n.expression, st);
    }
    TsExternalModuleReference(n, st, cb) {
        cb(n.expression, st);
    }
    TsFunctionType(n, st, cb) {
        for (const param of n.params) {
            cb(param, st);
        }
        cb(n.typeAnnotation, st);
        if (n.typeParams) {
            cb(n.typeParams, st);
        }
    }
    TsGetterSignature(n, st, cb) {
        cb(n.key, st);
        if (n.typeAnnotation) {
            cb(n.typeAnnotation, st);
        }
    }
    TsImportEqualsDeclaration(n, st, cb) {
        cb(n.id, st);
        cb(n.moduleRef, st);
    }
    TsImportType(n, st, cb) {
        cb(n.argument, st);
        if (n.qualifier) {
            cb(n.qualifier, st);
        }
        if (n.typeArguments) {
            cb(n.typeArguments, st);
        }
    }
    TsIndexedAccessType(n, st, cb) {
        cb(n.indexType, st);
        cb(n.objectType, st);
    }
    TsIndexSignature(n, st, cb) {
        for (const param of n.params) {
            cb(param, st);
        }
        if (n.typeAnnotation) {
            cb(n.typeAnnotation, st);
        }
    }
    TsInferType(n, st, cb) {
        cb(n.typeParam, st);
    }
    TsInstantiation(n, st, cb) {
        cb(n.expression, st);
        cb(n.typeArguments, st);
    }
    TsIntersectionType(n, st, cb) {
        for (const type of n.types) {
            cb(type, st);
        }
    }
    TsLiteralType(n, st, cb) {
        cb(n.literal, st);
    }
    TsMappedType(n, st, cb) {
        if (n.nameType) {
            cb(n.nameType, st);
        }
        if (n.typeAnnotation) {
            cb(n.typeAnnotation, st);
        }
        cb(n.typeParam, st);
    }
    TsMethodSignature(n, st, cb) {
        for (const param of n.params) {
            cb(param, st);
        }
        cb(n.key, st);
        if (n.typeAnn) {
            cb(n.typeAnn, st);
        }
        if (n.typeParams) {
            cb(n.typeParams, st);
        }
    }
    TsModuleBlock(n, st, cb) {
        for (const stmt of n.body) {
            cb(stmt, st);
        }
    }
    TsModuleDeclaration(n, st, cb) {
        cb(n.id, st);
        if (n.body) {
            cb(n.body, st);
        }
    }
    TsNamespaceDeclaration(n, st, cb) {
        cb(n.id, st);
        cb(n.body, st);
    }
    TsNamespaceExportDeclaration(n, st, cb) {
        cb(n.id, st);
    }
    TsNonNullExpression(n, st, cb) {
        cb(n.expression, st);
    }
    TsOptionalType(n, st, cb) {
        cb(n.typeAnnotation, st);
    }
    TsParameterProperty(n, st, cb) {
        for (const decorator of n.decorators ?? []) {
            cb(decorator, st);
        }
        cb(n.param, st);
    }
    TsParenthesizedType(n, st, cb) {
        cb(n.typeAnnotation, st);
    }
    TsQualifiedName(n, st, cb) {
        cb(n.left, st);
        cb(n.right, st);
    }
    TsRestType(n, st, cb) {
        cb(n.typeAnnotation, st);
    }
    TsSatisfiesExpression(n, st, cb) {
        cb(n.expression, st);
        cb(n.typeAnnotation, st);
    }
    TsSetterSignature(n, st, cb) {
        cb(n.key, st);
        cb(n.param, st);
    }
    TsThisType = ignore;
    TsTupleElement(n, st, cb) {
        if (n.label) {
            cb(n.label, st);
        }
        cb(n.ty, st);
    }
    TsTupleType(n, st, cb) {
        for (const el of n.elemTypes) {
            cb(el, st);
        }
    }
    TsTypeAliasDeclaration(n, st, cb) {
        cb(n.id, st);
        cb(n.typeAnnotation, st);
        if (n.typeParams) {
            cb(n.typeParams, st);
        }
    }
    TsType = ignore;
    TsTypeAnnotation(n, st, cb) {
        cb(n.typeAnnotation, st);
    }
    TsTypeParameter(n, st, cb) {
        cb(n.name, st);
        if (n.constraint) {
            cb(n.constraint, st);
        }
        if (n.default) {
            cb(n.default, st);
        }
    }
    TsTypeParameterDeclaration(n, st, cb) {
        for (const param of n.parameters) {
            cb(param, st);
        }
    }
    TsTypeAssertion(n, st, cb) {
        cb(n.expression, st);
        cb(n.typeAnnotation, st);
    }
    TsTypeElement = ignore;
    TsTypeLiteral(n, st, cb) {
        for (const member of n.members) {
            cb(member, st);
        }
    }
    TsTypeOperator(n, st, cb) {
        cb(n.typeAnnotation, st);
    }
    TsTypeParameterInstantiation(n, st, cb) {
        for (const param of n.params) {
            cb(param, st);
        }
    }
    TsTypeReference(n, st, cb) {
        cb(n.typeName, st);
        if (n.typeParams) {
            cb(n.typeParams, st);
        }
    }
    TsTypePredicate(n, st, cb) {
        cb(n.paramName, st);
        if (n.typeAnnotation) {
            cb(n.typeAnnotation, st);
        }
    }
    TsTypeQuery(n, st, cb) {
        cb(n.exprName, st);
        if (n.typeArguments) {
            cb(n.typeArguments, st);
        }
    }
    TsUnionType(n, st, cb) {
        for (const type of n.types) {
            cb(type, st);
        }
    }
    UnaryExpression(n, st, cb) {
        cb(n.argument, st);
    }
    UpdateExpression(n, st, cb) {
        cb(n.argument, st);
    }
    VariableDeclaration(n, st, cb) {
        for (const decl of n.declarations) {
            cb(decl, st);
        }
    }
    VariableDeclarator(n, st, cb) {
        cb(n.id, st);
        if (n.init) {
            cb(n.init, st);
        }
    }
    WhileStatement(n, st, cb) {
        cb(n.test, st);
        cb(n.body, st);
    }
    WithStatement(n, st, cb) {
        cb(n.object, st);
        cb(n.body, st);
    }
    YieldExpression(n, st, cb) {
        if (n.argument) {
            cb(n.argument, st);
        }
    }
}
exports.BaseVisitor = BaseVisitor;
exports.default = BaseVisitor;
