/*
    SPDX-FileCopyrightText: 2007 Friedrich W. H. Kossebau <kossebau@kde.org>

    SPDX-License-Identifier: LGPL-2.1-only OR LGPL-3.0-only OR LicenseRef-KDE-Accepted-LGPL
*/

#include "abstractbytearraystreamencoder.hpp"

// lib
#include <bytearrayview.hpp>
// Okteta Kasten core
#include <Kasten/Okteta/ByteArrayDocument>
#include <Kasten/Okteta/ByteArraySelection>
// Okteta core
#include <Okteta/AbstractByteArrayModel>
// Qt
#include <QBuffer>

namespace Kasten {

AbstractByteArrayStreamEncoder::AbstractByteArrayStreamEncoder(const QString& remoteTypeName,
                                                               const QString& remoteMimeType,
                                                               const QString& remoteClipboardMimeType)
    : AbstractModelStreamEncoder(remoteTypeName, remoteMimeType,
                                 remoteClipboardMimeType.isEmpty() ? QStringLiteral("text/plain") : remoteClipboardMimeType)
{}

AbstractByteArrayStreamEncoder::~AbstractByteArrayStreamEncoder() = default;

QString AbstractByteArrayStreamEncoder::modelTypeName(AbstractModel* model, const AbstractModelSelection* selection) const
{
    Q_UNUSED(selection)

    const auto * byteArrayDocument = model->findBaseModel<const ByteArrayDocument*>();

    return byteArrayDocument ? byteArrayDocument->typeName() : QString();
}

bool AbstractByteArrayStreamEncoder::encodeToStream(QIODevice* device,
                                                    AbstractModel* model, const AbstractModelSelection* selection)
{
    const auto* const byteArrayView = qobject_cast<const ByteArrayView*>(model);

    const ByteArrayDocument* const byteArrayDocument =
        byteArrayView ? qobject_cast<const ByteArrayDocument*>(byteArrayView->baseModel()) : nullptr;
    if (!byteArrayDocument) {
        return false;
    }

    const Okteta::AbstractByteArrayModel* const byteArray = byteArrayDocument->content();

    const ByteArraySelection* const byteArraySelection =
        selection ? static_cast<const ByteArraySelection*>(selection) : nullptr;

    const Okteta::AddressRange range = byteArraySelection && byteArraySelection->isValid() ?
                                       byteArraySelection->range() :
                                       Okteta::AddressRange::fromWidth(0, byteArray->size());

    const bool success = encodeDataToStream(device, byteArrayView, byteArray, range);

    return success;
}

QByteArray AbstractByteArrayStreamEncoder::previewData(AbstractModel* model, const AbstractModelSelection* selection)
{
    const auto* const byteArrayView = qobject_cast<const ByteArrayView*>(model);

    const ByteArrayDocument* const byteArrayDocument =
        byteArrayView ? qobject_cast<const ByteArrayDocument*>(byteArrayView->baseModel()) : nullptr;
    if (!byteArrayDocument) {
        return {};
    }

    const Okteta::AbstractByteArrayModel* const byteArray = byteArrayDocument->content();

    const ByteArraySelection* const byteArraySelection =
        selection ? static_cast<const ByteArraySelection*>(selection) : nullptr;

    Okteta::AddressRange range = byteArraySelection && byteArraySelection->isValid() ?
                                 byteArraySelection->range() :
                                 Okteta::AddressRange::fromWidth(0, byteArray->size());
    range.restrictEndByWidth(MaxPreviewSize);

    QByteArray data;
    QBuffer dataBuffer(&data);
    dataBuffer.open(QIODevice::WriteOnly);

    const bool success = encodeDataToStream(&dataBuffer, byteArrayView, byteArray, range);
    dataBuffer.close();

    return success ? data : QByteArray();
}

}

#include "moc_abstractbytearraystreamencoder.cpp"
