/*
 * Copyright (c) 2004-2019 Matthew R. Green
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "irc.h"
IRCII_RCSID("@(#)$eterna: cipher-test.c,v 1.15 2023/09/21 19:37:03 mrg Exp $");

#include "irccrypt.h"
#include "vars.h"
#include "ircaux.h"
#include "list.h"
#include "ctcp.h"
#include "output.h"
#include "newio.h"

#ifdef HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif /* HAVE_SYS_WAIT_H */

static void crypt_get_random_data(u_char *, size_t);

#include "aes.c"
#include "cast.c"

static	char	*progname;

static	void	usage(void);
static	void
usage(void)
{
	fprintf(stderr, "Usage: %s [aes|cast]\n", progname);
	exit(1);
}
	u_char crypt_me[] = "\
this is the thing to be encrypted.  oink!  we will now pass this string\
to the crypt function and then the decrypt function and make sure it\
comes out the same way!\
";

int
main(int argc, char *argv[])
{
	struct crypt_key k;
	u_char *s, *olds;
	int rv;
	size_t len;

	progname = argv[0];

	if (argc != 2)
		usage();

	if (strcmp(argv[1], "cast") == 0)
	{
		k.key = UP("this is the cast encryption key that we are going to use for the test");
		k.type = CAST_STRING;
		k.crypt = cast_encrypt_str;
		k.decrypt = cast_decrypt_str;
		k.setkey = cast_setkey;
		k.cookie = 0;
	}
	else if (strcmp(argv[1], "aes") == 0)
	{
		k.key = UP("this is the aes encryption key that we are going to use for the test");
		k.type = AES_STRING;
		k.crypt = aes_encrypt_str;
		k.decrypt = aes_decrypt_str;
		k.setkey = aes_setkey;
		k.cookie = 0;
	}
	else
		usage();

	len = my_strlen(crypt_me);
	(*k.setkey)(&k, 0);
	s = (*k.crypt)(&k, crypt_me, &len);
	if (s == NULL)
		printf("crypt failed\n");
	else
		printf("crypt worked\n");

	rv = my_strcmp(s, crypt_me);
	printf("comparison %s\n", rv == 0 ? "succeeded" : "failed");

	olds = s;
	s = (*k.decrypt)(&k, s, &len);
	new_free(&olds);
	if (s == NULL)
		printf("decrypt failed\n");
	else
		printf("decrypt worked\n");
	(*k.setkey)(&k, 1);

	rv = my_strcmp(s, crypt_me);
	printf("comparison %s\n", rv == 0 ? "succeeded" : "failed");

	new_free(&s);

	exit(0);
}

/* stolen routines */

static void
crypt_get_random_data(u_char *buf, size_t buflen)
{
	size_t	i;
#ifdef HAVE_DEV_RANDOM
	static	int	devrndfd = -1;
	size_t	remain = buflen;

	if (devrndfd == -1)
	{
		devrndfd = open(DEV_RANDOM_PATH, O_RDONLY);

		if (devrndfd == -1)
		{
			static int first = 1;

			if (first)
			{
				first = 0;
				printf("--- HELP!!!! crypt_dev_random_byte: can not open %s: %s",
				    DEV_RANDOM_PATH, strerror(errno));
				goto do_random_instead_no_alarm;
			}
		}
	}
	if (devrndfd == -2)
		goto do_random_instead_no_alarm;

	for (; remain > 0;) {
		alarm(2);
		ssize_t rv = read(devrndfd, buf, remain);
		alarm(0);
		if (rv > 0) {
			remain -= rv;
			continue;
		}
		/* if we were just interrupted, don't bail on /dev/random */
		if (errno == EINTR)
			printf("--- crypt_dev_random_byte: interrupt -- "
			     "using random()");
		else
			printf("--- HELP!  crypt_dev_random_byte(): read of %zu "
			     "bytes on %s failed: %s",
			     buflen, DEV_RANDOM_PATH, strerror(errno));
		printf("--- using random()");
		break;
	}

	alarm(0);
do_random_instead_no_alarm:
#endif
	for (i = 0; i < buflen; i++)
	{
		buf[i] = (random() & 255);
	}
}

void	*
new_malloc(size_t size)
{
	void	*ptr;

	if ((ptr = malloc(size)) == NULL)
	{
		printf("-1 0\n");
		exit(1);
	}
	return (ptr);
}

/*
 * new_free:  Why do this?  Why not?  Saves me a bit of trouble here and
 * there 
 */
void
new_free(void *ptr)
{
	void	**iptr = ptr;

	if (*iptr)
	{
		free(*iptr);
		*iptr = 0;
	}
}
