/*
 * Decompiled with CFR 0.152.
 */
package org.logstash.secret.cli;

import java.nio.CharBuffer;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.logstash.plugins.ConfigVariableExpander;
import org.logstash.secret.SecretIdentifier;
import org.logstash.secret.cli.Terminal;
import org.logstash.secret.store.SecretStore;
import org.logstash.secret.store.SecretStoreFactory;
import org.logstash.secret.store.SecretStoreUtil;
import org.logstash.secret.store.SecureConfig;

public class SecretStoreCli {
    private static final CharsetEncoder ASCII_ENCODER = StandardCharsets.US_ASCII.newEncoder();
    private final Terminal terminal;
    private final SecretStoreFactory secretStoreFactory;

    public SecretStoreCli(Terminal terminal) {
        this(terminal, SecretStoreFactory.fromEnvironment());
    }

    SecretStoreCli(Terminal terminal, SecretStoreFactory secretStoreFactory) {
        this.terminal = terminal;
        this.secretStoreFactory = secretStoreFactory;
    }

    public void command(String primaryCommand, SecureConfig config, String ... allArguments) {
        Optional<CommandLine> commandParseResult;
        this.terminal.writeLine("");
        try {
            commandParseResult = Command.parse(primaryCommand, allArguments);
        }
        catch (InvalidCommandException e) {
            this.terminal.writeLine(String.format("ERROR: %s", e.getMessage()));
            return;
        }
        if (commandParseResult.isEmpty()) {
            this.printHelp();
            return;
        }
        CommandLine commandLine = commandParseResult.get();
        switch (commandLine.getCommand()) {
            case CREATE: {
                if (commandLine.hasOption(CommandOptions.HELP)) {
                    this.terminal.writeLine("Creates a new keystore. For example: 'bin/logstash-keystore create'");
                    return;
                }
                if (this.secretStoreFactory.exists(config.clone())) {
                    this.terminal.write("An Logstash keystore already exists. Overwrite ? [y/N] ");
                    if (!SecretStoreCli.isYes(this.terminal.readLine())) break;
                    this.create(config);
                    break;
                }
                this.create(config);
                break;
            }
            case LIST: {
                if (commandLine.hasOption(CommandOptions.HELP)) {
                    this.terminal.writeLine("List all secret identifiers from the keystore. For example: `bin/logstash-keystore list`. Note - only the identifiers will be listed, not the secrets.");
                    return;
                }
                Collection<SecretIdentifier> ids = this.secretStoreFactory.load(config).list();
                List<String> keys = ids.stream().filter(id -> !id.equals(SecretStoreFactory.LOGSTASH_MARKER)).map(id -> id.getKey()).collect(Collectors.toList());
                Collections.sort(keys);
                keys.forEach(this.terminal::writeLine);
                break;
            }
            case ADD: {
                if (commandLine.hasOption(CommandOptions.HELP)) {
                    this.terminal.writeLine("Add secrets to the keystore. For example: `bin/logstash-keystore add my-secret`, at the prompt enter your secret. You will use the identifier ${my-secret} in your Logstash configuration.");
                    return;
                }
                if (commandLine.getArguments().isEmpty()) {
                    this.terminal.writeLine("ERROR: You must supply an identifier to add. (e.g. bin/logstash-keystore add my-secret)");
                    return;
                }
                if (this.secretStoreFactory.exists(config.clone())) {
                    SecretStore secretStore = this.secretStoreFactory.load(config);
                    for (String argument : commandLine.getArguments()) {
                        if (!ConfigVariableExpander.KEY_PATTERN.matcher(argument).matches()) {
                            throw new IllegalArgumentException(String.format("Invalid secret key name `%s` provided. %s", argument, "Key names are limited to ASCII letters (`a`-`z`, `A`-`Z`), numbers (`0`-`9`), underscores (`_`), and dots (`.`); they must be at least one character long and cannot begin with a number"));
                        }
                        SecretIdentifier id2 = new SecretIdentifier(argument);
                        byte[] existingValue = secretStore.retrieveSecret(id2);
                        if (existingValue != null) {
                            SecretStoreUtil.clearBytes(existingValue);
                            this.terminal.write(String.format("%s already exists. Overwrite ? [y/N] ", argument));
                            if (!SecretStoreCli.isYes(this.terminal.readLine())) continue;
                        }
                        String enterValueMessage = String.format("Enter value for %s: ", argument);
                        char[] secret = null;
                        while (secret == null) {
                            this.terminal.write(enterValueMessage);
                            char[] readSecret = this.terminal.readSecret();
                            if (readSecret == null || readSecret.length == 0) {
                                this.terminal.writeLine("ERROR: Value cannot be empty");
                                continue;
                            }
                            if (!ASCII_ENCODER.canEncode((CharSequence)CharBuffer.wrap(readSecret))) {
                                this.terminal.writeLine("ERROR: Value must contain only ASCII characters");
                                continue;
                            }
                            secret = readSecret;
                        }
                        this.add(secretStore, id2, SecretStoreUtil.asciiCharToBytes(secret));
                    }
                    break;
                }
                this.terminal.writeLine("ERROR: Logstash keystore not found. Use 'create' command to create one.");
                break;
            }
            case REMOVE: {
                if (commandLine.hasOption(CommandOptions.HELP)) {
                    this.terminal.writeLine("Remove secrets from the keystore. For example: `bin/logstash-keystore remove my-secret`");
                    return;
                }
                if (commandLine.getArguments().isEmpty()) {
                    this.terminal.writeLine("ERROR: You must supply a value to remove. (e.g. bin/logstash-keystore remove my-secret)");
                    return;
                }
                SecretStore secretStore = this.secretStoreFactory.load(config);
                for (String argument : commandLine.getArguments()) {
                    SecretIdentifier id3 = new SecretIdentifier(argument);
                    if (secretStore.containsSecret(id3)) {
                        secretStore.purgeSecret(id3);
                        this.terminal.writeLine(String.format("Removed '%s' from the Logstash keystore.", id3.getKey()));
                        continue;
                    }
                    this.terminal.writeLine(String.format("ERROR: '%s' does not exist in the Logstash keystore.", argument));
                }
                break;
            }
        }
    }

    private void printHelp() {
        this.terminal.writeLine("Usage:");
        this.terminal.writeLine("--------");
        this.terminal.writeLine("bin/logstash-keystore [option] command [argument]");
        this.terminal.writeLine("");
        this.terminal.writeLine("Commands:");
        this.terminal.writeLine("--------");
        this.terminal.writeLine("create - Creates a new Logstash keystore  (e.g. bin/logstash-keystore create)");
        this.terminal.writeLine("list   - List entries in the keystore  (e.g. bin/logstash-keystore list)");
        this.terminal.writeLine("add    - Add values to the keystore (e.g. bin/logstash-keystore add secret-one secret-two)");
        this.terminal.writeLine("remove - Remove values from the keystore  (e.g. bin/logstash-keystore remove secret-one secret-two)");
        this.terminal.writeLine("");
        this.terminal.writeLine("Argument:");
        this.terminal.writeLine("--------");
        this.terminal.writeLine("--help - Display command specific help  (e.g. bin/logstash-keystore add --help)");
        this.terminal.writeLine("");
        this.terminal.writeLine("Options:");
        this.terminal.writeLine("--------");
        this.terminal.writeLine("--path.settings - Set the directory for the keystore. This is should be the same directory as the logstash.yml settings file. The default is the config directory under Logstash home. (e.g. bin/logstash-keystore --path.settings /tmp/foo create)");
        this.terminal.writeLine("");
    }

    private void add(SecretStore secretStore, SecretIdentifier id, byte[] secret) {
        secretStore.persistSecret(id, secret);
        this.terminal.writeLine(String.format("Added '%s' to the Logstash keystore.", id.getKey()));
        SecretStoreUtil.clearBytes(secret);
    }

    private void create(SecureConfig config) {
        if (System.getenv("LOGSTASH_KEYSTORE_PASS") == null) {
            this.terminal.write(String.format("WARNING: The keystore password is not set. Please set the environment variable `%s`. Failure to do so will result in reduced security. Continue without password protection on the keystore? [y/N] ", "LOGSTASH_KEYSTORE_PASS"));
            if (SecretStoreCli.isYes(this.terminal.readLine())) {
                this.deleteThenCreate(config);
            }
        } else {
            this.deleteThenCreate(config);
        }
    }

    private void deleteThenCreate(SecureConfig config) {
        this.secretStoreFactory.delete(config.clone());
        this.secretStoreFactory.create(config.clone());
        char[] fileLocation = config.getPlainText("keystore.file");
        this.terminal.writeLine("Created Logstash keystore" + (String)(fileLocation == null ? "." : " at " + new String(fileLocation)));
    }

    private static boolean isYes(String response) {
        return "y".equalsIgnoreCase(response) || "yes".equalsIgnoreCase(response);
    }

    static enum Command {
        CREATE("create"),
        LIST("list"),
        ADD("add", List.of(CommandOptions.STDIN)),
        REMOVE("remove");

        private final Set<CommandOptions> validOptions = EnumSet.of(CommandOptions.HELP);
        private final String command;

        private Command(String command) {
            this.command = command;
        }

        private Command(String command, Collection<CommandOptions> validOptions) {
            this.command = command;
            this.validOptions.addAll(validOptions);
        }

        static Optional<CommandLine> parse(String command, String[] arguments) {
            Optional<Command> foundCommand = Arrays.stream(Command.values()).filter(c -> c.command.equals(command)).findFirst();
            return foundCommand.map(value -> new CommandLine((Command)((Object)value), arguments));
        }

        Set<CommandOptions> getValidOptions() {
            return Collections.unmodifiableSet(this.validOptions);
        }
    }

    static class InvalidCommandException
    extends RuntimeException {
        private static final long serialVersionUID = 8402825920204022022L;

        public InvalidCommandException(String message) {
            super(message);
        }
    }

    static class CommandLine {
        private final Command command;
        private final List<String> arguments;
        private final Set<CommandOptions> options;

        CommandLine(Command command, String[] arguments) {
            this.command = command;
            this.arguments = CommandLine.parseCommandArguments(command, arguments);
            this.options = CommandLine.parseCommandOptions(command, arguments);
        }

        private static List<String> parseCommandArguments(Command command, String[] arguments) {
            if (arguments == null || arguments.length == 0) {
                return List.of();
            }
            Set commandValidOptions = command.getValidOptions().stream().map(CommandOptions::getOption).collect(Collectors.toSet());
            String[] filteredArguments = arguments;
            for (int i = 0; i < arguments.length; ++i) {
                if (!commandValidOptions.contains(arguments[i])) continue;
                filteredArguments = Arrays.copyOfRange(arguments, 0, i);
                break;
            }
            return Arrays.asList(filteredArguments);
        }

        private static Set<CommandOptions> parseCommandOptions(Command command, String[] arguments) {
            EnumSet<CommandOptions> providedOptions = EnumSet.noneOf(CommandOptions.class);
            if (arguments == null) {
                return providedOptions;
            }
            for (String argument : arguments) {
                if (!argument.startsWith("--")) continue;
                Optional<CommandOptions> option = CommandOptions.fromString(argument);
                if (option.isEmpty() || !command.validOptions.contains((Object)option.get())) {
                    throw new InvalidCommandException(String.format("Unrecognized option '%s' for command '%s'", argument, command.command));
                }
                providedOptions.add(option.get());
            }
            return providedOptions;
        }

        boolean hasOption(CommandOptions option) {
            return this.options.contains((Object)option);
        }

        Command getCommand() {
            return this.command;
        }

        List<String> getArguments() {
            return Collections.unmodifiableList(this.arguments);
        }
    }

    static enum CommandOptions {
        HELP("help"),
        STDIN("stdin");

        private static final String PREFIX = "--";
        private final String option;

        private CommandOptions(String option) {
            this.option = String.format("%s%s", PREFIX, option);
        }

        static Optional<CommandOptions> fromString(String option) {
            if (option == null || !option.startsWith(PREFIX)) {
                return Optional.empty();
            }
            return Arrays.stream(CommandOptions.values()).filter(p -> p.option.equals(option)).findFirst();
        }

        String getOption() {
            return this.option;
        }
    }
}

