/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { describe, it, expect, beforeEach, afterEach, vi } from 'vitest';
import stripAnsi from 'strip-ansi';
import { act } from 'react';
import * as fs from 'node:fs';
import * as path from 'node:path';
import * as os from 'node:os';
import { renderHook, renderHookWithProviders, } from '../../../test-utils/render.js';
import { useTextBuffer, offsetToLogicalPos, logicalPosToOffset, textBufferReducer, findWordEndInLine, findNextWordStartInLine, findNextBigWordStartInLine, findPrevBigWordStartInLine, findBigWordEndInLine, isWordCharStrict, calculateTransformationsForLine, calculateTransformedLine, getTransformUnderCursor, getTransformedImagePath, } from './text-buffer.js';
import { cpLen } from '../../utils/textUtils.js';
const defaultVisualLayout = {
    visualLines: [''],
    logicalToVisualMap: [[[0, 0]]],
    visualToLogicalMap: [[0, 0]],
    transformedToLogicalMaps: [[]],
    visualToTransformedMap: [],
};
const initialState = {
    lines: [''],
    cursorRow: 0,
    cursorCol: 0,
    preferredCol: null,
    undoStack: [],
    redoStack: [],
    clipboard: null,
    selectionAnchor: null,
    viewportWidth: 80,
    viewportHeight: 24,
    transformationsByLine: [[]],
    visualLayout: defaultVisualLayout,
    pastedContent: {},
    expandedPaste: null,
};
/**
 * Helper to create a TextBufferState with properly calculated transformations.
 */
function createStateWithTransformations(partial) {
    const state = { ...initialState, ...partial };
    return {
        ...state,
        transformationsByLine: state.lines.map((l) => calculateTransformationsForLine(l)),
    };
}
describe('textBufferReducer', () => {
    afterEach(() => {
        vi.restoreAllMocks();
    });
    it('should return the initial state if state is undefined', () => {
        const action = { type: 'unknown_action' };
        const state = textBufferReducer(initialState, action);
        expect(state).toHaveOnlyValidCharacters();
        expect(state).toEqual(initialState);
    });
    describe('Big Word Navigation Helpers', () => {
        describe('findNextBigWordStartInLine (W)', () => {
            it('should skip non-whitespace and then whitespace', () => {
                expect(findNextBigWordStartInLine('hello world', 0)).toBe(6);
                expect(findNextBigWordStartInLine('hello.world test', 0)).toBe(12);
                expect(findNextBigWordStartInLine('   test', 0)).toBe(3);
                expect(findNextBigWordStartInLine('test   ', 0)).toBe(null);
            });
        });
        describe('findPrevBigWordStartInLine (B)', () => {
            it('should skip whitespace backwards then non-whitespace', () => {
                expect(findPrevBigWordStartInLine('hello world', 6)).toBe(0);
                expect(findPrevBigWordStartInLine('hello.world test', 12)).toBe(0);
                expect(findPrevBigWordStartInLine('   test', 3)).toBe(null); // At start of word
                expect(findPrevBigWordStartInLine('   test', 4)).toBe(3); // Inside word
                expect(findPrevBigWordStartInLine('test   ', 6)).toBe(0);
            });
        });
        describe('findBigWordEndInLine (E)', () => {
            it('should find end of current big word', () => {
                expect(findBigWordEndInLine('hello world', 0)).toBe(4);
                expect(findBigWordEndInLine('hello.world test', 0)).toBe(10);
                expect(findBigWordEndInLine('hello.world test', 11)).toBe(15);
            });
            it('should skip whitespace if currently on whitespace', () => {
                expect(findBigWordEndInLine('hello   world', 5)).toBe(12);
            });
            it('should find next big word end if at end of current', () => {
                expect(findBigWordEndInLine('hello world', 4)).toBe(10);
            });
        });
    });
    describe('set_text action', () => {
        it('should set new text and move cursor to the end', () => {
            const action = {
                type: 'set_text',
                payload: 'hello\nworld',
            };
            const state = textBufferReducer(initialState, action);
            expect(state).toHaveOnlyValidCharacters();
            expect(state.lines).toEqual(['hello', 'world']);
            expect(state.cursorRow).toBe(1);
            expect(state.cursorCol).toBe(5);
            expect(state.undoStack.length).toBe(1);
        });
        it('should not create an undo snapshot if pushToUndo is false', () => {
            const action = {
                type: 'set_text',
                payload: 'no undo',
                pushToUndo: false,
            };
            const state = textBufferReducer(initialState, action);
            expect(state).toHaveOnlyValidCharacters();
            expect(state.lines).toEqual(['no undo']);
            expect(state.undoStack.length).toBe(0);
        });
    });
    describe('insert action', () => {
        it('should insert a character', () => {
            const action = { type: 'insert', payload: 'a' };
            const state = textBufferReducer(initialState, action);
            expect(state).toHaveOnlyValidCharacters();
            expect(state.lines).toEqual(['a']);
            expect(state.cursorCol).toBe(1);
        });
        it('should insert a newline', () => {
            const stateWithText = { ...initialState, lines: ['hello'] };
            const action = { type: 'insert', payload: '\n' };
            const state = textBufferReducer(stateWithText, action);
            expect(state).toHaveOnlyValidCharacters();
            expect(state.lines).toEqual(['', 'hello']);
            expect(state.cursorRow).toBe(1);
            expect(state.cursorCol).toBe(0);
        });
    });
    describe('insert action with options', () => {
        it('should filter input using inputFilter option', () => {
            const action = { type: 'insert', payload: 'a1b2c3' };
            const options = {
                inputFilter: (text) => text.replace(/[0-9]/g, ''),
            };
            const state = textBufferReducer(initialState, action, options);
            expect(state.lines).toEqual(['abc']);
            expect(state.cursorCol).toBe(3);
        });
        it('should strip newlines when singleLine option is true', () => {
            const action = {
                type: 'insert',
                payload: 'hello\nworld',
            };
            const options = { singleLine: true };
            const state = textBufferReducer(initialState, action, options);
            expect(state.lines).toEqual(['helloworld']);
            expect(state.cursorCol).toBe(10);
        });
        it('should apply both inputFilter and singleLine options', () => {
            const action = {
                type: 'insert',
                payload: 'h\ne\nl\nl\no\n1\n2\n3',
            };
            const options = {
                singleLine: true,
                inputFilter: (text) => text.replace(/[0-9]/g, ''),
            };
            const state = textBufferReducer(initialState, action, options);
            expect(state.lines).toEqual(['hello']);
            expect(state.cursorCol).toBe(5);
        });
    });
    describe('add_pasted_content action', () => {
        it('should add content to pastedContent Record', () => {
            const action = {
                type: 'add_pasted_content',
                payload: { id: '[Pasted Text: 6 lines]', text: 'large content' },
            };
            const state = textBufferReducer(initialState, action);
            expect(state.pastedContent).toEqual({
                '[Pasted Text: 6 lines]': 'large content',
            });
        });
    });
    describe('backspace action', () => {
        it('should remove a character', () => {
            const stateWithText = {
                ...initialState,
                lines: ['a'],
                cursorRow: 0,
                cursorCol: 1,
            };
            const action = { type: 'backspace' };
            const state = textBufferReducer(stateWithText, action);
            expect(state).toHaveOnlyValidCharacters();
            expect(state.lines).toEqual(['']);
            expect(state.cursorCol).toBe(0);
        });
        it('should join lines if at the beginning of a line', () => {
            const stateWithText = {
                ...initialState,
                lines: ['hello', 'world'],
                cursorRow: 1,
                cursorCol: 0,
            };
            const action = { type: 'backspace' };
            const state = textBufferReducer(stateWithText, action);
            expect(state).toHaveOnlyValidCharacters();
            expect(state.lines).toEqual(['helloworld']);
            expect(state.cursorRow).toBe(0);
            expect(state.cursorCol).toBe(5);
        });
    });
    describe('atomic placeholder deletion', () => {
        describe('paste placeholders', () => {
            it('backspace at end of paste placeholder removes entire placeholder', () => {
                const placeholder = '[Pasted Text: 6 lines]';
                const stateWithPlaceholder = createStateWithTransformations({
                    lines: [placeholder],
                    cursorRow: 0,
                    cursorCol: placeholder.length, // cursor at end
                    pastedContent: {
                        [placeholder]: 'line1\nline2\nline3\nline4\nline5\nline6',
                    },
                });
                const action = { type: 'backspace' };
                const state = textBufferReducer(stateWithPlaceholder, action);
                expect(state).toHaveOnlyValidCharacters();
                expect(state.lines).toEqual(['']);
                expect(state.cursorCol).toBe(0);
                // pastedContent should be cleaned up
                expect(state.pastedContent[placeholder]).toBeUndefined();
            });
            it('delete at start of paste placeholder removes entire placeholder', () => {
                const placeholder = '[Pasted Text: 6 lines]';
                const stateWithPlaceholder = createStateWithTransformations({
                    lines: [placeholder],
                    cursorRow: 0,
                    cursorCol: 0, // cursor at start
                    pastedContent: {
                        [placeholder]: 'line1\nline2\nline3\nline4\nline5\nline6',
                    },
                });
                const action = { type: 'delete' };
                const state = textBufferReducer(stateWithPlaceholder, action);
                expect(state).toHaveOnlyValidCharacters();
                expect(state.lines).toEqual(['']);
                expect(state.cursorCol).toBe(0);
                // pastedContent should be cleaned up
                expect(state.pastedContent[placeholder]).toBeUndefined();
            });
            it('backspace inside paste placeholder does normal deletion', () => {
                const placeholder = '[Pasted Text: 6 lines]';
                const stateWithPlaceholder = createStateWithTransformations({
                    lines: [placeholder],
                    cursorRow: 0,
                    cursorCol: 10, // cursor in middle
                    pastedContent: {
                        [placeholder]: 'line1\nline2\nline3\nline4\nline5\nline6',
                    },
                });
                const action = { type: 'backspace' };
                const state = textBufferReducer(stateWithPlaceholder, action);
                expect(state).toHaveOnlyValidCharacters();
                // Should only delete one character
                expect(state.lines[0].length).toBe(placeholder.length - 1);
                expect(state.cursorCol).toBe(9);
                // pastedContent should NOT be cleaned up (placeholder is broken)
                expect(state.pastedContent[placeholder]).toBeDefined();
            });
        });
        describe('image placeholders', () => {
            it('backspace at end of image path removes entire path', () => {
                const imagePath = '@test.png';
                const stateWithImage = createStateWithTransformations({
                    lines: [imagePath],
                    cursorRow: 0,
                    cursorCol: imagePath.length, // cursor at end
                });
                const action = { type: 'backspace' };
                const state = textBufferReducer(stateWithImage, action);
                expect(state).toHaveOnlyValidCharacters();
                expect(state.lines).toEqual(['']);
                expect(state.cursorCol).toBe(0);
            });
            it('delete at start of image path removes entire path', () => {
                const imagePath = '@test.png';
                const stateWithImage = createStateWithTransformations({
                    lines: [imagePath],
                    cursorRow: 0,
                    cursorCol: 0, // cursor at start
                });
                const action = { type: 'delete' };
                const state = textBufferReducer(stateWithImage, action);
                expect(state).toHaveOnlyValidCharacters();
                expect(state.lines).toEqual(['']);
                expect(state.cursorCol).toBe(0);
            });
            it('backspace inside image path does normal deletion', () => {
                const imagePath = '@test.png';
                const stateWithImage = createStateWithTransformations({
                    lines: [imagePath],
                    cursorRow: 0,
                    cursorCol: 5, // cursor in middle
                });
                const action = { type: 'backspace' };
                const state = textBufferReducer(stateWithImage, action);
                expect(state).toHaveOnlyValidCharacters();
                // Should only delete one character
                expect(state.lines[0].length).toBe(imagePath.length - 1);
                expect(state.cursorCol).toBe(4);
            });
        });
        describe('undo behavior', () => {
            it('undo after placeholder deletion restores everything', () => {
                const placeholder = '[Pasted Text: 6 lines]';
                const pasteContent = 'line1\nline2\nline3\nline4\nline5\nline6';
                const stateWithPlaceholder = createStateWithTransformations({
                    lines: [placeholder],
                    cursorRow: 0,
                    cursorCol: placeholder.length,
                    pastedContent: { [placeholder]: pasteContent },
                });
                // Delete the placeholder
                const deleteAction = { type: 'backspace' };
                const stateAfterDelete = textBufferReducer(stateWithPlaceholder, deleteAction);
                expect(stateAfterDelete.lines).toEqual(['']);
                expect(stateAfterDelete.pastedContent[placeholder]).toBeUndefined();
                // Undo should restore
                const undoAction = { type: 'undo' };
                const stateAfterUndo = textBufferReducer(stateAfterDelete, undoAction);
                expect(stateAfterUndo).toHaveOnlyValidCharacters();
                expect(stateAfterUndo.lines).toEqual([placeholder]);
                expect(stateAfterUndo.pastedContent[placeholder]).toBe(pasteContent);
            });
        });
    });
    describe('undo/redo actions', () => {
        it('should undo and redo a change', () => {
            // 1. Insert text
            const insertAction = {
                type: 'insert',
                payload: 'test',
            };
            const stateAfterInsert = textBufferReducer(initialState, insertAction);
            expect(stateAfterInsert).toHaveOnlyValidCharacters();
            expect(stateAfterInsert.lines).toEqual(['test']);
            expect(stateAfterInsert.undoStack.length).toBe(1);
            // 2. Undo
            const undoAction = { type: 'undo' };
            const stateAfterUndo = textBufferReducer(stateAfterInsert, undoAction);
            expect(stateAfterUndo).toHaveOnlyValidCharacters();
            expect(stateAfterUndo.lines).toEqual(['']);
            expect(stateAfterUndo.undoStack.length).toBe(0);
            expect(stateAfterUndo.redoStack.length).toBe(1);
            // 3. Redo
            const redoAction = { type: 'redo' };
            const stateAfterRedo = textBufferReducer(stateAfterUndo, redoAction);
            expect(stateAfterRedo).toHaveOnlyValidCharacters();
            expect(stateAfterRedo.lines).toEqual(['test']);
            expect(stateAfterRedo.undoStack.length).toBe(1);
            expect(stateAfterRedo.redoStack.length).toBe(0);
        });
    });
    describe('create_undo_snapshot action', () => {
        it('should create a snapshot without changing state', () => {
            const stateWithText = {
                ...initialState,
                lines: ['hello'],
                cursorRow: 0,
                cursorCol: 5,
            };
            const action = { type: 'create_undo_snapshot' };
            const state = textBufferReducer(stateWithText, action);
            expect(state).toHaveOnlyValidCharacters();
            expect(state.lines).toEqual(['hello']);
            expect(state.cursorRow).toBe(0);
            expect(state.cursorCol).toBe(5);
            expect(state.undoStack.length).toBe(1);
            expect(state.undoStack[0].lines).toEqual(['hello']);
            expect(state.undoStack[0].cursorRow).toBe(0);
            expect(state.undoStack[0].cursorCol).toBe(5);
        });
    });
    describe('delete_word_left action', () => {
        const createSingleLineState = (text, col) => ({
            ...initialState,
            lines: [text],
            cursorRow: 0,
            cursorCol: col,
        });
        it.each([
            {
                input: 'hello world',
                cursorCol: 11,
                expectedLines: ['hello '],
                expectedCol: 6,
                desc: 'simple word',
            },
            {
                input: 'path/to/file',
                cursorCol: 12,
                expectedLines: ['path/to/'],
                expectedCol: 8,
                desc: 'path segment',
            },
            {
                input: 'variable_name',
                cursorCol: 13,
                expectedLines: ['variable_'],
                expectedCol: 9,
                desc: 'variable_name parts',
            },
        ])('should delete $desc', ({ input, cursorCol, expectedLines, expectedCol }) => {
            const state = textBufferReducer(createSingleLineState(input, cursorCol), { type: 'delete_word_left' });
            expect(state.lines).toEqual(expectedLines);
            expect(state.cursorCol).toBe(expectedCol);
        });
        it('should act like backspace at the beginning of a line', () => {
            const stateWithText = {
                ...initialState,
                lines: ['hello', 'world'],
                cursorRow: 1,
                cursorCol: 0,
            };
            const state = textBufferReducer(stateWithText, {
                type: 'delete_word_left',
            });
            expect(state.lines).toEqual(['helloworld']);
            expect(state.cursorRow).toBe(0);
            expect(state.cursorCol).toBe(5);
        });
    });
    describe('delete_word_right action', () => {
        const createSingleLineState = (text, col) => ({
            ...initialState,
            lines: [text],
            cursorRow: 0,
            cursorCol: col,
        });
        it.each([
            {
                input: 'hello world',
                cursorCol: 0,
                expectedLines: ['world'],
                expectedCol: 0,
                desc: 'simple word',
            },
            {
                input: 'variable_name',
                cursorCol: 0,
                expectedLines: ['_name'],
                expectedCol: 0,
                desc: 'variable_name parts',
            },
        ])('should delete $desc', ({ input, cursorCol, expectedLines, expectedCol }) => {
            const state = textBufferReducer(createSingleLineState(input, cursorCol), { type: 'delete_word_right' });
            expect(state.lines).toEqual(expectedLines);
            expect(state.cursorCol).toBe(expectedCol);
        });
        it('should delete path segments progressively', () => {
            const stateWithText = {
                ...initialState,
                lines: ['path/to/file'],
                cursorRow: 0,
                cursorCol: 0,
            };
            let state = textBufferReducer(stateWithText, {
                type: 'delete_word_right',
            });
            expect(state.lines).toEqual(['/to/file']);
            state = textBufferReducer(state, { type: 'delete_word_right' });
            expect(state.lines).toEqual(['to/file']);
        });
        it('should act like delete at the end of a line', () => {
            const stateWithText = {
                ...initialState,
                lines: ['hello', 'world'],
                cursorRow: 0,
                cursorCol: 5,
            };
            const state = textBufferReducer(stateWithText, {
                type: 'delete_word_right',
            });
            expect(state.lines).toEqual(['helloworld']);
            expect(state.cursorRow).toBe(0);
            expect(state.cursorCol).toBe(5);
        });
    });
    describe('toggle_paste_expansion action', () => {
        const placeholder = '[Pasted Text: 6 lines]';
        const content = 'line1\nline2\nline3\nline4\nline5\nline6';
        it('should expand a placeholder correctly', () => {
            const stateWithPlaceholder = createStateWithTransformations({
                lines: ['prefix ' + placeholder + ' suffix'],
                cursorRow: 0,
                cursorCol: 0,
                pastedContent: { [placeholder]: content },
            });
            const action = {
                type: 'toggle_paste_expansion',
                payload: { id: placeholder, row: 0, col: 7 },
            };
            const state = textBufferReducer(stateWithPlaceholder, action);
            expect(state.lines).toEqual([
                'prefix line1',
                'line2',
                'line3',
                'line4',
                'line5',
                'line6 suffix',
            ]);
            expect(state.expandedPaste?.id).toBe(placeholder);
            const info = state.expandedPaste;
            expect(info).toEqual({
                id: placeholder,
                startLine: 0,
                lineCount: 6,
                prefix: 'prefix ',
                suffix: ' suffix',
            });
            // Cursor should be at the end of expanded content (before suffix)
            expect(state.cursorRow).toBe(5);
            expect(state.cursorCol).toBe(5); // length of 'line6'
        });
        it('should collapse an expanded placeholder correctly', () => {
            const expandedState = createStateWithTransformations({
                lines: [
                    'prefix line1',
                    'line2',
                    'line3',
                    'line4',
                    'line5',
                    'line6 suffix',
                ],
                cursorRow: 5,
                cursorCol: 5,
                pastedContent: { [placeholder]: content },
                expandedPaste: {
                    id: placeholder,
                    startLine: 0,
                    lineCount: 6,
                    prefix: 'prefix ',
                    suffix: ' suffix',
                },
            });
            const action = {
                type: 'toggle_paste_expansion',
                payload: { id: placeholder, row: 0, col: 7 },
            };
            const state = textBufferReducer(expandedState, action);
            expect(state.lines).toEqual(['prefix ' + placeholder + ' suffix']);
            expect(state.expandedPaste).toBeNull();
            // Cursor should be at the end of the collapsed placeholder
            expect(state.cursorRow).toBe(0);
            expect(state.cursorCol).toBe(('prefix ' + placeholder).length);
        });
        it('should expand single-line content correctly', () => {
            const singleLinePlaceholder = '[Pasted Text: 10 chars]';
            const singleLineContent = 'some text';
            const stateWithPlaceholder = createStateWithTransformations({
                lines: [singleLinePlaceholder],
                cursorRow: 0,
                cursorCol: 0,
                pastedContent: { [singleLinePlaceholder]: singleLineContent },
            });
            const state = textBufferReducer(stateWithPlaceholder, {
                type: 'toggle_paste_expansion',
                payload: { id: singleLinePlaceholder, row: 0, col: 0 },
            });
            expect(state.lines).toEqual(['some text']);
            expect(state.cursorRow).toBe(0);
            expect(state.cursorCol).toBe(9);
        });
        it('should return current state if placeholder ID not found in pastedContent', () => {
            const action = {
                type: 'toggle_paste_expansion',
                payload: { id: 'unknown', row: 0, col: 0 },
            };
            const state = textBufferReducer(initialState, action);
            expect(state).toBe(initialState);
        });
        it('should preserve expandedPaste when lines change from edits outside the region', () => {
            // Start with an expanded paste at line 0 (3 lines long)
            const placeholder = '[Pasted Text: 3 lines]';
            const expandedState = createStateWithTransformations({
                lines: ['line1', 'line2', 'line3', 'suffix'],
                cursorRow: 3,
                cursorCol: 0,
                pastedContent: { [placeholder]: 'line1\nline2\nline3' },
                expandedPaste: {
                    id: placeholder,
                    startLine: 0,
                    lineCount: 3,
                    prefix: '',
                    suffix: '',
                },
            });
            expect(expandedState.expandedPaste).not.toBeNull();
            // Insert a newline at the end - this changes lines but is OUTSIDE the expanded region
            const stateAfterInsert = textBufferReducer(expandedState, {
                type: 'insert',
                payload: '\n',
            });
            // Lines changed, but expandedPaste should be PRESERVED and optionally shifted (no shift here since edit is after)
            expect(stateAfterInsert.expandedPaste).not.toBeNull();
            expect(stateAfterInsert.expandedPaste?.id).toBe(placeholder);
        });
    });
});
const getBufferState = (result) => {
    expect(result.current).toHaveOnlyValidCharacters();
    return {
        text: result.current.text,
        lines: [...result.current.lines], // Clone for safety
        cursor: [...result.current.cursor],
        allVisualLines: [...result.current.allVisualLines],
        viewportVisualLines: [...result.current.viewportVisualLines],
        visualCursor: [...result.current.visualCursor],
        visualScrollRow: result.current.visualScrollRow,
        preferredCol: result.current.preferredCol,
    };
};
describe('useTextBuffer', () => {
    let viewport;
    beforeEach(() => {
        viewport = { width: 10, height: 3 }; // Default viewport for tests
    });
    afterEach(() => {
        vi.restoreAllMocks();
    });
    describe('Initialization', () => {
        it('should initialize with empty text and cursor at (0,0) by default', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            const state = getBufferState(result);
            expect(state.text).toBe('');
            expect(state.lines).toEqual(['']);
            expect(state.cursor).toEqual([0, 0]);
            expect(state.allVisualLines).toEqual(['']);
            expect(state.viewportVisualLines).toEqual(['']);
            expect(state.visualCursor).toEqual([0, 0]);
            expect(state.visualScrollRow).toBe(0);
        });
        it('should initialize with provided initialText', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'hello',
                viewport,
            }));
            const state = getBufferState(result);
            expect(state.text).toBe('hello');
            expect(state.lines).toEqual(['hello']);
            expect(state.cursor).toEqual([0, 0]); // Default cursor if offset not given
            expect(state.allVisualLines).toEqual(['hello']);
            expect(state.viewportVisualLines).toEqual(['hello']);
            expect(state.visualCursor).toEqual([0, 0]);
        });
        it('should initialize with initialText and initialCursorOffset', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'hello\nworld',
                initialCursorOffset: 7, // Should be at 'o' in 'world'
                viewport,
            }));
            const state = getBufferState(result);
            expect(state.text).toBe('hello\nworld');
            expect(state.lines).toEqual(['hello', 'world']);
            expect(state.cursor).toEqual([1, 1]); // Logical cursor at 'o' in "world"
            expect(state.allVisualLines).toEqual(['hello', 'world']);
            expect(state.viewportVisualLines).toEqual(['hello', 'world']);
            expect(state.visualCursor[0]).toBe(1); // On the second visual line
            expect(state.visualCursor[1]).toBe(1); // At 'o' in "world"
        });
        it('should wrap visual lines', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'The quick brown fox jumps over the lazy dog.',
                initialCursorOffset: 2, // After '好'
                viewport: { width: 15, height: 4 },
            }));
            const state = getBufferState(result);
            expect(state.allVisualLines).toEqual([
                'The quick',
                'brown fox',
                'jumps over the',
                'lazy dog.',
            ]);
        });
        it('should wrap visual lines with multiple spaces', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'The  quick  brown fox    jumps over the lazy dog.',
                viewport: { width: 15, height: 4 },
            }));
            const state = getBufferState(result);
            // Including multiple spaces at the end of the lines like this is
            // consistent with Google docs behavior and makes it intuitive to edit
            // the spaces as needed.
            expect(state.allVisualLines).toEqual([
                'The  quick ',
                'brown fox   ',
                'jumps over the',
                'lazy dog.',
            ]);
        });
        it('should wrap visual lines even without spaces', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: '123456789012345ABCDEFG', // 4 chars, 12 bytes
                viewport: { width: 15, height: 2 },
            }));
            const state = getBufferState(result);
            // Including multiple spaces at the end of the lines like this is
            // consistent with Google docs behavior and makes it intuitive to edit
            // the spaces as needed.
            expect(state.allVisualLines).toEqual(['123456789012345', 'ABCDEFG']);
        });
        it('should initialize with multi-byte unicode characters and correct cursor offset', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: '你好世界', // 4 chars, 12 bytes
                initialCursorOffset: 2, // After '好'
                viewport: { width: 5, height: 2 },
            }));
            const state = getBufferState(result);
            expect(state.text).toBe('你好世界');
            expect(state.lines).toEqual(['你好世界']);
            expect(state.cursor).toEqual([0, 2]);
            // Visual: "你好" (width 4), "世"界" (width 4) with viewport width 5
            expect(state.allVisualLines).toEqual(['你好', '世界']);
            expect(state.visualCursor).toEqual([1, 0]);
        });
    });
    describe('Basic Editing', () => {
        it('insert: should insert a character and update cursor', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            act(() => result.current.insert('a'));
            let state = getBufferState(result);
            expect(state.text).toBe('a');
            expect(state.cursor).toEqual([0, 1]);
            expect(state.visualCursor).toEqual([0, 1]);
            act(() => result.current.insert('b'));
            state = getBufferState(result);
            expect(state.text).toBe('ab');
            expect(state.cursor).toEqual([0, 2]);
            expect(state.visualCursor).toEqual([0, 2]);
        });
        it('insert: should insert text in the middle of a line', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'abc',
                viewport,
            }));
            act(() => result.current.move('right'));
            act(() => result.current.insert('-NEW-'));
            const state = getBufferState(result);
            expect(state.text).toBe('a-NEW-bc');
            expect(state.cursor).toEqual([0, 6]);
        });
        it('insert: should use placeholder for large text paste', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            const largeText = '1\n2\n3\n4\n5\n6';
            act(() => result.current.insert(largeText, { paste: true }));
            const state = getBufferState(result);
            expect(state.text).toBe('[Pasted Text: 6 lines]');
            expect(result.current.pastedContent['[Pasted Text: 6 lines]']).toBe(largeText);
        });
        it('insert: should NOT use placeholder for large text if NOT a paste', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            const largeText = '1\n2\n3\n4\n5\n6';
            act(() => result.current.insert(largeText, { paste: false }));
            const state = getBufferState(result);
            expect(state.text).toBe(largeText);
        });
        it('insert: should clean up pastedContent when placeholder is deleted', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            const largeText = '1\n2\n3\n4\n5\n6';
            act(() => result.current.insert(largeText, { paste: true }));
            expect(result.current.pastedContent['[Pasted Text: 6 lines]']).toBe(largeText);
            // Delete the placeholder using setText
            act(() => result.current.setText(''));
            expect(Object.keys(result.current.pastedContent)).toHaveLength(0);
        });
        it('insert: should clean up pastedContent when placeholder is removed via atomic backspace', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            const largeText = '1\n2\n3\n4\n5\n6';
            act(() => result.current.insert(largeText, { paste: true }));
            expect(result.current.pastedContent['[Pasted Text: 6 lines]']).toBe(largeText);
            // Single backspace at end of placeholder removes entire placeholder
            act(() => {
                result.current.backspace();
            });
            expect(getBufferState(result).text).toBe('');
            // pastedContent is cleaned up when placeholder is deleted atomically
            expect(Object.keys(result.current.pastedContent)).toHaveLength(0);
        });
        it('newline: should create a new line and move cursor', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'ab',
                viewport,
            }));
            act(() => result.current.move('end')); // cursor at [0,2]
            act(() => result.current.newline());
            const state = getBufferState(result);
            expect(state.text).toBe('ab\n');
            expect(state.lines).toEqual(['ab', '']);
            expect(state.cursor).toEqual([1, 0]);
            expect(state.allVisualLines).toEqual(['ab', '']);
            expect(state.viewportVisualLines).toEqual(['ab', '']); // viewport height 3
            expect(state.visualCursor).toEqual([1, 0]); // On the new visual line
        });
        it('backspace: should delete char to the left or merge lines', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'a\nb',
                viewport,
            }));
            act(() => {
                result.current.move('down');
            });
            act(() => {
                result.current.move('end'); // cursor to [1,1] (end of 'b')
            });
            act(() => result.current.backspace()); // delete 'b'
            let state = getBufferState(result);
            expect(state.text).toBe('a\n');
            expect(state.cursor).toEqual([1, 0]);
            act(() => result.current.backspace()); // merge lines
            state = getBufferState(result);
            expect(state.text).toBe('a');
            expect(state.cursor).toEqual([0, 1]); // cursor after 'a'
            expect(state.allVisualLines).toEqual(['a']);
            expect(state.viewportVisualLines).toEqual(['a']);
            expect(state.visualCursor).toEqual([0, 1]);
        });
        it('del: should delete char to the right or merge lines', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'a\nb',
                viewport,
            }));
            // cursor at [0,0]
            act(() => result.current.del()); // delete 'a'
            let state = getBufferState(result);
            expect(state.text).toBe('\nb');
            expect(state.cursor).toEqual([0, 0]);
            act(() => result.current.del()); // merge lines (deletes newline)
            state = getBufferState(result);
            expect(state.text).toBe('b');
            expect(state.cursor).toEqual([0, 0]);
            expect(state.allVisualLines).toEqual(['b']);
            expect(state.viewportVisualLines).toEqual(['b']);
            expect(state.visualCursor).toEqual([0, 0]);
        });
    });
    describe('Drag and Drop File Paths', () => {
        let tempDir;
        beforeEach(() => {
            tempDir = fs.mkdtempSync(path.join(os.tmpdir(), 'gemini-cli-test-'));
        });
        afterEach(() => {
            fs.rmSync(tempDir, { recursive: true, force: true });
        });
        it('should prepend @ to a valid file path on insert', () => {
            const filePath = path.join(tempDir, 'file.txt');
            fs.writeFileSync(filePath, '');
            const { result } = renderHook(() => useTextBuffer({ viewport, escapePastedPaths: true }));
            act(() => result.current.insert(filePath, { paste: true }));
            expect(getBufferState(result).text).toBe(`@${filePath} `);
        });
        it('should not prepend @ to an invalid file path on insert', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            const notAPath = path.join(tempDir, 'non_existent.txt');
            act(() => result.current.insert(notAPath, { paste: true }));
            expect(getBufferState(result).text).toBe(notAPath);
        });
        it('should handle quoted paths', () => {
            const filePath = path.join(tempDir, 'file.txt');
            fs.writeFileSync(filePath, '');
            const { result } = renderHook(() => useTextBuffer({ viewport, escapePastedPaths: true }));
            const quotedPath = `'${filePath}'`;
            act(() => result.current.insert(quotedPath, { paste: true }));
            expect(getBufferState(result).text).toBe(`@${filePath} `);
        });
        it('should not prepend @ to short text that is not a path', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport, escapePastedPaths: true }));
            const shortText = 'ab';
            act(() => result.current.insert(shortText, { paste: true }));
            expect(getBufferState(result).text).toBe(shortText);
        });
        it('should prepend @ to multiple valid file paths on insert', () => {
            const file1 = path.join(tempDir, 'file1.txt');
            const file2 = path.join(tempDir, 'file2.txt');
            fs.writeFileSync(file1, '');
            fs.writeFileSync(file2, '');
            const { result } = renderHook(() => useTextBuffer({ viewport, escapePastedPaths: true }));
            const filePaths = `${file1} ${file2}`;
            act(() => result.current.insert(filePaths, { paste: true }));
            expect(getBufferState(result).text).toBe(`@${file1} @${file2} `);
        });
        it('should handle multiple paths with escaped spaces', () => {
            const file1 = path.join(tempDir, 'my file.txt');
            const file2 = path.join(tempDir, 'other.txt');
            fs.writeFileSync(file1, '');
            fs.writeFileSync(file2, '');
            const { result } = renderHook(() => useTextBuffer({ viewport, escapePastedPaths: true }));
            // Construct escaped path string: "/path/to/my\ file.txt /path/to/other.txt"
            const escapedFile1 = file1.replace(/ /g, '\\ ');
            const filePaths = `${escapedFile1} ${file2}`;
            act(() => result.current.insert(filePaths, { paste: true }));
            expect(getBufferState(result).text).toBe(`@${escapedFile1} @${file2} `);
        });
        it('should only prepend @ to valid paths in multi-path paste', () => {
            const validFile = path.join(tempDir, 'valid.txt');
            const invalidFile = path.join(tempDir, 'invalid.jpg');
            fs.writeFileSync(validFile, '');
            // Do not create invalidFile
            const { result } = renderHook(() => useTextBuffer({
                viewport,
                escapePastedPaths: true,
            }));
            const filePaths = `${validFile} ${invalidFile}`;
            act(() => result.current.insert(filePaths, { paste: true }));
            expect(getBufferState(result).text).toBe(`@${validFile} ${invalidFile} `);
        });
    });
    describe('Shell Mode Behavior', () => {
        it('should not prepend @ to valid file paths when shellModeActive is true', () => {
            const { result } = renderHook(() => useTextBuffer({
                viewport,
                escapePastedPaths: true,
                shellModeActive: true,
            }));
            const filePath = '/path/to/a/valid/file.txt';
            act(() => result.current.insert(filePath, { paste: true }));
            expect(getBufferState(result).text).toBe(filePath); // No @ prefix
        });
        it('should not prepend @ to quoted paths when shellModeActive is true', () => {
            const { result } = renderHook(() => useTextBuffer({
                viewport,
                escapePastedPaths: true,
                shellModeActive: true,
            }));
            const quotedFilePath = "'/path/to/a/valid/file.txt'";
            act(() => result.current.insert(quotedFilePath, { paste: true }));
            expect(getBufferState(result).text).toBe(quotedFilePath); // No @ prefix, keeps quotes
        });
        it('should behave normally with invalid paths when shellModeActive is true', () => {
            const { result } = renderHook(() => useTextBuffer({
                viewport,
                shellModeActive: true,
            }));
            const notAPath = 'this is just some text';
            act(() => result.current.insert(notAPath, { paste: true }));
            expect(getBufferState(result).text).toBe(notAPath);
        });
        it('should behave normally with short text when shellModeActive is true', () => {
            const { result } = renderHook(() => useTextBuffer({
                viewport,
                escapePastedPaths: true,
                shellModeActive: true,
            }));
            const shortText = 'ls';
            act(() => result.current.insert(shortText, { paste: true }));
            expect(getBufferState(result).text).toBe(shortText); // No @ prefix for short text
        });
    });
    describe('Cursor Movement', () => {
        it('move: left/right should work within and across visual lines (due to wrapping)', () => {
            // Text: "long line1next line2" (20 chars)
            // Viewport width 5. Word wrapping should produce:
            // "long " (5)
            // "line1" (5)
            // "next " (5)
            // "line2" (5)
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'long line1next line2', // Corrected: was 'long line1next line2'
                viewport: { width: 5, height: 4 },
            }));
            // Initial cursor [0,0] logical, visual [0,0] ("l" of "long ")
            act(() => result.current.move('right')); // visual [0,1] ("o")
            expect(getBufferState(result).visualCursor).toEqual([0, 1]);
            act(() => result.current.move('right')); // visual [0,2] ("n")
            act(() => result.current.move('right')); // visual [0,3] ("g")
            act(() => result.current.move('right')); // visual [0,4] (" ")
            expect(getBufferState(result).visualCursor).toEqual([0, 4]);
            act(() => result.current.move('right')); // visual [1,0] ("l" of "line1")
            expect(getBufferState(result).visualCursor).toEqual([1, 0]);
            expect(getBufferState(result).cursor).toEqual([0, 5]); // logical cursor
            act(() => result.current.move('left')); // visual [0,4] (" " of "long ")
            expect(getBufferState(result).visualCursor).toEqual([0, 4]);
            expect(getBufferState(result).cursor).toEqual([0, 4]); // logical cursor
        });
        it('move: up/down should preserve preferred visual column', () => {
            const text = 'abcde\nxy\n12345';
            const { result } = renderHook(() => useTextBuffer({
                initialText: text,
                viewport,
            }));
            expect(result.current.allVisualLines).toEqual(['abcde', 'xy', '12345']);
            // Place cursor at the end of "abcde" -> logical [0,5]
            act(() => {
                result.current.move('home'); // to [0,0]
            });
            for (let i = 0; i < 5; i++) {
                act(() => {
                    result.current.move('right'); // to [0,5]
                });
            }
            expect(getBufferState(result).cursor).toEqual([0, 5]);
            expect(getBufferState(result).visualCursor).toEqual([0, 5]);
            // Set preferredCol by moving up then down to the same spot, then test.
            act(() => {
                result.current.move('down'); // to xy, logical [1,2], visual [1,2], preferredCol should be 5
            });
            let state = getBufferState(result);
            expect(state.cursor).toEqual([1, 2]); // Logical cursor at end of 'xy'
            expect(state.visualCursor).toEqual([1, 2]); // Visual cursor at end of 'xy'
            expect(state.preferredCol).toBe(5);
            act(() => result.current.move('down')); // to '12345', preferredCol=5.
            state = getBufferState(result);
            expect(state.cursor).toEqual([2, 5]); // Logical cursor at end of '12345'
            expect(state.visualCursor).toEqual([2, 5]); // Visual cursor at end of '12345'
            expect(state.preferredCol).toBe(5); // Preferred col is maintained
            act(() => result.current.move('left')); // preferredCol should reset
            state = getBufferState(result);
            expect(state.preferredCol).toBe(null);
        });
        it('move: home/end should go to visual line start/end', () => {
            const initialText = 'line one\nsecond line';
            const { result } = renderHook(() => useTextBuffer({
                initialText,
                viewport: { width: 5, height: 5 },
            }));
            expect(result.current.allVisualLines).toEqual([
                'line',
                'one',
                'secon',
                'd',
                'line',
            ]);
            // Initial cursor [0,0] (start of "line")
            act(() => result.current.move('down')); // visual cursor from [0,0] to [1,0] ("o" of "one")
            act(() => result.current.move('right')); // visual cursor to [1,1] ("n" of "one")
            expect(getBufferState(result).visualCursor).toEqual([1, 1]);
            act(() => result.current.move('home')); // visual cursor to [1,0] (start of "one")
            expect(getBufferState(result).visualCursor).toEqual([1, 0]);
            act(() => result.current.move('end')); // visual cursor to [1,3] (end of "one")
            expect(getBufferState(result).visualCursor).toEqual([1, 3]); // "one" is 3 chars
        });
    });
    describe('Visual Layout & Viewport', () => {
        it('should wrap long lines correctly into visualLines', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'This is a very long line of text.', // 33 chars
                viewport: { width: 10, height: 5 },
            }));
            const state = getBufferState(result);
            // Expected visual lines with word wrapping (viewport width 10):
            // "This is a"
            // "very long"
            // "line of"
            // "text."
            expect(state.allVisualLines.length).toBe(4);
            expect(state.allVisualLines[0]).toBe('This is a');
            expect(state.allVisualLines[1]).toBe('very long');
            expect(state.allVisualLines[2]).toBe('line of');
            expect(state.allVisualLines[3]).toBe('text.');
        });
        it('should update visualScrollRow when visualCursor moves out of viewport', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'l1\nl2\nl3\nl4\nl5',
                viewport: { width: 5, height: 3 }, // Can show 3 visual lines
            }));
            // Initial: l1, l2, l3 visible. visualScrollRow = 0. visualCursor = [0,0]
            expect(getBufferState(result).visualScrollRow).toBe(0);
            expect(getBufferState(result).allVisualLines).toEqual([
                'l1',
                'l2',
                'l3',
                'l4',
                'l5',
            ]);
            expect(getBufferState(result).viewportVisualLines).toEqual([
                'l1',
                'l2',
                'l3',
            ]);
            act(() => result.current.move('down')); // vc=[1,0]
            act(() => result.current.move('down')); // vc=[2,0] (l3)
            expect(getBufferState(result).visualScrollRow).toBe(0);
            act(() => result.current.move('down')); // vc=[3,0] (l4) - scroll should happen
            // Now: l2, l3, l4 visible. visualScrollRow = 1.
            let state = getBufferState(result);
            expect(state.visualScrollRow).toBe(1);
            expect(state.allVisualLines).toEqual(['l1', 'l2', 'l3', 'l4', 'l5']);
            expect(state.viewportVisualLines).toEqual(['l2', 'l3', 'l4']);
            expect(state.visualCursor).toEqual([3, 0]);
            act(() => result.current.move('up')); // vc=[2,0] (l3)
            act(() => result.current.move('up')); // vc=[1,0] (l2)
            expect(getBufferState(result).visualScrollRow).toBe(1);
            act(() => result.current.move('up')); // vc=[0,0] (l1) - scroll up
            // Now: l1, l2, l3 visible. visualScrollRow = 0
            state = getBufferState(result); // Assign to the existing `state` variable
            expect(state.visualScrollRow).toBe(0);
            expect(state.allVisualLines).toEqual(['l1', 'l2', 'l3', 'l4', 'l5']);
            expect(state.viewportVisualLines).toEqual(['l1', 'l2', 'l3']);
            expect(state.visualCursor).toEqual([0, 0]);
        });
    });
    describe('Undo/Redo', () => {
        it('should undo and redo an insert operation', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            act(() => result.current.insert('a'));
            expect(getBufferState(result).text).toBe('a');
            act(() => result.current.undo());
            expect(getBufferState(result).text).toBe('');
            expect(getBufferState(result).cursor).toEqual([0, 0]);
            act(() => result.current.redo());
            expect(getBufferState(result).text).toBe('a');
            expect(getBufferState(result).cursor).toEqual([0, 1]);
        });
        it('should undo and redo a newline operation', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'test',
                viewport,
            }));
            act(() => result.current.move('end'));
            act(() => result.current.newline());
            expect(getBufferState(result).text).toBe('test\n');
            act(() => result.current.undo());
            expect(getBufferState(result).text).toBe('test');
            expect(getBufferState(result).cursor).toEqual([0, 4]);
            act(() => result.current.redo());
            expect(getBufferState(result).text).toBe('test\n');
            expect(getBufferState(result).cursor).toEqual([1, 0]);
        });
    });
    describe('Unicode Handling', () => {
        it('insert: should correctly handle multi-byte unicode characters', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            act(() => result.current.insert('你好'));
            const state = getBufferState(result);
            expect(state.text).toBe('你好');
            expect(state.cursor).toEqual([0, 2]); // Cursor is 2 (char count)
            expect(state.visualCursor).toEqual([0, 2]);
        });
        it('backspace: should correctly delete multi-byte unicode characters', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: '你好',
                viewport,
            }));
            act(() => result.current.move('end')); // cursor at [0,2]
            act(() => result.current.backspace()); // delete '好'
            let state = getBufferState(result);
            expect(state.text).toBe('你');
            expect(state.cursor).toEqual([0, 1]);
            act(() => result.current.backspace()); // delete '你'
            state = getBufferState(result);
            expect(state.text).toBe('');
            expect(state.cursor).toEqual([0, 0]);
        });
        it('move: left/right should treat multi-byte chars as single units for visual cursor', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: '🐶🐱',
                viewport: { width: 5, height: 1 },
            }));
            // Initial: visualCursor [0,0]
            act(() => result.current.move('right')); // visualCursor [0,1] (after 🐶)
            let state = getBufferState(result);
            expect(state.cursor).toEqual([0, 1]);
            expect(state.visualCursor).toEqual([0, 1]);
            act(() => result.current.move('right')); // visualCursor [0,2] (after 🐱)
            state = getBufferState(result);
            expect(state.cursor).toEqual([0, 2]);
            expect(state.visualCursor).toEqual([0, 2]);
            act(() => result.current.move('left')); // visualCursor [0,1] (before 🐱 / after 🐶)
            state = getBufferState(result);
            expect(state.cursor).toEqual([0, 1]);
            expect(state.visualCursor).toEqual([0, 1]);
        });
        it('move: up/down should work on wrapped lines (regression test)', () => {
            // Line that wraps into two visual lines
            // Viewport width 10. "0123456789ABCDE" (15 chars)
            // Visual Line 0: "0123456789"
            // Visual Line 1: "ABCDE"
            const { result } = renderHook(() => useTextBuffer({
                viewport: { width: 10, height: 5 },
            }));
            act(() => {
                result.current.setText('0123456789ABCDE');
            });
            // Cursor should be at the end: logical [0, 15], visual [1, 5]
            expect(getBufferState(result).cursor).toEqual([0, 15]);
            expect(getBufferState(result).visualCursor).toEqual([1, 5]);
            // Press Up arrow - should move to first visual line
            // This currently fails because handleInput returns false if cursorRow === 0
            let handledUp = false;
            act(() => {
                handledUp = result.current.handleInput({
                    name: 'up',
                    shift: false,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: false,
                    sequence: '\x1b[A',
                });
            });
            expect(handledUp).toBe(true);
            expect(getBufferState(result).visualCursor[0]).toBe(0);
            // Press Down arrow - should move back to second visual line
            // This would also fail if cursorRow is the last logical row
            let handledDown = false;
            act(() => {
                handledDown = result.current.handleInput({
                    name: 'down',
                    shift: false,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: false,
                    sequence: '\x1b[B',
                });
            });
            expect(handledDown).toBe(true);
            expect(getBufferState(result).visualCursor[0]).toBe(1);
        });
        it('moveToVisualPosition: should correctly handle wide characters (Chinese)', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: '你好', // 2 chars, width 4
                viewport: { width: 10, height: 1 },
            }));
            // '你' (width 2): visual 0-1. '好' (width 2): visual 2-3.
            // Click on '你' (first half, x=0) -> index 0
            act(() => result.current.moveToVisualPosition(0, 0));
            expect(getBufferState(result).cursor).toEqual([0, 0]);
            // Click on '你' (second half, x=1) -> index 1 (after first char)
            act(() => result.current.moveToVisualPosition(0, 1));
            expect(getBufferState(result).cursor).toEqual([0, 1]);
            // Click on '好' (first half, x=2) -> index 1 (before second char)
            act(() => result.current.moveToVisualPosition(0, 2));
            expect(getBufferState(result).cursor).toEqual([0, 1]);
            // Click on '好' (second half, x=3) -> index 2 (after second char)
            act(() => result.current.moveToVisualPosition(0, 3));
            expect(getBufferState(result).cursor).toEqual([0, 2]);
        });
    });
    describe('handleInput', () => {
        it('should insert printable characters', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            act(() => {
                result.current.handleInput({
                    name: 'h',
                    shift: false,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: true,
                    sequence: 'h',
                });
            });
            void act(() => result.current.handleInput({
                name: 'i',
                shift: false,
                alt: false,
                ctrl: false,
                cmd: false,
                insertable: true,
                sequence: 'i',
            }));
            expect(getBufferState(result).text).toBe('hi');
        });
        it('should handle "Enter" key as newline', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            act(() => {
                result.current.handleInput({
                    name: 'return',
                    shift: false,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: true,
                    sequence: '\r',
                });
            });
            expect(getBufferState(result).lines).toEqual(['', '']);
        });
        it('should handle Ctrl+J as newline', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            act(() => {
                result.current.handleInput({
                    name: 'j',
                    shift: false,
                    alt: false,
                    ctrl: true,
                    cmd: false,
                    insertable: false,
                    sequence: '\n',
                });
            });
            expect(getBufferState(result).lines).toEqual(['', '']);
        });
        it('should do nothing for a tab key press', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            act(() => {
                result.current.handleInput({
                    name: 'tab',
                    shift: false,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: false,
                    sequence: '\t',
                });
            });
            expect(getBufferState(result).text).toBe('');
        });
        it('should do nothing for a shift tab key press', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            act(() => {
                result.current.handleInput({
                    name: 'tab',
                    shift: true,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: false,
                    sequence: '\u001b[9;2u',
                });
            });
            expect(getBufferState(result).text).toBe('');
        });
        it('should handle CLEAR_INPUT (Ctrl+C)', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'hello',
                viewport,
            }));
            expect(getBufferState(result).text).toBe('hello');
            let handled = false;
            act(() => {
                handled = result.current.handleInput({
                    name: 'c',
                    shift: false,
                    alt: false,
                    ctrl: true,
                    cmd: false,
                    insertable: false,
                    sequence: '\u0003',
                });
            });
            expect(handled).toBe(true);
            expect(getBufferState(result).text).toBe('');
        });
        it('should NOT handle CLEAR_INPUT if buffer is empty', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            let handled = true;
            act(() => {
                handled = result.current.handleInput({
                    name: 'c',
                    shift: false,
                    alt: false,
                    ctrl: true,
                    cmd: false,
                    insertable: false,
                    sequence: '\u0003',
                });
            });
            expect(handled).toBe(false);
        });
        it('should handle "Backspace" key', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'a',
                viewport,
            }));
            act(() => result.current.move('end'));
            act(() => {
                result.current.handleInput({
                    name: 'backspace',
                    shift: false,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: false,
                    sequence: '\x7f',
                });
            });
            expect(getBufferState(result).text).toBe('');
        });
        it('should handle multiple delete characters in one input', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'abcde',
                viewport,
            }));
            act(() => result.current.move('end')); // cursor at the end
            expect(getBufferState(result).cursor).toEqual([0, 5]);
            act(() => {
                result.current.handleInput({
                    name: 'backspace',
                    shift: false,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: false,
                    sequence: '\x7f',
                });
                result.current.handleInput({
                    name: 'backspace',
                    shift: false,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: false,
                    sequence: '\x7f',
                });
                result.current.handleInput({
                    name: 'backspace',
                    shift: false,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: false,
                    sequence: '\x7f',
                });
            });
            expect(getBufferState(result).text).toBe('ab');
            expect(getBufferState(result).cursor).toEqual([0, 2]);
        });
        it('should handle inserts that contain delete characters', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'abcde',
                viewport,
            }));
            act(() => result.current.move('end')); // cursor at the end
            expect(getBufferState(result).cursor).toEqual([0, 5]);
            act(() => {
                result.current.insert('\x7f\x7f\x7f');
            });
            expect(getBufferState(result).text).toBe('ab');
            expect(getBufferState(result).cursor).toEqual([0, 2]);
        });
        it('should handle inserts with a mix of regular and delete characters', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'abcde',
                viewport,
            }));
            act(() => result.current.move('end')); // cursor at the end
            expect(getBufferState(result).cursor).toEqual([0, 5]);
            act(() => {
                result.current.insert('\x7fI\x7f\x7fNEW');
            });
            expect(getBufferState(result).text).toBe('abcNEW');
            expect(getBufferState(result).cursor).toEqual([0, 6]);
        });
        it('should handle arrow keys for movement', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'ab',
                viewport,
            }));
            act(() => result.current.move('end')); // cursor [0,2]
            act(() => {
                result.current.handleInput({
                    name: 'left',
                    shift: false,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: false,
                    sequence: '\x1b[D',
                });
            });
            expect(getBufferState(result).cursor).toEqual([0, 1]);
            act(() => {
                result.current.handleInput({
                    name: 'right',
                    shift: false,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: false,
                    sequence: '\x1b[C',
                });
            });
            expect(getBufferState(result).cursor).toEqual([0, 2]);
        });
        it('should strip ANSI escape codes when pasting text', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            const textWithAnsi = '\x1B[31mHello\x1B[0m \x1B[32mWorld\x1B[0m';
            // Simulate pasting by calling handleInput with a string longer than 1 char
            act(() => {
                result.current.handleInput({
                    name: '',
                    shift: false,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: true,
                    sequence: textWithAnsi,
                });
            });
            expect(getBufferState(result).text).toBe('Hello World');
        });
        it('should handle VSCode terminal Shift+Enter as newline', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            act(() => {
                result.current.handleInput({
                    name: 'return',
                    shift: true,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: true,
                    sequence: '\r',
                });
            }); // Simulates Shift+Enter in VSCode terminal
            expect(getBufferState(result).lines).toEqual(['', '']);
        });
        it('should correctly handle repeated pasting of long text', () => {
            const longText = `not only five centuries, but also the leap into electronic typesetting, remaining essentially unchanged. It was popularised in the 1960s with the release of Letraset sheets containing Lorem Ipsum passages, and more recently with desktop publishing software like Aldus PageMaker including versions of Lorem Ipsum.

Why do we use it?
It is a long established fact that a reader will be distracted by the readable content of a page when looking at its layout. The point of using Lorem Ipsum is that it has a more-or-less normal distribution of letters, as opposed to using 'Content here, content here', making it look like readable English. Many desktop publishing packages and web page editors now use Lorem Ipsum as their default model text, and a search for 'lorem ipsum' will uncover many web sites still in their infancy. Various versions have evolved over the years, sometimes by accident, sometimes on purpose (injected humour and the like).

Where does it come from?
Contrary to popular belief, Lorem Ipsum is not simply random text. It has roots in a piece of classical Latin literature from 45 BC, making it over 2000 years old. Richard McClintock, a Latin professor at Hampden-Sydney College in Virginia, looked up one of the more obscure Latin words, consectetur, from a Lore
`;
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            // Simulate pasting the long text multiple times
            act(() => {
                result.current.insert(longText, { paste: true });
                result.current.insert(longText, { paste: true });
                result.current.insert(longText, { paste: true });
            });
            const state = getBufferState(result);
            // Check that the text is the result of three concatenations of unique placeholders.
            // Now that ID generation is in the reducer, they are correctly unique even when batched.
            expect(state.lines).toStrictEqual([
                '[Pasted Text: 8 lines][Pasted Text: 8 lines #2][Pasted Text: 8 lines #3]',
            ]);
            expect(result.current.pastedContent['[Pasted Text: 8 lines]']).toBe(longText);
            expect(result.current.pastedContent['[Pasted Text: 8 lines #2]']).toBe(longText);
            expect(result.current.pastedContent['[Pasted Text: 8 lines #3]']).toBe(longText);
            const expectedCursorPos = offsetToLogicalPos(state.text, state.text.length);
            expect(state.cursor).toEqual(expectedCursorPos);
        });
    });
    // More tests would be needed for:
    // - setText, replaceRange
    // - deleteWordLeft, deleteWordRight
    // - More complex undo/redo scenarios
    // - Selection and clipboard (copy/paste) - might need clipboard API mocks or internal state check
    // - openInExternalEditor (heavy mocking of fs, child_process, os)
    // - All edge cases for visual scrolling and wrapping with different viewport sizes and text content.
    describe('replaceRange', () => {
        it('should replace a single-line range with single-line text', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: '@pac',
                viewport,
            }));
            act(() => result.current.replaceRange(0, 1, 0, 4, 'packages'));
            const state = getBufferState(result);
            expect(state.text).toBe('@packages');
            expect(state.cursor).toEqual([0, 9]); // cursor after 'typescript'
        });
        it('should replace a multi-line range with single-line text', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'hello\nworld\nagain',
                viewport,
            }));
            act(() => result.current.replaceRange(0, 2, 1, 3, ' new ')); // replace 'llo\nwor' with ' new '
            const state = getBufferState(result);
            expect(state.text).toBe('he new ld\nagain');
            expect(state.cursor).toEqual([0, 7]); // cursor after ' new '
        });
        it('should delete a range when replacing with an empty string', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'hello world',
                viewport,
            }));
            act(() => result.current.replaceRange(0, 5, 0, 11, '')); // delete ' world'
            const state = getBufferState(result);
            expect(state.text).toBe('hello');
            expect(state.cursor).toEqual([0, 5]);
        });
        it('should handle replacing at the beginning of the text', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'world',
                viewport,
            }));
            act(() => result.current.replaceRange(0, 0, 0, 0, 'hello '));
            const state = getBufferState(result);
            expect(state.text).toBe('hello world');
            expect(state.cursor).toEqual([0, 6]);
        });
        it('should handle replacing at the end of the text', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'hello',
                viewport,
            }));
            act(() => result.current.replaceRange(0, 5, 0, 5, ' world'));
            const state = getBufferState(result);
            expect(state.text).toBe('hello world');
            expect(state.cursor).toEqual([0, 11]);
        });
        it('should handle replacing the entire buffer content', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'old text',
                viewport,
            }));
            act(() => result.current.replaceRange(0, 0, 0, 8, 'new text'));
            const state = getBufferState(result);
            expect(state.text).toBe('new text');
            expect(state.cursor).toEqual([0, 8]);
        });
        it('should correctly replace with unicode characters', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'hello *** world',
                viewport,
            }));
            act(() => result.current.replaceRange(0, 6, 0, 9, '你好'));
            const state = getBufferState(result);
            expect(state.text).toBe('hello 你好 world');
            expect(state.cursor).toEqual([0, 8]); // after '你好'
        });
        it('should handle invalid range by returning false and not changing text', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'test',
                viewport,
            }));
            act(() => {
                result.current.replaceRange(0, 5, 0, 3, 'fail'); // startCol > endCol in same line
            });
            expect(getBufferState(result).text).toBe('test');
            act(() => {
                result.current.replaceRange(1, 0, 0, 0, 'fail'); // startRow > endRow
            });
            expect(getBufferState(result).text).toBe('test');
        });
        it('replaceRange: multiple lines with a single character', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'first\nsecond\nthird',
                viewport,
            }));
            act(() => result.current.replaceRange(0, 2, 2, 3, 'X')); // Replace 'rst\nsecond\nthi'
            const state = getBufferState(result);
            expect(state.text).toBe('fiXrd');
            expect(state.cursor).toEqual([0, 3]); // After 'X'
        });
        it('should replace a single-line range with multi-line text', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'one two three',
                viewport,
            }));
            // Replace "two" with "new\nline"
            act(() => result.current.replaceRange(0, 4, 0, 7, 'new\nline'));
            const state = getBufferState(result);
            expect(state.lines).toEqual(['one new', 'line three']);
            expect(state.text).toBe('one new\nline three');
            expect(state.cursor).toEqual([1, 4]); // cursor after 'line'
        });
    });
    describe('Input Sanitization', () => {
        const createInput = (sequence) => ({
            name: '',
            shift: false,
            alt: false,
            ctrl: false,
            cmd: false,
            insertable: true,
            sequence,
        });
        it.each([
            {
                input: '\x1B[31mHello\x1B[0m \x1B[32mWorld\x1B[0m',
                expected: 'Hello World',
                desc: 'ANSI escape codes',
            },
            {
                input: 'H\x07e\x08l\x0Bl\x0Co',
                expected: 'Hello',
                desc: 'control characters',
            },
            {
                input: '\u001B[4mH\u001B[0mello',
                expected: 'Hello',
                desc: 'mixed ANSI and control characters',
            },
            {
                input: '\u001B[4mPasted\u001B[4m Text',
                expected: 'Pasted Text',
                desc: 'pasted text with ANSI',
            },
        ])('should strip $desc from input', ({ input, expected }) => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            act(() => {
                result.current.handleInput(createInput(input));
            });
            expect(getBufferState(result).text).toBe(expected);
        });
        it('should not strip standard characters or newlines', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            const validText = 'Hello World\nThis is a test.';
            act(() => {
                result.current.handleInput(createInput(validText));
            });
            expect(getBufferState(result).text).toBe(validText);
        });
        it('should sanitize large text (>5000 chars) and strip unsafe characters', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            const unsafeChars = '\x07\x08\x0B\x0C';
            const largeTextWithUnsafe = 'safe text'.repeat(600) + unsafeChars + 'more safe text';
            expect(largeTextWithUnsafe.length).toBeGreaterThan(5000);
            act(() => {
                result.current.handleInput({
                    name: '',
                    shift: false,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: true,
                    sequence: largeTextWithUnsafe,
                });
            });
            const resultText = getBufferState(result).text;
            expect(resultText).not.toContain('\x07');
            expect(resultText).not.toContain('\x08');
            expect(resultText).not.toContain('\x0B');
            expect(resultText).not.toContain('\x0C');
            expect(resultText).toContain('safe text');
            expect(resultText).toContain('more safe text');
        });
        it('should sanitize large ANSI text (>5000 chars) and strip escape codes', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            const largeTextWithAnsi = '\x1B[31m' +
                'red text'.repeat(800) +
                '\x1B[0m' +
                '\x1B[32m' +
                'green text'.repeat(200) +
                '\x1B[0m';
            expect(largeTextWithAnsi.length).toBeGreaterThan(5000);
            act(() => {
                result.current.handleInput({
                    name: '',
                    shift: false,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: true,
                    sequence: largeTextWithAnsi,
                });
            });
            const resultText = getBufferState(result).text;
            expect(resultText).not.toContain('\x1B[31m');
            expect(resultText).not.toContain('\x1B[32m');
            expect(resultText).not.toContain('\x1B[0m');
            expect(resultText).toContain('red text');
            expect(resultText).toContain('green text');
        });
        it('should not strip popular emojis', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            const emojis = '🐍🐳🦀🦄';
            act(() => {
                result.current.handleInput({
                    name: '',
                    shift: false,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: true,
                    sequence: emojis,
                });
            });
            expect(getBufferState(result).text).toBe(emojis);
        });
    });
    describe('inputFilter', () => {
        it('should filter input based on the provided filter function', () => {
            const { result } = renderHook(() => useTextBuffer({
                viewport,
                inputFilter: (text) => text.replace(/[^0-9]/g, ''),
            }));
            act(() => result.current.insert('a1b2c3'));
            expect(getBufferState(result).text).toBe('123');
        });
        it('should handle empty result from filter', () => {
            const { result } = renderHook(() => useTextBuffer({
                viewport,
                inputFilter: (text) => text.replace(/[^0-9]/g, ''),
            }));
            act(() => result.current.insert('abc'));
            expect(getBufferState(result).text).toBe('');
        });
        it('should filter pasted text', () => {
            const { result } = renderHook(() => useTextBuffer({
                viewport,
                inputFilter: (text) => text.toUpperCase(),
            }));
            act(() => result.current.insert('hello', { paste: true }));
            expect(getBufferState(result).text).toBe('HELLO');
        });
        it('should not filter newlines if they are allowed by the filter', () => {
            const { result } = renderHook(() => useTextBuffer({
                viewport,
                inputFilter: (text) => text, // Allow everything including newlines
            }));
            act(() => result.current.insert('a\nb'));
            // The insert function splits by newline and inserts separately if it detects them.
            // If the filter allows them, they should be handled correctly by the subsequent logic in insert.
            expect(getBufferState(result).text).toBe('a\nb');
        });
        it('should filter before newline check in insert', () => {
            const { result } = renderHook(() => useTextBuffer({
                viewport,
                inputFilter: (text) => text.replace(/\n/g, ''), // Filter out newlines
            }));
            act(() => result.current.insert('a\nb'));
            expect(getBufferState(result).text).toBe('ab');
        });
    });
    describe('stripAnsi', () => {
        it('should correctly strip ANSI escape codes', () => {
            const textWithAnsi = '\x1B[31mHello\x1B[0m World';
            expect(stripAnsi(textWithAnsi)).toBe('Hello World');
        });
        it('should handle multiple ANSI codes', () => {
            const textWithMultipleAnsi = '\x1B[1m\x1B[34mBold Blue\x1B[0m Text';
            expect(stripAnsi(textWithMultipleAnsi)).toBe('Bold Blue Text');
        });
        it('should not modify text without ANSI codes', () => {
            const plainText = 'Plain text';
            expect(stripAnsi(plainText)).toBe('Plain text');
        });
        it('should handle empty string', () => {
            expect(stripAnsi('')).toBe('');
        });
    });
    describe('Memoization', () => {
        it('should keep action references stable across re-renders', () => {
            const { result, rerender } = renderHook(() => useTextBuffer({ viewport }));
            const initialInsert = result.current.insert;
            const initialBackspace = result.current.backspace;
            const initialMove = result.current.move;
            const initialHandleInput = result.current.handleInput;
            rerender();
            expect(result.current.insert).toBe(initialInsert);
            expect(result.current.backspace).toBe(initialBackspace);
            expect(result.current.move).toBe(initialMove);
            expect(result.current.handleInput).toBe(initialHandleInput);
        });
        it('should have memoized actions that operate on the latest state', () => {
            const { result } = renderHook(() => useTextBuffer({ viewport }));
            // Store a reference to the memoized insert function.
            const memoizedInsert = result.current.insert;
            // Update the buffer state.
            act(() => {
                result.current.insert('hello');
            });
            expect(getBufferState(result).text).toBe('hello');
            // Now, call the original memoized function reference.
            act(() => {
                memoizedInsert(' world');
            });
            // It should have operated on the updated state.
            expect(getBufferState(result).text).toBe('hello world');
        });
    });
    describe('singleLine mode', () => {
        it('should not insert a newline character when singleLine is true', () => {
            const { result } = renderHook(() => useTextBuffer({
                viewport,
                singleLine: true,
            }));
            act(() => result.current.insert('\n'));
            const state = getBufferState(result);
            expect(state.text).toBe('');
            expect(state.lines).toEqual(['']);
        });
        it('should not create a new line when newline() is called and singleLine is true', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'ab',
                viewport,
                singleLine: true,
            }));
            act(() => result.current.move('end')); // cursor at [0,2]
            act(() => result.current.newline());
            const state = getBufferState(result);
            expect(state.text).toBe('ab');
            expect(state.lines).toEqual(['ab']);
            expect(state.cursor).toEqual([0, 2]);
        });
        it('should not handle "Enter" key as newline when singleLine is true', () => {
            const { result } = renderHook(() => useTextBuffer({
                viewport,
                singleLine: true,
            }));
            act(() => {
                result.current.handleInput({
                    name: 'return',
                    shift: false,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: true,
                    sequence: '\r',
                });
            });
            expect(getBufferState(result).lines).toEqual(['']);
        });
        it('should not print anything for function keys when singleLine is true', () => {
            const { result } = renderHook(() => useTextBuffer({
                viewport,
                singleLine: true,
            }));
            act(() => {
                result.current.handleInput({
                    name: 'f1',
                    shift: false,
                    alt: false,
                    ctrl: false,
                    cmd: false,
                    insertable: false,
                    sequence: '\u001bOP',
                });
            });
            expect(getBufferState(result).lines).toEqual(['']);
        });
        it('should strip newlines from pasted text when singleLine is true', () => {
            const { result } = renderHook(() => useTextBuffer({
                viewport,
                singleLine: true,
            }));
            act(() => result.current.insert('hello\nworld', { paste: true }));
            const state = getBufferState(result);
            expect(state.text).toBe('helloworld');
            expect(state.lines).toEqual(['helloworld']);
        });
    });
});
describe('offsetToLogicalPos', () => {
    it.each([
        { text: 'any text', offset: 0, expected: [0, 0], desc: 'offset 0' },
        { text: 'hello', offset: 0, expected: [0, 0], desc: 'single line start' },
        { text: 'hello', offset: 2, expected: [0, 2], desc: 'single line middle' },
        { text: 'hello', offset: 5, expected: [0, 5], desc: 'single line end' },
        { text: 'hello', offset: 10, expected: [0, 5], desc: 'beyond end clamps' },
        {
            text: 'a\n\nc',
            offset: 0,
            expected: [0, 0],
            desc: 'empty lines - first char',
        },
        {
            text: 'a\n\nc',
            offset: 1,
            expected: [0, 1],
            desc: 'empty lines - end of first',
        },
        {
            text: 'a\n\nc',
            offset: 2,
            expected: [1, 0],
            desc: 'empty lines - empty line',
        },
        {
            text: 'a\n\nc',
            offset: 3,
            expected: [2, 0],
            desc: 'empty lines - last line start',
        },
        {
            text: 'a\n\nc',
            offset: 4,
            expected: [2, 1],
            desc: 'empty lines - last line end',
        },
        {
            text: 'hello\n',
            offset: 5,
            expected: [0, 5],
            desc: 'newline end - before newline',
        },
        {
            text: 'hello\n',
            offset: 6,
            expected: [1, 0],
            desc: 'newline end - after newline',
        },
        {
            text: 'hello\n',
            offset: 7,
            expected: [1, 0],
            desc: 'newline end - beyond',
        },
        {
            text: '\nhello',
            offset: 0,
            expected: [0, 0],
            desc: 'newline start - first line',
        },
        {
            text: '\nhello',
            offset: 1,
            expected: [1, 0],
            desc: 'newline start - second line',
        },
        {
            text: '\nhello',
            offset: 3,
            expected: [1, 2],
            desc: 'newline start - middle of second',
        },
        { text: '', offset: 0, expected: [0, 0], desc: 'empty string at 0' },
        { text: '', offset: 5, expected: [0, 0], desc: 'empty string beyond' },
        {
            text: '你好\n世界',
            offset: 0,
            expected: [0, 0],
            desc: 'unicode - start',
        },
        {
            text: '你好\n世界',
            offset: 1,
            expected: [0, 1],
            desc: 'unicode - after first char',
        },
        {
            text: '你好\n世界',
            offset: 2,
            expected: [0, 2],
            desc: 'unicode - end first line',
        },
        {
            text: '你好\n世界',
            offset: 3,
            expected: [1, 0],
            desc: 'unicode - second line start',
        },
        {
            text: '你好\n世界',
            offset: 4,
            expected: [1, 1],
            desc: 'unicode - second line middle',
        },
        {
            text: '你好\n世界',
            offset: 5,
            expected: [1, 2],
            desc: 'unicode - second line end',
        },
        {
            text: '你好\n世界',
            offset: 6,
            expected: [1, 2],
            desc: 'unicode - beyond',
        },
        {
            text: 'abc\ndef',
            offset: 3,
            expected: [0, 3],
            desc: 'at newline - end of line',
        },
        {
            text: 'abc\ndef',
            offset: 4,
            expected: [1, 0],
            desc: 'at newline - after newline',
        },
        { text: '🐶🐱', offset: 0, expected: [0, 0], desc: 'emoji - start' },
        { text: '🐶🐱', offset: 1, expected: [0, 1], desc: 'emoji - middle' },
        { text: '🐶🐱', offset: 2, expected: [0, 2], desc: 'emoji - end' },
    ])('should handle $desc', ({ text, offset, expected }) => {
        expect(offsetToLogicalPos(text, offset)).toEqual(expected);
    });
    describe('multi-line text', () => {
        const text = 'hello\nworld\n123';
        it.each([
            { offset: 0, expected: [0, 0], desc: 'start of first line' },
            { offset: 3, expected: [0, 3], desc: 'middle of first line' },
            { offset: 5, expected: [0, 5], desc: 'end of first line' },
            { offset: 6, expected: [1, 0], desc: 'start of second line' },
            { offset: 8, expected: [1, 2], desc: 'middle of second line' },
            { offset: 11, expected: [1, 5], desc: 'end of second line' },
            { offset: 12, expected: [2, 0], desc: 'start of third line' },
            { offset: 13, expected: [2, 1], desc: 'middle of third line' },
            { offset: 15, expected: [2, 3], desc: 'end of third line' },
            { offset: 20, expected: [2, 3], desc: 'beyond end' },
        ])('should return $expected for $desc (offset $offset)', ({ offset, expected }) => {
            expect(offsetToLogicalPos(text, offset)).toEqual(expected);
        });
    });
});
describe('logicalPosToOffset', () => {
    it('should convert row/col position to offset correctly', () => {
        const lines = ['hello', 'world', '123'];
        // Line 0: "hello" (5 chars)
        expect(logicalPosToOffset(lines, 0, 0)).toBe(0); // Start of 'hello'
        expect(logicalPosToOffset(lines, 0, 3)).toBe(3); // 'l' in 'hello'
        expect(logicalPosToOffset(lines, 0, 5)).toBe(5); // End of 'hello'
        // Line 1: "world" (5 chars), offset starts at 6 (5 + 1 for newline)
        expect(logicalPosToOffset(lines, 1, 0)).toBe(6); // Start of 'world'
        expect(logicalPosToOffset(lines, 1, 2)).toBe(8); // 'r' in 'world'
        expect(logicalPosToOffset(lines, 1, 5)).toBe(11); // End of 'world'
        // Line 2: "123" (3 chars), offset starts at 12 (5 + 1 + 5 + 1)
        expect(logicalPosToOffset(lines, 2, 0)).toBe(12); // Start of '123'
        expect(logicalPosToOffset(lines, 2, 1)).toBe(13); // '2' in '123'
        expect(logicalPosToOffset(lines, 2, 3)).toBe(15); // End of '123'
    });
    it('should handle empty lines', () => {
        const lines = ['a', '', 'c'];
        expect(logicalPosToOffset(lines, 0, 0)).toBe(0); // 'a'
        expect(logicalPosToOffset(lines, 0, 1)).toBe(1); // End of 'a'
        expect(logicalPosToOffset(lines, 1, 0)).toBe(2); // Empty line
        expect(logicalPosToOffset(lines, 2, 0)).toBe(3); // 'c'
        expect(logicalPosToOffset(lines, 2, 1)).toBe(4); // End of 'c'
    });
    it('should handle single empty line', () => {
        const lines = [''];
        expect(logicalPosToOffset(lines, 0, 0)).toBe(0);
    });
    it('should be inverse of offsetToLogicalPos', () => {
        const lines = ['hello', 'world', '123'];
        const text = lines.join('\n');
        // Test round-trip conversion
        for (let offset = 0; offset <= text.length; offset++) {
            const [row, col] = offsetToLogicalPos(text, offset);
            const convertedOffset = logicalPosToOffset(lines, row, col);
            expect(convertedOffset).toBe(offset);
        }
    });
    it('should handle out-of-bounds positions', () => {
        const lines = ['hello'];
        // Beyond end of line
        expect(logicalPosToOffset(lines, 0, 10)).toBe(5); // Clamps to end of line
        // Beyond array bounds - should clamp to the last line
        expect(logicalPosToOffset(lines, 5, 0)).toBe(0); // Clamps to start of last line (row 0)
        expect(logicalPosToOffset(lines, 5, 10)).toBe(5); // Clamps to end of last line
    });
});
const createTestState = (lines, cursorRow, cursorCol, viewportWidth = 80) => {
    const text = lines.join('\n');
    let state = textBufferReducer(initialState, {
        type: 'set_text',
        payload: text,
    });
    state = textBufferReducer(state, {
        type: 'set_cursor',
        payload: { cursorRow, cursorCol, preferredCol: null },
    });
    state = textBufferReducer(state, {
        type: 'set_viewport',
        payload: { width: viewportWidth, height: 24 },
    });
    return state;
};
describe('textBufferReducer vim operations', () => {
    describe('vim_delete_line', () => {
        it('should delete a single line including newline in multi-line text', () => {
            const state = createTestState(['line1', 'line2', 'line3'], 1, 2);
            const action = {
                type: 'vim_delete_line',
                payload: { count: 1 },
            };
            const result = textBufferReducer(state, action);
            expect(result).toHaveOnlyValidCharacters();
            // After deleting line2, we should have line1 and line3, with cursor on line3 (now at index 1)
            expect(result.lines).toEqual(['line1', 'line3']);
            expect(result.cursorRow).toBe(1);
            expect(result.cursorCol).toBe(0);
        });
        it('should delete multiple lines when count > 1', () => {
            const state = createTestState(['line1', 'line2', 'line3', 'line4'], 1, 0);
            const action = {
                type: 'vim_delete_line',
                payload: { count: 2 },
            };
            const result = textBufferReducer(state, action);
            expect(result).toHaveOnlyValidCharacters();
            // Should delete line2 and line3, leaving line1 and line4
            expect(result.lines).toEqual(['line1', 'line4']);
            expect(result.cursorRow).toBe(1);
            expect(result.cursorCol).toBe(0);
        });
        it('should clear single line content when only one line exists', () => {
            const state = createTestState(['only line'], 0, 5);
            const action = {
                type: 'vim_delete_line',
                payload: { count: 1 },
            };
            const result = textBufferReducer(state, action);
            expect(result).toHaveOnlyValidCharacters();
            // Should clear the line content but keep the line
            expect(result.lines).toEqual(['']);
            expect(result.cursorRow).toBe(0);
            expect(result.cursorCol).toBe(0);
        });
        it('should handle deleting the last line properly', () => {
            const state = createTestState(['line1', 'line2'], 1, 0);
            const action = {
                type: 'vim_delete_line',
                payload: { count: 1 },
            };
            const result = textBufferReducer(state, action);
            expect(result).toHaveOnlyValidCharacters();
            // Should delete the last line completely, not leave empty line
            expect(result.lines).toEqual(['line1']);
            expect(result.cursorRow).toBe(0);
            expect(result.cursorCol).toBe(0);
        });
        it('should handle deleting all lines and maintain valid state for subsequent paste', () => {
            const state = createTestState(['line1', 'line2', 'line3', 'line4'], 0, 0);
            // Delete all 4 lines with 4dd
            const deleteAction = {
                type: 'vim_delete_line',
                payload: { count: 4 },
            };
            const afterDelete = textBufferReducer(state, deleteAction);
            expect(afterDelete).toHaveOnlyValidCharacters();
            // After deleting all lines, should have one empty line
            expect(afterDelete.lines).toEqual(['']);
            expect(afterDelete.cursorRow).toBe(0);
            expect(afterDelete.cursorCol).toBe(0);
            // Now paste multiline content - this should work correctly
            const pasteAction = {
                type: 'insert',
                payload: 'new1\nnew2\nnew3\nnew4',
            };
            const afterPaste = textBufferReducer(afterDelete, pasteAction);
            expect(afterPaste).toHaveOnlyValidCharacters();
            // All lines including the first one should be present
            expect(afterPaste.lines).toEqual(['new1', 'new2', 'new3', 'new4']);
            expect(afterPaste.cursorRow).toBe(3);
            expect(afterPaste.cursorCol).toBe(4);
        });
    });
});
describe('Unicode helper functions', () => {
    describe('findWordEndInLine with Unicode', () => {
        it('should handle combining characters', () => {
            // café with combining accent
            const cafeWithCombining = 'cafe\u0301';
            const result = findWordEndInLine(cafeWithCombining + ' test', 0);
            expect(result).toBe(3); // End of 'café' at base character 'e', not combining accent
        });
        it('should handle precomposed characters with diacritics', () => {
            // café with precomposed é (U+00E9)
            const cafePrecomposed = 'café';
            const result = findWordEndInLine(cafePrecomposed + ' test', 0);
            expect(result).toBe(3); // End of 'café' at precomposed character 'é'
        });
        it('should return null when no word end found', () => {
            const result = findWordEndInLine('   ', 0);
            expect(result).toBeNull(); // No word end found in whitespace-only string string
        });
    });
    describe('findNextWordStartInLine with Unicode', () => {
        it('should handle right-to-left text', () => {
            const result = findNextWordStartInLine('hello مرحبا world', 0);
            expect(result).toBe(6); // Start of Arabic word
        });
        it('should handle Chinese characters', () => {
            const result = findNextWordStartInLine('hello 你好 world', 0);
            expect(result).toBe(6); // Start of Chinese word
        });
        it('should return null at end of line', () => {
            const result = findNextWordStartInLine('hello', 10);
            expect(result).toBeNull();
        });
        it('should handle combining characters', () => {
            // café with combining accent + next word
            const textWithCombining = 'cafe\u0301 test';
            const result = findNextWordStartInLine(textWithCombining, 0);
            expect(result).toBe(6); // Start of 'test' after 'café ' (combining char makes string longer)
        });
        it('should handle precomposed characters with diacritics', () => {
            // café with precomposed é + next word
            const textPrecomposed = 'café test';
            const result = findNextWordStartInLine(textPrecomposed, 0);
            expect(result).toBe(5); // Start of 'test' after 'café '
        });
    });
    describe('isWordCharStrict with Unicode', () => {
        it('should return true for ASCII word characters', () => {
            expect(isWordCharStrict('a')).toBe(true);
            expect(isWordCharStrict('Z')).toBe(true);
            expect(isWordCharStrict('0')).toBe(true);
            expect(isWordCharStrict('_')).toBe(true);
        });
        it('should return false for punctuation', () => {
            expect(isWordCharStrict('.')).toBe(false);
            expect(isWordCharStrict(',')).toBe(false);
            expect(isWordCharStrict('!')).toBe(false);
        });
        it('should return true for non-Latin scripts', () => {
            expect(isWordCharStrict('你')).toBe(true); // Chinese character
            expect(isWordCharStrict('م')).toBe(true); // Arabic character
        });
        it('should return false for whitespace', () => {
            expect(isWordCharStrict(' ')).toBe(false);
            expect(isWordCharStrict('\t')).toBe(false);
        });
    });
    describe('cpLen with Unicode', () => {
        it('should handle combining characters', () => {
            expect(cpLen('é')).toBe(1); // Precomposed
            expect(cpLen('e\u0301')).toBe(2); // e + combining acute
        });
        it('should handle Chinese and Arabic text', () => {
            expect(cpLen('hello 你好 world')).toBe(14); // 5 + 1 + 2 + 1 + 5 = 14
            expect(cpLen('hello مرحبا world')).toBe(17);
        });
    });
    describe('useTextBuffer CJK Navigation', () => {
        const viewport = { width: 80, height: 24 };
        it('should navigate by word in Chinese', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: '你好世界',
                initialCursorOffset: 4, // End of string
                viewport,
            }));
            // Initial state: cursor at end (index 2 in code points if 4 is length? wait. length is 2 code points? No. '你好世界' length is 4.)
            // '你好世界' length is 4. Code points length is 4.
            // Move word left
            act(() => {
                result.current.move('wordLeft');
            });
            // Should be at start of "世界" (index 2)
            // "你好世界" -> "你好" | "世界"
            expect(result.current.cursor[1]).toBe(2);
            // Move word left again
            act(() => {
                result.current.move('wordLeft');
            });
            // Should be at start of "你好" (index 0)
            expect(result.current.cursor[1]).toBe(0);
            // Move word left again (should stay at 0)
            act(() => {
                result.current.move('wordLeft');
            });
            expect(result.current.cursor[1]).toBe(0);
            // Move word right
            act(() => {
                result.current.move('wordRight');
            });
            // Should be at end of "你好" (index 2)
            expect(result.current.cursor[1]).toBe(2);
            // Move word right again
            act(() => {
                result.current.move('wordRight');
            });
            // Should be at end of "世界" (index 4)
            expect(result.current.cursor[1]).toBe(4);
            // Move word right again (should stay at end)
            act(() => {
                result.current.move('wordRight');
            });
            expect(result.current.cursor[1]).toBe(4);
        });
        it('should navigate mixed English and Chinese', () => {
            const { result } = renderHook(() => useTextBuffer({
                initialText: 'Hello你好World',
                initialCursorOffset: 10, // End
                viewport,
            }));
            // Hello (5) + 你好 (2) + World (5) = 12 chars.
            // initialCursorOffset 10? 'Hello你好World'.length is 12.
            // Let's set it to end.
            act(() => {
                result.current.move('end');
            });
            expect(result.current.cursor[1]).toBe(12);
            // wordLeft -> start of "World" (index 7)
            act(() => result.current.move('wordLeft'));
            expect(result.current.cursor[1]).toBe(7);
            // wordLeft -> start of "你好" (index 5)
            act(() => result.current.move('wordLeft'));
            expect(result.current.cursor[1]).toBe(5);
            // wordLeft -> start of "Hello" (index 0)
            act(() => result.current.move('wordLeft'));
            expect(result.current.cursor[1]).toBe(0);
            // wordLeft -> start of line (should stay at 0)
            act(() => result.current.move('wordLeft'));
            expect(result.current.cursor[1]).toBe(0);
        });
    });
});
describe('Transformation Utilities', () => {
    afterEach(() => {
        vi.restoreAllMocks();
    });
    describe('getTransformedImagePath', () => {
        it('should transform a simple image path', () => {
            expect(getTransformedImagePath('@test.png')).toBe('[Image test.png]');
        });
        it('should handle paths with directories', () => {
            expect(getTransformedImagePath('@path/to/image.jpg')).toBe('[Image image.jpg]');
        });
        it('should truncate long filenames', () => {
            expect(getTransformedImagePath('@verylongfilename1234567890.png')).toBe('[Image ...1234567890.png]');
        });
        it('should handle different image extensions', () => {
            expect(getTransformedImagePath('@test.jpg')).toBe('[Image test.jpg]');
            expect(getTransformedImagePath('@test.jpeg')).toBe('[Image test.jpeg]');
            expect(getTransformedImagePath('@test.gif')).toBe('[Image test.gif]');
            expect(getTransformedImagePath('@test.webp')).toBe('[Image test.webp]');
            expect(getTransformedImagePath('@test.svg')).toBe('[Image test.svg]');
            expect(getTransformedImagePath('@test.bmp')).toBe('[Image test.bmp]');
        });
        it('should handle POSIX-style forward-slash paths on any platform', () => {
            const input = '@C:/Users/foo/screenshots/image2x.png';
            expect(getTransformedImagePath(input)).toBe('[Image image2x.png]');
        });
        it('should handle Windows-style backslash paths on any platform', () => {
            const input = '@C:\\Users\\foo\\screenshots\\image2x.png';
            expect(getTransformedImagePath(input)).toBe('[Image image2x.png]');
        });
        it('should handle escaped spaces in paths', () => {
            const input = '@path/to/my\\ file.png';
            expect(getTransformedImagePath(input)).toBe('[Image my file.png]');
        });
    });
    describe('getTransformationsForLine', () => {
        it('should find transformations in a line', () => {
            const line = 'Check out @test.png and @another.jpg';
            const result = calculateTransformationsForLine(line);
            expect(result).toHaveLength(2);
            expect(result[0]).toMatchObject({
                logicalText: '@test.png',
                collapsedText: '[Image test.png]',
            });
            expect(result[1]).toMatchObject({
                logicalText: '@another.jpg',
                collapsedText: '[Image another.jpg]',
            });
        });
        it('should handle no transformations', () => {
            const line = 'Just some regular text';
            const result = calculateTransformationsForLine(line);
            expect(result).toEqual([]);
        });
        it('should handle empty line', () => {
            const result = calculateTransformationsForLine('');
            expect(result).toEqual([]);
        });
        it('should keep adjacent image paths as separate transformations', () => {
            const line = '@a.png@b.png@c.png';
            const result = calculateTransformationsForLine(line);
            expect(result).toHaveLength(3);
            expect(result[0].logicalText).toBe('@a.png');
            expect(result[1].logicalText).toBe('@b.png');
            expect(result[2].logicalText).toBe('@c.png');
        });
        it('should handle multiple transformations in a row', () => {
            const line = '@a.png @b.png @c.png';
            const result = calculateTransformationsForLine(line);
            expect(result).toHaveLength(3);
        });
    });
    describe('getTransformUnderCursor', () => {
        const transformations = [
            {
                logStart: 5,
                logEnd: 14,
                logicalText: '@test.png',
                collapsedText: '[Image @test.png]',
                type: 'image',
            },
            {
                logStart: 20,
                logEnd: 31,
                logicalText: '@another.jpg',
                collapsedText: '[Image @another.jpg]',
                type: 'image',
            },
        ];
        it('should find transformation when cursor is inside it', () => {
            const result = getTransformUnderCursor(0, 7, [transformations]);
            expect(result).toEqual(transformations[0]);
        });
        it('should find transformation when cursor is at start', () => {
            const result = getTransformUnderCursor(0, 5, [transformations]);
            expect(result).toEqual(transformations[0]);
        });
        it('should NOT find transformation when cursor is at end', () => {
            const result = getTransformUnderCursor(0, 14, [transformations]);
            expect(result).toBeNull();
        });
        it('should return null when cursor is not on a transformation', () => {
            const result = getTransformUnderCursor(0, 2, [transformations]);
            expect(result).toBeNull();
        });
        it('should handle empty transformations array', () => {
            const result = getTransformUnderCursor(0, 5, []);
            expect(result).toBeNull();
        });
        it('regression: should not find paste transformation when clicking one character after it', () => {
            const pasteId = '[Pasted Text: 5 lines]';
            const line = pasteId + ' suffix';
            const transformations = calculateTransformationsForLine(line);
            const pasteTransform = transformations.find((t) => t.type === 'paste');
            expect(pasteTransform).toBeDefined();
            const endPos = pasteTransform.logEnd;
            // Position strictly at end should be null
            expect(getTransformUnderCursor(0, endPos, [transformations])).toBeNull();
            // Position inside should be found
            expect(getTransformUnderCursor(0, endPos - 1, [transformations])).toEqual(pasteTransform);
        });
    });
    describe('calculateTransformedLine', () => {
        it('should transform a line with one transformation', () => {
            const line = 'Check out @test.png';
            const transformations = calculateTransformationsForLine(line);
            const result = calculateTransformedLine(line, 0, [0, 0], transformations);
            expect(result.transformedLine).toBe('Check out [Image test.png]');
            expect(result.transformedToLogMap).toHaveLength(27); // Length includes all characters in the transformed line
            // Test that we have proper mappings
            expect(result.transformedToLogMap[0]).toBe(0); // 'C'
            expect(result.transformedToLogMap[9]).toBe(9); // ' ' before transformation
        });
        it('should handle cursor inside transformation', () => {
            const line = 'Check out @test.png';
            const transformations = calculateTransformationsForLine(line);
            // Cursor at '@' (position 10 in the line)
            const result = calculateTransformedLine(line, 0, [0, 10], transformations);
            // Should show full path when cursor is on it
            expect(result.transformedLine).toBe('Check out @test.png');
            // When expanded, each character maps to itself
            expect(result.transformedToLogMap[10]).toBe(10); // '@'
        });
        it('should handle line with no transformations', () => {
            const line = 'Just some text';
            const result = calculateTransformedLine(line, 0, [0, 0], []);
            expect(result.transformedLine).toBe(line);
            // Each visual position should map directly to logical position + trailing
            expect(result.transformedToLogMap).toHaveLength(15); // 14 chars + 1 trailing
            expect(result.transformedToLogMap[0]).toBe(0);
            expect(result.transformedToLogMap[13]).toBe(13);
            expect(result.transformedToLogMap[14]).toBe(14); // Trailing position
        });
        it('should handle empty line', () => {
            const result = calculateTransformedLine('', 0, [0, 0], []);
            expect(result.transformedLine).toBe('');
            expect(result.transformedToLogMap).toEqual([0]); // Just the trailing position
        });
    });
    describe('Layout Caching and Invalidation', () => {
        it.each([
            {
                desc: 'via setText',
                actFn: (result) => result.current.setText('changed line'),
                expected: 'changed line',
            },
            {
                desc: 'via replaceRange',
                actFn: (result) => result.current.replaceRange(0, 0, 0, 13, 'changed line'),
                expected: 'changed line',
            },
        ])('should invalidate cache when line content changes $desc', ({ actFn, expected }) => {
            const viewport = { width: 80, height: 24 };
            const { result } = renderHookWithProviders(() => useTextBuffer({
                initialText: 'original line',
                viewport,
                escapePastedPaths: true,
            }));
            const originalLayout = result.current.visualLayout;
            act(() => {
                actFn(result);
            });
            expect(result.current.visualLayout).not.toBe(originalLayout);
            expect(result.current.allVisualLines[0]).toBe(expected);
        });
        it('should invalidate cache when viewport width changes', () => {
            const viewport = { width: 80, height: 24 };
            const { result, rerender } = renderHookWithProviders(({ vp }) => useTextBuffer({
                initialText: 'a very long line that will wrap when the viewport is small',
                viewport: vp,
                escapePastedPaths: true,
            }), { initialProps: { vp: viewport } });
            const originalLayout = result.current.visualLayout;
            // Shrink viewport to force wrapping change
            rerender({ vp: { width: 10, height: 24 } });
            expect(result.current.visualLayout).not.toBe(originalLayout);
            expect(result.current.allVisualLines.length).toBeGreaterThan(1);
        });
        it('should correctly handle cursor expansion/collapse in cached layout', () => {
            const viewport = { width: 80, height: 24 };
            const text = 'Check @image.png here';
            const { result } = renderHookWithProviders(() => useTextBuffer({
                initialText: text,
                viewport,
                escapePastedPaths: true,
            }));
            // Cursor at start (collapsed)
            act(() => {
                result.current.moveToOffset(0);
            });
            expect(result.current.allVisualLines[0]).toContain('[Image image.png]');
            // Move cursor onto the @path (expanded)
            act(() => {
                result.current.moveToOffset(7); // onto @
            });
            expect(result.current.allVisualLines[0]).toContain('@image.png');
            expect(result.current.allVisualLines[0]).not.toContain('[Image image.png]');
            // Move cursor away (collapsed again)
            act(() => {
                result.current.moveToOffset(0);
            });
            expect(result.current.allVisualLines[0]).toContain('[Image image.png]');
        });
        it('should reuse cache for unchanged lines during editing', () => {
            const viewport = { width: 80, height: 24 };
            const initialText = 'line 1\nline 2\nline 3';
            const { result } = renderHookWithProviders(() => useTextBuffer({
                initialText,
                viewport,
                escapePastedPaths: true,
            }));
            const layout1 = result.current.visualLayout;
            // Edit line 1
            act(() => {
                result.current.moveToOffset(0);
                result.current.insert('X');
            });
            const layout2 = result.current.visualLayout;
            expect(layout2).not.toBe(layout1);
            // Verify that visual lines for line 2 and 3 (indices 1 and 2 in visualLines)
            // are identical in content if not in object reference (the arrays are rebuilt, but contents are cached)
            expect(result.current.allVisualLines[1]).toBe('line 2');
            expect(result.current.allVisualLines[2]).toBe('line 3');
        });
    });
    describe('Scroll Regressions', () => {
        const scrollViewport = { width: 80, height: 5 };
        it('should not show empty viewport when collapsing a large paste that was scrolled', () => {
            const largeContent = 'line1\nline2\nline3\nline4\nline5\nline6\nline7\nline8\nline9\nline10';
            const placeholder = '[Pasted Text: 10 lines]';
            const { result } = renderHook(() => useTextBuffer({
                initialText: placeholder,
                viewport: scrollViewport,
            }));
            // Setup: paste large content
            act(() => {
                result.current.setText('');
                result.current.insert(largeContent, { paste: true });
            });
            // Expand it
            act(() => {
                result.current.togglePasteExpansion(placeholder, 0, 0);
            });
            // Verify scrolled state
            expect(result.current.visualScrollRow).toBe(5);
            // Collapse it
            act(() => {
                result.current.togglePasteExpansion(placeholder, 9, 0);
            });
            // Verify viewport is NOT empty immediately (clamping in useMemo)
            expect(result.current.allVisualLines.length).toBe(1);
            expect(result.current.viewportVisualLines.length).toBe(1);
            expect(result.current.viewportVisualLines[0]).toBe(placeholder);
        });
    });
});
//# sourceMappingURL=text-buffer.test.js.map