/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode;

import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.management.NotCompliantMBeanException;
import javax.management.ObjectName;
import javax.management.StandardMBean;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.DF;
import org.apache.hadoop.fs.DU;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.BlockListAsLongs;
import org.apache.hadoop.hdfs.protocol.BlockLocalPathInfo;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.RecoveryInProgressException;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.datanode.BlockMetadataHeader;
import org.apache.hadoop.hdfs.server.datanode.BlockVolumeChoosingPolicy;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.DataStorage;
import org.apache.hadoop.hdfs.server.datanode.DatanodeUtil;
import org.apache.hadoop.hdfs.server.datanode.FSDatasetAsyncDiskService;
import org.apache.hadoop.hdfs.server.datanode.FSDatasetInterface;
import org.apache.hadoop.hdfs.server.datanode.FinalizedReplica;
import org.apache.hadoop.hdfs.server.datanode.ReplicaAlreadyExistsException;
import org.apache.hadoop.hdfs.server.datanode.ReplicaBeingWritten;
import org.apache.hadoop.hdfs.server.datanode.ReplicaInPipeline;
import org.apache.hadoop.hdfs.server.datanode.ReplicaInPipelineInterface;
import org.apache.hadoop.hdfs.server.datanode.ReplicaInfo;
import org.apache.hadoop.hdfs.server.datanode.ReplicaNotFoundException;
import org.apache.hadoop.hdfs.server.datanode.ReplicaUnderRecovery;
import org.apache.hadoop.hdfs.server.datanode.ReplicaWaitingToBeRecovered;
import org.apache.hadoop.hdfs.server.datanode.ReplicasMap;
import org.apache.hadoop.hdfs.server.datanode.RoundRobinVolumesPolicy;
import org.apache.hadoop.hdfs.server.datanode.metrics.FSDatasetMBean;
import org.apache.hadoop.hdfs.server.protocol.BlockRecoveryCommand;
import org.apache.hadoop.hdfs.server.protocol.InterDatanodeProtocol;
import org.apache.hadoop.hdfs.server.protocol.ReplicaRecoveryInfo;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.metrics2.util.MBeans;
import org.apache.hadoop.util.DataChecksum;
import org.apache.hadoop.util.DiskChecker;
import org.apache.hadoop.util.ReflectionUtils;

@InterfaceAudience.Private
class FSDataset
implements FSDatasetInterface {
    private final DataNode datanode;
    final FSVolumeSet volumes;
    private final int maxBlocksPerDir;
    final ReplicasMap volumeMap;
    final FSDatasetAsyncDiskService asyncDiskService;
    private final int validVolsRequired;
    private final Object statsLock = new Object();
    final boolean supportAppends;
    private ObjectName mbeanName;

    private static boolean isUnlinkTmpFile(File f) {
        String name = f.getName();
        return name.endsWith(".unlinked");
    }

    private static File getOrigFile(File unlinkTmpFile) {
        String fileName = unlinkTmpFile.getName();
        return new File(unlinkTmpFile.getParentFile(), fileName.substring(0, fileName.length() - ".unlinked".length()));
    }

    protected File getMetaFile(ExtendedBlock b) throws IOException {
        return DatanodeUtil.getMetaFile(this.getBlockFile(b), b.getGenerationStamp());
    }

    private static long getGenerationStampFromFile(File[] listdir, File blockFile) {
        String blockName = blockFile.getName();
        for (int j = 0; j < listdir.length; ++j) {
            String path = listdir[j].getName();
            if (!path.startsWith(blockName) || blockFile == listdir[j]) continue;
            return Block.getGenerationStamp(listdir[j].getName());
        }
        DataNode.LOG.warn((Object)("Block " + blockFile + " does not have a metafile!"));
        return 0L;
    }

    private static long parseGenerationStamp(File blockFile, File metaFile) throws IOException {
        String metaname = metaFile.getName();
        String gs = metaname.substring(blockFile.getName().length() + 1, metaname.length() - ".meta".length());
        try {
            return Long.parseLong(gs);
        }
        catch (NumberFormatException nfe) {
            throw (IOException)new IOException("blockFile=" + blockFile + ", metaFile=" + metaFile).initCause(nfe);
        }
    }

    @Override
    public List<FSDatasetInterface.FSVolumeInterface> getVolumes() {
        return this.volumes.volumes;
    }

    @Override
    public synchronized Block getStoredBlock(String bpid, long blkid) throws IOException {
        File blockfile = this.getFile(bpid, blkid);
        if (blockfile == null) {
            return null;
        }
        File metafile = DatanodeUtil.findMetaFile(blockfile);
        return new Block(blkid, blockfile.length(), FSDataset.parseGenerationStamp(blockfile, metafile));
    }

    ReplicaInfo fetchReplicaInfo(String bpid, long blockId) {
        ReplicaInfo r = this.volumeMap.get(bpid, blockId);
        if (r == null) {
            return null;
        }
        switch (r.getState()) {
            case FINALIZED: {
                return new FinalizedReplica((FinalizedReplica)r);
            }
            case RBW: {
                return new ReplicaBeingWritten((ReplicaBeingWritten)r);
            }
            case RWR: {
                return new ReplicaWaitingToBeRecovered((ReplicaWaitingToBeRecovered)r);
            }
            case RUR: {
                return new ReplicaUnderRecovery((ReplicaUnderRecovery)r);
            }
            case TEMPORARY: {
                return new ReplicaInPipeline((ReplicaInPipeline)r);
            }
        }
        return null;
    }

    @Override
    public boolean metaFileExists(ExtendedBlock b) throws IOException {
        return this.getMetaFile(b).exists();
    }

    @Override
    public long getMetaDataLength(ExtendedBlock b) throws IOException {
        File checksumFile = this.getMetaFile(b);
        return checksumFile.length();
    }

    @Override
    public FSDatasetInterface.MetaDataInputStream getMetaDataInputStream(ExtendedBlock b) throws IOException {
        File checksumFile = this.getMetaFile(b);
        return new FSDatasetInterface.MetaDataInputStream(new FileInputStream(checksumFile), checksumFile.length());
    }

    private FSDataset(DataNode datanode, DataStorage storage, Configuration conf) throws IOException {
        this.datanode = datanode;
        this.maxBlocksPerDir = conf.getInt("dfs.datanode.numblocks", 64);
        this.supportAppends = conf.getBoolean("dfs.support.append", true);
        int volFailuresTolerated = conf.getInt("dfs.datanode.failed.volumes.tolerated", 0);
        String[] dataDirs = conf.getTrimmedStrings("dfs.datanode.data.dir");
        int volsConfigured = dataDirs == null ? 0 : dataDirs.length;
        int volsFailed = volsConfigured - storage.getNumStorageDirs();
        this.validVolsRequired = volsConfigured - volFailuresTolerated;
        if (volFailuresTolerated < 0 || volFailuresTolerated >= volsConfigured) {
            throw new DiskChecker.DiskErrorException("Invalid volume failure  config value: " + volFailuresTolerated);
        }
        if (volsFailed > volFailuresTolerated) {
            throw new DiskChecker.DiskErrorException("Too many failed volumes - current valid volumes: " + storage.getNumStorageDirs() + ", volumes configured: " + volsConfigured + ", volumes failed: " + volsFailed + ", volume failures tolerated: " + volFailuresTolerated);
        }
        ArrayList<FSDatasetInterface.FSVolumeInterface> volArray = new ArrayList<FSDatasetInterface.FSVolumeInterface>(storage.getNumStorageDirs());
        for (int idx = 0; idx < storage.getNumStorageDirs(); ++idx) {
            File dir = storage.getStorageDir(idx).getCurrentDir();
            volArray.add(new FSVolume(dir, conf));
            DataNode.LOG.info((Object)("FSDataset added volume - " + dir));
        }
        this.volumeMap = new ReplicasMap(this);
        BlockVolumeChoosingPolicy blockChooserImpl = (BlockVolumeChoosingPolicy)ReflectionUtils.newInstance((Class)conf.getClass("dfs.datanode.block.volume.choice.policy", RoundRobinVolumesPolicy.class, BlockVolumeChoosingPolicy.class), (Configuration)conf);
        this.volumes = new FSVolumeSet(volArray, volsFailed, blockChooserImpl);
        this.volumes.getVolumeMap(this.volumeMap);
        File[] roots = new File[storage.getNumStorageDirs()];
        for (int idx = 0; idx < storage.getNumStorageDirs(); ++idx) {
            roots[idx] = storage.getStorageDir(idx).getCurrentDir();
        }
        this.asyncDiskService = new FSDatasetAsyncDiskService(this, roots);
        this.registerMBean(storage.getStorageID());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long getDfsUsed() throws IOException {
        Object object = this.statsLock;
        synchronized (object) {
            return this.volumes.getDfsUsed();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long getBlockPoolUsed(String bpid) throws IOException {
        Object object = this.statsLock;
        synchronized (object) {
            return this.volumes.getBlockPoolUsed(bpid);
        }
    }

    @Override
    public boolean hasEnoughResource() {
        return this.getVolumes().size() >= this.validVolsRequired;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long getCapacity() throws IOException {
        Object object = this.statsLock;
        synchronized (object) {
            return this.volumes.getCapacity();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long getRemaining() throws IOException {
        Object object = this.statsLock;
        synchronized (object) {
            return this.volumes.getRemaining();
        }
    }

    @Override
    public int getNumFailedVolumes() {
        return this.volumes.numberOfFailedVolumes();
    }

    @Override
    public long getLength(ExtendedBlock b) throws IOException {
        return this.getBlockFile(b).length();
    }

    private File getBlockFile(ExtendedBlock b) throws IOException {
        return this.getBlockFile(b.getBlockPoolId(), b.getLocalBlock());
    }

    File getBlockFile(String bpid, Block b) throws IOException {
        File f = this.validateBlockFile(bpid, b);
        if (f == null) {
            if (InterDatanodeProtocol.LOG.isDebugEnabled()) {
                InterDatanodeProtocol.LOG.debug((Object)("b=" + b + ", volumeMap=" + this.volumeMap));
            }
            throw new IOException("Block " + b + " is not valid.");
        }
        return f;
    }

    @Override
    public InputStream getBlockInputStream(ExtendedBlock b) throws IOException {
        File f = this.getBlockFileNoExistsCheck(b);
        try {
            return new FileInputStream(f);
        }
        catch (FileNotFoundException fnfe) {
            throw new IOException("Block " + b + " is not valid. " + "Expected block file at " + f + " does not exist.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private File getBlockFileNoExistsCheck(ExtendedBlock b) throws IOException {
        File f;
        FSDataset fSDataset = this;
        synchronized (fSDataset) {
            f = this.getFile(b.getBlockPoolId(), b.getLocalBlock().getBlockId());
        }
        if (f == null) {
            throw new IOException("Block " + b + " is not valid");
        }
        return f;
    }

    @Override
    public InputStream getBlockInputStream(ExtendedBlock b, long seekOffset) throws IOException {
        RandomAccessFile blockInFile;
        File blockFile = this.getBlockFileNoExistsCheck(b);
        try {
            blockInFile = new RandomAccessFile(blockFile, "r");
        }
        catch (FileNotFoundException fnfe) {
            throw new IOException("Block " + b + " is not valid. " + "Expected block file at " + blockFile + " does not exist.");
        }
        if (seekOffset > 0L) {
            blockInFile.seek(seekOffset);
        }
        return new FileInputStream(blockInFile.getFD());
    }

    ReplicaInfo getReplicaInfo(ExtendedBlock b) throws ReplicaNotFoundException {
        ReplicaInfo info = this.volumeMap.get(b.getBlockPoolId(), b.getLocalBlock());
        if (info == null) {
            throw new ReplicaNotFoundException("Cannot append to a non-existent replica " + b);
        }
        return info;
    }

    private ReplicaInfo getReplicaInfo(String bpid, long blkid) throws ReplicaNotFoundException {
        ReplicaInfo info = this.volumeMap.get(bpid, blkid);
        if (info == null) {
            throw new ReplicaNotFoundException("Cannot append to a non-existent replica " + bpid + ":" + blkid);
        }
        return info;
    }

    @Override
    public synchronized FSDatasetInterface.BlockInputStreams getTmpInputStreams(ExtendedBlock b, long blkOffset, long ckoff) throws IOException {
        ReplicaInfo info = this.getReplicaInfo(b);
        File blockFile = info.getBlockFile();
        RandomAccessFile blockInFile = new RandomAccessFile(blockFile, "r");
        if (blkOffset > 0L) {
            blockInFile.seek(blkOffset);
        }
        File metaFile = info.getMetaFile();
        RandomAccessFile metaInFile = new RandomAccessFile(metaFile, "r");
        if (ckoff > 0L) {
            metaInFile.seek(ckoff);
        }
        return new FSDatasetInterface.BlockInputStreams(new FileInputStream(blockInFile.getFD()), new FileInputStream(metaInFile.getFD()));
    }

    public boolean unlinkBlock(ExtendedBlock block, int numLinks) throws IOException {
        ReplicaInfo info = this.getReplicaInfo(block);
        return info.unlinkBlock(numLinks);
    }

    private static File moveBlockFiles(Block b, File srcfile, File destdir) throws IOException {
        File dstmeta;
        File dstfile = new File(destdir, b.getBlockName());
        File srcmeta = DatanodeUtil.getMetaFile(srcfile, b.getGenerationStamp());
        if (!srcmeta.renameTo(dstmeta = DatanodeUtil.getMetaFile(dstfile, b.getGenerationStamp()))) {
            throw new IOException("Failed to move meta file for " + b + " from " + srcmeta + " to " + dstmeta);
        }
        if (!srcfile.renameTo(dstfile)) {
            throw new IOException("Failed to move block file for " + b + " from " + srcfile + " to " + dstfile.getAbsolutePath());
        }
        if (DataNode.LOG.isDebugEnabled()) {
            DataNode.LOG.debug((Object)("addBlock: Moved " + srcmeta + " to " + dstmeta));
            DataNode.LOG.debug((Object)("addBlock: Moved " + srcfile + " to " + dstfile));
        }
        return dstfile;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void truncateBlock(File blockFile, File metaFile, long oldlen, long newlen) throws IOException {
        DataNode.LOG.info((Object)("truncateBlock: blockFile=" + blockFile + ", metaFile=" + metaFile + ", oldlen=" + oldlen + ", newlen=" + newlen));
        if (newlen == oldlen) {
            return;
        }
        if (newlen > oldlen) {
            throw new IOException("Cannout truncate block to from oldlen (=" + oldlen + ") to newlen (=" + newlen + ")");
        }
        DataChecksum dcs = BlockMetadataHeader.readHeader(metaFile).getChecksum();
        int checksumsize = dcs.getChecksumSize();
        int bpc = dcs.getBytesPerChecksum();
        long n = (newlen - 1L) / (long)bpc + 1L;
        long newmetalen = (long)BlockMetadataHeader.getHeaderSize() + n * (long)checksumsize;
        long lastchunkoffset = (n - 1L) * (long)bpc;
        int lastchunksize = (int)(newlen - lastchunkoffset);
        byte[] b = new byte[Math.max(lastchunksize, checksumsize)];
        RandomAccessFile blockRAF = new RandomAccessFile(blockFile, "rw");
        try {
            blockRAF.setLength(newlen);
            blockRAF.seek(lastchunkoffset);
            blockRAF.readFully(b, 0, lastchunksize);
        }
        finally {
            blockRAF.close();
        }
        dcs.update(b, 0, lastchunksize);
        dcs.writeValue(b, 0, false);
        RandomAccessFile metaRAF = new RandomAccessFile(metaFile, "rw");
        try {
            metaRAF.setLength(newmetalen);
            metaRAF.seek(newmetalen - (long)checksumsize);
            metaRAF.write(b, 0, checksumsize);
        }
        finally {
            metaRAF.close();
        }
    }

    @Override
    public synchronized ReplicaInPipelineInterface append(ExtendedBlock b, long newGS, long expectedBlockLen) throws IOException {
        if (newGS < b.getGenerationStamp()) {
            throw new IOException("The new generation stamp " + newGS + " should be greater than the replica " + b + "'s generation stamp");
        }
        ReplicaInfo replicaInfo = this.getReplicaInfo(b);
        DataNode.LOG.info((Object)("Appending to replica " + replicaInfo));
        if (replicaInfo.getState() != HdfsServerConstants.ReplicaState.FINALIZED) {
            throw new ReplicaNotFoundException("Cannot append to an unfinalized replica " + b);
        }
        if (replicaInfo.getNumBytes() != expectedBlockLen) {
            throw new IOException("Corrupted replica " + replicaInfo + " with a length of " + replicaInfo.getNumBytes() + " expected length is " + expectedBlockLen);
        }
        return this.append(b.getBlockPoolId(), (FinalizedReplica)replicaInfo, newGS, b.getNumBytes());
    }

    private synchronized ReplicaBeingWritten append(String bpid, FinalizedReplica replicaInfo, long newGS, long estimateBlockLen) throws IOException {
        replicaInfo.unlinkBlock(1);
        File blkfile = replicaInfo.getBlockFile();
        FSVolume v = (FSVolume)replicaInfo.getVolume();
        if (v.getAvailable() < estimateBlockLen - replicaInfo.getNumBytes()) {
            throw new DiskChecker.DiskOutOfSpaceException("Insufficient space for appending to " + replicaInfo);
        }
        File newBlkFile = new File(v.getRbwDir(bpid), replicaInfo.getBlockName());
        File oldmeta = replicaInfo.getMetaFile();
        ReplicaBeingWritten newReplicaInfo = new ReplicaBeingWritten(replicaInfo.getBlockId(), replicaInfo.getNumBytes(), newGS, v, newBlkFile.getParentFile(), Thread.currentThread());
        File newmeta = newReplicaInfo.getMetaFile();
        if (DataNode.LOG.isDebugEnabled()) {
            DataNode.LOG.debug((Object)("Renaming " + oldmeta + " to " + newmeta));
        }
        if (!oldmeta.renameTo(newmeta)) {
            throw new IOException("Block " + replicaInfo + " reopen failed. " + " Unable to move meta file  " + oldmeta + " to rbw dir " + newmeta);
        }
        if (DataNode.LOG.isDebugEnabled()) {
            DataNode.LOG.debug((Object)("Renaming " + blkfile + " to " + newBlkFile));
            DataNode.LOG.debug((Object)("Old block file length is " + blkfile.length()));
        }
        if (!blkfile.renameTo(newBlkFile)) {
            if (!newmeta.renameTo(oldmeta)) {
                DataNode.LOG.warn((Object)("Cannot move meta file " + newmeta + "back to the finalized directory " + oldmeta));
            }
            throw new IOException("Block " + replicaInfo + " reopen failed. " + " Unable to move block file " + blkfile + " to rbw dir " + newBlkFile);
        }
        this.volumeMap.add(bpid, newReplicaInfo);
        return newReplicaInfo;
    }

    private ReplicaInfo recoverCheck(ExtendedBlock b, long newGS, long expectedBlockLen) throws IOException {
        ReplicaInfo replicaInfo = this.getReplicaInfo(b.getBlockPoolId(), b.getBlockId());
        if (replicaInfo.getState() != HdfsServerConstants.ReplicaState.FINALIZED && replicaInfo.getState() != HdfsServerConstants.ReplicaState.RBW) {
            throw new ReplicaNotFoundException("Cannot recover append/close to a replica that's not FINALIZED and not RBW " + replicaInfo);
        }
        long replicaGenerationStamp = replicaInfo.getGenerationStamp();
        if (replicaGenerationStamp < b.getGenerationStamp() || replicaGenerationStamp > newGS) {
            throw new ReplicaNotFoundException("Cannot append to a replica with unexpeted generation stamp " + replicaGenerationStamp + ". Expected GS range is [" + b.getGenerationStamp() + ", " + newGS + "].");
        }
        long replicaLen = replicaInfo.getNumBytes();
        if (replicaInfo.getState() == HdfsServerConstants.ReplicaState.RBW) {
            ReplicaBeingWritten rbw = (ReplicaBeingWritten)replicaInfo;
            rbw.stopWriter();
            rbw.setWriter(Thread.currentThread());
            if (replicaLen != rbw.getBytesOnDisk() || replicaLen != rbw.getBytesAcked()) {
                throw new ReplicaAlreadyExistsException("RBW replica " + replicaInfo + "bytesRcvd(" + rbw.getNumBytes() + "), bytesOnDisk(" + rbw.getBytesOnDisk() + "), and bytesAcked(" + rbw.getBytesAcked() + ") are not the same.");
            }
        }
        if (replicaLen != expectedBlockLen) {
            throw new IOException("Corrupted replica " + replicaInfo + " with a length of " + replicaLen + " expected length is " + expectedBlockLen);
        }
        return replicaInfo;
    }

    @Override
    public synchronized ReplicaInPipelineInterface recoverAppend(ExtendedBlock b, long newGS, long expectedBlockLen) throws IOException {
        DataNode.LOG.info((Object)("Recover failed append to " + b));
        ReplicaInfo replicaInfo = this.recoverCheck(b, newGS, expectedBlockLen);
        if (replicaInfo.getState() == HdfsServerConstants.ReplicaState.FINALIZED) {
            return this.append(b.getBlockPoolId(), (FinalizedReplica)replicaInfo, newGS, b.getNumBytes());
        }
        this.bumpReplicaGS(replicaInfo, newGS);
        return (ReplicaBeingWritten)replicaInfo;
    }

    @Override
    public void recoverClose(ExtendedBlock b, long newGS, long expectedBlockLen) throws IOException {
        DataNode.LOG.info((Object)("Recover failed close " + b));
        ReplicaInfo replicaInfo = this.recoverCheck(b, newGS, expectedBlockLen);
        this.bumpReplicaGS(replicaInfo, newGS);
        if (replicaInfo.getState() == HdfsServerConstants.ReplicaState.RBW) {
            this.finalizeReplica(b.getBlockPoolId(), replicaInfo);
        }
    }

    private void bumpReplicaGS(ReplicaInfo replicaInfo, long newGS) throws IOException {
        long oldGS = replicaInfo.getGenerationStamp();
        File oldmeta = replicaInfo.getMetaFile();
        replicaInfo.setGenerationStamp(newGS);
        File newmeta = replicaInfo.getMetaFile();
        if (DataNode.LOG.isDebugEnabled()) {
            DataNode.LOG.debug((Object)("Renaming " + oldmeta + " to " + newmeta));
        }
        if (!oldmeta.renameTo(newmeta)) {
            replicaInfo.setGenerationStamp(oldGS);
            throw new IOException("Block " + replicaInfo + " reopen failed. " + " Unable to move meta file  " + oldmeta + " to " + newmeta);
        }
    }

    @Override
    public synchronized ReplicaInPipelineInterface createRbw(ExtendedBlock b) throws IOException {
        ReplicaInfo replicaInfo = this.volumeMap.get(b.getBlockPoolId(), b.getBlockId());
        if (replicaInfo != null) {
            throw new ReplicaAlreadyExistsException("Block " + b + " already exists in state " + (Object)((Object)replicaInfo.getState()) + " and thus cannot be created.");
        }
        FSVolume v = this.volumes.getNextVolume(b.getNumBytes());
        File f = v.createRbwFile(b.getBlockPoolId(), b.getLocalBlock());
        ReplicaBeingWritten newReplicaInfo = new ReplicaBeingWritten(b.getBlockId(), b.getGenerationStamp(), v, f.getParentFile());
        this.volumeMap.add(b.getBlockPoolId(), newReplicaInfo);
        return newReplicaInfo;
    }

    @Override
    public synchronized ReplicaInPipelineInterface recoverRbw(ExtendedBlock b, long newGS, long minBytesRcvd, long maxBytesRcvd) throws IOException {
        DataNode.LOG.info((Object)("Recover the RBW replica " + b));
        ReplicaInfo replicaInfo = this.getReplicaInfo(b.getBlockPoolId(), b.getBlockId());
        if (replicaInfo.getState() != HdfsServerConstants.ReplicaState.RBW) {
            throw new ReplicaNotFoundException("Cannot recover a non-RBW replica " + replicaInfo);
        }
        ReplicaBeingWritten rbw = (ReplicaBeingWritten)replicaInfo;
        DataNode.LOG.info((Object)("Recovering replica " + rbw));
        rbw.stopWriter();
        rbw.setWriter(Thread.currentThread());
        long replicaGenerationStamp = rbw.getGenerationStamp();
        if (replicaGenerationStamp < b.getGenerationStamp() || replicaGenerationStamp > newGS) {
            throw new ReplicaNotFoundException("Cannot append to a replica with unexpeted generation stamp " + b + ". Expected GS range is [" + b.getGenerationStamp() + ", " + newGS + "].");
        }
        long bytesAcked = rbw.getBytesAcked();
        long numBytes = rbw.getNumBytes();
        if (bytesAcked < minBytesRcvd || numBytes > maxBytesRcvd) {
            throw new ReplicaNotFoundException("Unmatched length replica " + replicaInfo + ": BytesAcked = " + bytesAcked + " BytesRcvd = " + numBytes + " are not in the range of [" + minBytesRcvd + ", " + maxBytesRcvd + "].");
        }
        if (numBytes > bytesAcked) {
            File replicafile = rbw.getBlockFile();
            FSDataset.truncateBlock(replicafile, rbw.getMetaFile(), numBytes, bytesAcked);
            rbw.setNumBytes(bytesAcked);
            rbw.setLastChecksumAndDataLen(bytesAcked, null);
        }
        this.bumpReplicaGS(rbw, newGS);
        return rbw;
    }

    @Override
    public synchronized ReplicaInPipelineInterface convertTemporaryToRbw(ExtendedBlock b) throws IOException {
        long blockId = b.getBlockId();
        long expectedGs = b.getGenerationStamp();
        long visible = b.getNumBytes();
        DataNode.LOG.info((Object)("Convert replica " + b + " from Temporary to RBW, visible length=" + visible));
        ReplicaInfo r = this.volumeMap.get(b.getBlockPoolId(), blockId);
        if (r == null) {
            throw new ReplicaNotFoundException("Cannot append to a non-existent replica " + b);
        }
        if (r.getState() != HdfsServerConstants.ReplicaState.TEMPORARY) {
            throw new ReplicaAlreadyExistsException("r.getState() != ReplicaState.TEMPORARY, r=" + r);
        }
        ReplicaInPipeline temp = (ReplicaInPipeline)r;
        if (temp.getGenerationStamp() != expectedGs) {
            throw new ReplicaAlreadyExistsException("temp.getGenerationStamp() != expectedGs = " + expectedGs + ", temp=" + temp);
        }
        long numBytes = temp.getNumBytes();
        if (numBytes < visible) {
            throw new IOException(numBytes + " = numBytes < visible = " + visible + ", temp=" + temp);
        }
        FSVolume v = (FSVolume)temp.getVolume();
        if (v == null) {
            throw new IOException("r.getVolume() = null, temp=" + temp);
        }
        BlockPoolSlice bpslice = v.getBlockPoolSlice(b.getBlockPoolId());
        File dest = FSDataset.moveBlockFiles(b.getLocalBlock(), temp.getBlockFile(), bpslice.getRbwDir());
        ReplicaBeingWritten rbw = new ReplicaBeingWritten(blockId, numBytes, expectedGs, v, dest.getParentFile(), Thread.currentThread());
        rbw.setBytesAcked(visible);
        this.volumeMap.add(b.getBlockPoolId(), rbw);
        return rbw;
    }

    @Override
    public synchronized ReplicaInPipelineInterface createTemporary(ExtendedBlock b) throws IOException {
        ReplicaInfo replicaInfo = this.volumeMap.get(b.getBlockPoolId(), b.getBlockId());
        if (replicaInfo != null) {
            throw new ReplicaAlreadyExistsException("Block " + b + " already exists in state " + (Object)((Object)replicaInfo.getState()) + " and thus cannot be created.");
        }
        FSVolume v = this.volumes.getNextVolume(b.getNumBytes());
        File f = v.createTmpFile(b.getBlockPoolId(), b.getLocalBlock());
        ReplicaInPipeline newReplicaInfo = new ReplicaInPipeline(b.getBlockId(), b.getGenerationStamp(), v, f.getParentFile());
        this.volumeMap.add(b.getBlockPoolId(), newReplicaInfo);
        return newReplicaInfo;
    }

    @Override
    public void adjustCrcChannelPosition(ExtendedBlock b, FSDatasetInterface.BlockWriteStreams streams, int checksumSize) throws IOException {
        FileOutputStream file = (FileOutputStream)streams.checksumOut;
        FileChannel channel = file.getChannel();
        long oldPos = channel.position();
        long newPos = oldPos - (long)checksumSize;
        if (DataNode.LOG.isDebugEnabled()) {
            DataNode.LOG.debug((Object)("Changing meta file offset of block " + b + " from " + oldPos + " to " + newPos));
        }
        channel.position(newPos);
    }

    synchronized File createTmpFile(FSVolume vol, String bpid, Block blk) throws IOException {
        if (vol == null) {
            ReplicaInfo replica = this.volumeMap.get(bpid, blk);
            if (replica != null) {
                vol = (FSVolume)this.volumeMap.get(bpid, blk).getVolume();
            }
            if (vol == null) {
                throw new IOException("Could not find volume for block " + blk);
            }
        }
        return vol.createTmpFile(bpid, blk);
    }

    @Override
    public synchronized void finalizeBlock(ExtendedBlock b) throws IOException {
        if (Thread.interrupted()) {
            throw new IOException("Cannot finalize block from Interrupted Thread");
        }
        ReplicaInfo replicaInfo = this.getReplicaInfo(b);
        if (replicaInfo.getState() == HdfsServerConstants.ReplicaState.FINALIZED) {
            return;
        }
        this.finalizeReplica(b.getBlockPoolId(), replicaInfo);
    }

    private synchronized FinalizedReplica finalizeReplica(String bpid, ReplicaInfo replicaInfo) throws IOException {
        FinalizedReplica newReplicaInfo = null;
        if (replicaInfo.getState() == HdfsServerConstants.ReplicaState.RUR && ((ReplicaUnderRecovery)replicaInfo).getOrignalReplicaState() == HdfsServerConstants.ReplicaState.FINALIZED) {
            newReplicaInfo = (FinalizedReplica)((ReplicaUnderRecovery)replicaInfo).getOriginalReplica();
        } else {
            FSVolume v = (FSVolume)replicaInfo.getVolume();
            File f = replicaInfo.getBlockFile();
            if (v == null) {
                throw new IOException("No volume for temporary file " + f + " for block " + replicaInfo);
            }
            File dest = v.addBlock(bpid, replicaInfo, f);
            newReplicaInfo = new FinalizedReplica(replicaInfo, v, dest.getParentFile());
        }
        this.volumeMap.add(bpid, newReplicaInfo);
        return newReplicaInfo;
    }

    @Override
    public synchronized void unfinalizeBlock(ExtendedBlock b) throws IOException {
        ReplicaInfo replicaInfo = this.volumeMap.get(b.getBlockPoolId(), b.getLocalBlock());
        if (replicaInfo != null && replicaInfo.getState() == HdfsServerConstants.ReplicaState.TEMPORARY) {
            this.volumeMap.remove(b.getBlockPoolId(), b.getLocalBlock());
            if (this.delBlockFromDisk(replicaInfo.getBlockFile(), replicaInfo.getMetaFile(), b.getLocalBlock())) {
                DataNode.LOG.warn((Object)("Block " + b + " unfinalized and removed. "));
            }
        }
    }

    private boolean delBlockFromDisk(File blockFile, File metaFile, Block b) {
        if (blockFile == null) {
            DataNode.LOG.warn((Object)("No file exists for block: " + b));
            return true;
        }
        if (!blockFile.delete()) {
            DataNode.LOG.warn((Object)("Not able to delete the block file: " + blockFile));
            return false;
        }
        if (metaFile != null && !metaFile.delete()) {
            DataNode.LOG.warn((Object)("Not able to delete the meta block file: " + metaFile));
            return false;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public BlockListAsLongs getBlockReport(String bpid) {
        int size = this.volumeMap.size(bpid);
        ArrayList<ReplicaInfo> finalized = new ArrayList<ReplicaInfo>(size);
        ArrayList<ReplicaInfo> uc = new ArrayList<ReplicaInfo>();
        if (size == 0) {
            return new BlockListAsLongs(finalized, uc);
        }
        FSDataset fSDataset = this;
        synchronized (fSDataset) {
            block9: for (ReplicaInfo b : this.volumeMap.replicas(bpid)) {
                switch (b.getState()) {
                    case FINALIZED: {
                        finalized.add(b);
                        continue block9;
                    }
                    case RBW: 
                    case RWR: {
                        uc.add(b);
                        continue block9;
                    }
                    case RUR: {
                        ReplicaUnderRecovery rur = (ReplicaUnderRecovery)b;
                        uc.add(rur.getOriginalReplica());
                        continue block9;
                    }
                    case TEMPORARY: {
                        continue block9;
                    }
                }
                assert (false) : "Illegal ReplicaInfo state.";
            }
            return new BlockListAsLongs(finalized, uc);
        }
    }

    @Override
    public synchronized List<Block> getFinalizedBlocks(String bpid) {
        ArrayList<Block> finalized = new ArrayList<Block>(this.volumeMap.size(bpid));
        for (ReplicaInfo b : this.volumeMap.replicas(bpid)) {
            if (b.getState() != HdfsServerConstants.ReplicaState.FINALIZED) continue;
            finalized.add(new Block(b));
        }
        return finalized;
    }

    @Override
    public boolean isValidBlock(ExtendedBlock b) {
        return this.isValid(b, HdfsServerConstants.ReplicaState.FINALIZED);
    }

    @Override
    public boolean isValidRbw(ExtendedBlock b) {
        return this.isValid(b, HdfsServerConstants.ReplicaState.RBW);
    }

    private boolean isValid(ExtendedBlock b, HdfsServerConstants.ReplicaState state) {
        ReplicaInfo replicaInfo = this.volumeMap.get(b.getBlockPoolId(), b.getLocalBlock());
        return replicaInfo != null && replicaInfo.getState() == state && replicaInfo.getBlockFile().exists();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    File validateBlockFile(String bpid, Block b) throws IOException {
        File f;
        FSDataset fSDataset = this;
        synchronized (fSDataset) {
            f = this.getFile(bpid, b.getBlockId());
        }
        if (f != null) {
            if (f.exists()) {
                return f;
            }
            this.datanode.checkDiskError();
        }
        if (InterDatanodeProtocol.LOG.isDebugEnabled()) {
            InterDatanodeProtocol.LOG.debug((Object)("b=" + b + ", f=" + f));
        }
        return null;
    }

    static void checkReplicaFiles(ReplicaInfo r) throws IOException {
        File f = r.getBlockFile();
        if (!f.exists()) {
            throw new FileNotFoundException("File " + f + " not found, r=" + r);
        }
        if (r.getBytesOnDisk() != f.length()) {
            throw new IOException("File length mismatched.  The length of " + f + " is " + f.length() + " but r=" + r);
        }
        File metafile = DatanodeUtil.getMetaFile(f, r.getGenerationStamp());
        if (!metafile.exists()) {
            throw new IOException("Metafile " + metafile + " does not exist, r=" + r);
        }
        if (metafile.length() == 0L) {
            throw new IOException("Metafile " + metafile + " is empty, r=" + r);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void invalidate(String bpid, Block[] invalidBlks) throws IOException {
        boolean error = false;
        for (int i = 0; i < invalidBlks.length; ++i) {
            FSVolume v;
            File f = null;
            FSDataset fSDataset = this;
            synchronized (fSDataset) {
                f = this.getFile(bpid, invalidBlks[i].getBlockId());
                ReplicaInfo dinfo = this.volumeMap.get(bpid, invalidBlks[i]);
                if (dinfo == null || dinfo.getGenerationStamp() != invalidBlks[i].getGenerationStamp()) {
                    DataNode.LOG.warn((Object)("Unexpected error trying to delete block " + invalidBlks[i] + ". BlockInfo not found in volumeMap."));
                    error = true;
                    continue;
                }
                v = (FSVolume)dinfo.getVolume();
                if (f == null) {
                    DataNode.LOG.warn((Object)("Unexpected error trying to delete block " + invalidBlks[i] + ". Block not found in blockMap." + (v == null ? " " : " Block found in volumeMap.")));
                    error = true;
                    continue;
                }
                if (v == null) {
                    DataNode.LOG.warn((Object)("Unexpected error trying to delete block " + invalidBlks[i] + ". No volume for this block." + " Block found in blockMap. " + f + "."));
                    error = true;
                    continue;
                }
                File parent = f.getParentFile();
                if (parent == null) {
                    DataNode.LOG.warn((Object)("Unexpected error trying to delete block " + invalidBlks[i] + ". Parent not found for file " + f + "."));
                    error = true;
                    continue;
                }
                HdfsServerConstants.ReplicaState replicaState = dinfo.getState();
                if (replicaState == HdfsServerConstants.ReplicaState.FINALIZED || replicaState == HdfsServerConstants.ReplicaState.RUR && ((ReplicaUnderRecovery)dinfo).getOrignalReplicaState() == HdfsServerConstants.ReplicaState.FINALIZED) {
                    v.clearPath(bpid, parent);
                }
                this.volumeMap.remove(bpid, invalidBlks[i]);
            }
            File metaFile = DatanodeUtil.getMetaFile(f, invalidBlks[i].getGenerationStamp());
            this.asyncDiskService.deleteAsync(v, f, metaFile, new ExtendedBlock(bpid, invalidBlks[i]));
        }
        if (error) {
            throw new IOException("Error in deleting blocks.");
        }
    }

    public void notifyNamenodeDeletedBlock(ExtendedBlock block) {
        this.datanode.notifyNamenodeDeletedBlock(block);
    }

    @Override
    public synchronized boolean contains(ExtendedBlock block) {
        long blockId = block.getLocalBlock().getBlockId();
        return this.getFile(block.getBlockPoolId(), blockId) != null;
    }

    File getFile(String bpid, long blockId) {
        ReplicaInfo info = this.volumeMap.get(bpid, blockId);
        if (info != null) {
            return info.getBlockFile();
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void checkDataDir() throws DiskChecker.DiskErrorException {
        long totalBlocks = 0L;
        long removedBlocks = 0L;
        List failedVols = this.volumes.checkDirs();
        if (failedVols == null) {
            return;
        }
        long mlsec = System.currentTimeMillis();
        FSDataset fSDataset = this;
        synchronized (fSDataset) {
            for (FSVolume fv : failedVols) {
                for (String bpid : fv.map.keySet()) {
                    Iterator<ReplicaInfo> ib = this.volumeMap.replicas(bpid).iterator();
                    while (ib.hasNext()) {
                        ReplicaInfo b = ib.next();
                        ++totalBlocks;
                        if (b.getVolume() != fv) continue;
                        DataNode.LOG.warn((Object)("Removing replica " + bpid + ":" + b.getBlockId() + " on failed volume " + fv.currentDir.getAbsolutePath()));
                        ib.remove();
                        ++removedBlocks;
                    }
                }
            }
        }
        mlsec = System.currentTimeMillis() - mlsec;
        DataNode.LOG.warn((Object)("Removed " + removedBlocks + " out of " + totalBlocks + "(took " + mlsec + " millisecs)"));
        StringBuilder sb = new StringBuilder();
        for (FSVolume fv : failedVols) {
            sb.append(fv.currentDir.getAbsolutePath() + ";");
        }
        throw new DiskChecker.DiskErrorException("DataNode failed volumes:" + sb);
    }

    @Override
    public String toString() {
        return "FSDataset{dirpath='" + this.volumes + "'}";
    }

    void registerMBean(String storageId) {
        String storageName = storageId == null || storageId.equals("") ? "UndefinedStorageId" + DFSUtil.getRandom().nextInt() : storageId;
        try {
            StandardMBean bean = new StandardMBean(this, FSDatasetMBean.class);
            this.mbeanName = MBeans.register((String)"DataNode", (String)("FSDatasetState-" + storageName), (Object)bean);
        }
        catch (NotCompliantMBeanException e) {
            DataNode.LOG.warn((Object)"Error registering FSDatasetState MBean", (Throwable)e);
        }
        DataNode.LOG.info((Object)"Registered FSDatasetState MBean");
    }

    @Override
    public void shutdown() {
        if (this.mbeanName != null) {
            MBeans.unregister((ObjectName)this.mbeanName);
        }
        if (this.asyncDiskService != null) {
            this.asyncDiskService.shutdown();
        }
        if (this.volumes != null) {
            this.volumes.shutdown();
        }
    }

    @Override
    public String getStorageInfo() {
        return this.toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void checkAndUpdate(String bpid, long blockId, File diskFile, File diskMetaFile, FSDatasetInterface.FSVolumeInterface vol) {
        Block corruptBlock = null;
        FSDataset fSDataset = this;
        synchronized (fSDataset) {
            long diskGS;
            ReplicaInfo memBlockInfo = this.volumeMap.get(bpid, blockId);
            if (memBlockInfo != null && memBlockInfo.getState() != HdfsServerConstants.ReplicaState.FINALIZED) {
                return;
            }
            long l = diskGS = diskMetaFile != null && diskMetaFile.exists() ? Block.getGenerationStamp(diskMetaFile.getName()) : 0L;
            if (diskFile == null || !diskFile.exists()) {
                if (memBlockInfo == null) {
                    if (diskMetaFile != null && diskMetaFile.exists() && diskMetaFile.delete()) {
                        DataNode.LOG.warn((Object)("Deleted a metadata file without a block " + diskMetaFile.getAbsolutePath()));
                    }
                    return;
                }
                if (!memBlockInfo.getBlockFile().exists()) {
                    this.volumeMap.remove(bpid, blockId);
                    if (this.datanode.blockScanner != null) {
                        this.datanode.blockScanner.deleteBlock(bpid, new Block(blockId));
                    }
                    DataNode.LOG.warn((Object)("Removed block " + blockId + " from memory with missing block file on the disk"));
                    if (diskMetaFile != null && diskMetaFile.exists() && diskMetaFile.delete()) {
                        DataNode.LOG.warn((Object)("Deleted a metadata file for the deleted block " + diskMetaFile.getAbsolutePath()));
                    }
                }
                return;
            }
            if (memBlockInfo == null) {
                FinalizedReplica diskBlockInfo = new FinalizedReplica(blockId, diskFile.length(), diskGS, vol, diskFile.getParentFile());
                this.volumeMap.add(bpid, diskBlockInfo);
                if (this.datanode.blockScanner != null) {
                    this.datanode.blockScanner.addBlock(new ExtendedBlock(bpid, diskBlockInfo));
                }
                DataNode.LOG.warn((Object)("Added missing block to memory " + diskBlockInfo));
                return;
            }
            File memFile = memBlockInfo.getBlockFile();
            if (memFile.exists()) {
                if (memFile.compareTo(diskFile) != 0) {
                    DataNode.LOG.warn((Object)("Block file " + memFile.getAbsolutePath() + " does not match file found by scan " + diskFile.getAbsolutePath()));
                }
            } else {
                DataNode.LOG.warn((Object)("Block file in volumeMap " + memFile.getAbsolutePath() + " does not exist. Updating it to the file found during scan " + diskFile.getAbsolutePath()));
                memBlockInfo.setDir(diskFile.getParentFile());
                memFile = diskFile;
                DataNode.LOG.warn((Object)("Updating generation stamp for block " + blockId + " from " + memBlockInfo.getGenerationStamp() + " to " + diskGS));
                memBlockInfo.setGenerationStamp(diskGS);
            }
            if (memBlockInfo.getGenerationStamp() != diskGS) {
                File memMetaFile = DatanodeUtil.getMetaFile(diskFile, memBlockInfo.getGenerationStamp());
                if (memMetaFile.exists()) {
                    if (memMetaFile.compareTo(diskMetaFile) != 0) {
                        DataNode.LOG.warn((Object)("Metadata file in memory " + memMetaFile.getAbsolutePath() + " does not match file found by scan " + diskMetaFile.getAbsolutePath()));
                    }
                } else {
                    long gs = diskMetaFile != null && diskMetaFile.exists() && diskMetaFile.getParent().equals(memFile.getParent()) ? diskGS : 0L;
                    DataNode.LOG.warn((Object)("Updating generation stamp for block " + blockId + " from " + memBlockInfo.getGenerationStamp() + " to " + gs));
                    memBlockInfo.setGenerationStamp(gs);
                }
            }
            if (memBlockInfo.getNumBytes() != memFile.length()) {
                corruptBlock = new Block(memBlockInfo);
                DataNode.LOG.warn((Object)("Updating size of block " + blockId + " from " + memBlockInfo.getNumBytes() + " to " + memFile.length()));
                memBlockInfo.setNumBytes(memFile.length());
            }
        }
        if (corruptBlock != null) {
            DataNode.LOG.warn((Object)("Reporting the block " + corruptBlock + " as corrupt due to length mismatch"));
            try {
                this.datanode.reportBadBlocks(new ExtendedBlock(bpid, corruptBlock));
            }
            catch (IOException e) {
                DataNode.LOG.warn((Object)("Failed to repot bad block " + corruptBlock), (Throwable)e);
            }
        }
    }

    @Override
    @Deprecated
    public ReplicaInfo getReplica(String bpid, long blockId) {
        return this.volumeMap.get(bpid, blockId);
    }

    @Override
    public synchronized String getReplicaString(String bpid, long blockId) {
        ReplicaInfo r = this.volumeMap.get(bpid, blockId);
        return r == null ? "null" : ((Object)r).toString();
    }

    @Override
    public synchronized ReplicaRecoveryInfo initReplicaRecovery(BlockRecoveryCommand.RecoveringBlock rBlock) throws IOException {
        return FSDataset.initReplicaRecovery(rBlock.getBlock().getBlockPoolId(), this.volumeMap, rBlock.getBlock().getLocalBlock(), rBlock.getNewGenerationStamp());
    }

    static ReplicaRecoveryInfo initReplicaRecovery(String bpid, ReplicasMap map, Block block, long recoveryId) throws IOException {
        ReplicaUnderRecovery rur;
        ReplicaInfo replica = map.get(bpid, block.getBlockId());
        DataNode.LOG.info((Object)("initReplicaRecovery: block=" + block + ", recoveryId=" + recoveryId + ", replica=" + replica));
        if (replica == null) {
            return null;
        }
        if (replica instanceof ReplicaInPipeline) {
            ReplicaInPipeline rip = (ReplicaInPipeline)replica;
            rip.stopWriter();
            if (rip.getBytesOnDisk() < rip.getVisibleLength()) {
                throw new IOException("THIS IS NOT SUPPOSED TO HAPPEN: getBytesOnDisk() < getVisibleLength(), rip=" + rip);
            }
            FSDataset.checkReplicaFiles(rip);
        }
        if (replica.getGenerationStamp() < block.getGenerationStamp()) {
            throw new IOException("replica.getGenerationStamp() < block.getGenerationStamp(), block=" + block + ", replica=" + replica);
        }
        if (replica.getGenerationStamp() >= recoveryId) {
            throw new IOException("THIS IS NOT SUPPOSED TO HAPPEN: replica.getGenerationStamp() >= recoveryId = " + recoveryId + ", block=" + block + ", replica=" + replica);
        }
        if (replica.getState() == HdfsServerConstants.ReplicaState.RUR) {
            rur = (ReplicaUnderRecovery)replica;
            if (rur.getRecoveryID() >= recoveryId) {
                throw new RecoveryInProgressException("rur.getRecoveryID() >= recoveryId = " + recoveryId + ", block=" + block + ", rur=" + rur);
            }
            long oldRecoveryID = rur.getRecoveryID();
            rur.setRecoveryID(recoveryId);
            DataNode.LOG.info((Object)("initReplicaRecovery: update recovery id for " + block + " from " + oldRecoveryID + " to " + recoveryId));
        } else {
            rur = new ReplicaUnderRecovery(replica, recoveryId);
            map.add(bpid, rur);
            DataNode.LOG.info((Object)("initReplicaRecovery: changing replica state for " + block + " from " + (Object)((Object)replica.getState()) + " to " + (Object)((Object)rur.getState())));
        }
        return rur.createInfo();
    }

    @Override
    public synchronized ReplicaInfo updateReplicaUnderRecovery(ExtendedBlock oldBlock, long recoveryId, long newlength) throws IOException {
        ReplicaInfo replica = this.volumeMap.get(oldBlock.getBlockPoolId(), oldBlock.getBlockId());
        DataNode.LOG.info((Object)("updateReplica: block=" + oldBlock + ", recoveryId=" + recoveryId + ", length=" + newlength + ", replica=" + replica));
        if (replica == null) {
            throw new ReplicaNotFoundException(oldBlock);
        }
        if (replica.getState() != HdfsServerConstants.ReplicaState.RUR) {
            throw new IOException("replica.getState() != " + (Object)((Object)HdfsServerConstants.ReplicaState.RUR) + ", replica=" + replica);
        }
        if (replica.getBytesOnDisk() != oldBlock.getNumBytes()) {
            throw new IOException("THIS IS NOT SUPPOSED TO HAPPEN: replica.getBytesOnDisk() != block.getNumBytes(), block=" + oldBlock + ", replica=" + replica);
        }
        FSDataset.checkReplicaFiles(replica);
        FinalizedReplica finalized = this.updateReplicaUnderRecovery(oldBlock.getBlockPoolId(), (ReplicaUnderRecovery)replica, recoveryId, newlength);
        FSDataset.checkReplicaFiles(finalized);
        return finalized;
    }

    private FinalizedReplica updateReplicaUnderRecovery(String bpid, ReplicaUnderRecovery rur, long recoveryId, long newlength) throws IOException {
        if (rur.getRecoveryID() != recoveryId) {
            throw new IOException("rur.getRecoveryID() != recoveryId = " + recoveryId + ", rur=" + rur);
        }
        this.bumpReplicaGS(rur, recoveryId);
        File replicafile = rur.getBlockFile();
        if (rur.getNumBytes() < newlength) {
            throw new IOException("rur.getNumBytes() < newlength = " + newlength + ", rur=" + rur);
        }
        if (rur.getNumBytes() > newlength) {
            rur.unlinkBlock(1);
            FSDataset.truncateBlock(replicafile, rur.getMetaFile(), rur.getNumBytes(), newlength);
            rur.setNumBytes(newlength);
        }
        return this.finalizeReplica(bpid, rur);
    }

    @Override
    public synchronized long getReplicaVisibleLength(ExtendedBlock block) throws IOException {
        ReplicaInfo replica = this.getReplicaInfo(block.getBlockPoolId(), block.getBlockId());
        if (replica.getGenerationStamp() < block.getGenerationStamp()) {
            throw new IOException("replica.getGenerationStamp() < block.getGenerationStamp(), block=" + block + ", replica=" + replica);
        }
        return replica.getVisibleLength();
    }

    @Override
    public synchronized void addBlockPool(String bpid, Configuration conf) throws IOException {
        DataNode.LOG.info((Object)("Adding block pool " + bpid));
        this.volumes.addBlockPool(bpid, conf);
        this.volumeMap.initBlockPool(bpid);
        this.volumes.getVolumeMap(bpid, this.volumeMap);
    }

    @Override
    public synchronized void shutdownBlockPool(String bpid) {
        DataNode.LOG.info((Object)("Removing block pool " + bpid));
        this.volumeMap.cleanUpBlockPool(bpid);
        this.volumes.removeBlockPool(bpid);
    }

    @Override
    public String[] getBlockPoolList() {
        return this.volumeMap.getBlockPoolList();
    }

    private Collection<VolumeInfo> getVolumeInfo() {
        ArrayList<VolumeInfo> info = new ArrayList<VolumeInfo>();
        for (FSDatasetInterface.FSVolumeInterface v : this.volumes.volumes) {
            FSVolume volume = (FSVolume)v;
            long used = 0L;
            long free = 0L;
            try {
                used = volume.getDfsUsed();
                free = volume.getAvailable();
            }
            catch (IOException e) {
                DataNode.LOG.warn((Object)e.getMessage());
                used = 0L;
                free = 0L;
            }
            info.add(new VolumeInfo(volume.toString(), used, free, volume.getReserved()));
        }
        return info;
    }

    @Override
    public Map<String, Object> getVolumeInfoMap() {
        HashMap<String, Object> info = new HashMap<String, Object>();
        Collection<VolumeInfo> volumes = this.getVolumeInfo();
        for (VolumeInfo v : volumes) {
            HashMap<String, Long> innerInfo = new HashMap<String, Long>();
            innerInfo.put("usedSpace", v.usedSpace);
            innerInfo.put("freeSpace", v.freeSpace);
            innerInfo.put("reservedSpace", v.reservedSpace);
            info.put(v.directory, innerInfo);
        }
        return info;
    }

    @Override
    public synchronized void deleteBlockPool(String bpid, boolean force) throws IOException {
        if (!force) {
            for (FSDatasetInterface.FSVolumeInterface volume : this.volumes.volumes) {
                if (((FSVolume)volume).isBPDirEmpty(bpid)) continue;
                DataNode.LOG.warn((Object)(bpid + " has some block files, cannot delete unless forced"));
                throw new IOException("Cannot delete block pool, it contains some block files");
            }
        }
        for (FSDatasetInterface.FSVolumeInterface volume : this.volumes.volumes) {
            ((FSVolume)volume).deleteBPDirectories(bpid, force);
        }
    }

    @Override
    public BlockLocalPathInfo getBlockLocalPathInfo(ExtendedBlock block) throws IOException {
        File datafile = this.getBlockFile(block);
        File metafile = DatanodeUtil.getMetaFile(datafile, block.getGenerationStamp());
        BlockLocalPathInfo info = new BlockLocalPathInfo(block, datafile.getAbsolutePath(), metafile.getAbsolutePath());
        return info;
    }

    private static class VolumeInfo {
        final String directory;
        final long usedSpace;
        final long freeSpace;
        final long reservedSpace;

        VolumeInfo(String dir, long usedSpace, long freeSpace, long reservedSpace) {
            this.directory = dir;
            this.usedSpace = usedSpace;
            this.freeSpace = freeSpace;
            this.reservedSpace = reservedSpace;
        }
    }

    static class FSVolumeSet {
        private volatile List<FSDatasetInterface.FSVolumeInterface> volumes = null;
        BlockVolumeChoosingPolicy blockChooser;
        int numFailedVolumes;

        FSVolumeSet(List<FSDatasetInterface.FSVolumeInterface> volumes, int failedVols, BlockVolumeChoosingPolicy blockChooser) {
            this.volumes = Collections.unmodifiableList(volumes);
            this.blockChooser = blockChooser;
            this.numFailedVolumes = failedVols;
        }

        private int numberOfFailedVolumes() {
            return this.numFailedVolumes;
        }

        synchronized FSVolume getNextVolume(long blockSize) throws IOException {
            return (FSVolume)this.blockChooser.chooseVolume(this.volumes, blockSize);
        }

        private long getDfsUsed() throws IOException {
            long dfsUsed = 0L;
            for (FSDatasetInterface.FSVolumeInterface v : this.volumes) {
                dfsUsed += ((FSVolume)v).getDfsUsed();
            }
            return dfsUsed;
        }

        private long getBlockPoolUsed(String bpid) throws IOException {
            long dfsUsed = 0L;
            for (FSDatasetInterface.FSVolumeInterface v : this.volumes) {
                dfsUsed += ((FSVolume)v).getBlockPoolUsed(bpid);
            }
            return dfsUsed;
        }

        private long getCapacity() throws IOException {
            long capacity = 0L;
            for (FSDatasetInterface.FSVolumeInterface v : this.volumes) {
                capacity += ((FSVolume)v).getCapacity();
            }
            return capacity;
        }

        private long getRemaining() throws IOException {
            long remaining = 0L;
            for (FSDatasetInterface.FSVolumeInterface vol : this.volumes) {
                remaining += vol.getAvailable();
            }
            return remaining;
        }

        private void getVolumeMap(ReplicasMap volumeMap) throws IOException {
            for (FSDatasetInterface.FSVolumeInterface v : this.volumes) {
                ((FSVolume)v).getVolumeMap(volumeMap);
            }
        }

        private void getVolumeMap(String bpid, ReplicasMap volumeMap) throws IOException {
            for (FSDatasetInterface.FSVolumeInterface v : this.volumes) {
                ((FSVolume)v).getVolumeMap(bpid, volumeMap);
            }
        }

        private synchronized List<FSVolume> checkDirs() {
            ArrayList<FSVolume> removedVols = null;
            ArrayList<FSDatasetInterface.FSVolumeInterface> volumeList = new ArrayList<FSDatasetInterface.FSVolumeInterface>(this.volumes);
            for (int idx = 0; idx < volumeList.size(); ++idx) {
                FSVolume fsv = (FSVolume)volumeList.get(idx);
                try {
                    fsv.checkDirs();
                    continue;
                }
                catch (DiskChecker.DiskErrorException e) {
                    DataNode.LOG.warn((Object)("Removing failed volume " + fsv + ": "), (Throwable)e);
                    if (removedVols == null) {
                        removedVols = new ArrayList<FSVolume>(1);
                    }
                    removedVols.add(fsv);
                    fsv.shutdown();
                    volumeList.set(idx, null);
                    ++this.numFailedVolumes;
                }
            }
            if (removedVols != null && removedVols.size() > 0) {
                ArrayList<FSDatasetInterface.FSVolumeInterface> newVols = new ArrayList<FSDatasetInterface.FSVolumeInterface>();
                for (FSDatasetInterface.FSVolumeInterface vol : volumeList) {
                    if (vol == null) continue;
                    newVols.add(vol);
                }
                this.volumes = Collections.unmodifiableList(newVols);
                DataNode.LOG.info((Object)("Completed FSVolumeSet.checkDirs. Removed " + removedVols.size() + " volumes. List of current volumes: " + this));
            }
            return removedVols;
        }

        public String toString() {
            return this.volumes.toString();
        }

        private void addBlockPool(String bpid, Configuration conf) throws IOException {
            for (FSDatasetInterface.FSVolumeInterface v : this.volumes) {
                ((FSVolume)v).addBlockPool(bpid, conf);
            }
        }

        private void removeBlockPool(String bpid) {
            for (FSDatasetInterface.FSVolumeInterface v : this.volumes) {
                ((FSVolume)v).shutdownBlockPool(bpid);
            }
        }

        private void shutdown() {
            for (FSDatasetInterface.FSVolumeInterface volume : this.volumes) {
                if (volume == null) continue;
                ((FSVolume)volume).shutdown();
            }
        }
    }

    class FSVolume
    implements FSDatasetInterface.FSVolumeInterface {
        private final Map<String, BlockPoolSlice> map = new HashMap<String, BlockPoolSlice>();
        private final File currentDir;
        private final DF usage;
        private final long reserved;

        FSVolume(File currentDir, Configuration conf) throws IOException {
            this.reserved = conf.getLong("dfs.datanode.du.reserved", 0L);
            this.currentDir = currentDir;
            File parent = currentDir.getParentFile();
            this.usage = new DF(parent, conf);
        }

        File getCurrentDir() {
            return this.currentDir;
        }

        File getRbwDir(String bpid) throws IOException {
            BlockPoolSlice bp = this.getBlockPoolSlice(bpid);
            return bp.getRbwDir();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void decDfsUsed(String bpid, long value) {
            FSDataset fSDataset = FSDataset.this;
            synchronized (fSDataset) {
                BlockPoolSlice bp = this.map.get(bpid);
                if (bp != null) {
                    bp.decDfsUsed(value);
                }
            }
        }

        long getDfsUsed() throws IOException {
            long dfsUsed = 0L;
            Set<Map.Entry<String, BlockPoolSlice>> set = this.map.entrySet();
            for (Map.Entry<String, BlockPoolSlice> entry : set) {
                dfsUsed += entry.getValue().getDfsUsed();
            }
            return dfsUsed;
        }

        long getBlockPoolUsed(String bpid) throws IOException {
            return this.getBlockPoolSlice(bpid).getDfsUsed();
        }

        long getCapacity() throws IOException {
            long remaining = this.usage.getCapacity() - this.reserved;
            return remaining > 0L ? remaining : 0L;
        }

        @Override
        public long getAvailable() throws IOException {
            long available;
            long remaining = this.getCapacity() - this.getDfsUsed();
            if (remaining > (available = this.usage.getAvailable())) {
                remaining = available;
            }
            return remaining > 0L ? remaining : 0L;
        }

        @Override
        public String getBasePath() {
            return this.currentDir.getParent();
        }

        long getReserved() {
            return this.reserved;
        }

        String getMount() throws IOException {
            return this.usage.getMount();
        }

        private BlockPoolSlice getBlockPoolSlice(String bpid) throws IOException {
            BlockPoolSlice bp = this.map.get(bpid);
            if (bp == null) {
                throw new IOException("block pool " + bpid + " is not found");
            }
            return bp;
        }

        @Override
        public File getDirectory(String bpid) throws IOException {
            return this.getBlockPoolSlice(bpid).getDirectory();
        }

        @Override
        public File getFinalizedDir(String bpid) throws IOException {
            return this.getBlockPoolSlice(bpid).getFinalizedDir();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public String[] getBlockPoolList() {
            FSDataset fSDataset = FSDataset.this;
            synchronized (fSDataset) {
                return this.map.keySet().toArray(new String[this.map.keySet().size()]);
            }
        }

        File createTmpFile(String bpid, Block b) throws IOException {
            BlockPoolSlice bp = this.getBlockPoolSlice(bpid);
            return bp.createTmpFile(b);
        }

        File createRbwFile(String bpid, Block b) throws IOException {
            BlockPoolSlice bp = this.getBlockPoolSlice(bpid);
            return bp.createRbwFile(b);
        }

        File addBlock(String bpid, Block b, File f) throws IOException {
            BlockPoolSlice bp = this.getBlockPoolSlice(bpid);
            return bp.addBlock(b, f);
        }

        void checkDirs() throws DiskChecker.DiskErrorException {
            Set<Map.Entry<String, BlockPoolSlice>> set = this.map.entrySet();
            for (Map.Entry<String, BlockPoolSlice> entry : set) {
                entry.getValue().checkDirs();
            }
        }

        void getVolumeMap(ReplicasMap volumeMap) throws IOException {
            Set<Map.Entry<String, BlockPoolSlice>> set = this.map.entrySet();
            for (Map.Entry<String, BlockPoolSlice> entry : set) {
                entry.getValue().getVolumeMap(volumeMap);
            }
        }

        void getVolumeMap(String bpid, ReplicasMap volumeMap) throws IOException {
            BlockPoolSlice bp = this.getBlockPoolSlice(bpid);
            bp.getVolumeMap(volumeMap);
        }

        private void addToReplicasMap(String bpid, ReplicasMap volumeMap, File dir, boolean isFinalized) throws IOException {
            BlockPoolSlice bp = this.getBlockPoolSlice(bpid);
            bp.addToReplicasMap(volumeMap, dir, isFinalized);
        }

        void clearPath(String bpid, File f) throws IOException {
            BlockPoolSlice bp = this.getBlockPoolSlice(bpid);
            bp.clearPath(f);
        }

        public String toString() {
            return this.currentDir.getAbsolutePath();
        }

        public void shutdown() {
            Set<Map.Entry<String, BlockPoolSlice>> set = this.map.entrySet();
            for (Map.Entry<String, BlockPoolSlice> entry : set) {
                entry.getValue().shutdown();
            }
        }

        public void addBlockPool(String bpid, Configuration conf) throws IOException {
            File bpdir = new File(this.currentDir, bpid);
            BlockPoolSlice bp = new BlockPoolSlice(bpid, this, bpdir, conf);
            this.map.put(bpid, bp);
        }

        public void shutdownBlockPool(String bpid) {
            BlockPoolSlice bp = this.map.get(bpid);
            if (bp != null) {
                bp.shutdown();
            }
            this.map.remove(bpid);
        }

        private boolean isBPDirEmpty(String bpid) throws IOException {
            File volumeCurrentDir = this.getCurrentDir();
            File bpDir = new File(volumeCurrentDir, bpid);
            File bpCurrentDir = new File(bpDir, "current");
            File finalizedDir = new File(bpCurrentDir, "finalized");
            File rbwDir = new File(bpCurrentDir, "rbw");
            if (finalizedDir.exists() && FileUtil.list((File)finalizedDir).length != 0) {
                return false;
            }
            return !rbwDir.exists() || FileUtil.list((File)rbwDir).length == 0;
        }

        private void deleteBPDirectories(String bpid, boolean force) throws IOException {
            File volumeCurrentDir = this.getCurrentDir();
            File bpDir = new File(volumeCurrentDir, bpid);
            if (!bpDir.isDirectory()) {
                return;
            }
            File tmpDir = new File(bpDir, "tmp");
            File bpCurrentDir = new File(bpDir, "current");
            File finalizedDir = new File(bpCurrentDir, "finalized");
            File rbwDir = new File(bpCurrentDir, "rbw");
            if (force) {
                FileUtil.fullyDelete((File)bpDir);
            } else {
                if (!rbwDir.delete()) {
                    throw new IOException("Failed to delete " + rbwDir);
                }
                if (!finalizedDir.delete()) {
                    throw new IOException("Failed to delete " + finalizedDir);
                }
                FileUtil.fullyDelete((File)tmpDir);
                for (File f : FileUtil.listFiles((File)bpCurrentDir)) {
                    if (f.delete()) continue;
                    throw new IOException("Failed to delete " + f);
                }
                if (!bpCurrentDir.delete()) {
                    throw new IOException("Failed to delete " + bpCurrentDir);
                }
                for (File f : FileUtil.listFiles((File)bpDir)) {
                    if (f.delete()) continue;
                    throw new IOException("Failed to delete " + f);
                }
                if (!bpDir.delete()) {
                    throw new IOException("Failed to delete " + bpDir);
                }
            }
        }
    }

    private class BlockPoolSlice {
        private final String bpid;
        private final FSVolume volume;
        private final File currentDir;
        private final FSDir finalizedDir;
        private final File rbwDir;
        private final File tmpDir;
        private final DU dfsUsage;

        BlockPoolSlice(String bpid, FSVolume volume, File bpDir, Configuration conf) throws IOException {
            this.bpid = bpid;
            this.volume = volume;
            this.currentDir = new File(bpDir, "current");
            File finalizedDir = new File(this.currentDir, "finalized");
            this.tmpDir = new File(bpDir, "tmp");
            if (this.tmpDir.exists()) {
                FileUtil.fullyDelete((File)this.tmpDir);
            }
            this.rbwDir = new File(this.currentDir, "rbw");
            if (this.rbwDir.exists() && !FSDataset.this.supportAppends) {
                FileUtil.fullyDelete((File)this.rbwDir);
            }
            this.finalizedDir = new FSDir(finalizedDir);
            if (!this.rbwDir.mkdirs() && !this.rbwDir.isDirectory()) {
                throw new IOException("Mkdirs failed to create " + this.rbwDir.toString());
            }
            if (!this.tmpDir.mkdirs() && !this.tmpDir.isDirectory()) {
                throw new IOException("Mkdirs failed to create " + this.tmpDir.toString());
            }
            this.dfsUsage = new DU(bpDir, conf);
            this.dfsUsage.start();
        }

        File getDirectory() {
            return this.currentDir.getParentFile();
        }

        File getFinalizedDir() {
            return this.finalizedDir.dir;
        }

        File getRbwDir() {
            return this.rbwDir;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void decDfsUsed(long value) {
            FSDataset fSDataset = FSDataset.this;
            synchronized (fSDataset) {
                this.dfsUsage.decDfsUsed(value);
            }
        }

        long getDfsUsed() throws IOException {
            return this.dfsUsage.getUsed();
        }

        File createTmpFile(Block b) throws IOException {
            File f = new File(this.tmpDir, b.getBlockName());
            return DatanodeUtil.createTmpFile(b, f);
        }

        File createRbwFile(Block b) throws IOException {
            File f = new File(this.rbwDir, b.getBlockName());
            return DatanodeUtil.createTmpFile(b, f);
        }

        File addBlock(Block b, File f) throws IOException {
            File blockFile = this.finalizedDir.addBlock(b, f);
            File metaFile = DatanodeUtil.getMetaFile(blockFile, b.getGenerationStamp());
            this.dfsUsage.incDfsUsed(b.getNumBytes() + metaFile.length());
            return blockFile;
        }

        void checkDirs() throws DiskChecker.DiskErrorException {
            this.finalizedDir.checkDirTree();
            DiskChecker.checkDir((File)this.tmpDir);
            DiskChecker.checkDir((File)this.rbwDir);
        }

        void getVolumeMap(ReplicasMap volumeMap) throws IOException {
            this.finalizedDir.getVolumeMap(this.bpid, volumeMap, this.volume);
            this.addToReplicasMap(volumeMap, this.rbwDir, false);
        }

        private void addToReplicasMap(ReplicasMap volumeMap, File dir, boolean isFinalized) throws IOException {
            File[] blockFiles;
            for (File blockFile : blockFiles = FileUtil.listFiles((File)dir)) {
                if (!Block.isBlockFilename(blockFile)) continue;
                long genStamp = FSDataset.getGenerationStampFromFile(blockFiles, blockFile);
                long blockId = Block.filename2id(blockFile.getName());
                ReplicaInfo newReplica = null;
                newReplica = isFinalized ? new FinalizedReplica(blockId, blockFile.length(), genStamp, this.volume, blockFile.getParentFile()) : new ReplicaWaitingToBeRecovered(blockId, this.validateIntegrityAndSetLength(blockFile, genStamp), genStamp, this.volume, blockFile.getParentFile());
                ReplicaInfo oldReplica = volumeMap.add(this.bpid, newReplica);
                if (oldReplica == null) continue;
                DataNode.LOG.warn((Object)("Two block files with the same block id exist on disk: " + oldReplica.getBlockFile() + " and " + blockFile));
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        private long validateIntegrityAndSetLength(File blockFile, long genStamp) {
            long l;
            FileInputStream blockIn;
            DataInputStream checksumIn;
            block16: {
                long l2;
                block14: {
                    checksumIn = null;
                    blockIn = null;
                    try {
                        int checksumSize;
                        DataChecksum checksum;
                        int bytesPerChecksum;
                        long numChunks;
                        File metaFile = DatanodeUtil.getMetaFile(blockFile, genStamp);
                        long blockFileLen = blockFile.length();
                        long metaFileLen = metaFile.length();
                        int crcHeaderLen = DataChecksum.getChecksumHeaderSize();
                        if (!blockFile.exists() || blockFileLen == 0L || !metaFile.exists() || metaFileLen < (long)crcHeaderLen) {
                            long l3 = 0L;
                            return l3;
                        }
                        checksumIn = new DataInputStream(new BufferedInputStream(new FileInputStream(metaFile), HdfsConstants.IO_FILE_BUFFER_SIZE));
                        BlockMetadataHeader header = BlockMetadataHeader.readHeader(checksumIn);
                        short version = header.getVersion();
                        if (version != 1) {
                            DataNode.LOG.warn((Object)("Wrong version (" + version + ") for metadata file " + metaFile + " ignoring ..."));
                        }
                        if ((numChunks = Math.min((blockFileLen + (long)(bytesPerChecksum = (checksum = header.getChecksum()).getBytesPerChecksum()) - 1L) / (long)bytesPerChecksum, (metaFileLen - (long)crcHeaderLen) / (long)(checksumSize = checksum.getChecksumSize()))) == 0L) {
                            l2 = 0L;
                            IOUtils.closeStream((Closeable)checksumIn);
                            break block14;
                        }
                        IOUtils.skipFully((InputStream)checksumIn, (long)((numChunks - 1L) * (long)checksumSize));
                        blockIn = new FileInputStream(blockFile);
                        long lastChunkStartPos = (numChunks - 1L) * (long)bytesPerChecksum;
                        IOUtils.skipFully((InputStream)blockIn, (long)lastChunkStartPos);
                        int lastChunkSize = (int)Math.min((long)bytesPerChecksum, blockFileLen - lastChunkStartPos);
                        byte[] buf = new byte[lastChunkSize + checksumSize];
                        checksumIn.readFully(buf, lastChunkSize, checksumSize);
                        IOUtils.readFully((InputStream)blockIn, (byte[])buf, (int)0, (int)lastChunkSize);
                        checksum.update(buf, 0, lastChunkSize);
                        long validFileLength = checksum.compare(buf, lastChunkSize) ? lastChunkStartPos + (long)lastChunkSize : lastChunkStartPos;
                        if (blockFile.length() > validFileLength) {
                            RandomAccessFile blockRAF = new RandomAccessFile(blockFile, "rw");
                            try {
                                blockRAF.setLength(validFileLength);
                            }
                            finally {
                                blockRAF.close();
                            }
                        }
                        l = validFileLength;
                        IOUtils.closeStream((Closeable)checksumIn);
                        break block16;
                    }
                    catch (IOException e) {
                        DataNode.LOG.warn((Object)e);
                        long l4 = 0L;
                        return l4;
                    }
                }
                IOUtils.closeStream(blockIn);
                return l2;
            }
            IOUtils.closeStream((Closeable)blockIn);
            return l;
            finally {
                IOUtils.closeStream(checksumIn);
                IOUtils.closeStream(blockIn);
            }
        }

        void clearPath(File f) {
            this.finalizedDir.clearPath(f);
        }

        public String toString() {
            return this.currentDir.getAbsolutePath();
        }

        public void shutdown() {
            this.dfsUsage.shutdown();
        }
    }

    private class FSDir {
        final File dir;
        int numBlocks = 0;
        FSDir[] children;
        int lastChildIdx = 0;

        private FSDir(File dir) throws IOException {
            this.dir = dir;
            this.children = null;
            if (!dir.exists()) {
                if (!dir.mkdirs()) {
                    throw new IOException("Mkdirs failed to create " + dir.toString());
                }
            } else {
                File[] files = FileUtil.listFiles((File)dir);
                ArrayList<FSDir> dirList = new ArrayList<FSDir>();
                for (int idx = 0; idx < files.length; ++idx) {
                    if (files[idx].isDirectory()) {
                        dirList.add(new FSDir(files[idx]));
                        continue;
                    }
                    if (!Block.isBlockFilename(files[idx])) continue;
                    ++this.numBlocks;
                }
                if (dirList.size() > 0) {
                    this.children = dirList.toArray(new FSDir[dirList.size()]);
                }
            }
        }

        private File addBlock(Block b, File src) throws IOException {
            File file = this.addBlock(b, src, false, false);
            return file != null ? file : this.addBlock(b, src, true, true);
        }

        private File addBlock(Block b, File src, boolean createOk, boolean resetIdx) throws IOException {
            if (this.numBlocks < FSDataset.this.maxBlocksPerDir) {
                File dest = FSDataset.moveBlockFiles(b, src, this.dir);
                ++this.numBlocks;
                return dest;
            }
            if (this.lastChildIdx < 0 && resetIdx) {
                this.lastChildIdx = DFSUtil.getRandom().nextInt(this.children.length);
            }
            if (this.lastChildIdx >= 0 && this.children != null) {
                for (int i = 0; i < this.children.length; ++i) {
                    int idx = (this.lastChildIdx + i) % this.children.length;
                    File file = this.children[idx].addBlock(b, src, false, resetIdx);
                    if (file == null) continue;
                    this.lastChildIdx = idx;
                    return file;
                }
                this.lastChildIdx = -1;
            }
            if (!createOk) {
                return null;
            }
            if (this.children == null || this.children.length == 0) {
                this.children = new FSDir[FSDataset.this.maxBlocksPerDir];
                for (int idx = 0; idx < FSDataset.this.maxBlocksPerDir; ++idx) {
                    this.children[idx] = new FSDir(new File(this.dir, "subdir" + idx));
                }
            }
            this.lastChildIdx = DFSUtil.getRandom().nextInt(this.children.length);
            return this.children[this.lastChildIdx].addBlock(b, src, true, false);
        }

        private void getVolumeMap(String bpid, ReplicasMap volumeMap, FSVolume volume) throws IOException {
            if (this.children != null) {
                for (int i = 0; i < this.children.length; ++i) {
                    this.children[i].getVolumeMap(bpid, volumeMap, volume);
                }
            }
            this.recoverTempUnlinkedBlock();
            volume.addToReplicasMap(bpid, volumeMap, this.dir, true);
        }

        private void recoverTempUnlinkedBlock() throws IOException {
            File[] files;
            for (File file : files = FileUtil.listFiles((File)this.dir)) {
                if (!FSDataset.isUnlinkTmpFile(file)) continue;
                File blockFile = FSDataset.getOrigFile(file);
                if (blockFile.exists()) {
                    if (file.delete()) continue;
                    throw new IOException("Unable to cleanup unlinked tmp file " + file);
                }
                if (file.renameTo(blockFile)) continue;
                throw new IOException("Unable to cleanup detached file " + file);
            }
        }

        private void checkDirTree() throws DiskChecker.DiskErrorException {
            DiskChecker.checkDir((File)this.dir);
            if (this.children != null) {
                for (int i = 0; i < this.children.length; ++i) {
                    this.children[i].checkDirTree();
                }
            }
        }

        private void clearPath(File f) {
            String[] dirNames;
            String root = this.dir.getAbsolutePath();
            String dir = f.getAbsolutePath();
            if (dir.startsWith(root) && this.clearPath(f, dirNames = dir.substring(root.length()).split(File.separator + "subdir"), 1)) {
                return;
            }
            this.clearPath(f, null, -1);
        }

        private boolean clearPath(File f, String[] dirNames, int idx) {
            if ((dirNames == null || idx == dirNames.length) && this.dir.compareTo(f) == 0) {
                --this.numBlocks;
                return true;
            }
            if (dirNames != null) {
                int childIdx;
                if (idx > dirNames.length - 1 || this.children == null) {
                    return false;
                }
                try {
                    childIdx = Integer.parseInt(dirNames[idx]);
                }
                catch (NumberFormatException ignored) {
                    return false;
                }
                return childIdx >= 0 && childIdx < this.children.length ? this.children[childIdx].clearPath(f, dirNames, idx + 1) : false;
            }
            if (this.children != null) {
                for (int i = 0; i < this.children.length; ++i) {
                    if (!this.children[i].clearPath(f, null, -1)) continue;
                    return true;
                }
            }
            return false;
        }

        public String toString() {
            return "FSDir{dir=" + this.dir + ", children=" + (this.children == null ? null : Arrays.asList(this.children)) + "}";
        }
    }

    static class Factory
    extends FSDatasetInterface.Factory {
        Factory() {
        }

        @Override
        public FSDatasetInterface createFSDatasetInterface(DataNode datanode, DataStorage storage, Configuration conf) throws IOException {
            return new FSDataset(datanode, storage, conf);
        }
    }
}

