/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode;

import java.io.BufferedReader;
import java.io.Closeable;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.server.datanode.BlockSender;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.FSDatasetInterface;
import org.apache.hadoop.hdfs.util.DataTransferThrottler;
import org.apache.hadoop.io.IOUtils;

class BlockPoolSliceScanner {
    public static final Log LOG = LogFactory.getLog(BlockPoolSliceScanner.class);
    private static final int MAX_SCAN_RATE = 0x800000;
    private static final int MIN_SCAN_RATE = 0x100000;
    static final long DEFAULT_SCAN_PERIOD_HOURS = 504L;
    private final String blockPoolId;
    private static final String dateFormatString = "yyyy-MM-dd HH:mm:ss,SSS";
    static final String verificationLogFile = "dncp_block_verification.log";
    static final int verficationLogLimit = 5;
    private long scanPeriod = 1814400000L;
    private DataNode datanode;
    private final FSDatasetInterface dataset;
    private TreeSet<BlockScanInfo> blockInfoSet;
    private HashMap<Block, BlockScanInfo> blockMap;
    private HashMap<Long, Integer> processedBlocks;
    private long totalScans = 0L;
    private long totalScanErrors = 0L;
    private long totalTransientErrors = 0L;
    private long totalBlocksScannedInLastRun = 0L;
    private long currentPeriodStart = System.currentTimeMillis();
    private long bytesLeft = 0L;
    private long totalBytesToScan = 0L;
    private LogFileHandler verificationLog;
    private DataTransferThrottler throttler = null;

    BlockPoolSliceScanner(DataNode datanode, FSDatasetInterface dataset, Configuration conf, String bpid) {
        this.datanode = datanode;
        this.dataset = dataset;
        this.blockPoolId = bpid;
        this.scanPeriod = conf.getInt("dfs.datanode.scan.period.hours", 0);
        if (this.scanPeriod <= 0L) {
            this.scanPeriod = 504L;
        }
        this.scanPeriod *= 3600000L;
        LOG.info((Object)("Periodic Block Verification scan initialized with interval " + this.scanPeriod + "."));
    }

    String getBlockPoolId() {
        return this.blockPoolId;
    }

    synchronized boolean isInitialized() {
        return this.throttler != null;
    }

    private void updateBytesToScan(long len, long lastScanTime) {
        this.totalBytesToScan += len;
        if (lastScanTime < this.currentPeriodStart) {
            this.bytesLeft += len;
        }
    }

    private synchronized void addBlockInfo(BlockScanInfo info) {
        boolean added = this.blockInfoSet.add(info);
        this.blockMap.put(info.block, info);
        if (added) {
            this.updateBytesToScan(info.block.getNumBytes(), info.lastScanTime);
        }
    }

    private synchronized void delBlockInfo(BlockScanInfo info) {
        boolean exists = this.blockInfoSet.remove(info);
        this.blockMap.remove(info.block);
        if (exists) {
            this.updateBytesToScan(-info.block.getNumBytes(), info.lastScanTime);
        }
    }

    private synchronized void updateBlockInfo(LogEntry e) {
        BlockScanInfo info = this.blockMap.get(new Block(e.blockId, 0L, e.genStamp));
        if (info != null && e.verificationTime > 0L && info.lastScanTime < e.verificationTime) {
            this.delBlockInfo(info);
            info.lastScanTime = e.verificationTime;
            info.lastScanType = ScanType.VERIFICATION_SCAN;
            this.addBlockInfo(info);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void init() throws IOException {
        List<Block> arr = this.dataset.getFinalizedBlocks(this.blockPoolId);
        Collections.shuffle(arr);
        this.blockInfoSet = new TreeSet();
        this.blockMap = new HashMap();
        long scanTime = -1L;
        for (Block block : arr) {
            BlockScanInfo info = new BlockScanInfo(block);
            --scanTime;
            info.lastScanTime = info.lastScanTime;
            this.addBlockInfo(info);
        }
        File dir = null;
        List<FSDatasetInterface.FSVolumeInterface> volumes = this.dataset.getVolumes();
        for (FSDatasetInterface.FSVolumeInterface vol : volumes) {
            File bpDir = vol.getDirectory(this.blockPoolId);
            if (!LogFileHandler.isFilePresent(bpDir, verificationLogFile)) continue;
            dir = bpDir;
            break;
        }
        if (dir == null) {
            dir = volumes.get(0).getDirectory(this.blockPoolId);
        }
        try {
            this.verificationLog = new LogFileHandler(dir, verificationLogFile, 100);
        }
        catch (IOException e) {
            LOG.warn((Object)"Could not open verfication log. Verification times are not stored.");
        }
        BlockPoolSliceScanner blockPoolSliceScanner = this;
        synchronized (blockPoolSliceScanner) {
            this.throttler = new DataTransferThrottler(200L, 0x800000L);
        }
    }

    private synchronized long getNewBlockScanTime() {
        long period = Math.min(this.scanPeriod, (long)(Math.max(this.blockMap.size(), 1) * 600) * 1000L);
        int periodInt = Math.abs((int)period);
        return System.currentTimeMillis() - this.scanPeriod + (long)DFSUtil.getRandom().nextInt(periodInt);
    }

    synchronized void addBlock(ExtendedBlock block) {
        if (!this.isInitialized()) {
            return;
        }
        BlockScanInfo info = this.blockMap.get(block.getLocalBlock());
        if (info != null) {
            LOG.warn((Object)("Adding an already existing block " + block));
            this.delBlockInfo(info);
        }
        info = new BlockScanInfo(block.getLocalBlock());
        info.lastScanTime = this.getNewBlockScanTime();
        this.addBlockInfo(info);
        this.adjustThrottler();
    }

    synchronized void deleteBlock(Block block) {
        if (!this.isInitialized()) {
            return;
        }
        BlockScanInfo info = this.blockMap.get(block);
        if (info != null) {
            this.delBlockInfo(info);
        }
    }

    synchronized long getLastScanTime(Block block) {
        if (!this.isInitialized()) {
            return 0L;
        }
        BlockScanInfo info = this.blockMap.get(block);
        return info == null ? 0L : info.lastScanTime;
    }

    void deleteBlocks(Block[] blocks) {
        for (Block b : blocks) {
            this.deleteBlock(b);
        }
    }

    private synchronized void updateScanStatus(Block block, ScanType type, boolean scanOk) {
        if (!this.isInitialized()) {
            return;
        }
        BlockScanInfo info = this.blockMap.get(block);
        if (info != null) {
            this.delBlockInfo(info);
        } else {
            info = new BlockScanInfo(block);
        }
        long now = System.currentTimeMillis();
        info.lastScanType = type;
        info.lastScanTime = now;
        info.lastScanOk = scanOk;
        this.addBlockInfo(info);
        if (!scanOk) {
            return;
        }
        LogFileHandler log = this.verificationLog;
        if (log != null) {
            log.appendLine(now, block.getGenerationStamp(), block.getBlockId());
        }
    }

    private void handleScanFailure(ExtendedBlock block) {
        LOG.info((Object)("Reporting bad block " + block));
        try {
            this.datanode.reportBadBlocks(block);
        }
        catch (IOException ie) {
            LOG.warn((Object)("Cannot report bad block=" + block.getBlockId()));
        }
    }

    private synchronized void adjustThrottler() {
        long timeLeft = this.currentPeriodStart + this.scanPeriod - System.currentTimeMillis();
        long bw = Math.max(this.bytesLeft * 1000L / timeLeft, 0x100000L);
        this.throttler.setBandwidth(Math.min(bw, 0x800000L));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifyBlock(ExtendedBlock block) {
        BlockSender blockSender = null;
        for (int i = 0; i < 2; ++i) {
            boolean second = i > 0;
            try {
                this.adjustThrottler();
                blockSender = new BlockSender(block, 0L, -1L, false, true, this.datanode, null);
                DataOutputStream out = new DataOutputStream((OutputStream)new IOUtils.NullOutputStream());
                blockSender.sendBlock(out, null, this.throttler);
                LOG.info((Object)((second ? "Second " : "") + "Verification succeeded for " + block));
                if (second) {
                    ++this.totalTransientErrors;
                }
                this.updateScanStatus(block.getLocalBlock(), ScanType.VERIFICATION_SCAN, true);
            }
            catch (IOException e) {
                block10: {
                    block9: {
                        block8: {
                            try {
                                this.updateScanStatus(block.getLocalBlock(), ScanType.VERIFICATION_SCAN, false);
                                if (this.dataset.contains(block)) break block8;
                                LOG.info((Object)(block + " is no longer in the dataset."));
                                this.deleteBlock(block.getLocalBlock());
                            }
                            catch (Throwable throwable) {
                                IOUtils.closeStream(blockSender);
                                this.datanode.getMetrics().incrBlocksVerified();
                                ++this.totalScans;
                                throw throwable;
                            }
                            IOUtils.closeStream((Closeable)blockSender);
                            this.datanode.getMetrics().incrBlocksVerified();
                            ++this.totalScans;
                            return;
                        }
                        if (!(e instanceof FileNotFoundException)) break block9;
                        LOG.info((Object)("Verification failed for " + block + ". It may be due to race with write."));
                        this.deleteBlock(block.getLocalBlock());
                        IOUtils.closeStream((Closeable)blockSender);
                        this.datanode.getMetrics().incrBlocksVerified();
                        ++this.totalScans;
                        return;
                    }
                    LOG.warn((Object)((second ? "Second " : "First ") + "Verification failed for " + block), (Throwable)e);
                    if (!second) break block10;
                    ++this.totalScanErrors;
                    this.datanode.getMetrics().incrBlockVerificationFailures();
                    this.handleScanFailure(block);
                    IOUtils.closeStream((Closeable)blockSender);
                    this.datanode.getMetrics().incrBlocksVerified();
                    ++this.totalScans;
                    return;
                }
                IOUtils.closeStream((Closeable)blockSender);
                this.datanode.getMetrics().incrBlocksVerified();
                ++this.totalScans;
                continue;
            }
            IOUtils.closeStream((Closeable)blockSender);
            this.datanode.getMetrics().incrBlocksVerified();
            ++this.totalScans;
            return;
        }
    }

    private synchronized long getEarliestScanTime() {
        if (!this.blockInfoSet.isEmpty()) {
            return this.blockInfoSet.first().lastScanTime;
        }
        return Long.MAX_VALUE;
    }

    private synchronized boolean isFirstBlockProcessed() {
        long blockId;
        return !this.blockInfoSet.isEmpty() && this.processedBlocks.get(blockId = this.blockInfoSet.first().block.getBlockId()) != null && this.processedBlocks.get(blockId) == 1;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void verifyFirstBlock() {
        Block block = null;
        BlockPoolSliceScanner blockPoolSliceScanner = this;
        synchronized (blockPoolSliceScanner) {
            if (!this.blockInfoSet.isEmpty()) {
                block = this.blockInfoSet.first().block;
            }
        }
        if (block != null) {
            this.verifyBlock(new ExtendedBlock(this.blockPoolId, block));
            this.processedBlocks.put(block.getBlockId(), 1);
        }
    }

    long getBlocksScannedInLastRun() {
        return this.totalBlocksScannedInLastRun;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean assignInitialVerificationTimes() {
        BlockScanInfo info;
        int numBlocks = 1;
        LogFileHandler log = null;
        BlockPoolSliceScanner blockPoolSliceScanner = this;
        synchronized (blockPoolSliceScanner) {
            log = this.verificationLog;
            numBlocks = Math.max(this.blockMap.size(), 1);
        }
        long now = System.currentTimeMillis();
        LogFileHandler.Reader[] logReader = new LogFileHandler.Reader[2];
        try {
            if (log != null) {
                logReader[0] = log.getCurrentFileReader();
                logReader[1] = log.getPreviousFileReader();
            }
        }
        catch (IOException e) {
            LOG.warn((Object)"Could not read previous verification times", (Throwable)e);
        }
        try {
            for (LogFileHandler.Reader reader : logReader) {
                while (logReader != null && reader.hasNext()) {
                    if (!this.datanode.shouldRun || this.datanode.blockScanner.blockScannerThread.isInterrupted()) {
                        boolean bl = false;
                        return bl;
                    }
                    LogEntry entry = LogEntry.parseEntry(reader.next());
                    if (entry == null) continue;
                    this.updateBlockInfo(entry);
                    if (now - entry.verificationTime >= this.scanPeriod || (info = this.blockMap.get(new Block(entry.blockId, 0L, entry.genStamp))) == null) continue;
                    if (this.processedBlocks.get(entry.blockId) == null) {
                        this.updateBytesLeft(-info.block.getNumBytes());
                        this.processedBlocks.put(entry.blockId, 1);
                    }
                    if (reader.file != log.prevFile) continue;
                    log.appendLine(entry.verificationTime, entry.genStamp, entry.blockId);
                }
            }
        }
        finally {
            IOUtils.closeStream((Closeable)logReader[0]);
            IOUtils.closeStream((Closeable)logReader[1]);
        }
        long verifyInterval = (long)Math.min((double)this.scanPeriod / 2.0 / (double)numBlocks, 600000.0);
        long lastScanTime = System.currentTimeMillis() - this.scanPeriod;
        BlockPoolSliceScanner blockPoolSliceScanner2 = this;
        synchronized (blockPoolSliceScanner2) {
            if (!this.blockInfoSet.isEmpty()) {
                while (true) {
                    info = this.blockInfoSet.first();
                    if (info.lastScanTime >= 0L) break;
                    this.delBlockInfo(info);
                    info.lastScanTime = lastScanTime;
                    lastScanTime += verifyInterval;
                    this.addBlockInfo(info);
                }
            }
        }
        return true;
    }

    private synchronized void updateBytesLeft(long len) {
        this.bytesLeft += len;
    }

    static File getCurrentFile(FSDatasetInterface.FSVolumeInterface vol, String bpid) throws IOException {
        return LogFileHandler.getCurrentFile(vol.getDirectory(bpid), verificationLogFile);
    }

    private synchronized void startNewPeriod() {
        LOG.info((Object)("Starting a new period : work left in prev period : " + String.format("%.2f%%", this.totalBytesToScan == 0L ? 0.0 : (double)this.bytesLeft * 100.0 / (double)this.totalBytesToScan)));
        this.bytesLeft = this.totalBytesToScan;
        this.currentPeriodStart = System.currentTimeMillis();
    }

    void scanBlockPoolSlice() {
        this.startNewPeriod();
        if (this.processedBlocks != null) {
            this.totalBlocksScannedInLastRun = this.processedBlocks.size();
        }
        this.processedBlocks = new HashMap();
        if (this.verificationLog != null) {
            try {
                this.verificationLog.openCurFile();
            }
            catch (FileNotFoundException ex) {
                LOG.warn((Object)"Could not open current file");
            }
        }
        if (!this.assignInitialVerificationTimes()) {
            return;
        }
        this.scan();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void scan() {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Starting to scan blockpool: " + this.blockPoolId));
        }
        try {
            this.adjustThrottler();
            while (this.datanode.shouldRun && !Thread.interrupted() && this.datanode.isBPServiceAlive(this.blockPoolId)) {
                long now = System.currentTimeMillis();
                BlockPoolSliceScanner blockPoolSliceScanner = this;
                synchronized (blockPoolSliceScanner) {
                    if (now >= this.currentPeriodStart + this.scanPeriod) {
                        this.startNewPeriod();
                    }
                }
                if (now - this.getEarliestScanTime() >= this.scanPeriod || !this.blockInfoSet.isEmpty() && !this.isFirstBlockProcessed()) {
                    this.verifyFirstBlock();
                    continue;
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)"All remaining blocks were processed recently, so this run is complete");
                }
                break;
            }
        }
        catch (RuntimeException e) {
            LOG.warn((Object)"RuntimeException during BlockPoolScanner.scan()", (Throwable)e);
            throw e;
        }
        finally {
            this.cleanUp();
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Done scanning block pool: " + this.blockPoolId));
            }
        }
    }

    private synchronized void cleanUp() {
        if (this.verificationLog != null) {
            try {
                this.verificationLog.roll();
            }
            catch (IOException ex) {
                LOG.warn((Object)"Received exception: ", (Throwable)ex);
                this.verificationLog.close();
            }
        }
    }

    synchronized void printBlockReport(StringBuilder buffer, boolean summaryOnly) {
        long oneHour = 3600000L;
        long oneDay = 24L * oneHour;
        long oneWeek = 7L * oneDay;
        long fourWeeks = 4L * oneWeek;
        int inOneHour = 0;
        int inOneDay = 0;
        int inOneWeek = 0;
        int inFourWeeks = 0;
        int inScanPeriod = 0;
        int neverScanned = 0;
        SimpleDateFormat dateFormat = new SimpleDateFormat(dateFormatString);
        int total = this.blockInfoSet.size();
        long now = System.currentTimeMillis();
        Date date = new Date();
        for (BlockScanInfo info : this.blockInfoSet) {
            long scanTime = info.getLastScanTime();
            long diff = now - scanTime;
            if (diff <= oneHour) {
                ++inOneHour;
            }
            if (diff <= oneDay) {
                ++inOneDay;
            }
            if (diff <= oneWeek) {
                ++inOneWeek;
            }
            if (diff <= fourWeeks) {
                ++inFourWeeks;
            }
            if (diff <= this.scanPeriod) {
                ++inScanPeriod;
            }
            if (scanTime <= 0L) {
                ++neverScanned;
            }
            if (summaryOnly) continue;
            date.setTime(scanTime);
            String scanType = info.lastScanType == ScanType.VERIFICATION_SCAN ? "local" : "none";
            buffer.append(String.format("%-26s : status : %-6s type : %-6s scan time : %-15d %s\n", info.block, info.lastScanOk ? "ok" : "failed", scanType, scanTime, scanTime <= 0L ? "not yet verified" : dateFormat.format(date)));
        }
        double pctPeriodLeft = (double)(this.scanPeriod + this.currentPeriodStart - now) * 100.0 / (double)this.scanPeriod;
        double pctProgress = this.totalBytesToScan == 0L ? 100.0 : (double)(this.totalBytesToScan - this.bytesLeft) * 100.0 / (double)this.totalBytesToScan;
        buffer.append(String.format("\nTotal Blocks                 : %6d\nVerified in last hour        : %6d\nVerified in last day         : %6d\nVerified in last week        : %6d\nVerified in last four weeks  : %6d\nVerified in SCAN_PERIOD      : %6d\nNot yet verified             : %6d\nVerified since restart       : %6d\nScans since restart          : %6d\nScan errors since restart    : %6d\nTransient scan errors        : %6d\nCurrent scan rate limit KBps : %6d\nProgress this period         : %6.0f%%\nTime left in cur period      : %6.2f%%\n", total, inOneHour, inOneDay, inOneWeek, inFourWeeks, inScanPeriod, neverScanned, this.totalScans, this.totalScans, this.totalScanErrors, this.totalTransientErrors, Math.round((double)this.throttler.getBandwidth() / 1024.0), pctProgress, pctPeriodLeft));
    }

    private static class LogFileHandler {
        private static final String curFileSuffix = ".curr";
        private static final String prevFileSuffix = ".prev";
        private final DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss,SSS");
        private File curFile;
        private File prevFile;
        private PrintStream out;

        static File getCurrentFile(File dir, String filePrefix) {
            return new File(dir, filePrefix + curFileSuffix);
        }

        public Reader getPreviousFileReader() throws IOException {
            return new Reader(this.prevFile);
        }

        public Reader getCurrentFileReader() throws IOException {
            return new Reader(this.curFile);
        }

        static boolean isFilePresent(File dir, String filePrefix) {
            return new File(dir, filePrefix + curFileSuffix).exists() || new File(dir, filePrefix + prevFileSuffix).exists();
        }

        LogFileHandler(File dir, String filePrefix, int maxNumLines) throws IOException {
            this.curFile = new File(dir, filePrefix + curFileSuffix);
            this.prevFile = new File(dir, filePrefix + prevFileSuffix);
        }

        synchronized boolean appendLine(String line) {
            if (this.out == null) {
                return false;
            }
            this.out.println();
            this.out.print(line);
            return true;
        }

        boolean appendLine(long verificationTime, long genStamp, long blockId) {
            return this.appendLine("date=\"" + this.dateFormat.format(new Date(verificationTime)) + "\"\t " + "time=\"" + verificationTime + "\"\t " + "genstamp=\"" + genStamp + "\"\t " + "id=\"" + blockId + "\"");
        }

        private synchronized void openCurFile() throws FileNotFoundException {
            this.close();
            this.out = new PrintStream(new FileOutputStream(this.curFile, true));
        }

        private void roll() throws IOException {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Rolling current file: " + this.curFile.getAbsolutePath() + " to previous file: " + this.prevFile.getAbsolutePath()));
            }
            if (!this.prevFile.delete() && this.prevFile.exists()) {
                throw new IOException("Could not delete " + this.prevFile);
            }
            this.close();
            if (!this.curFile.renameTo(this.prevFile)) {
                throw new IOException("Could not rename " + this.curFile + " to " + this.prevFile);
            }
        }

        synchronized void close() {
            if (this.out != null) {
                this.out.close();
                this.out = null;
            }
        }

        private static class Reader
        implements Iterator<String>,
        Closeable {
            BufferedReader reader = null;
            File file;
            String line;
            boolean closed = false;

            private Reader(File file) throws IOException {
                this.file = file;
                this.readNext();
            }

            private boolean openFile() throws IOException {
                if (this.file == null) {
                    return false;
                }
                if (this.reader != null) {
                    this.reader.close();
                    this.reader = null;
                }
                if (this.file.exists()) {
                    this.reader = new BufferedReader(new FileReader(this.file));
                    return true;
                }
                return false;
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            private void readNext() throws IOException {
                this.line = null;
                if (this.reader == null) {
                    this.openFile();
                }
                try {
                    if (this.reader != null && (this.line = this.reader.readLine()) != null) {
                        return;
                    }
                }
                finally {
                    if (!this.hasNext()) {
                        this.close();
                    }
                }
            }

            @Override
            public boolean hasNext() {
                return this.line != null;
            }

            @Override
            public String next() {
                String curLine = this.line;
                try {
                    this.readNext();
                }
                catch (IOException e) {
                    LOG.info((Object)"Could not read next line in LogHandler", (Throwable)e);
                }
                return curLine;
            }

            @Override
            public void remove() {
                throw new RuntimeException("remove() is not supported.");
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void close() throws IOException {
                if (!this.closed) {
                    try {
                        if (this.reader != null) {
                            this.reader.close();
                        }
                    }
                    finally {
                        this.file = null;
                        this.reader = null;
                        this.closed = true;
                    }
                }
            }
        }
    }

    private static class LogEntry {
        long blockId = -1L;
        long verificationTime = -1L;
        long genStamp = 0L;
        private static Pattern entryPattern = Pattern.compile("\\G\\s*([^=\\p{Space}]+)=\"(.*?)\"\\s*");

        private LogEntry() {
        }

        static LogEntry parseEntry(String line) {
            LogEntry entry = new LogEntry();
            Matcher matcher = entryPattern.matcher(line);
            while (matcher.find()) {
                String name = matcher.group(1);
                String value = matcher.group(2);
                try {
                    if (name.equals("id")) {
                        entry.blockId = Long.valueOf(value);
                        continue;
                    }
                    if (name.equals("time")) {
                        entry.verificationTime = Long.valueOf(value);
                        continue;
                    }
                    if (!name.equals("genstamp")) continue;
                    entry.genStamp = Long.valueOf(value);
                }
                catch (NumberFormatException nfe) {
                    LOG.warn((Object)("Cannot parse line: " + line), (Throwable)nfe);
                    return null;
                }
            }
            return entry;
        }
    }

    static class BlockScanInfo
    implements Comparable<BlockScanInfo> {
        Block block;
        long lastScanTime = 0L;
        ScanType lastScanType = ScanType.NONE;
        boolean lastScanOk = true;

        BlockScanInfo(Block block) {
            this.block = block;
        }

        public int hashCode() {
            return this.block.hashCode();
        }

        public boolean equals(Object other) {
            return other instanceof BlockScanInfo && this.compareTo((BlockScanInfo)other) == 0;
        }

        long getLastScanTime() {
            return this.lastScanType == ScanType.NONE ? 0L : this.lastScanTime;
        }

        @Override
        public int compareTo(BlockScanInfo other) {
            long t1 = this.lastScanTime;
            long t2 = other.lastScanTime;
            return t1 < t2 ? -1 : (t1 > t2 ? 1 : this.block.compareTo(other.block));
        }
    }

    private static enum ScanType {
        VERIFICATION_SCAN,
        NONE;

    }
}

