package org.herac.tuxguitar.song.managers;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.herac.tuxguitar.song.models.Measure;
import org.herac.tuxguitar.song.models.MeasureHeader;
import org.herac.tuxguitar.song.models.RGBColor;
import org.herac.tuxguitar.song.models.SongChannel;
import org.herac.tuxguitar.song.models.SongTrack;

public class SongTrackManager {
	private SongManager songManager;
	
	public SongTrackManager(SongManager songManager){
		this.songManager = songManager;
	}
	
	
	
    public int countMeasures(SongTrack track){
    	return (track.getMeasures().size());
    }
	
	   
    public Measure getFirstMeasure(SongTrack track){
        Measure firstMeasure = null;    
        for(int i = 0;i < track.getMeasures().size();i++){
            Measure currMeasure = (Measure)track.getMeasures().get(i);
            if(firstMeasure == null || (currMeasure.getStart() < firstMeasure.getStart())){
                firstMeasure = currMeasure;
            }
        }        
        return firstMeasure;
    }     

    public Measure getLastMeasure(SongTrack track){
        int lastIndex = track.getMeasures().size() - 1; 
        return (Measure)track.getMeasures().get(lastIndex);
    }      
    
    public Measure getPrevMeasure(SongTrack track,Measure measure){
        int prevIndex = measure.getNumber() - 1; 
        if(prevIndex > 0){
            return (Measure)track.getMeasures().get(prevIndex - 1);
        }
        return null;
    }      
    
    public Measure getNextMeasure(SongTrack track,Measure measure){
        int nextIndex = measure.getNumber(); 
        if(nextIndex < track.getMeasures().size()){
            return (Measure)track.getMeasures().get(nextIndex);
        }
        return null;
    }       
  
    public Measure getMeasureAt(SongTrack track,long start){
        Iterator it = track.getMeasures().iterator();
        while(it.hasNext()){
            Measure measure = (Measure)it.next();  
            long measureStart = measure.getStart();
            long measureLength = measure.getLength();
            if(start >= measureStart && start < measureStart + measureLength){
                return measure;
            }
        }     
        return null;
    }           
    
    public Measure getMeasure(SongTrack track,int number){
        for (int i = 0; i < track.getMeasures().size(); i++) {
            Measure measure = (Measure) track.getMeasures().get(i);            
            if(measure.getNumber() == number){
                return measure;
            }
        }     
        return null;
    }   
    
    /**
     * Retorna Todos los desde Start hasta el final del compas
     */
    public List getMeasuresBeforeEnd(SongTrack track,long fromStart) {
        List measures = new ArrayList();        
        Iterator it = track.getMeasures().iterator();
        while(it.hasNext()){           
            Measure currMeasure = (Measure)it.next();
            if (currMeasure.getStart() >= fromStart) {
                measures.add(currMeasure);
            }
        }
        return measures;
    }         
    
    /**
     * Retorna Todos los desde Start hasta el final del compas
     */
    public List getMeasuresBetween(SongTrack track,long p1,long p2) {
        List measures = new ArrayList();        
        Iterator it = track.getMeasures().iterator();
        while(it.hasNext()){           
            Measure currMeasure = (Measure)it.next();
            long start = currMeasure.getStart();
            if (start >= p1 && start <= p2) {
                measures.add(currMeasure);
            }
        }
        return measures;
    }          
    
    public void addNewMeasureBeforeEnd(SongTrack track,MeasureHeader header){        
        addNewMeasureAfter(track,header,getLastMeasure(track));
    }    
    

    

    /**
     * Agrega un Compas
     */
    public void addNewMeasureAfter(SongTrack track,MeasureHeader header,Measure measure){      
        int clef = measure.getClef();
        int keySignature = measure.getKeySignature();
        
        addMeasure(track,new Measure(header, new ArrayList(), new ArrayList(),clef,keySignature));
    }

    
    public List copyMeasures(SongTrack track,long p1,long p2){
        List measures = new ArrayList();
        
        Iterator it = getMeasuresBetween(track,p1,p2).iterator();
        while(it.hasNext()){
            Measure measure = (Measure)it.next();
            measures.add(measure);
        }
        return measures;
    }
    
    /**
     * Agrega un Compas
     */
    public void insertMeasures(SongTrack track,List measures,int fromNumber){
        long start = -1;
        long end = 0;        
        if(!measures.isEmpty()){
            start = ((Measure)measures.get(0)).getStart(); 
            end = ((Measure)measures.get(measures.size() - 1)).getStart() + ((Measure)measures.get(measures.size() - 1)).getLength();

            Iterator it = measures.iterator();
            while(it.hasNext()){
                Measure measure = (Measure)it.next();
                addMeasure(track,(measure.getNumber() - 1),measure);
                fromNumber ++;
            }        
        }
    }    
        
    
    /**
     * Agrega un Compas
     */
    public void addMeasure(SongTrack track,Measure measure){
        track.getMeasures().add(measure);
    }    
    
    /**
     * Agrega un Compas
     */
    public void addMeasure(SongTrack track,int index,Measure measure){
        track.getMeasures().add(index,measure);
    }        

    public void removeLastMeasure(SongTrack track){    
        removeMeasure(track,getLastMeasure(track));
    }
    
    public void removeMeasure(SongTrack track,long start){    
        removeMeasure(track,getMeasureAt(track,start));
    }
    
    public void removeMeasure(SongTrack track,Measure measure){        
        long start = measure.getStart();
        long length = measure.getLength();
    
        track.getMeasures().remove(measure.getNumber() - 1);  
    }    
    
    public Measure replaceMeasure(SongTrack track,Measure newMeasure){    	
    	Measure measure = getMeasureAt(track,newMeasure.getStart());
    	measure.makeEqual(newMeasure);    	

    	return measure;
    }
    
    /**
     * Mueve el compas
     */    
    public void moveMeasure(SongTrack track,Measure measure,long theMove,int numberMove){                
        songManager.getMeasureManager().moveAllNotes(measure,theMove);
        songManager.getMeasureManager().moveAllSilences(measure,theMove);
    }
    
    public void changeKeySignature(SongTrack track,long start,int keySignature,boolean toEnd){
    	changeKeySignature(track,getMeasureAt(track,start),keySignature,toEnd);
    }
    
    /**
     * Cambia el Key Signature
     */
    public void changeKeySignature(SongTrack track,Measure measure,int keySignature,boolean toEnd){        
        //asigno el nuevo Key
        measure.setKeySignature(keySignature);
        
        if(toEnd){
            List measures = getMeasuresBeforeEnd(track,measure.getStart() + 1);
            Iterator it = measures.iterator();
            while(it.hasNext()){
                Measure nextMeasure = (Measure)it.next();
                nextMeasure.setKeySignature(keySignature);
            }            
        }                
    }    
    
    
    public void changeClef(SongTrack track,long start,int clef,boolean toEnd){
    	changeClef(track,getMeasureAt(track,start),clef,toEnd);
    }
    
    /**
     * Cambia el Clef
     */
    public void changeClef(SongTrack track,Measure measure,int clef,boolean toEnd){        
        //asigno el nuevo clef
        measure.setClef(clef);         

        if(toEnd){
            List measures = getMeasuresBeforeEnd(track,measure.getStart() + 1);
            Iterator it = measures.iterator();
            while(it.hasNext()){
                Measure nextMeasure = (Measure)it.next();
                nextMeasure.setClef(clef);  
            }            
        }        
     
    }        

    public void changeInfo(SongTrack track,String name,RGBColor color,int offset){         
        track.setName(name);
        track.setColor(color);
        track.setOffset(offset);
    }
    
    
    public void changeInstrumentStrings(SongTrack track,List strings){
        if(strings.size() < track.getStrings().size()){
            removeNotesAfterString(track,strings.size());            
        }                
        track.setStrings(strings);
    }
    
    public void removeNotesAfterString(SongTrack track,int string){
        Iterator it = track.getMeasures().iterator();
        while(it.hasNext()){
            Measure measure = (Measure)it.next();
            
            songManager.getMeasureManager().removeNotesAfterString(measure,string);
        }          
    }
    
    public void changeInstrument(SongTrack track,int instrument,boolean percusion){
    	track.getChannel().setInstrument((short)instrument);
        if(percusion){
        	track.getChannel().setChannel(SongChannel.DEFAULT_PERCUSION_CHANNEL);
        	track.getChannel().setEffectChannel(SongChannel.DEFAULT_PERCUSION_CHANNEL);        	
        	track.setStrings(SongManager.createPercusionStrings(track.getStrings().size()));
        }else{
        	if(track.getChannel().isPercusionChannel()){
        		SongChannel tempChannel = this.songManager.getFreeChannel((short)instrument,false);
        		track.getChannel().setChannel(tempChannel.getChannel());
        		track.getChannel().setEffectChannel(tempChannel.getEffectChannel()); 
        	}
        }
        this.songManager.updateChannel(track.getChannel());
    }

    
    public void autoCompleteSilences(SongTrack track){
        Iterator it = track.getMeasures().iterator();
        while(it.hasNext()){
            Measure measure = (Measure)it.next();            
            this.songManager.getMeasureManager().autoCompleteSilences(measure);
        }
    }
    
    /**
     * Retorna true si es el primer compas
     */         
    public boolean isFirstMeasure(Measure measure){
        return (measure.getNumber() == 1);
    }

    /**
     * Retorna true si es el ultimo compas
     */         
    public boolean isLastMeasure(SongTrack track,Measure measure){
        return (countMeasures(track) == measure.getNumber());
    }
}
