package org.herac.tuxguitar.io.pt;

import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.herac.tuxguitar.song.managers.SongManager;
import org.herac.tuxguitar.song.models.BendEffect;
import org.herac.tuxguitar.song.models.Duration;
import org.herac.tuxguitar.song.models.Measure;
import org.herac.tuxguitar.song.models.MeasureHeader;
import org.herac.tuxguitar.song.models.Note;
import org.herac.tuxguitar.song.models.NoteEffect;
import org.herac.tuxguitar.song.models.RGBColor;
import org.herac.tuxguitar.song.models.Song;
import org.herac.tuxguitar.song.models.SongChannel;
import org.herac.tuxguitar.song.models.SongTrack;
import org.herac.tuxguitar.song.models.Tempo;
import org.herac.tuxguitar.song.models.TimeSignature;

public abstract class PTBInputStream {
	protected static final int TRACK_TYPE_GUITAR = 1;
	
	protected static final int TRACK_TYPE_BASS = 2;
	
	private static final int ERROR_MARGIN = 50;	
	
	private static final BendEffect[] DEFAULT_BENDS = makeDefaultBends();
	
    private DataInputStream dataInputStream;
    
    protected String version;    
    protected List headers;
    protected List tracks;
    protected List trackInfos;
    protected int lastUsedChannel = 0;
    
    public PTBInputStream(FileInputStream file) throws FileNotFoundException {
        this.dataInputStream = new DataInputStream(file);
        this.headers = new ArrayList();
        this.tracks = new ArrayList();
        this.trackInfos = new ArrayList();
    }

    public PTBInputStream(String fileName) throws FileNotFoundException {        
        this(new FileInputStream(new File(fileName)));             
    }          
    
    public abstract boolean isSupportedVersion(String version);

    public abstract boolean isSupportedVersion();
	
	
    protected SongTrack getTrack(int number,int trackType){
    	Iterator it = this.tracks.iterator();
    	while(it.hasNext()){
    		SongTrack track = (SongTrack)it.next();
    		if(track.getNumber() == number){
    			return track;
    		}
    	}
    	lastUsedChannel = (lastUsedChannel < (SongManager.MAX_CHANNELS - 1))?lastUsedChannel+1:lastUsedChannel;
    	lastUsedChannel = (lastUsedChannel == SongChannel.DEFAULT_PERCUSION_CHANNEL)?lastUsedChannel+1:lastUsedChannel;
    	
    	SongChannel channel = new SongChannel((short)lastUsedChannel,(short)lastUsedChannel,(short)0);
    	SongTrack track = new SongTrack(number,new String(),channel,new ArrayList(),new ArrayList(),0,(RGBColor)RGBColor.RED);    	
    	TrackInfo trackInfo = null;
    	
    	//pongo por defecto el primer track info.
    	it = trackInfos.iterator();
    	while(it.hasNext()){
    		TrackInfo info = (TrackInfo)it.next();
    		//if(info.trackInfoNumber == 0 && info.type == trackType){
    		if(info.type == trackType && (trackInfo == null || info.trackInfoNumber < trackInfo.trackInfoNumber)){
    			trackInfo = info;
    		}
    	}    	
    	if(trackInfo != null){    		    		
    		track.setName(trackInfo.getName());
    		track.setStrings(trackInfo.getStrings());    		    		
    		track.getChannel().setInstrument((short)trackInfo.getChannel().getInstrument());
    		track.getChannel().setVolume((short)trackInfo.getChannel().getVolume());
    		track.getChannel().setBalance((short)trackInfo.getChannel().getBalance());
    	}
    	tracks.add(track);
    	
    	return track;
    }
    
    
	
    protected MeasureHeader getHeader(long start){
		if(this.headers == null){
			this.headers = new ArrayList();
		}
		
		Iterator it = headers.iterator();
		while(it.hasNext()){
			MeasureHeader header = (MeasureHeader)it.next();
			if((start + ERROR_MARGIN) >= header.getStart() && (start + ERROR_MARGIN) < header.getStart() + header.getLength()){
				return header;
			}
		}		
		MeasureHeader last = getLastHeader();		
		int lastNumber = (last != null)?last.getNumber() + 1:1;
		long lastStart = (last != null)?(last.getStart() + last.getLength()):1000;
		TimeSignature lastTimeSignature = (last != null)?(TimeSignature)last.getTimeSignature().clone():new TimeSignature(4,new Duration(Duration.QUARTER));
		Tempo lastTempo = (last != null)?(Tempo)last.getTempo().clone():new Tempo(120);				
		int tripletFeel = MeasureHeader.TRIPLET_FEEL_NONE;
		MeasureHeader header = new MeasureHeader(lastNumber,lastStart,lastTimeSignature,lastTempo,null,tripletFeel,false,0);
		
		this.headers.add(header);
		
		if(header.getStart() + header.getLength() <= start){		
			return header;
		}
		return getHeader(start);
	}
	
	protected MeasureHeader getLastHeader(){
		if(this.headers != null && !this.headers.isEmpty()){
			return (MeasureHeader)this.headers.get(this.headers.size() - 1);
		}		
		return null;
	}
	
	protected Measure getMeasure(SongTrack track,long start){				
		Iterator it = track.getMeasures().iterator();
		while(it.hasNext()){
			Measure measure = (Measure)it.next();
			if((start + ERROR_MARGIN) >= measure.getStart() && (start + ERROR_MARGIN) < measure.getStart() + measure.getLength()){
				return measure;
			}
		}				
		getHeader(start);
		for(int i = 0;i < headers.size();i++){
			boolean exist = false;
			MeasureHeader header = (MeasureHeader)headers.get(i);			
			for(int j = 0;j < track.getMeasures().size();j++){
				Measure measure = (Measure)track.getMeasures().get(j);				
				if(measure.getHeader().equals(header)){
					exist = true;
				}
			}
			if(!exist){
				track.getMeasures().add(new Measure(header,new ArrayList(),new ArrayList(),1,0));		
			}
		}
		return getMeasure(track,start);
	}	
	
	protected void makeTrackChannels(int fromNumber,int trackType,/*List trackInfoHelpers,*/List guitarInHelpers){
		Iterator it = guitarInHelpers.iterator();
		while(it.hasNext()){
			GuitarInHelper guitarIn = (GuitarInHelper)it.next();
			for(int i = 0;i < trackInfos.size();i ++){
				TrackInfo info = (TrackInfo)trackInfos.get(i);
				if(info.getTrackInfoNumber() == guitarIn.getTrackInfo() && info.getType() == trackType){
					SongTrack track = getTrack(guitarIn.staff + fromNumber,trackType);
					track.setName(info.getName());
					track.setStrings(info.getStrings());					
		    		track.getChannel().setInstrument((short)info.getChannel().getInstrument());
		    		track.getChannel().setVolume((short)info.getChannel().getVolume());
		    		track.getChannel().setBalance((short)info.getChannel().getBalance());					
				}
			}
		}
	}
	
	protected void makeSectionNotes(long start,List trackHelpers,List barHelpers,List tempoHelpers){
		for(int i = 0;i < trackHelpers.size();i ++){
			TrackHelper track = (TrackHelper)trackHelpers.get(i);
			makeTrackNotes(start,track,barHelpers,tempoHelpers);		
		}
	}
	
	private void makeTrackNotes(long start,TrackHelper track,List barHelpers,List tempoHelpers){
		int lastBar = -1;
		for(int z = 0;z < barHelpers.size();z ++){
			BarHelper bar = (BarHelper)barHelpers.get(z);			
			int length1 = 0;
			int length2 = 0;
							
			makeTempoMarker(bar.getPosition(),start,track.getTrackNumber(),tempoHelpers);
			
			for(int j = 0;j < track.getVoice1().size();j ++){					
				BeatHelper beat = (BeatHelper)track.getVoice1().get(j);
				makeTempoMarker(beat.getPosition(),start + length1,track.getTrackNumber(),tempoHelpers);
				length1 += makeBeatNotes(start + length1,track,beat,lastBar,bar.getPosition());
			}

			for(int j = 0;j < track.getVoice2().size();j ++){
				BeatHelper beat = (BeatHelper)track.getVoice2().get(j);
				makeTempoMarker(beat.getPosition(),start + length2,track.getTrackNumber(),tempoHelpers);
				length2 += makeBeatNotes(start + length2,track,beat,lastBar,bar.getPosition());
			}
			
			lastBar = bar.getPosition();
			start += Math.max(length1,length2);			
			
		}
		int length1 = 0;
		int length2 = 0;		
		for(int j = 0;j < track.getVoice1().size();j ++){					
			BeatHelper beat = (BeatHelper)track.getVoice1().get(j);
			makeTempoMarker(beat.getPosition(),start + length1,track.getTrackNumber(),tempoHelpers);
			length1 += makeBeatNotes(start + length1,track,beat,lastBar,-1);			
		}

		for(int j = 0;j < track.getVoice2().size();j ++){
			BeatHelper beat = (BeatHelper)track.getVoice2().get(j);
			makeTempoMarker(beat.getPosition(),start + length2,track.getTrackNumber(),tempoHelpers);
			length2 += makeBeatNotes(start + length2,track,beat,lastBar,-1);
		}		
	}

	private long makeBeatNotes(long start,TrackHelper track,BeatHelper beat,int minPos,int maxPos){
		if(!beat.isAcciaccatura() && beat.getPosition() > minPos && (beat.getPosition() < maxPos || maxPos < 0)){
			Measure measure = getMeasure(getTrack(track.getTrackNumber(),track.getTrackType()),start);
			int diff = (int)((measure.getStart() > start)?measure.getStart() - start:0);
			for(int k = 0;k < beat.getNotes().size();k++){
				NoteHelper helper = (NoteHelper)beat.getNotes().get(k);
				
				
				int value = helper.getValue();
				int string = helper.getString();
				int velocity = 64;
				boolean tied = helper.isTied();
				Duration duration = (Duration)beat.getDuration().clone();				
				NoteEffect effect = new NoteEffect();
				effect.setVibrato(beat.isVibrato());
				effect.setDeadNote(helper.isDead());
				effect.setHammer(helper.isHammer());
				effect.setSlide(helper.isSlide());
				effect.setBend((!tied && helper.getBend() > 0 && helper.getBend() < DEFAULT_BENDS.length)?(BendEffect)DEFAULT_BENDS[helper.getBend() - 1].clone():null);
				
				addNote(measure,new Note(value,start + diff,duration,velocity,string,tied,effect));
				//measure.addNote(new Note(value,start + diff,duration,velocity,string,tied,effect));
			}		
			return beat.getDuration().getTime() + diff;
		}		
		return 0;
	}
	
	private void addNote(Measure measure,Note note){
		Iterator it = measure.getNotes().iterator();
		while(it.hasNext()){
			Note n = (Note)it.next();
			if(n.getStart() < note.getStart() && (n.getStart() + ERROR_MARGIN) > note.getStart()){
				note.setStart(n.getStart());
			}else if(n.getStart() > note.getStart() && (n.getStart() - ERROR_MARGIN) < note.getStart()){
				note.setStart(n.getStart());
			}
		}
		measure.addNote(note);
	}
	
	private void makeTempoMarker(int position,long start,int track,List tempoHelpers){
		//solo si es la primer pista
		if(track == 1){
			Iterator it = tempoHelpers.iterator();
			while(it.hasNext()){
				TempoHelper helper = (TempoHelper)it.next();
				if(helper.getPosition() == position){
					MeasureHeader header = getHeader(start); 
					header.getTempo().setValue(helper.getTempo());
					header.setTripletFeel(helper.getTripletFeel());
				}
			}
		}
	}
		
	protected Song checkSong(Song song)throws IOException{
		return new SongAdjuster().process(song);
	}    
    
    
	
	/*
	protected BeatHelper getBeatHelper(List beatHelpers,int position,long start,long minStart){
    	BeatHelper helper = null;
    	for(int i = 0;i < beatHelpers.size();i++){
    		BeatHelper currentBeat = (BeatHelper)beatHelpers.get(i);
    		if(currentBeat.position == position){
    			helper = currentBeat;
    			break;
    		}
    	}
    	        
    	if(helper == null){
    		helper = new BeatHelper();    		
    		helper.position = position;
    		//helper.notes = new ArrayList();
    		
    		if(start > 0){
    			helper.start = start;
    		}else{
    			BeatHelper prev = getPreviousBeatHelper(beatHelpers,position);
    			helper.start = (prev != null)?prev.start + prev.duration.getTime():minStart;
    		}
    		
    		beatHelpers.add(helper);
    	}
    	return helper;
	}
	
	protected BeatHelper getPreviousBeatHelper(List beatHelpers,int position){
    	BeatHelper previous = null;
    	for(int i = 0;i < beatHelpers.size();i++){
    		BeatHelper beat = (BeatHelper)beatHelpers.get(i);
    		if(beat.position < position){
    			if(previous == null || beat.position > previous.position){
    				previous = beat;
    			}
    		}
    	}
    	return previous;
	}
	
	protected BeatHelper getNextBeatHelper(List beatHelpers,int position){
    	BeatHelper next = null;
    	for(int i = 0;i < beatHelpers.size();i++){
    		BeatHelper beat = (BeatHelper)beatHelpers.get(i);
    		if(beat.position > position){
    			if(next == null || next.position > beat.position){
    				next = beat;
    			}
    		}
    	}
    	return next;
	}	
    */
    
    
    private static BendEffect[] makeDefaultBends(){
    	BendEffect[] defaultBends = new BendEffect[8];
        
        defaultBends[0] = new BendEffect();        
        defaultBends[0].addPoint(0,0);
        defaultBends[0].addPoint(6,8);
        defaultBends[0].addPoint(12,8);               
        
        defaultBends[1] = new BendEffect();                 
        defaultBends[1].addPoint(0,0);
        defaultBends[1].addPoint(3,8);
        defaultBends[1].addPoint(6,8);
        defaultBends[1].addPoint(9,0);
        defaultBends[1].addPoint(12,0);
        
        defaultBends[2] = new BendEffect();        
        defaultBends[2].addPoint(0,0);
        defaultBends[2].addPoint(6,8);
        defaultBends[2].addPoint(12,8);         
                
        defaultBends[3] = new BendEffect();
        defaultBends[3].addPoint(0,8);
        defaultBends[3].addPoint(12,8);

        defaultBends[4] = new BendEffect();        
        defaultBends[4].addPoint(0,8);
        defaultBends[4].addPoint(4,8);
        defaultBends[4].addPoint(8,0);
        defaultBends[4].addPoint(12,0);
        
        defaultBends[5] = new BendEffect();
        defaultBends[5].addPoint(0,8);
        defaultBends[5].addPoint(12,8);
        
        defaultBends[6] = new BendEffect();        
        defaultBends[6].addPoint(0,8);
        defaultBends[6].addPoint(4,8);
        defaultBends[6].addPoint(8,0);
        defaultBends[6].addPoint(12,0);
        
        defaultBends[7] = new BendEffect();        
        defaultBends[7].addPoint(0,8);
        defaultBends[7].addPoint(4,8);
        defaultBends[7].addPoint(8,0);
        defaultBends[7].addPoint(12,0);
        
        return defaultBends;
    }       
	
	
    
	
    protected int readByte(){
        try {
            return this.dataInputStream.read();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return 0;
    }
          
    protected String readString(int length){
        try {
            byte[] bytes = new byte[length];
            for(int i = 0;i < bytes.length; i++){
            	bytes[i] = this.dataInputStream.readByte();
            }
            return new String(bytes);
        } catch (IOException e) {            
            e.printStackTrace();
        }
        return null;
        
    }        
    
    protected String readString(){
        try {        	
        	int length = this.dataInputStream.read();        	
        	return readString(((length < 255)?length:readUInt()));
        } catch (IOException e) {            
            e.printStackTrace();
        }
        return null;
        
    }   
    
    protected int readUCType(){
    	return readByte();
    }
    
    protected boolean readUChar(){
        try {
            return this.dataInputStream.readBoolean();
        } catch (IOException e) {            
            e.printStackTrace();
        }
        return false;
    }
    
    protected int readUInt(){
        try {
        	byte[] b = {0, 0};
        	this.dataInputStream.read(b);
        	return ((b[1] & 0xff) << 8) | (b[0] & 0xff);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return 0;
    }
    
    protected int readULong(){
        try {
        	//return this.dataInputStream.readInt(); 
        	
        	byte[] b = new byte[4];
        	this.dataInputStream.read(b);
        	return ((b[3] & 0xff) << 24) | ((b[2] & 0xff) << 16) | ((b[1] & 0xff) << 8) | (b[0] & 0xff);        	
        } catch (IOException e) {
            e.printStackTrace();
        }
        return 0;
    }
    
    
    protected void debug(int i){
    	System.out.println(i);
    }
    
    protected void debug(String s){
    	System.out.println(s);
    }
    
    
    
    
    
    
    
    
    
    
    
    
    
    

    protected class TrackHelper{
    	private int trackNumber;
    	private int trackType;
    	private List voice1 = new ArrayList();
    	private List voice2 = new ArrayList();
    	
    	public TrackHelper(int trackNumber,int trackType){
    		this.trackNumber = trackNumber;
    		this.trackType = trackType;
    		this.voice1 = new ArrayList();
    		this.voice2 = new ArrayList();
    	}

		public int getTrackNumber() {
			return trackNumber;
		}

		public int getTrackType() {
			return trackType;
		}
		
		public List getVoice1() {
			return voice1;
		}

		public List getVoice2() {
			return voice2;
		}
	
    }
	
	protected class BeatHelper{
		private int position;		
		private Duration duration;
		private List notes;
		private boolean acciaccatura;
		private boolean vibrato;
		
    	public BeatHelper(){ 
    		this.notes = new ArrayList();
    	}

		public Duration getDuration() {
			return duration;
		}

		public void setDuration(Duration duration) {
			this.duration = duration;
		}

		public int getPosition() {
			return position;
		}

		public void setPosition(int position) {
			this.position = position;
		}

		public void addNote(NoteHelper note){
			this.notes.add(note);
		}
		
		public List getNotes(){
			return this.notes;
		}

		public boolean isAcciaccatura() {
			return acciaccatura;
		}

		public void setAcciaccatura(boolean acciaccatura) {
			this.acciaccatura = acciaccatura;
		}

		public boolean isVibrato() {
			return vibrato;
		}

		public void setVibrato(boolean vibrato) {
			this.vibrato = vibrato;
		}
		
		
	}
	
	protected class NoteHelper{
		private int value;
		private int string;
		private boolean tied;
		private boolean dead;
		private boolean hammer;
		private boolean slide;
		private int bend;
		
		
		public NoteHelper(){
		}
		
		public int getString() {
			return string;
		}
		public void setString(int string) {
			this.string = string;
		}
		public int getValue() {
			return value;
		}
		public void setValue(int value) {
			this.value = value;
		}

		public boolean isDead() {
			return dead;
		}

		public void setDead(boolean dead) {
			this.dead = dead;
		}

		public boolean isTied() {
			return tied;
		}

		public void setTied(boolean tied) {
			this.tied = tied;
		}

		public int getBend() {
			return bend;
		}

		public void setBend(int bend) {
			this.bend = bend;
		}

		public boolean isHammer() {
			return hammer;
		}

		public void setHammer(boolean hammer) {
			this.hammer = hammer;
		}

		public boolean isSlide() {
			return slide;
		}

		public void setSlide(boolean slide) {
			this.slide = slide;
		}		
		
		
	}
	
	protected class BarHelper{
		private int position;
		
		public BarHelper(int position){
			this.position = position;
		}
		
		public int getPosition() {
			return position;
		}
		
	}	
	
	protected class TempoHelper{
		int position;
		int tempo;
		int tripletFeel;
		
		public TempoHelper(int position,int tempo,int tripletFeel){
			this.position = position;
			this.tempo = tempo;
			this.tripletFeel = tripletFeel;
		}

		public int getPosition() {
			return position;
		}

		public int getTempo() {
			return tempo;
		}
		
		public int getTripletFeel() {
			return tripletFeel;
		}
		
	}
    
    protected class TrackInfo{    	
    	private int type;
    	private int trackInfoNumber;
    	private SongChannel channel;
    	private List strings;
    	private String name;
    	
    	public TrackInfo(){    		
    	}
    	
    	
		public SongChannel getChannel() {
			return channel;
		}

		public void setChannel(SongChannel channel) {
			this.channel = channel;
		}

		public List getStrings() {
			return strings;
		}

		public void setStrings(List strings) {
			this.strings = strings;
		}

		public int getTrackInfoNumber() {
			return trackInfoNumber;
		}

		public void setTrackInfoNumber(int trackInfoNumber) {
			this.trackInfoNumber = trackInfoNumber;
		}

		public int getType() {
			return type;
		}

		public void setType(int type) {
			this.type = type;
		}

		public String getName() {
			return name;
		}

		public void setName(String name) {
			this.name = name;
		}    	
    }
    
    protected class GuitarInHelper{
    	private int staff;
    	private int trackInfo;
    	
    	public GuitarInHelper(int staff,int trackInfo){
    		this.staff = staff;
    		this.trackInfo = trackInfo;
    	}

		public int getStaff() {
			return staff;
		}

		public int getTrackInfo() {
			return trackInfo;
		}    	
    }
    
    protected class SongInfo{
	    private String name;    
	    private String interpret;
	    private String album;
	    private String author;
	    
	    public SongInfo(){	    	
	    }
	    
		public String getAlbum() {
			return album;
		}
		public void setAlbum(String album) {
			this.album = album;
		}
		public String getAuthor() {
			return author;
		}
		public void setAuthor(String author) {
			this.author = author;
		}
		public String getInterpret() {
			return interpret;
		}
		public void setInterpret(String interpret) {
			this.interpret = interpret;
		}
		public String getName() {
			return name;
		}
		public void setName(String name) {
			this.name = name;
		}		    	    
    }
    
    
    
    private class SongAdjuster{    	
    	public SongAdjuster(){    		
    	}
    	
    	public Song process(Song song)throws IOException{		
    		completeMeasures(song);
    		adjustDurations(song);
    		
    		return song;
    	}      	
    	
    	public void completeMeasures(Song song)throws IOException{		
    		int headerCount = song.getMeasureHeaders().size();
    		for(int i = 0;i < song.getTracks().size();i ++){
    			SongTrack track = (SongTrack)song.getTracks().get(i);	    			
    			while(track.getMeasures().size() < headerCount){				
    				long start = Duration.QUARTER_TIME;
    				Measure lastMeasure = (Measure)((!track.getMeasures().isEmpty())?track.getMeasures().get(track.getMeasures().size() - 1) :null);
    				if(lastMeasure != null){
    					start = (lastMeasure.getStart() + lastMeasure.getLength());
    				}
    				track.getMeasures().add(new Measure(getHeader(start),new ArrayList(),new ArrayList(),1,0));
    			}
    		}    		
    		if(song.getMeasureHeaders().isEmpty() || song.getTracks().isEmpty()){
    			throw new IOException("Empty Song");
    		}
    	}      	    	
    	
    	public void adjustDurations(Song song){
    		for(int i = 0;i < song.getTracks().size();i ++){
    			SongTrack track = (SongTrack)song.getTracks().get(i);    			
        		for(int j = 0;j < track.getMeasures().size();j ++){
        			Measure measure = (Measure)track.getMeasures().get(j);    			    			
        			if(!measure.getNotes().isEmpty()){
        				orderNotes(measure);    		
        				long start = ((Note)measure.getNotes().get(0)).getStart();
        				long measureStart = measure.getStart();
        				long measureLength = measure.getLength();
        				while(start < (measureStart + measureLength)){
        					List notesAtBeat = getNotesAtBeat(measure,start);
        					long maxLength = getMaxLength(measure,start);
        					normalizeNotes(notesAtBeat,maxLength);
        					start += maxLength;
        				}
        			}
        		}
    		}    		
    	}
    	
    	private void normalizeNotes(List notes,long maxLength){
    		Duration beatDuration = null;
    		
    		Iterator it = notes.iterator();
    		while(it.hasNext()){
    			Note note = (Note)it.next();
    			long noteDuration = note.getDuration().getTime();			
    			if(noteDuration <= maxLength && (beatDuration == null || noteDuration > beatDuration.getTime())){
    				beatDuration = note.getDuration();
    			}
    		}		
    		if(beatDuration == null){
    			beatDuration = Duration.fromTime(maxLength);
    		}
    		if(beatDuration != null){			
    			it = notes.iterator();
    			while(it.hasNext()){
    				Note note = (Note)it.next();
    				note.setDuration((Duration)beatDuration.clone());
    			}
    		}
    	}    	
    	
    	
    	private List getNotesAtBeat(Measure measure,long start){
    		List notes = new ArrayList();
    		Iterator it = measure.getNotes().iterator();
    		while(it.hasNext()){
    			Note note = (Note)it.next();
    			if(note.getStart() == start){
    				notes.add(note);
    			}
    		}
    		return notes;
    	}
    		
    	private long getMaxLength(Measure measure,long start){
    		long nextStart = -1;
    		Iterator it = measure.getNotes().iterator();
    		while(it.hasNext()){
    			Note note = (Note)it.next();
    			if(note.getStart() > start && (nextStart < 0 || note.getStart() < nextStart)){
    				nextStart = note.getStart();
    			}
    		}
    		if(nextStart < 0){
    			nextStart = (measure.getStart() + measure.getLength());
    		}
    		return (nextStart - start);
    	}
    	
        private void orderNotes(Measure measure){
        	int noteCount = measure.getNotes().size();
            for(int i = 0;i < noteCount;i++){
                Note minNote = null;
                for(int noteIdx = i;noteIdx < noteCount;noteIdx++){
                    Note note = (Note)measure.getNotes().get(noteIdx);
                    if(minNote == null){
                        minNote = note;
                    }else if(note.getStart() < minNote.getStart()){
                    	minNote = note;
                    }else if(note.getStart() == minNote.getStart() && note.getString() < minNote.getString()){
                    	minNote = note;
                    }
                }
                measure.getNotes().remove(minNote);
                measure.getNotes().add(i,minNote);
            }
        }
        
    }    

}
