package org.herac.tuxguitar.io.gp;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.herac.tuxguitar.io.gp.GPFormatException;
import org.herac.tuxguitar.song.models.BendEffect;
import org.herac.tuxguitar.song.models.Duration;
import org.herac.tuxguitar.song.models.InstrumentString;
import org.herac.tuxguitar.song.models.Marker;
import org.herac.tuxguitar.song.models.Measure;
import org.herac.tuxguitar.song.models.MeasureHeader;
import org.herac.tuxguitar.song.models.Note;
import org.herac.tuxguitar.song.models.NoteEffect;
import org.herac.tuxguitar.song.models.Song;
import org.herac.tuxguitar.song.models.SongChannel;
import org.herac.tuxguitar.song.models.SongTrack;
import org.herac.tuxguitar.song.models.Tempo;
import org.herac.tuxguitar.song.models.TimeSignature;
import org.herac.tuxguitar.song.models.RGBColor;

public class GP5InputStream extends InputStream {
    private static final String supportedVersions[] = { "FICHIER GUITAR PRO v5.00" };
    private String version;
    private InputStream inputStream;
    private int offset;

    public GP5InputStream(InputStream inputStream) {
        super();
        this.inputStream = inputStream;
        this.offset = 0;
    }

    public GP5InputStream(String fileName) throws FileNotFoundException {
        this(new FileInputStream(new File(fileName)));
    }

    public void close() throws IOException {
        this.inputStream.close();
    }

    public int read() throws IOException {
        this.offset++;
        return this.inputStream.read();
    }

    
    private void readVersion(){
		try {
			if(this.version == null){
				this.version = readStringByte(30);
			}
		} catch (IOException e) {
			this.version = "NOT_SUPPORTED";
		}    	
    }
    
    public static boolean isSupportedVersion(String version) {
        for (int i = 0; i < supportedVersions.length; i++) {
            if (version.equals(supportedVersions[i])) {
                return true;
            }
        }
        return false;
    }
    
    public boolean isSupportedVersion(){
    	try{
    		readVersion();
    		return isSupportedVersion(version);
    	}catch(Exception e){
    		return false;
    	}catch(Error e){
    		return false;
    	}   
    }        

    public Song readSong() throws IOException, GPFormatException {
    	readVersion();
        if (!isSupportedVersion(version)) {
            throw new GPFormatException("Unsuported Version");
        }

        String title = readStringIntegerPlusOne();

        String subtitle = readStringIntegerPlusOne();

        String interpret = readStringIntegerPlusOne();

        String album = readStringIntegerPlusOne();
        
        String songAuthor = readStringIntegerPlusOne();

        String music = readStringIntegerPlusOne();
        
        String copyright = readStringIntegerPlusOne();

        String tab = readStringIntegerPlusOne();

        String instructions = readStringIntegerPlusOne();

        int nbNotes = readInt();
        String note = "";
        for (int i = 0; i < nbNotes; i++) {
            note += readStringIntegerPlusOne();
            note += "\n";
        }
        
        int trackNumber = readInt();
        for (int i = 0; i < 5; i++) {
        	int measureNumber = readInt();
        	String lines = readStringInteger();        	
        }        

        skipBytes(30);
        for (int i = 0; i < 11; i++) {
        	readInt();
        	readStringByte(0);
        }        
        
        
        int tempoValue = readInt();
        int key = readByte();
        int octave = readInt();
        
        
        List channels = new ArrayList();
        int[] instruments = new int[64];
        for (int i = 0; i < 64; i++) {
        	channels.add(new SongChannel((short)i,
    					(short)i, 
    					(short)readInt(),
    					toChannelShort(readByte()),
    					toChannelShort(readByte()),
    					toChannelShort(readByte()),
    					toChannelShort(readByte()),
    					toChannelShort(readByte()),
    					toChannelShort(readByte()),
    					false,
    					false));       
            byte[] b = { 0, 0 };
            read(b);
        }
        
        TimeSignature timeSignature = new TimeSignature(4, new Duration(4));

        skipBytes(42);
        
        int numberOfMeasures = readInt();
        int numberOfTracks = readInt();
        
        List headers = new ArrayList();
        if (numberOfMeasures > 0) {
            for (int i = 0; i < numberOfMeasures; i++) {
            	if(i > 0 ){
            		skipBytes(1);
            	}
                MeasureHeader header = createMeasureHeader((i + 1),timeSignature);
                headers.add(header);
            }
        }

        List tracks = new ArrayList();
        for (int i = 0; i < numberOfTracks; i++) {
            tracks.add(createTrack((i + 1), channels));
        }
        
        skipBytes(2);
        
        long nextMeasureStart = 1000;        
        for (int i = 0; i < numberOfMeasures; i++) {
            MeasureHeader currHeader = (MeasureHeader) headers.get(i);
            Tempo tempo = new Tempo(tempoValue);
            currHeader.setStart(nextMeasureStart);
            for (int j = 0; j < numberOfTracks; j++) {
                SongTrack track = (SongTrack) tracks.get(j);
                Measure measure = new Measure(currHeader,new ArrayList(),new ArrayList(),1,0);                
                addMeasureComponents(track.getStrings(), measure, track.getMeasures(), tempo);
                currHeader.setTempo(tempo);
                track.getMeasures().add(measure);
                skipBytes(1);
            }
            tempoValue = tempo.getValue();
            nextMeasureStart += currHeader.getLength();            
        }
        
        return new Song(title,interpret,album,songAuthor,tracks,headers,Song.MAX_VOLUME);
    }

    private long addNotes(long start, List notes, List trackStrings, List currTrackMeasures, Tempo tempo) throws IOException,
            GPFormatException {
        NoteEffect effect = new NoteEffect();
        
        int header = readUnsignedByte();

        if ((header & 0x40) != 0) {
            int beatStatus = readUnsignedByte();
            boolean emptyBeat = (beatStatus == 0x00);
            boolean restBeat = (beatStatus == 0x02);
        }

        boolean dottedNotes = ((header & 0x01) != 0);

        Duration duration = parseDuration(readByte());
        duration.setDotted(dottedNotes);

        if ((header & 0x20) != 0) {
            int tuplet = readInt();
            //-------Verifico el tupleto--------------------
            switch (tuplet) {
            case 3:
                duration.getTupleto().setEnters(3);
                duration.getTupleto().setTimes(2);
                break;
            case 5:
                duration.getTupleto().setEnters(5);
                duration.getTupleto().setTimes(4);
                break;                
            case 6:
                duration.getTupleto().setEnters(6);
                duration.getTupleto().setTimes(4);
                break;
            case 7:
                duration.getTupleto().setEnters(7);
                duration.getTupleto().setTimes(4);
                break;
            case 9:
                duration.getTupleto().setEnters(9);
                duration.getTupleto().setTimes(8);
                break;
            case 10:
                duration.getTupleto().setEnters(10);
                duration.getTupleto().setTimes(8);
                break;
            case 11:
                duration.getTupleto().setEnters(11);
                duration.getTupleto().setTimes(8);
                break;
            case 12:
                duration.getTupleto().setEnters(12);
                duration.getTupleto().setTimes(8);
                break;                  
            }
        }

        if ((header & 0x02) != 0) {
            readChordDiagram();
        }

        if ((header & 0x04) != 0) {
            String text = readStringIntegerPlusOne();
        }

        if ((header & 0x08) != 0) {
            readBeatEffects((NoteEffect)effect.clone(),duration);
        }

        if ((header & 0x10) != 0) {
            readMixChange(tempo);
        }

        int stringsPlayed = readUnsignedByte();
        List strings = getPlayedStrings(stringsPlayed, trackStrings);

        for (int i = strings.size() - 1; i >= 0; i--) {
            InstrumentString string = (InstrumentString) strings.get(i);
            Note note = parseNote(start, string, duration, notes, currTrackMeasures,(NoteEffect)effect.clone());
            if (note != null) {
                notes.add(note);
            }
        }

        skipBytes(1);
        int read = readByte();
        if (read >= 8 && read < 32) {
        	skipBytes(1);
        }
        
        
        return duration.getTime();
    }

    private List getPlayedStrings(int stringsPlayed, List trackStrings) {
        List strings = new ArrayList();
        if ((stringsPlayed & (1 << 0)) != 0 && trackStrings.size() > 6) {
            strings.add(((InstrumentString) trackStrings.get(6)).clone());
        }
        if ((stringsPlayed & (1 << 1)) != 0 && trackStrings.size() > 5) {
            strings.add(((InstrumentString) trackStrings.get(5)).clone());
        }
        if ((stringsPlayed & (1 << 2)) != 0 && trackStrings.size() > 4) {
            strings.add(((InstrumentString) trackStrings.get(4)).clone());
        }
        if ((stringsPlayed & (1 << 3)) != 0 && trackStrings.size() > 3) {
            strings.add(((InstrumentString) trackStrings.get(3)).clone());
        }
        if ((stringsPlayed & (1 << 4)) != 0 && trackStrings.size() > 2) {
            strings.add(((InstrumentString) trackStrings.get(2)).clone());
        }
        if ((stringsPlayed & (1 << 5)) != 0 && trackStrings.size() > 1) {
            strings.add(((InstrumentString) trackStrings.get(1)).clone());
        }
        if ((stringsPlayed & (1 << 6)) != 0 && trackStrings.size() > 0) {
            strings.add(((InstrumentString) trackStrings.get(0)).clone());
        }
        return strings;
    }

    private Duration parseDuration(byte value) {
        Duration duration = null;
        switch (value) {
        case -2:
            duration = new Duration(Duration.WHOLE);
            break;
        case -1:
            duration = new Duration(Duration.HALF);
            break;
        case 0:
            duration = new Duration(Duration.QUARTER);
            break;
        case 1:
            duration = new Duration(Duration.EIGHTH);
            break;
        case 2:
            duration = new Duration(Duration.SIXTEENTH);
            break;
        case 3:
            duration = new Duration(Duration.THIRTY_SECOND);
            break;
        case 4:
            duration = new Duration(Duration.SIXTY_FOURTH);
            break;
        }
        if(duration == null){
        	duration = new Duration(Duration.QUARTER);
        	System.err.println("Incorrect Duration. Forcing duration Quarter as Default.");
        }        
        return duration;
    }

    private int getTiedNoteValue(int string, List notes, List measures) {
        if (!notes.isEmpty()) {
            for (int nIdx = notes.size() - 1; nIdx >= 0; nIdx--) {
                Note note = (Note) notes.get(nIdx);
                if (note.getString() == string) {
                    return note.getValue();
                }
            }
        }
        if (!measures.isEmpty()) {
            for (int mIdx = measures.size() - 1; mIdx >= 0; mIdx--) {
                Measure measure = (Measure) measures.get(mIdx);
                for (int nIdx = measure.getNotes().size() - 1; nIdx >= 0; nIdx--) {
                    Note note = (Note) measure.getNotes().get(nIdx);
                    if (note.getString() == string) {
                        return note.getValue();
                    }
                }
            }
        }
        return -1;
    }

    private boolean readBoolean() throws IOException {
        return (read() == 1);
    }

    private byte readByte() throws IOException {
        return (byte) read();
    }

    private RGBColor readColor() throws IOException {
        int r = readUnsignedByte();
        int g = readUnsignedByte();
        int b = readUnsignedByte();
        read();
        
        return new RGBColor(r,g,b);
    }

    private int readInt() throws IOException {
        int integer = 0;
        byte[] b = { 0, 0, 0, 0 };

        read(b);
        integer = ((b[3] & 0xff) << 24) | ((b[2] & 0xff) << 16) | ((b[1] & 0xff) << 8) | (b[0] & 0xff);

        return integer;
    }
    
    private Marker readMarker(int measure) throws IOException {
        String name = readStringIntegerPlusOne();
        RGBColor color = readColor();
        
        return new Marker(measure,name,color);
    }    

    private MeasureHeader createMeasureHeader(int number,TimeSignature currTimeSignature) throws IOException {
        int header = readUnsignedByte();

        int numerator = 0;
        if ((header & 0x01) != 0) {
            numerator = readByte();
        }

        int denominator = 0;
        if ((header & 0x02) != 0) {
            denominator = readByte();
        }

        boolean repeatStart = ((header & 0x04) != 0);

        int numberOfRepetitions = 0;
        if ((header & 0x08) != 0) {
            numberOfRepetitions = readByte();
        }

        Marker marker = null;
        if ((header & 0x20) != 0) {
        	marker = readMarker(number);
        }        
        
        int NumberOfAlternateEnding = 0;
        if ((header & 0x10) != 0) {
            NumberOfAlternateEnding = readByte();
        }

        if ((header & 0x40) != 0) {
            int type = readByte();
            readByte();
        }

        if ((header & 0x80) != 0) {
        }
        
        boolean doubleBar = ((header & 0x80) != 0);

        if (numerator > 0) {
            currTimeSignature.setNumerator(numerator);
        }
        if (denominator > 0) {
            currTimeSignature.setDenominator(new Duration(denominator));
        }

        //--------------------------------
        if(NumberOfAlternateEnding == 0){
        	skipBytes(1);	
        }                
        
        if ((header & 0x01) != 0) {
        	skipBytes(4);          	
        }        
        
        int tripletFeel = MeasureHeader.TRIPLET_FEEL_NONE;
        int gpTripletFeel = readByte(); 
        if(gpTripletFeel == 1){
        	tripletFeel = MeasureHeader.TRIPLET_FEEL_EIGHTH;
        }else if(gpTripletFeel == 2){
        	tripletFeel = MeasureHeader.TRIPLET_FEEL_SIXTEENTH;
        }
        
        return new MeasureHeader(number,0,(TimeSignature) currTimeSignature.clone(), new Tempo(120),marker,tripletFeel,repeatStart,numberOfRepetitions);
    }

    private void addMeasureComponents(List trackStrings, Measure measure, List currTrackMeasures, Tempo tempo) throws IOException,GPFormatException {
        //voice 1
    	long nextNoteStart = measure.getStart();
        int numberOfBeats = readInt();
        for (int i = 0; i < numberOfBeats; i++) {
            nextNoteStart += addNotes(nextNoteStart, measure.getNotes(), trackStrings, currTrackMeasures, tempo);
        }
        int noteCount = measure.getNotes().size();
        
        //voice 2
        nextNoteStart = measure.getStart();
        numberOfBeats = readInt();
        for (int i = 0; i < numberOfBeats; i++) {
            nextNoteStart += addNotes(nextNoteStart, measure.getNotes(), trackStrings, currTrackMeasures, tempo);
        }
        
        //join voices
        if(noteCount < measure.getNotes().size()){                
        	new JoinVoicesHelper(measure).process();
        }
    }

    private Note parseNote(long start, InstrumentString string, Duration currDuration, List currMeasureNotes, List currTrackMeasures,NoteEffect effect)
            throws IOException {
        int header = readUnsignedByte();

        boolean accentuated = ((header & 0x40) != 0);
        boolean dotted = ((header & 0x02) != 0);
        boolean ghostNote = ((header & 0x04) != 0);

        boolean tiedNote = false;
        boolean deadNote = false;
        if ((header & 0x20) != 0) {
            int noteType = readUnsignedByte();
            tiedNote = (noteType == 0x02);
            effect.setDeadNote((noteType == 0x03));
        }

        if ((header & 0x01) != 0) {            
        	//byte duration = readByte();
            //byte tuplet = readByte();
        }

        if ((header & 0x10) != 0) {
            byte dynamic = readByte();
        }

        byte numberOfFret = 0;
        if ((header & 0x20) != 0) {
            numberOfFret = readByte();
        }

        if ((header & 0x80) != 0) {
            byte fingeringLeftHand = readByte();
            byte fingeringRightHand = readByte();
        }

        //cuando la duracion de la nota no es 100%
        if ((header & 0x01) != 0) {
        	skipBytes(8);
        }
        
        skipBytes(1);
        if ((header & 0x08) != 0) {
            readNoteEffects(effect,currDuration);
        }        
        
        int value = numberOfFret;
        if (numberOfFret >= 0 || tiedNote) {
            if (tiedNote) {
                value = getTiedNoteValue(string.getNumber(), currMeasureNotes, currTrackMeasures);
            }

            return new Note(value, start, (Duration) currDuration.clone(), 64, string.getNumber(),tiedNote,effect);
        }

        return null;
    }

    private String readStringByte(int expectedLength) throws IOException {
        byte[] bytes;
        int realLength = readUnsignedByte();

        if (expectedLength != 0) {
        	bytes = new byte[expectedLength];
        } else {
        	bytes = new byte[realLength];
        }
        read(bytes);
        
        realLength = (realLength >= 0)?realLength:expectedLength;
        return new String(bytes, 0, realLength);  
    }

    private String readStringInteger() throws IOException {
        byte[] b;
        String str;
        int length = readInt();

        b = new byte[length];
        read(b);

        str = new String(b);
        return str;
    }

    private String readStringIntegerPlusOne() throws IOException {
        byte[] b;
        String str;
        int lengthPlusOne = readInt();
        int length = lengthPlusOne - 1;

        if (length != read()) {
            throw new IOException();
        }

        b = new byte[length];
        read(b);

        str = new String(b);
        return str;
    }

    private SongTrack createTrack(long number, List channels) throws IOException {
    	int test = 0;
    	byte bb = 0;
    	
        int header = readUnsignedByte();

        boolean isDrumsTrack = ((header & 0x01) != 0);
        boolean is12StringedGuitarTrack = ((header & 0x02) != 0);
        boolean isBanjoTrack = ((header & 0x04) != 0);
                
        skipBytes(1);
        String name = readStringByte(40);        

        
        int numberOfStrings = readInt();       
        
        List strings = new ArrayList(numberOfStrings);

        for (int i = 0; i < 7; i++) {
            int tunning = readInt();
            if (numberOfStrings > i) {
                strings.add(new InstrumentString(i + 1, tunning));
            }
        }

        int port = readInt();

        int channelIndex = readInt();

        int effects = readInt();

        int numberOfFrets = readInt();

        int capo = readInt();

        RGBColor color = readColor();
        
        skipBytes(44);
        
        return new SongTrack(number,name,parseChannel(channels,channelIndex,effects), new ArrayList(), strings,0,color);
    }
    
    private SongChannel parseChannel(List channels, int channelIndex,int effectChannel) {
    	SongChannel channel = (SongChannel) channels.get(channelIndex - 1);

        int instrument = channel.getInstrument();
        if (instrument == -1) {
            channel.setInstrument((short)0);
        }
        if(!channel.isPercusionChannel()){
        	channel.setEffectChannel((short)(effectChannel - 1));
        }
        
        return channel;
    }


    private int readUnsignedByte() throws IOException {
        return read();
    }

    private void readChordType() throws IOException {
        readUnsignedByte();
    }

    private void readRoot() throws IOException {
        readByte();
    }

    private void readTonalityType(int numBytes) throws IOException {
        if (numBytes == 1) {
            int type = readUnsignedByte();
        } else if (numBytes == 4) {
            int type = readInt();
        }
    }

    private String readChordName() throws IOException {
        byte[] nameB;
        char[] nameC;
        int i;
        int max;

        nameB = new byte[21];
        nameC = new char[20];
        read(nameB, 0, 21);
        max = 20;
        if (nameB[0] < max) {
            max = nameB[0];
        }

        for (i = 1; i <= max; i++) {
            nameC[i - 1] = (char) nameB[i];
        }
        return (String.valueOf(nameC, 0, max));
    }

    private void readChordDiagram() throws IOException, GPFormatException {
        int header;
        long aux;
        long skip;
        int i;

        header = readUnsignedByte();

        if ((header & 0x01) == 0) {
            throw new GPFormatException("Cannot Read Chord Diagram");
        }

        boolean sharp = readBoolean();

        this.skip(3);

        readRoot();

        readChordType();

        int nineElevenThirteen = readUnsignedByte();

        int bass = readInt();

        readTonalityType(4);

        int addedNote = readUnsignedByte();

        String name = readChordName();

        this.skip(2);

        readTonalityType(1);

        readTonalityType(1);

        readTonalityType(1);

        int baseFret = readInt();

        for (i = 1; i <= 7; i++) {
            int fret = readInt();
        }

        int numBarres = readUnsignedByte();

        for (i = 1; i <= 5; i++) {
            int fretOfBarre = readUnsignedByte();
        }
        for (i = 1; i <= 5; i++) {
            int barreStart = readUnsignedByte();
        }
        for (i = 1; i <= 5; i++) {
            int barreEnd = readUnsignedByte();
        }

        aux = this.skip(8);

        for (i = 1; i <= 7; i++) {
            int fingering = readByte();
        }
        boolean chordFingeringDisplayed = readBoolean();
    }

    private void readGraceNote() throws IOException {
    	skipBytes(5);
    }

    private void readBend(NoteEffect effect,Duration duration) throws IOException {
        byte type = readByte();
        int value = readInt();
        
        BendEffect bend = new BendEffect();        
        int numPoints = readInt();
        for (int i = 0; i < numPoints; i++) {
            
            int bendPosition = readInt();
            int bendValue = readInt();
            byte bendVibrato = readByte();
            
            bend.addPoint((int)(bendPosition * BendEffect.MAX_POSITION_LENGTH / 60),(bendValue * 8 / 100));           
        }
        if(!bend.getPoints().isEmpty()){
            effect.setBend(bend);
        }        
    }

    private void readNoteEffects(NoteEffect noteEffect,Duration duration) throws IOException {
        int header1;
        int header2;
        int b;

        header1 = readUnsignedByte();
        header2 = readUnsignedByte();

        if ((header1 & 0x01) != 0) {
            readBend(noteEffect,duration);
        }

        if ((header1 & 0x10) != 0) {
            readGraceNote();
        }

        if ((header2 & 0x04) != 0) {
            readUnsignedByte();
        }

        if ((header2 & 0x08) != 0) {
            noteEffect.setSlide(true);
            readByte();            
        }

        if ((header2 & 0x10) != 0) {
        	readArtificialHarmonic();
        }

        if ((header2 & 0x20) != 0) {
            byte fret = readByte();
            byte period = readByte();
        }

        if ((header1 & 0x08) != 0) {
        }

        if ((header1 & 0x02) != 0) {
            noteEffect.setHammer(true);
        }

        if ((header2 & 0x40) != 0) {
            noteEffect.setVibrato(true);
        }

        if ((header2 & 0x02) != 0) {
        }

        if ((header2 & 0x01) != 0) {
        }
    }

    private void readBeatEffects(NoteEffect noteEffect,Duration currDuration) throws IOException {
        int header[] = { 0, 0 };

        header[0] = readUnsignedByte();
        header[1] = readUnsignedByte();

        if ((header[0] & 0x20) != 0) {
            int effect = readUnsignedByte();
            switch (effect) {
            case 0:
                break;
            case 1:
                break;
            case 2:
                break;
            case 3:
                break;
            default:
                //throw new IOException();
            }
        }

        if ((header[1] & 0x04) != 0) {
            readBend(noteEffect,currDuration);
        }

        if ((header[0] & 0x40) != 0) {
            int durationValue = readByte();
            durationValue = readByte();
        }

        if ((header[1] & 0x01) != 0) {
            //Rasgueado
        }

        if ((header[1] & 0x02) != 0) {
            //Pickstroke
            readByte();
        }
    }

    private void readMixChange(Tempo currentTempo) throws IOException { 
    	int instrument = readByte();
    	
    	skipBytes(16);
    	int volume = readByte();
    	int pan = readByte();
    	int chorus = readByte();
    	int reverb = readByte();
    	int phaser = readByte();
    	int tremolo = readByte();    
    	String tempoName = readStringInteger();    	
    	int tempo = readInt();
    	if(volume >= 0){
    		int type = readByte();	
    	}
    	if(pan >= 0){
    		int type = readByte();	
    	}
    	if(chorus >= 0){
    		int type = readByte();	
    	}
    	if(reverb >= 0){
    		int type = readByte();	
    	}
    	if(phaser >= 0){
    		int type = readByte();	
    	}
    	if(tremolo >= 0){
    		int type = readByte();	
    	}    	
    	if(tempo >= 0){
    		currentTempo.setValue(tempo);
    		int type = readByte();	
    	}        	
    	int applyToAllTracks = readByte(); 
    	skipBytes(1);    	
    }
    
    private void readArtificialHarmonic() throws IOException{    	
    	int read = readByte();
    	if(read == 2){
    		skipBytes(3);	
    	}else if(read == 3){
    		skipBytes(1);
    	}
    }
    
    private short toChannelShort(byte b){
    	short s = (short)b;
    	s = (short)((s * (short)127) / (short)16);
    	return (s <= 127)?s:127;
    }

    private void skipBytes(int count) throws IOException{
    	for(int i = 0;i < count;i ++){
    		readByte();
    	}
    }
    

    private void skipBytes(int count,boolean debug) throws IOException{
    	System.out.println("---------");
    	for(int i = 0;i < count;i ++){
    		int b = readByte();
    		System.out.println(b);
    	}
    	System.out.println("---------");
    }
}
class JoinVoicesHelper{
	private Measure measure;
	
	public JoinVoicesHelper(Measure measure){
		this.measure = measure;
	}
	
	public void process(){
		if(!measure.getNotes().isEmpty()){
			orderNotes();
		
			long start = ((Note)measure.getNotes().get(0)).getStart();
			long measureStart = measure.getStart();
			long measureLength = measure.getLength();
			while(start < (measureStart + measureLength)){
				List notesAtBeat = getNotesAtBeat(start);
				long maxLength = getMaxLength(start);
				normalizeNotes(notesAtBeat,maxLength);
				start += maxLength;
			}
		}
	}
	
	private void normalizeNotes(List notes,long maxLength){
		Duration beatDuration = null;
		
		Iterator it = notes.iterator();
		while(it.hasNext()){
			Note note = (Note)it.next();
			long noteDuration = note.getDuration().getTime();			
			if(noteDuration <= maxLength && (beatDuration == null || noteDuration > beatDuration.getTime())){
				beatDuration = note.getDuration();
			}
		}		
		if(beatDuration == null){
			beatDuration = Duration.fromTime(maxLength);
		}
		if(beatDuration != null){			
			it = notes.iterator();
			while(it.hasNext()){
				Note note = (Note)it.next();
				note.setDuration((Duration)beatDuration.clone());
			}
		}
	}
	
	private List getNotesAtBeat(long start){
		List notes = new ArrayList();
		Iterator it = measure.getNotes().iterator();
		while(it.hasNext()){
			Note note = (Note)it.next();
			if(note.getStart() == start){
				notes.add(note);
			}
		}
		return notes;
	}
		
	private long getMaxLength(long start){
		long nextStart = -1;
		Iterator it = measure.getNotes().iterator();
		while(it.hasNext()){
			Note note = (Note)it.next();
			if(note.getStart() > start && (nextStart < 0 || note.getStart() < nextStart)){
				nextStart = note.getStart();
			}
		}
		if(nextStart < 0){
			nextStart = (measure.getStart() + measure.getLength());
		}
		return (nextStart - start);
	}
	
    private void orderNotes(){
    	int noteCount = measure.getNotes().size();
        for(int i = 0;i < noteCount;i++){
            Note minNote = null;
            for(int noteIdx = i;noteIdx < noteCount;noteIdx++){
                Note note = (Note)measure.getNotes().get(noteIdx);
                if(minNote == null){
                    minNote = note;
                }else if(note.getStart() < minNote.getStart()){
                	minNote = note;
                }else if(note.getStart() == minNote.getStart() && note.getString() < minNote.getString()){
                	minNote = note;
                }
            }
            measure.getNotes().remove(minNote);
            measure.getNotes().add(i,minNote);
        }
    }
}
